/* grepmap
 * arch-tag: 0f26769d-a614-4bbf-bc70-a70a9905c1ee
 *
 * usbmap.c - parse modules.usbmap
 *
 * Copyright © 2004 Canonical Ltd.
 * Author: Scott James Remnant <scott@canonical.com>.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif /* HAVE_CONFIG_H */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "grepmap.h"


/* Flags specifying what to match */
typedef enum {
	USB_MATCH_VENDOR        = (1 << 0),
	USB_MATCH_PRODUCT       = (1 << 1),
	USB_MATCH_DEV_LO        = (1 << 2),
	USB_MATCH_DEV_HI        = (1 << 3),
	USB_MATCH_DEV_CLASS     = (1 << 4),
	USB_MATCH_DEV_SUBCLASS  = (1 << 5),
	USB_MATCH_DEV_PROTOCOL  = (1 << 6),
	USB_MATCH_INT_CLASS     = (1 << 7),
	USB_MATCH_INT_SUBCLASS  = (1 << 8),
	USB_MATCH_INT_PROTOCOL  = (1 << 9)
} UsbMatchFlags;

/* USB Device structure */
typedef struct {
	UsbMatchFlags match_flags;
	unsigned int  vendor, product;
	unsigned int  dev, dev_lo, dev_hi;
	unsigned int  dev_class, dev_subclass, dev_protocol;
	unsigned int  int_class, int_subclass, int_protocol;
} UsbDevice;


/**
 * usbmap_modules:
 * @mapf: open map file to read from.
 * @file: filename or map file, or '-' for standard input.
 * @args: arguments (vendor, product, dev,
 *                   dev_class, dev_subclass, dev_protocol,
 *                   int_class, int_subclass, int_protocol)
 *
 * Handles the modules.usbmap file looking for a matching entry for
 * the device given, printing the module name to stdout if found.
 *
 * Returns: 0 on success, non-zero on failure.
 **/
int
usbmap_modules (FILE       *mapf,
		const char *file,
		char       *argv[])
{
	UsbDevice  dev;
	char      *line;
	int        lineno = 0, ret = 0;

	ret |= parse_hex (argv[0], &dev.vendor);
	ret |= parse_hex (argv[1], &dev.product);
	ret |= parse_hex (argv[2], &dev.dev);
	ret |= parse_hex (argv[3], &dev.dev_class);
	ret |= parse_hex (argv[4], &dev.dev_subclass);
	ret |= parse_hex (argv[5], &dev.dev_protocol);
	ret |= parse_hex (argv[6], &dev.int_class);
	ret |= parse_hex (argv[7], &dev.int_subclass);
	ret |= parse_hex (argv[8], &dev.int_protocol);
	if (ret) {
		fprintf (stderr, "%s: %s\n", program_name,
			 _("arguments must be in hexadecimal format"));
		suggest_help ();
		return 1;
	}

	ret = 1;
	while ((line = fgets_alloc (mapf)) != NULL) {
		UsbDevice map;
		int       s;

		++lineno;
		if ((line[0] == '#') || (line[0] == '\0'))
			continue;

		s = sscanf (line, "%*s 0x%x 0x%x 0x%x 0x%x 0x%x 0x%x 0x%x 0x%x 0x%x 0x%x 0x%x",
			    (unsigned int *)&map.match_flags,
			    &map.vendor, &map.product,
			    &map.dev_lo, &map.dev_hi,
			    &map.dev_class, &map.dev_subclass,
			    &map.dev_protocol,
			    &map.int_class, &map.int_subclass,
			    &map.int_protocol);
		if (s < 11) {
			fprintf (stderr, "%s:%s:%d: %s\n", program_name, file,
				 lineno, _("unrecognised line format"));
			continue;
		}

		if (FLAG_SET(map.match_flags, USB_MATCH_VENDOR)
		    && (map.vendor != dev.vendor))
			continue;

		if (FLAG_SET(map.match_flags, USB_MATCH_PRODUCT)
		    && (map.product != dev.product))
			continue;

		if (FLAG_SET(map.match_flags, USB_MATCH_DEV_LO)
		    && (map.dev_lo > dev.dev))
			continue;

		if (FLAG_SET(map.match_flags, USB_MATCH_DEV_HI)
		    && (map.dev_hi < dev.dev))
			continue;

		if (FLAG_SET(map.match_flags, USB_MATCH_DEV_CLASS)
		    && (map.dev_class != dev.dev_class))
			continue;

		if (FLAG_SET(map.match_flags, USB_MATCH_DEV_SUBCLASS)
		    && (map.dev_subclass != dev.dev_subclass))
			continue;

		if (FLAG_SET(map.match_flags, USB_MATCH_DEV_PROTOCOL)
		    && (map.dev_protocol != dev.dev_protocol))
			continue;

		if (FLAG_SET(map.match_flags, USB_MATCH_INT_CLASS)
		    && (map.int_class != dev.int_class))
			continue;

		if (FLAG_SET(map.match_flags, USB_MATCH_INT_SUBCLASS)
		    && (map.int_subclass != dev.int_subclass))
			continue;

		if (FLAG_SET(map.match_flags, USB_MATCH_INT_PROTOCOL)
		    && (map.int_protocol != dev.int_protocol))
			continue;

		line[strcspn (line, " \t\r\n")] = '\0';
		printf ("%s\n", line);
		ret = 0;
	}

	return ret;
}
