/* grepmap
 *
 * inputmap.c - parse modules.inputmap
 *
 * Copyright © 2005 Canonical Ltd.
 * Author: Scott James Remnant <scott@ubuntu.com>.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif /* HAVE_CONFIG_H */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "grepmap.h"


/* Flags specifying what to match */
typedef enum {
	INPUT_MATCH_BUS     = (1 << 0),
	INPUT_MATCH_VENDOR  = (1 << 1),
	INPUT_MATCH_PRODUCT = (1 << 2),
	INPUT_MATCH_VERSION = (1 << 3),
	INPUT_MATCH_EVBIT   = (1 << 4),
	INPUT_MATCH_KEYBIT  = (1 << 5),
	INPUT_MATCH_RELBIT  = (1 << 6),
	INPUT_MATCH_ABSBIT  = (1 << 7),
	INPUT_MATCH_MSCBIT  = (1 << 8),
	INPUT_MATCH_LEDBIT  = (1 << 9),
	INPUT_MATCH_SNDBIT  = (1 << 10),
	INPUT_MATCH_FFBIT   = (1 << 11)
} InputMatchFlags;

/* Length of arrays, as these have different types on 64-bit platforms,
 * they also need to be different lengths.
 */
#if LONG_ARRAYS
enum {
	EVBIT_LEN  = 1,
	KEYBIT_LEN = 8,
	RELBIT_LEN = 1,
	ABSBIT_LEN = 1,
	MSCBIT_LEN = 1,
	LEDBIT_LEN = 1,
	SNDBIT_LEN = 1,
	FFBIT_LEN  = 2
};
#else /* LONG_ARRAYS */
enum {
	EVBIT_LEN  = 1,
	KEYBIT_LEN = 16,
	RELBIT_LEN = 1,
	ABSBIT_LEN = 2,
	MSCBIT_LEN = 1,
	LEDBIT_LEN = 1,
	SNDBIT_LEN = 1,
	FFBIT_LEN  = 4
};
#endif /* LONG_ARRAYS */

/* Input Device structure */
typedef struct {
	InputMatchFlags match_flags;
	unsigned int    bus, vendor, product, version;
	BitArray        evbit[EVBIT_LEN], keybit[KEYBIT_LEN];
	BitArray        relbit[RELBIT_LEN], absbit[ABSBIT_LEN];
	BitArray        mscbit[MSCBIT_LEN], ledbit[LEDBIT_LEN];
	BitArray        sndbit[SNDBIT_LEN], ffbit[FFBIT_LEN];
} InputDevice;


/**
 * inputmap_modules:
 * @mapf: open map file to read from.
 * @file: filename or map file, or '-' for standard input.
 * @args: arguments (bus, vendor, product, version,
 *                   evbit, keybit, relbit, absbit,
 *                   mscbit, ledbit, sndbit, ffbit)
 *
 * Handles the modules.inputmap file looking for a matching entry for
 * the device given, printing the module name to stdout if found.
 *
 * Returns: 0 on success, non-zero on failure.
 **/
int
inputmap_modules (FILE       *mapf,
		  const char *file,
		  char       *argv[])
{
	InputDevice  dev;
	char      *line;
	int        lineno = 0, ret = 0;

	ret |= parse_hex (argv[0], &dev.bus);
	ret |= parse_hex (argv[1], &dev.vendor);
	ret |= parse_hex (argv[2], &dev.product);
	ret |= parse_hex (argv[3], &dev.version);
	ret |= parse_array (argv[4], dev.evbit, EVBIT_LEN);
	ret |= parse_array (argv[5], dev.keybit, KEYBIT_LEN);
	ret |= parse_array (argv[6], dev.relbit, RELBIT_LEN);
	ret |= parse_array (argv[7], dev.absbit, ABSBIT_LEN);
	ret |= parse_array (argv[8], dev.mscbit, MSCBIT_LEN);
	ret |= parse_array (argv[9], dev.ledbit, LEDBIT_LEN);
	ret |= parse_array (argv[10], dev.sndbit, SNDBIT_LEN);
	ret |= parse_array (argv[11], dev.ffbit, FFBIT_LEN);
	if (ret) {
		fprintf (stderr, "%s: %s\n", program_name,
			 _("arguments must be in hexadecimal format"));
		suggest_help ();
		return 2;
	}

	ret = 1;
	while ((line = fgets_alloc (mapf)) != NULL) {
		InputDevice map;

		++lineno;
		if ((line[0] == '#') || (line[0] == '\0'))
			continue;

		if (parse_line (line, "xxxxxaaaaaaaa",
				(unsigned int *)&map.match_flags, &map.bus,
				&map.vendor, &map.product, &map.version,
				map.evbit, EVBIT_LEN, map.keybit, KEYBIT_LEN,
				map.relbit, RELBIT_LEN, map.absbit, ABSBIT_LEN,
				map.mscbit, MSCBIT_LEN, map.ledbit, LEDBIT_LEN,
				map.sndbit, SNDBIT_LEN, map.ffbit, FFBIT_LEN))
		{
			fprintf (stderr, "%s:%s:%d: %s\n", program_name, file,
				 lineno, _("unrecognised line format"));
			continue;
		}

		if (FLAG_SET(map.match_flags, INPUT_MATCH_BUS)
		    && (map.bus != dev.bus))
			continue;

		if (FLAG_SET(map.match_flags, INPUT_MATCH_VENDOR)
		    && (map.vendor != dev.vendor))
			continue;

		if (FLAG_SET(map.match_flags, INPUT_MATCH_PRODUCT)
		    && (map.product != dev.product))
			continue;

		/* NOTE: this isn't what the kernel does, but is what hotplug
		 * does.  I'm being bug-compatible with the latter.
		 *
		 * It's also noteworthy that hotplug itself suggests it's
		 * going to do a < comparison, but then does this.  *shrug*
		 */
		if (FLAG_SET(map.match_flags, INPUT_MATCH_VERSION)
		    && (map.version >= dev.version))
			continue;

		if (FLAG_SET(map.match_flags, INPUT_MATCH_EVBIT)
		    && match_array (map.evbit, dev.evbit, EVBIT_LEN))
			continue;

		if (FLAG_SET(map.match_flags, INPUT_MATCH_KEYBIT)
		    && match_array (map.keybit, dev.keybit, KEYBIT_LEN))
			continue;

		if (FLAG_SET(map.match_flags, INPUT_MATCH_RELBIT)
		    && match_array (map.relbit, dev.relbit, RELBIT_LEN))
			continue;

		if (FLAG_SET(map.match_flags, INPUT_MATCH_ABSBIT)
		    && match_array (map.absbit, dev.absbit, ABSBIT_LEN))
			continue;

		if (FLAG_SET(map.match_flags, INPUT_MATCH_MSCBIT)
		    && match_array (map.mscbit, dev.mscbit, MSCBIT_LEN))
			continue;

		if (FLAG_SET(map.match_flags, INPUT_MATCH_LEDBIT)
		    && match_array (map.ledbit, dev.ledbit, LEDBIT_LEN))
			continue;

		if (FLAG_SET(map.match_flags, INPUT_MATCH_SNDBIT)
		    && match_array (map.sndbit, dev.sndbit, SNDBIT_LEN))
			continue;

		if (FLAG_SET(map.match_flags, INPUT_MATCH_FFBIT)
		    && match_array (map.ffbit, dev.ffbit, FFBIT_LEN))
			continue;

		line[strcspn (line, " \t\r\n")] = '\0';
		printf ("%s\n", line);
		ret = 0;
	}

	return ret;
}
