/*
 * Copyright (C) 2011 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3, as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <njpatel@gmail.com>
 */

public class StreamEntry : Gtk.Alignment
{
  private Gtk.VBox _vbox;
  private Gtk.Alignment _align;
  public GwibberGtk.InputTextView text_view;
  private Gtk.Label _spacing;
  public Gtk.Image private;

  private Gwibber.Service _service;

  private uint _anim_id = 0;
  private float _anim_offset = 0.0f;
  private int64 _anim_start = 0;

  public GwibberGtk.AccountTargetBar target_bar;

  private bool _showing = false;
  public bool showing {
    get {
      return _showing;
    }
    set {
      _showing = value;
      if (_showing)
        show ();
      else
      {
        private.hide ();
        target_bar.selected = null;
        text_view.hide ();
        _spacing.show ();
      }
    }
  }

  public StreamEntry ()
  {
    Object ();
  }

  construct
  {
    set_reallocate_redraws (true);

    _vbox = new Gtk.VBox (false, 0);
    add (_vbox);

    _align = new Gtk.Alignment (0.0f, 0.0f, 1.0f, 1.0f);
    _align.set_padding (18, 18, 18, 18);
    _vbox.pack_start (_align, false, false, 0);

    var box = new Gtk.VBox (false, 4);
    _align.add (box);
    _service = new Gwibber.Service ();

    var group = new Gtk.SizeGroup (Gtk.SizeGroupMode.VERTICAL);

    _spacing = new Gtk.Label (" ");
    box.pack_start (_spacing, false, true, 0);
    _spacing.set_no_show_all (true);
    _spacing.show ();
    group.add_widget (_spacing);
    
    text_view = new GwibberGtk.InputTextView (_service);
    text_view.set_wrap_mode (Gtk.WrapMode.WORD_CHAR);
    text_view.set_border_window_size (Gtk.TextWindowType.LEFT, 0);
    text_view.set_border_window_size (Gtk.TextWindowType.RIGHT, 0);
    text_view.set_border_window_size (Gtk.TextWindowType.TOP, 0);
    text_view.set_border_window_size (Gtk.TextWindowType.BOTTOM, 0);
    text_view.set_no_show_all (true);
    group.add_widget (text_view);
    box.pack_start (text_view, false, true, 0);
    text_view.key_press_event.connect ((t, e)=>
    {
      if (e.keyval == 65307) // Escape
      {
        showing = false;
        return true;
      }
      return false;
    });

    Gdk.RGBA color;
    color.red = 34/255.0f;
    color.green = 34/255.0f;
    color.blue = 34/255.0f;
    color.alpha = 1.0f;
    text_view.override_background_color (Gtk.StateFlags.NORMAL, color);

    color.red = 1.0f;
    color.green = 1.0f;
    color.blue = 1.0f;
    text_view.override_color (Gtk.StateFlags.NORMAL, color);
    text_view.fg_color = { 0, 65535, 65535, 65535 };

    color.alpha =0.5f;
    text_view.override_background_color (Gtk.StateFlags.SELECTED, color);

    color.red = 0.0f;
    color.green = 0.0f;
    color.blue = 0.0f;
    color.alpha = 1.0f;
    text_view.override_color (Gtk.StateFlags.SELECTED, color);
    
    var hbox = new Gtk.HBox (false, 12);
    box.pack_start (hbox, false, false, 0);

    private = new Gtk.Image.from_icon_name ("status_lock", Gtk.IconSize.MENU);
    private.set_no_show_all (true);

    hbox.pack_end (private, false, false, 0);

    target_bar = new GwibberGtk.AccountTargetBar ();
    hbox.pack_start (target_bar, true, true, 0);
    color.red = 1.0f;
    color.green = 1.0f;
    color.blue = 1.0f;
    color.alpha = 0.5f;
    target_bar.override_color (Gtk.StateFlags.NORMAL, color);
    target_bar.count.override_color (Gtk.StateFlags.NORMAL, color);
    
    target_bar.send.clicked.connect (()=>
    {
      text_view.submit ();
    });

    text_view.submit.connect (() =>
    {
      showing = false;
    });

    _align.size_allocate.connect_after (()=> { queue_draw (); });
    text_view.size_allocate.connect_after (()=> { queue_draw (); });
    box.size_allocate.connect_after (()=> { queue_draw (); });

    text_view.get_buffer ().changed.connect (() => 
    {
      target_bar.set_counter(text_view.get_buffer ().get_char_count ());
    });

    var eb = new Gtk.EventBox ();
    eb.set_visible_window (false);
    eb.set_above_child (false);
    _vbox.pack_end (eb, true, true, 0);
    eb.button_press_event.connect (()=> { showing = false; return true; });
    
    notify["showing"].connect (()=>
    {
      if (_anim_id != 0)
        Source.remove (_anim_id);

      _anim_start = G.get_monotonic_time ();

        _anim_id = Timeout.add (15, ()=>
        {
          float LENGTH = 120000.0f;
          int64 diff = G.get_monotonic_time () - _anim_start;
          float progress = diff/LENGTH;

          queue_draw ();
          
          if (_showing)
          {
            _anim_offset = (_anim_offset + (1.0f-_anim_offset) * progress);
          }
          else
          {
            _anim_offset = _anim_offset - (_anim_offset * progress);
          }

          if (diff > LENGTH)
          {
            _anim_id = 0;
            _anim_offset = _showing ? 1.0f : 0.0f;
            text_view.visible = _showing;
            _spacing.visible = !_showing;
            set_visible (_showing);
            return false;
          }

          return true;
        });
    });
  }

  public override bool draw (Cairo.Context cr)
  {
    cr.push_group ();

    cr.save ();

    Gtk.Allocation a;
    _align.get_allocation (out a);

    cr.translate (0, (-1 * a.height) + (a.height * _anim_offset));

    a.height -= 12;

    var r = 4.0f; //radius
    cr.rectangle (a.x, a.y, a.width, a.height);

    var pat = new Cairo.Pattern.linear (0, 0, 0, _align.get_allocated_height ());
    pat.add_color_stop_rgba (0, 69/255.0, 69/255.0, 69/255.0, 0.85);
    pat.add_color_stop_rgba (1.0, 20/255.0, 20/255.0, 20/255.0, 0.85);
    cr.set_source (pat);
    cr.fill ();

    cr.set_source_rgba (0.0f, 0.0f, 0.0f, 0.3f);
    cr.rectangle (a.x, a.y, a.width, 1);
    cr.fill ();

    cr.set_source_rgba (1.0f, 1.0f, 1.0f, 0.2f);
    cr.rectangle (a.x, a.y + 1, a.width, 1);
    cr.fill ();

    cr.set_source_rgba (1.0f, 1.0f, 1.0f, 0.2f);
    cr.rectangle (a.x, a.y + a.height-2, a.width, 1);
    cr.fill ();

    cr.set_source_rgba (0.0f, 0.0f, 0.0f, 0.3f);
    cr.rectangle (a.x, a.y + a.height-1, a.width, 1);
    cr.fill ();

    // Shadow (we'll make it nicer later)
    pat = new Cairo.Pattern.linear (a.x, a.y + a.height, a.x, a.y + a.height + 12);
    pat.add_color_stop_rgba (0.0f, 0.0f, 0.0f, 0.0f, 0.4f);
    pat.add_color_stop_rgba (1.0f, 0.0f, 0.0f, 0.0f, 0.0f);
    cr.rectangle (a.x, a.y + a.height, a.width, 12);
    cr.set_source (pat);
    cr.fill ();

    // Text View
    if (text_view.visible)
      text_view.get_allocation (out a);
    else
      _spacing.get_allocation (out a);

    pat = new Cairo.Pattern.linear (a.x, a.y, a.x, a.y + a.height);
    pat.add_color_stop_rgba (0.99, 24/255.0f, 24/255.0f, 24/255.0f, 1.0);
    pat.add_color_stop_rgba (1.0, 90/255.0f, 90/255.0f, 90/255.0f, 0.9);
    cr.set_source (pat);

    cr.rectangle (a.x - 1,
                  a.y - 1,
                  a.width + 2,
                  a.height + 2);

    cr.fill ();

    cr.move_to (a.x -1, a.y-1);
    cr.line_to (a.x -1 + a.width + 2, a.y-1);
    cr.set_source_rgba (10/255.0f, 10/255.0f, 10/255.0f, 1.0f);
    cr.stroke ();

    propagate_draw (_vbox, cr);

    cr.restore ();

    cr.pop_group_to_source ();
    cr.paint_with_alpha (_anim_offset);
    return true;
  }
}
