/*
 * Copyright (C) 2010 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3, as published
 * by the Free Software Foundation.

 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Ken VanDine <ken.vandine@canonical.com>
 */

public class Client : Gtk.Window
{
  private Gwibber.Streams streams_service;
  private Gwibber.Service _service;
  private GLib.Settings state_settings;
  public Gee.HashMap <string, Dee.Model?> streams_map;
  public Gee.HashMap <string, TabBarItem?> items_map;
  public TabBar tabbar;
  public GwibberGtk.StreamView view;
  public Gtk.UIManager manager;
  private StatusBar _status_bar;
  private int _sort_order = 1;

  public int sort_order {
    get { return _sort_order; }
    set {
      if (value != _sort_order)
      {
        _sort_order = value;
        foreach (var v in items_map.values)
        {
          var _view = v.get_view () as GwibberGtk.StreamView;
          if (_view.sort_order != _sort_order)
            _view.sort_order = _sort_order;
        }
      }
    }
  }


  public Client ()
  {
    Object ();
  }

  construct
  {
    state_settings = new GLib.Settings ("org.gwibber.state");
    streams_map = new Gee.HashMap<string?, Dee.Model?> ();
    items_map = new Gee.HashMap<string?, TabBarItem?> ();

    var icon_theme = Gtk.IconTheme.get_default ();
    icon_theme.prepend_search_path (Config.PKGDATADIR + "/ui/icons");

    // Prepend local icon path if running from a source checkout
    var local_icon_path = GLib.Path.build_path (Path.DIR_SEPARATOR_S, Environment.get_current_dir (), "data/icons");
    if (GLib.FileUtils.test (local_icon_path, GLib.FileTest.IS_DIR))
    {
      icon_theme.prepend_search_path (local_icon_path);
    }

    configure_event.connect (on_configure_event);
    delete_event.connect (() => {
      this.hide ();
      Idle.add (() => { this.destroy (); return false; });
      return true;
    });

    set_name(Config.PACKAGE);
    set_icon_name("gwibber");
    set_title("Gwibber");
    set_wmclass("gwibber", "Gwibber");

    set_default_size(400,800);

    // Move to last known location and resize
    move(state_settings.get_int("position-x"), state_settings.get_int("position-y"));
    resize(state_settings.get_int("width"), state_settings.get_int("height"));

    // Tab Bar
    tabbar = new TabBar ();

    var main_box = new Gtk.Box (Gtk.Orientation.VERTICAL, 0);

    _status_bar = new StatusBar ();
    /* We don't need this yet
    _status_bar.add_button (Gtk.Stock.OK, Gtk.ResponseType.OK);
    _status_bar.response.connect((r) => {
      _status_bar.showing = false;
    });
    */
    main_box.pack_end (_status_bar, false, false, 0);
    _status_bar.notify["showing"].connect (()=>
    {
      debug ("StatusBar showing changed");
    });

    _service = new Gwibber.Service ();
    _service.loading_started.connect((source) => {
      debug ("Loading started");
      if (!_status_bar.showing)
      {
        _status_bar.reset ();
        _status_bar.set_message_type (Gtk.MessageType.OTHER);
        _status_bar.message = _("Refreshing");
        _status_bar.showing = true;
      }
    });
    _service.loading_complete.connect((source) => {
      debug ("Loading completed");
      if (_status_bar.showing)
        _status_bar.showing = false;
    });

    main_box.pack_end (tabbar, true, true, 0);
    add (main_box);

    streams_service = new Gwibber.Streams ();

    streams_map["home"] = streams_service.stream_filter_model (streams_service.stream_model, "home");

    var home_item = new HomeItem (streams_map);
    view = (GwibberGtk.StreamView)home_item.get_view ();

    tabbar.add_item (home_item);
    items_map["home"] = home_item;

    streams_map["messages"] = streams_service.stream_filter_model (streams_service.stream_model, "messages");
    streams_map["replies"] = streams_service.stream_filter_model (streams_service.stream_model, "replies");
    streams_map["private"] = streams_service.stream_filter_model (streams_service.stream_model, "private");
    streams_map["images"] = streams_service.stream_filter_model (streams_service.stream_model, "images");
    streams_map["videos"] = streams_service.stream_filter_model (streams_service.stream_model, "videos");
    streams_map["links"] = streams_service.stream_filter_model (streams_service.stream_model, "links");
    streams_map["public"] = streams_service.stream_filter_model (streams_service.stream_model, "public");

    var messages_item = new MessagesItem (streams_map);
    tabbar.add_item (messages_item);
    items_map["messages"] = messages_item;

    var replies_item = new RepliesItem (streams_map);
    tabbar.add_item (replies_item);
    items_map["replies"] = replies_item;

    var private_item = new PrivateItem (streams_map);
    tabbar.add_item (private_item);
    items_map["private"] = private_item;

    var public_item = new PublicItem (streams_map);
    tabbar.add_item (public_item);
    items_map["public"] = public_item;

    var attach_item = new AttachmentsItem (streams_map);
    tabbar.add_item (attach_item);
    items_map["attach"] = attach_item;

    var searches_item = new SearchesItem (streams_service);
    tabbar.add_item (searches_item as TabBarItem);
    items_map["searches"] = searches_item;

    searches_item.raise.connect ((item) => {
      searches_item.clicked ();
    });

    var users_item = new UsersItem (streams_service);
    tabbar.add_item (users_item as TabBarItem);
    items_map["users"] = users_item;

    users_item.raise.connect ((item) => {
      users_item.clicked ();
    });

    view.notify["prepared"].connect (() => {
        sort_order = state_settings.get_int("stream-sort-order");
      });

    state_settings.changed.connect ((key) => {
      if (key == "stream-sort-order") {
         if (sort_order != state_settings.get_int("stream-sort-order"))
           sort_order = state_settings.get_int("stream-sort-order");
      }
    });
    messages_item.clicked ();

    Idle.add (() => {
      var menubar = create_menus ();
      main_box.pack_start(menubar, false, true, 0);
      return false;
    });

  }

  public Gtk.MenuBar create_menus ()
  {
    /**** Create menus ****/

    Gtk.ActionEntry[] entries = new Gtk.ActionEntry[0];

    Gtk.ActionEntry menu = {"Gwibber", null, _("_Gwibber"), null, null, null};
    entries += menu;
    menu = {"View", null, _("_View"), null, null, null};
    entries += menu;
    menu = {"Edit", null, _("_Edit"), null, null, null};
    entries += menu;
    menu = {"Help", null, _("_Help"), null, null, null};
    entries += menu;
    menu = {"refresh", Gtk.Stock.REFRESH, _("_Refresh"), "F5", null, on_refresh};
    entries += menu;
    menu = {"quit", Gtk.Stock.QUIT, _("_Quit"), "<ctrl>Q", null, on_quit};
    entries += menu;
    menu = {"sort", null, _("_Sort"), null, null, null};
    entries += menu;
    menu = {"accounts", null, _("_Accounts"), "<ctrl><shift>A", null,on_accounts};
    entries += menu;
    menu = {"preferences", Gtk.Stock.PREFERENCES, _("_Preferences"), "<ctrl>P", null, on_preferences};
    entries += menu;
    menu = {"about", Gtk.Stock.ABOUT, _("_About"), null, null, on_about};
    entries += menu;

    var sort_entries = new Gtk.RadioActionEntry[0];
    Gtk.RadioActionEntry radio = {"ascending", Gtk.Stock.SORT_ASCENDING, _("_Ascending"), null, null, 0};
    sort_entries += radio;
    radio = {"descending", Gtk.Stock.SORT_DESCENDING, _("_Descending"), null, null, 1};
    sort_entries += radio;

    string ui = """
    <ui>
      <menubar name="MenuBar">
        <menu action="Gwibber">
          <menuitem action="refresh" />
          <separator/>
          <menuitem action="quit" />
        </menu>

        <menu action="View">
          <menu action="sort">
            <menuitem action="ascending" />
            <menuitem action="descending" />
          </menu>
        </menu>

        <menu action="Edit">
          <menuitem action="accounts" />
          <menuitem name="preferences" action="preferences" />
        </menu>

        <menu action="Help">
          <menuitem action="about" />
        </menu>
      </menubar>
    </ui>
    """;

    var main_group = new Gtk.ActionGroup("client");
    main_group.add_actions(entries, this);
    main_group.add_radio_actions(sort_entries, sort_order, ((a,c) => {
      Gtk.RadioAction current = (Gtk.RadioAction) c;
      state_settings.set_int("stream-sort-order", current.get_current_value ());
      }));

    var manager = new Gtk.UIManager();

    try {
        manager.add_ui_from_string(ui, -1);
    } catch (Error e) { error("%s", e.message); }

    manager.insert_action_group(main_group, 0);

    var menubar = manager.get_widget("/MenuBar") as Gtk.MenuBar;
    add_accel_group(manager.get_accel_group());

    /**** End menu setup ****/
    
    return menubar;
  }


  static void on_refresh() {
    var service = new Gwibber.Service();
    service.refresh();
  }

  static void on_preferences() {
    GLib.Pid pid;
    try {
    GLib.Process.spawn_async(null, {"gwibber-preferences"}, null,
      GLib.SpawnFlags.SEARCH_PATH, null, out pid);
    } catch {
    }
  }

  static void on_accounts() {
    GLib.Pid pid;
    try {
    GLib.Process.spawn_async(null, {"gwibber-accounts"}, null,
      GLib.SpawnFlags.SEARCH_PATH, null, out pid);
    } catch {

    }
  }

  
  static void on_about() {

    string license = _("Gwibber is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation; either version 2 of the License, or (at your option) any later version.\n\nGwibber is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more details.\n\nYou should have received a copy of the GNU General Public License along with Gwibber; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-130159 USA");

    string[] authors = {"Ken VanDine <ken@vandine.org>", "Neil Jagdish Patel <njpatel@gmail.com>"};
    var about_dialog = new Gtk.AboutDialog ();
    about_dialog.set_program_name("Gwibber");
    about_dialog.set_logo_icon_name("gwibber");
    about_dialog.set_version(Config.VERSION);
    about_dialog.set_website("http://www.gwibber.com");
    about_dialog.set_license(license);
    about_dialog.set_wrap_license(true);
    about_dialog.set_authors (authors);
    about_dialog.run ();
    about_dialog.destroy ();
  }

  private void on_quit ()
  {
    destroy ();
    var service = new Gwibber.Service();

    /* Check if the unity lens is running and only shutdown the 
    *  service if it isn't
    */
    bool has_owner = false;
    try {
      string name = "com.canonical.Unity.Lens.Gwibber";
      DBusConnection bus = Bus.get_sync (BusType.SESSION);
      Variant result = bus.call_sync ("org.freedesktop.DBus",
                                      "/org/freedesktop/dbus",
                                      "org.freedesktop.DBus",
                                      "NameHasOwner",
                                      new Variant ("(s)", name),
                                      new VariantType ("(b)"),
                                      DBusCallFlags.NO_AUTO_START,
                                      -1);
      result.get ("(b)", out has_owner);
    } catch (Error e) {
      warning ("Unable to decide if '%s' is running: %s", name, e.message);
    }

    if (!has_owner)
    {
      debug ("Lens isn't running");
      service.quit ();
    }

  }

  bool on_configure_event (Gdk.EventConfigure event) {
    int width, height, x, y;
    get_size(out width, out height);
    get_position (out x, out y);
    state_settings.set_int("width", width);
    state_settings.set_int("height", height);
    state_settings.set_int("position-x", x);
    state_settings.set_int("position-y", y);
    return false;
  }
}

public class Main : Gtk.Application
{
  static string stream;

  static Client main_window;

  const OptionEntry[] options = {
    {"stream", 's', 0, OptionArg.STRING, ref stream, N_("Stream"), N_("STREAM")},
    {null}
  };

  public Main (string app_id, ApplicationFlags flags)
  {
    GLib.Object (application_id: app_id, flags: flags);
  }

  public void on_activate ()
  {
    if (get_windows () != null)
      main_window.present ();
    else
    {
      main_window = new Client ();

      add_window (main_window);

      Environment.set_application_name (Config.PACKAGE);

      Gtk.IconTheme.get_default ().append_search_path (GLib.Path.build_filename (Config.PKGDATADIR, "ui/icons"));

      main_window.set_application (this);
      main_window.show_all ();

      var accounts_service = new Gwibber.Accounts();
      var accounts_list = accounts_service.list ();
      if (accounts_list.length() == 0)
      {
        GLib.Pid pid;
        try {
          GLib.Process.spawn_async(null, {"gwibber-accounts"}, null,
          GLib.SpawnFlags.SEARCH_PATH, null, out pid);
        } catch {
        }
      }
    }
  }

  public int on_command_line (ApplicationCommandLine command_line)
  {
    var args = command_line.get_arguments ();
    unowned string[] arguments = args;
    try {
      var context = new OptionContext (_("— Gwibber Client"));
      context.set_help_enabled (false);
      context.add_main_entries (options, Config.GETTEXT_PACKAGE);
      context.parse (ref arguments);
    }
    catch (OptionError error) {
      command_line.set_exit_status (1);
    }

    activate ();

    var items_map = main_window.items_map;
    if (stream != null)
    {
      if ((main_window.tabbar is TabBar) && (items_map[stream] is TabBarItem))
        main_window.tabbar.on_tab_clicked(items_map[stream]);
    }

    command_line.set_exit_status (0);
    return command_line.get_exit_status ();
  }
}

public int main (string[] args)
{
  Gtk.init (ref args);

  Intl.bindtextdomain(Config.GETTEXT_PACKAGE, Config.LOCALE_DIR);
  Intl.bind_textdomain_codeset(Config.GETTEXT_PACKAGE, "UTF-8");
  Intl.textdomain(Config.GETTEXT_PACKAGE);

  var app = new Main ("org.gwibber.client", GLib.ApplicationFlags.HANDLES_COMMAND_LINE);

  app.activate.connect (app.on_activate);
  app.command_line.connect (app.on_command_line);
  int result = app.run (args);
  return result;
}
