/*
 * Copyright(C) 2010 Canonical Ltd.
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License
 * version 3.0 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3.0 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library. If not, see
 * <http://www.gnu.org/licenses/>.
 *
 * Authored by Ken VaDine <ken.vandine@canonical.com>
 */

namespace Gwibber
{

    [DBus (name = "org.freedesktop.DBus")]
    internal interface DBusIface : GLib.Object {
        public signal void NameOwnerChanged(string name, string oldowner, string newowner);
        public abstract async string[] list_names() throws GLib.IOError;
    }

    public class Utils : GLib.Object
    {
	private DBusIface bus;
	private string interface_name;

        /** 
            Utils::available:
	    @arg0: bool
            Emitted when the DBus service becomes available
        */

	internal signal void available(bool up);

	private void on_name_owner_changed (string name, string old_owner, string new_owner)
	{
		if (name == interface_name)
		{
			if (new_owner != "" && old_owner == "")
			{
				debug ("%s is available", name);
				available (true);
			}
			if (old_owner != "" && new_owner == "")
			{
				available (false);
				debug ("%s is no longer available", name);
			}
		}
	}

	private async void check_names (string[] names)
	{
		foreach (string name in names) {
			if (name == interface_name) {
				available (true);
				break;
                	}
        	}
	}

	private async bool on_ready ()
	{
		try {
			var n = yield bus.list_names ();
			check_names(n);
		} catch (GLib.IOError e) {
			warning ("Can't list: %s", e.message);
		}
		return false;
	}

	public string generate_time_string(uint t)
        {
		time_t _t = time_t ();
		Time n = Time.gm (_t);
		Time lt = Time.local (_t);
		n.isdst = lt.isdst;
		int now = (int)n.mktime ();
		int d = (int)(now - t);
		if(d < 60)
			return _("a few seconds ago");
		if(d < 3600)
			return ngettext("%i minute ago", "%i minutes ago", (int) d/60).printf(d/60);
		if(d < 86400)
			return ngettext("%i hour ago", "%i hours ago", (int) d/3600).printf(d/3600);
                return ngettext("%i day ago", "%i days ago", (int) d/86400).printf(d/86400);
        }

        public string? avatar_path(string url)
        {
            string _avatar_cache_image = Path.build_path(Path.DIR_SEPARATOR_S, Environment.get_user_cache_dir(), "gwibber/avatars", GLib.Checksum.compute_for_string(GLib.ChecksumType.SHA1, url));
            var file = File.new_for_path (_avatar_cache_image);
            if (file.query_exists ())
              return _avatar_cache_image;
            else
              return null;
        }


	private async void do_setup (string name) throws GLib.IOError
	{
		bus = yield Bus.get_proxy<DBusIface>(BusType.SESSION,
		                                     "org.freedesktop.DBus",
		                                     "/org/freedesktop/DBus");
		bus.NameOwnerChanged.connect(on_name_owner_changed);
		
		yield on_ready();
	}

	internal void setup (string name) throws GLib.IOError
	{
		interface_name = name;
		/* we should use name the name argument to make this more dynamic */
		do_setup.begin(name);
	}
    }
}
