/*

  Broadcom BCM43xx wireless driver

  Copyright (c) 2005 Martin Langer <martin-langer@gmx.de>,
                     Stefano Brivio <st3@riseup.net>
                     Danny van Dyk <kugelfang@gentoo.org>
                     Andreas Jaggi <andreas.jaggi@waterwave.ch>
  Copyright (c) 2005, 2006 Michael Buesch <mbuesch@freenet.de>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; see the file COPYING.  If not, write to
  the Free Software Foundation, Inc., 51 Franklin Steet, Fifth Floor,
  Boston, MA 02110-1301, USA.

*/

#include "bcm43xx.h"
#include "bcm43xx_vstack.h"
#include "bcm43xx_phy.h"
#include "bcm43xx_tables.h"


void bcm43xx_vstack_free(struct bcm43xx_vstack *stack)
{
	if (!stack)
		return;
	kfree(stack->items);
	stack->items = NULL;
	stack->nr_items = 0;
	stack->top = 0;
}

void bcm43xx_vstack_cleanup(struct bcm43xx_vstack *stack)
{
	stack->top = 0;
}

int bcm43xx_vstack_alloc(struct bcm43xx_wldev *dev,
			 struct bcm43xx_vstack *stack,
			 u8 size,
			 gfp_t gfp_flags)
{
	stack->items = kcalloc(size,
			       sizeof(struct bcm43xx_vstack_item),
			       gfp_flags);
	if (!stack->items)
		return -ENOMEM;
	assert(size <= 254);
	stack->nr_items = size;
	stack->dev = dev;
	bcm43xx_vstack_cleanup(stack);

	return 0;
}

void bcm43xx_vstack_save(struct bcm43xx_vstack *stack,
			 enum bcm43xx_vstack_type type,
			 u16 offset, u16 value)
{
	struct bcm43xx_vstack_item *top;

	if (BCM43xx_DEBUG) {
		if (unlikely((offset & 0xC000) != 0)) {
			printk(KERN_ERR PFX "bcm43xx_vstack_save() upper two bits "
					    "in \"offset\" set. They are used for "
					    "type encoding and must be free.\n");
			dump_stack();
			return;
		}
		assert((type & ~0x3UL) == 0);
		assert(stack->nr_items <= 254);
		if (unlikely(stack->top >= stack->nr_items)) {
			printk(KERN_ERR PFX "bcm43xx_vstack_save() overflow\n");
			dump_stack();
			return;
		}
	}

	top = &(stack->items[stack->top]);
	top->value = value;
	top->offset_type = offset;
	top->offset_type |= (type << 14);
	stack->top++;
}

u16 bcm43xx_vstack_restore(struct bcm43xx_vstack *stack,
			   enum bcm43xx_vstack_type type,
			   u16 offset)
{
	struct bcm43xx_vstack_item *item;
	int i;

	assert(stack->top <= stack->nr_items);
	item = &(stack->items[0]);
	for (i = 0; i < stack->top + 1; i++, item++) {
		if ((item->offset_type & 0x3FFF) != offset)
			continue;
		if (((item->offset_type & 0xC000) >> 14) != type)
			continue;
		return item->value;
	}

	if (BCM43xx_DEBUG) {
		printk(KERN_ERR PFX "bcm43xx_vstack_restore() value "
				    "(%d, 0x%04X) not found\n",
		       type, offset);
		dump_stack();
	}

	return 0;
}

void bcm43xx_phy_stacksave(struct bcm43xx_vstack *stack,
			   u16 offset)
{
	u16 v;

	v = bcm43xx_phy_read(stack->dev, offset);
	bcm43xx_vstack_save(stack, BCM43xx_VSTACK_PHY,
			    offset, v);
}

void bcm43xx_phy_stackrestore(struct bcm43xx_vstack *stack,
			      u16 offset)
{
	u16 v;

	v = bcm43xx_vstack_restore(stack, BCM43xx_VSTACK_PHY,
				   offset);
	bcm43xx_phy_write(stack->dev, offset, v);
}

void bcm43xx_radio_stacksave(struct bcm43xx_vstack *stack,
			     u16 offset)
{
	u16 v;

	v = bcm43xx_radio_read16(stack->dev, offset);
	bcm43xx_vstack_save(stack, BCM43xx_VSTACK_RADIO,
			    offset, v);
}

void bcm43xx_radio_stackrestore(struct bcm43xx_vstack *stack,
				u16 offset)
{
	u16 v;

	v = bcm43xx_vstack_restore(stack, BCM43xx_VSTACK_RADIO,
				   offset);
	bcm43xx_radio_write16(stack->dev, offset, v);
}

void bcm43xx_ilt_stacksave(struct bcm43xx_vstack *stack,
			   u16 offset)
{
	u16 v;

	v = bcm43xx_ofdmtab_read16(stack->dev, offset, 0);
	bcm43xx_vstack_save(stack, BCM43xx_VSTACK_ILT,
			    offset, v);
}

void bcm43xx_ilt_stackrestore(struct bcm43xx_vstack *stack,
			      u16 offset)
{
	u16 v;

	v = bcm43xx_vstack_restore(stack, BCM43xx_VSTACK_ILT,
				   offset);
	bcm43xx_ofdmtab_write16(stack->dev, offset, 0, v);
}

void bcm43xx_mmio_stacksave(struct bcm43xx_vstack *stack,
			    u16 offset)
{
	u16 v;

	v = bcm43xx_read16(stack->dev, offset);
	bcm43xx_vstack_save(stack, BCM43xx_VSTACK_MMIO,
			    offset, v);
}

void bcm43xx_mmio_stackrestore(struct bcm43xx_vstack *stack,
			       u16 offset)
{
	u16 v;

	v = bcm43xx_vstack_restore(stack, BCM43xx_VSTACK_MMIO,
				   offset);
	bcm43xx_write16(stack->dev, offset, v);
}
