#
# Copyright 2009 Canonical Ltd.
#
# Written by:
#     Gustavo Niemeyer <gustavo.niemeyer@canonical.com>
#     Sidnei da Silva <sidnei.da.silva@canonical.com>
#
# This file is part of the Image Store Proxy.
#
# This program is free software: you can redistribute it and/or modify it 
# under the terms of the GNU General Public License version 3, as published 
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but 
# WITHOUT ANY WARRANTY; without even the implied warranties of 
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR 
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along 
# with this program.  If not, see <http://www.gnu.org/licenses/>.
#
from weakref import ref
import traceback
import thread


class Event(object):
    """Base class for all events."""


class EventBus(object):

    def __init__(self):
        self._handlers = {}
        self._handlersLock = thread.allocate_lock()

    def addHandler(self, event_type, handler, *args, **kwargs):
        # setdefault() is inherently thread-safe in CPython, and this
        # is the only place we change it, so we don't have to lock it.
        assert callable(handler), \
               ("%r is not callable, thus not a handler" % handler)
        handlers = self._handlers.setdefault(event_type, [])

        self._handlersLock.acquire()
        try:
            # If this fails, things are pretty bad. :-)
            handlers.append((ref(handler), args, kwargs))
        finally:
            self._handlersLock.release()

    def fire(self, event):
        # No need to lock, since we're not changing anything here, and
        # removal of elements from handlers is dealt with copy-on-write
        # on removal, and appending on inclusion, which makes it
        # no-lock-iteration-friendly.
        handlers = self._handlers.get(type(event), ())
        clearDeadRefs = False
        for handlerRef, args, kwargs in handlers:
            handler = handlerRef()
            if handler is not None:
                try:
                    handler(event, *args, **kwargs)
                except Exception, e:
                    traceback.print_exc(e)
            else:
                clearDeadRefs = True

        if clearDeadRefs:
            self._handlersLock.acquire()
            try:
                # Must retrieve the current list again since another thread
                # might have replaced the real list while we were iterating
                # above, and then appended another handler which wasn't seen.
                handlers = self._handlers.get(type(event), ())

                # Then replace by a *new* list, otherwise we'd screw up
                # another thread going over the loop above.
                self._handlers[type(event)] = [triple for triple in handlers
                                               if triple[0]() is not None]
            finally:
                self._handlersLock.release()
