/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%            DDDD   EEEEE   CCCC   OOO   RRRR    AAA   TTTTT  EEEEE           %
%            D   D  E      C      O   O  R   R  A   A    T    E               %
%            D   D  EEE    C      O   O  RRRR   AAAAA    T    EEE             %
%            D   D  E      C      O   O  R R    A   A    T    E               %
%            DDDD   EEEEE   CCCC   OOO   R  R   A   A    T    EEEEE           %
%                                                                             %
%                                                                             %
%                     ImageMagick Image Decoration Methods                    %
%                                                                             %
%                                Software Design                              %
%                                  John Cristy                                %
%                                   July 1992                                 %
%                                                                             %
%                                                                             %
%  Copyright 1999-2004 ImageMagick Studio LLC, a non-profit organization      %
%  dedicated to making software imaging solutions freely available.           %
%                                                                             %
%  You may not use this file except in compliance with the License.  You may  %
%  obtain a copy of the License at                                            %
%                                                                             %
%    http://www.imagemagick.org/www/Copyright.html                            %
%                                                                             %
%  Unless required by applicable law or agreed to in writing, software        %
%  distributed under the License is distributed on an "AS IS" BASIS,          %
%  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.   %
%  See the License for the specific language governing permissions and        %
%  limitations under the License.                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%
%
*/

/*
  Include declarations.
*/
#include "magick/studio.h"
#include "magick/decorate.h"
#include "magick/error.h"
#include "magick/image.h"
#include "magick/memory_.h"
#include "magick/monitor.h"

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   B o r d e r I m a g e                                                     %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  BorderImage() surrounds the image with a border of the color defined by
%  the bordercolor member of the image structure.  The width and height
%  of the border are defined by the corresponding members of the border_info
%  structure.
%
%  The format of the BorderImage method is:
%
%      Image *BorderImage(const Image *image,const RectangleInfo *border_info,
%        ExceptionInfo *exception)
%
%  A description of each parameter follows:
%
%    o image: The image.
%
%    o border_info:  Define the width and height of the border.
%
%    o exception: Return any errors or warnings in this structure.
%
%
*/
MagickExport Image *BorderImage(const Image *image,
  const RectangleInfo *border_info,ExceptionInfo *exception)
{
  Image
    *border_image,
    *clone_image;

  FrameInfo
    frame_info;

  assert(image != (const Image *) NULL);
  assert(image->signature == MagickSignature);
  if (image->debug != MagickFalse)
    (void) LogMagickEvent(TraceEvent,GetMagickModule(),image->filename);
  assert(border_info != (RectangleInfo *) NULL);
  frame_info.width=image->columns+(border_info->width << 1);
  frame_info.height=image->rows+(border_info->height << 1);
  frame_info.x=(long) border_info->width;
  frame_info.y=(long) border_info->height;
  frame_info.inner_bevel=0;
  frame_info.outer_bevel=0;
  clone_image=CloneImage(image,0,0,MagickTrue,exception);
  if (clone_image == (Image *) NULL)
    return((Image *) NULL);
  clone_image->matte_color=image->border_color;
  border_image=FrameImage(clone_image,&frame_info,exception);
  DestroyImage(clone_image);
  if (border_image != (Image *) NULL)
    border_image->matte_color=image->matte_color;
  return(border_image);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   F r a m e I m a g e                                                       %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  FrameImage() adds a simulated three-dimensional border around the image.
%  The color of the border is defined by the matte_color member of image.
%  Members width and height of frame_info specify the border width of the
%  vertical and horizontal sides of the frame.  Members inner and outer
%  indicate the width of the inner and outer shadows of the frame.
%
%  The format of the FrameImage method is:
%
%      Image *FrameImage(const Image *image,const FrameInfo *frame_info,
%        ExceptionInfo *exception)
%
%  A description of each parameter follows:
%
%    o image: The image.
%
%    o frame_info: Define the width and height of the frame and its bevels.
%
%    o exception: Return any errors or warnings in this structure.
%
%
*/
MagickExport Image *FrameImage(const Image *image,const FrameInfo *frame_info,
  ExceptionInfo *exception)
{
#define FrameImageTag  "Frame/Image"

  Image
    *frame_image;

  long
    y;

  register const PixelPacket
    *p;

  register long
    x;

  register PixelPacket
    *q;

  PixelPacket
    accentuate,
    highlight,
    matte,
    shadow,
    trough;

  unsigned long
    bevel_width,
    height,
    width;

  /*
    Check frame geometry.
  */
  assert(image != (Image *) NULL);
  assert(image->signature == MagickSignature);
  if (image->debug != MagickFalse)
    (void) LogMagickEvent(TraceEvent,GetMagickModule(),image->filename);
  assert(frame_info != (FrameInfo *) NULL);
  bevel_width=(unsigned long) (frame_info->outer_bevel+frame_info->inner_bevel);
  width=frame_info->width-frame_info->x-bevel_width;
  height=frame_info->height-frame_info->y-bevel_width;
  if ((width < image->columns) || (height < image->rows))
    ThrowImageException(OptionError,"FrameIsLessThanImageSize");
  /*
    Initialize framed image attributes.
  */
  frame_image=CloneImage(image,frame_info->width,frame_info->height,MagickTrue,
    exception);
  if (frame_image == (Image *) NULL)
    return(MagickFalse);
  frame_image->storage_class=DirectClass;
  frame_image->matte|=(frame_image->matte_color.opacity != OpaqueOpacity);
  /*
    Initialize 3D effects color.
  */
  matte=image->matte_color;
  accentuate.red=(Quantum) (QuantumScale*(((MagickRealType) MaxRGB-
    AccentuateModulate)*matte.red+((MagickRealType) MaxRGB*
    AccentuateModulate))+0.5);
  accentuate.green=(Quantum) (QuantumScale*(((MagickRealType) MaxRGB-
    AccentuateModulate)*matte.green+((MagickRealType) MaxRGB*
    AccentuateModulate))+0.5);
  accentuate.blue=(Quantum) (QuantumScale*(((MagickRealType) MaxRGB-
    AccentuateModulate)*matte.blue+((MagickRealType) MaxRGB*
    AccentuateModulate))+0.5);
  accentuate.opacity=matte.opacity;
  highlight.red=(Quantum) (QuantumScale*(((MagickRealType) MaxRGB-
    HighlightModulate)*matte.red+((MagickRealType) MaxRGB*
    HighlightModulate))+0.5);
  highlight.green=(Quantum) (QuantumScale*(((MagickRealType) MaxRGB-
    HighlightModulate)*matte.green+((MagickRealType) MaxRGB*
    HighlightModulate))+0.5);
  highlight.blue=(Quantum) (QuantumScale*(((MagickRealType) MaxRGB-
    HighlightModulate)*matte.blue+((MagickRealType) MaxRGB*
    HighlightModulate))+0.5);
  highlight.opacity=matte.opacity;
  shadow.red=(Quantum) (QuantumScale*((MagickRealType) matte.red*
    ShadowModulate)+0.5);
  shadow.green=(Quantum) (QuantumScale*((MagickRealType) matte.green*
    ShadowModulate)+0.5);
  shadow.blue=(Quantum) (QuantumScale*((MagickRealType) matte.blue*
    ShadowModulate)+0.5);
  shadow.opacity=matte.opacity;
  trough.red=(Quantum) (QuantumScale*((MagickRealType) matte.red*
    TroughModulate)+0.5);
  trough.green=(Quantum) (QuantumScale*((MagickRealType) matte.green*
    TroughModulate)+0.5);
  trough.blue=(Quantum) (QuantumScale*((MagickRealType) matte.blue*
    TroughModulate)+0.5);
  trough.opacity=matte.opacity;
  /*
    Draw top of ornamental border.
  */
  height=(unsigned long) (frame_info->outer_bevel+(frame_info->y-bevel_width)+
    frame_info->inner_bevel);
  q=SetImagePixels(frame_image,0,0,frame_image->columns,height);
  if (q != (PixelPacket *) NULL)
    {
      for (y=0; y < (long) frame_info->outer_bevel; y++)
      {
        for (x=0; x < (long) (frame_image->columns-y); x++)
          if (x < y)
            *q++=highlight;
          else
            *q++=accentuate;
        for ( ; x < (long) frame_image->columns; x++)
          *q++=shadow;
      }
      for (y=0; y < (long) (frame_info->y-bevel_width); y++)
      {
        for (x=0; x < (long) frame_info->outer_bevel; x++)
          *q++=highlight;
        width=frame_image->columns-2*frame_info->outer_bevel;
        for (x=0; x < (long) width; x++)
          *q++=matte;
        for (x=0; x < (long) frame_info->outer_bevel; x++)
          *q++=shadow;
      }
      for (y=0; y < (long) frame_info->inner_bevel; y++)
      {
        for (x=0; x < (long) frame_info->outer_bevel; x++)
          *q++=highlight;
        for (x=0; x < (long) (frame_info->x-bevel_width); x++)
          *q++=matte;
        width=image->columns+((unsigned long) frame_info->inner_bevel << 1)-y;
        for (x=0; x < (long) width; x++)
          if (x < y)
            *q++=shadow;
          else
            *q++=trough;
        for ( ; x < (long) (image->columns+2*frame_info->inner_bevel); x++)
          *q++=highlight;
        width=frame_info->width-frame_info->x-image->columns-bevel_width;
        for (x=0; x < (long) width; x++)
          *q++=matte;
        for (x=0; x < (long) frame_info->outer_bevel; x++)
          *q++=shadow;
      }
      (void) SyncImagePixels(frame_image);
    }
  /*
    Draw sides of ornamental border.
  */
  for (y=0; y < (long) image->rows; y++)
  {
    /*
      Initialize scanline with border color.
    */
    p=AcquireImagePixels(image,0,y,image->columns,1,exception);
    q=SetImagePixels(frame_image,0,frame_info->y+y,frame_image->columns,1);
    if ((p == (const PixelPacket *) NULL) || (q == (PixelPacket *) NULL))
      break;
    for (x=0; x < (long) frame_info->outer_bevel; x++)
      *q++=highlight;
    for (x=0; x < (long) (frame_info->x-bevel_width); x++)
      *q++=matte;
    for (x=0; x < (long) frame_info->inner_bevel; x++)
      *q++=shadow;
    /*
      Transfer scanline.
    */
    (void) CopyMagickMemory(q,p,(size_t) image->columns*sizeof(PixelPacket));
    q+=image->columns;
    for (x=0; x < (long) frame_info->inner_bevel; x++)
      *q++=highlight;
    width=frame_info->width-frame_info->x-image->columns-bevel_width;
    for (x=0; x < (long) width; x++)
      *q++=matte;
    for (x=0; x < (long) frame_info->outer_bevel; x++)
      *q++=shadow;
    if (SyncImagePixels(frame_image) == MagickFalse)
      break;
    if (QuantumTick(y,image->rows) != 0)
      if (MagickMonitor(FrameImageTag,y,image->rows,exception) == MagickFalse)
        break;
  }
  /*
    Draw bottom of ornamental border.
  */
  height=(unsigned long) (frame_info->inner_bevel+frame_info->height-
    frame_info->y-image->rows-bevel_width+frame_info->outer_bevel);
  q=SetImagePixels(frame_image,0,(long) (frame_image->rows-height),
    frame_image->columns,height);
  if (q == (PixelPacket *) NULL)
    return(frame_image);
  for (y=frame_info->inner_bevel-1; y >= 0; y--)
  {
    for (x=0; x < (long) frame_info->outer_bevel; x++)
      *q++=highlight;
    for (x=0; x < (long) (frame_info->x-bevel_width); x++)
      *q++=matte;
    for (x=0; x < y; x++)
      *q++=shadow;
    for ( ; x < (long) (image->columns+2*frame_info->inner_bevel); x++)
      if (x >= (long) (image->columns+2*frame_info->inner_bevel-y))
        *q++=highlight;
      else
        *q++=accentuate;
    width=frame_info->width-frame_info->x-image->columns-bevel_width;
    for (x=0; x < (long) width; x++)
      *q++=matte;
    for (x=0; x < (long) frame_info->outer_bevel; x++)
      *q++=shadow;
  }
  height=frame_info->height-frame_info->y-image->rows-bevel_width;
  for (y=0; y < (long) height; y++)
  {
    for (x=0; x < (long) frame_info->outer_bevel; x++)
      *q++=highlight;
    for (x=0; x < (long) (frame_image->columns-2*frame_info->outer_bevel); x++)
      *q++=matte;
    for (x=0; x < (long) frame_info->outer_bevel; x++)
      *q++=shadow;
  }
  for (y=frame_info->outer_bevel-1; y >= 0; y--)
  {
    for (x=0; x < y; x++)
      *q++=highlight;
    for ( ; x < (long) frame_image->columns; x++)
      if (x >= (long) (frame_image->columns-y))
        *q++=shadow;
      else
        *q++=trough;
  }
  (void) SyncImagePixels(frame_image);
  return(frame_image);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   R a i s e I m a g e                                                       %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  RaiseImage() creates a simulated three-dimensional button-like effect
%  by lightening and darkening the edges of the image.  Members width and
%  height of raise_info define the width of the vertical and horizontal
%  edge of the effect.
%
%  The format of the RaiseImage method is:
%
%      MagickBooleanType RaiseImage(Image *image,
%        const RectangleInfo *raise_info,const MagickBooleanType raise)
%
%  A description of each parameter follows:
%
%    o image: The image.
%
%    o raise_info: Define the width and height of the raise area.
%
%    o raise: A value other than zero creates a 3-D raise effect,
%      otherwise it has a lowered effect.
%
%
*/
MagickExport MagickBooleanType RaiseImage(Image *image,
  const RectangleInfo *raise_info,const MagickBooleanType raise)
{
#define AccentuateFactor  ScaleCharToQuantum(135)
#define HighlightFactor  ScaleCharToQuantum(190)
#define ShadowFactor  ScaleCharToQuantum(190)
#define RaiseImageTag  "Raise/Image"
#define TroughFactor  ScaleCharToQuantum(135)

  MagickBooleanType
    status;

  Quantum
    foreground,
    background;

  long
    y;

  register long
    x;

  register PixelPacket
    *q;

  assert(image != (Image *) NULL);
  assert(image->signature == MagickSignature);
  if (image->debug != MagickFalse)
    (void) LogMagickEvent(TraceEvent,GetMagickModule(),image->filename);
  assert(raise_info != (RectangleInfo *) NULL);
  if ((image->columns <= (raise_info->width << 1)) ||
      (image->rows <= (raise_info->height << 1)))
    ThrowBinaryException(OptionError,"ImageSizeMustExceedBevelWidth",
      image->filename);
  foreground=MaxRGB;
  background=0;
  if (raise == MagickFalse)
    {
      foreground=0;
      background=MaxRGB;
    }
  image->storage_class=DirectClass;
  for (y=0; y < (long) raise_info->height; y++)
  {
    q=GetImagePixels(image,0,y,image->columns,1);
    if (q == (PixelPacket *) NULL)
      break;
    for (x=0; x < y; x++)
    {
      q->red=(Quantum) (QuantumScale*((MagickRealType) q->red*
        HighlightFactor+(MagickRealType) foreground*(MaxRGB-
        HighlightFactor))+0.5);
      q->green=(Quantum) (QuantumScale*((MagickRealType) q->green*
        HighlightFactor+(MagickRealType) foreground*(MaxRGB-
        HighlightFactor))+0.5);
      q->blue=(Quantum) (QuantumScale*((MagickRealType) q->blue*
        HighlightFactor+(MagickRealType) foreground*(MaxRGB-
        HighlightFactor))+0.5);
      q++;
    }
    for ( ; x < (long) (image->columns-y); x++)
    {
      q->red=(Quantum) (QuantumScale*((MagickRealType) q->red*AccentuateFactor+
        (MagickRealType) foreground*(MaxRGB-AccentuateFactor))+0.5);
      q->green=(Quantum) (QuantumScale*((MagickRealType) q->green*
        AccentuateFactor+(MagickRealType) foreground*(MaxRGB-
        AccentuateFactor))+0.5);
      q->blue=(Quantum) (QuantumScale*((MagickRealType) q->blue*
        AccentuateFactor+(MagickRealType) foreground*(MaxRGB-
        AccentuateFactor))+0.5);
      q++;
    }
    for ( ; x < (long) image->columns; x++)
    {
      q->red=(Quantum) (QuantumScale*((MagickRealType) q->red*ShadowFactor+
        (MagickRealType) background*(MaxRGB-ShadowFactor))+0.5);
      q->green=(Quantum) (QuantumScale*((MagickRealType) q->green*ShadowFactor+
        (MagickRealType) background*(MaxRGB-ShadowFactor))+0.5);
      q->blue=(Quantum) (QuantumScale*((MagickRealType) q->blue*ShadowFactor+
        (MagickRealType) background*(MaxRGB-ShadowFactor))+0.5);
      q++;
    }
    if (SyncImagePixels(image) == MagickFalse)
      break;
    if (QuantumTick(y,image->rows) != 0)
      if (MagickMonitor(RaiseImageTag,y,image->rows,&image->exception) == MagickFalse)
        break;
  }
  for ( ; y < (long) (image->rows-raise_info->height); y++)
  {
    q=GetImagePixels(image,0,y,image->columns,1);
    if (q == (PixelPacket *) NULL)
      break;
    for (x=0; x < (long) raise_info->width; x++)
    {
      q->red=(Quantum) (QuantumScale*((MagickRealType) q->red*
        HighlightFactor+(MagickRealType) foreground*(MaxRGB-
        HighlightFactor))+0.5);
      q->green=(Quantum) (QuantumScale*((MagickRealType) q->green*
        HighlightFactor+(MagickRealType) foreground*(MaxRGB-
        HighlightFactor))+0.5);
      q->blue=(Quantum) (QuantumScale*((MagickRealType) q->blue*
        HighlightFactor+(MagickRealType) foreground*(MaxRGB-
        HighlightFactor))+0.5);
      q++;
    }
    for ( ; x < (long) (image->columns-raise_info->width); x++)
      q++;
    for ( ; x < (long) image->columns; x++)
    {
      q->red=(Quantum) (QuantumScale*((MagickRealType) q->red*ShadowFactor+
        (MagickRealType) background*(MaxRGB-ShadowFactor))+0.5);
      q->green=(Quantum) (QuantumScale*((MagickRealType) q->green*ShadowFactor+
        (MagickRealType) background*(MaxRGB-ShadowFactor))+0.5);
      q->blue=(Quantum) (QuantumScale*((MagickRealType) q->blue*ShadowFactor+
        (MagickRealType) background*(MaxRGB-ShadowFactor))+0.5);
      q++;
    }
    if (SyncImagePixels(image) == MagickFalse)
      break;
    if (QuantumTick(y,image->rows) != 0)
      if (MagickMonitor(RaiseImageTag,y,image->rows,&image->exception) == MagickFalse)
        break;
  }
  for ( ; y < (long) image->rows; y++)
  {
    q=GetImagePixels(image,0,y,image->columns,1);
    if (q == (PixelPacket *) NULL)
      break;
    for (x=0; x < (long) (image->rows-y); x++)
    {
      q->red=(Quantum) (QuantumScale*((MagickRealType) q->red*
        HighlightFactor+(MagickRealType) foreground*(MaxRGB-
        HighlightFactor))+0.5);
      q->green=(Quantum) (QuantumScale*((MagickRealType) q->green*
        HighlightFactor+(MagickRealType) foreground*(MaxRGB-
        HighlightFactor))+0.5);
      q->blue=(Quantum) (QuantumScale*((MagickRealType) q->blue*
        HighlightFactor+(MagickRealType) foreground*(MaxRGB-
        HighlightFactor))+0.5);
      q++;
    }
    for ( ; x < (long) (image->columns-(image->rows-y)); x++)
    {
      q->red=(Quantum) (QuantumScale*((MagickRealType) q->red*TroughFactor+
        (MagickRealType) background*(MaxRGB-TroughFactor))+0.5);
      q->green=(Quantum) (QuantumScale*((MagickRealType) q->green*TroughFactor+
        (MagickRealType) background*(MaxRGB-TroughFactor))+0.5);
      q->blue=(Quantum) (QuantumScale*((MagickRealType) q->blue*TroughFactor+
        (MagickRealType) background*(MaxRGB-TroughFactor))+0.5);
      q++;
    }
    for ( ; x < (long) image->columns; x++)
    {
      q->red=(Quantum) (QuantumScale*((MagickRealType) q->red*ShadowFactor+
        (MagickRealType) background*(MaxRGB-ShadowFactor))+0.5);
      q->green=(Quantum) (QuantumScale*((MagickRealType) q->green*ShadowFactor+
        (MagickRealType) background*(MaxRGB-ShadowFactor))+0.5);
      q->blue=(Quantum) (QuantumScale*((MagickRealType) q->blue*ShadowFactor+
        (MagickRealType) background*(MaxRGB-ShadowFactor))+0.5);
      q++;
    }
    if (SyncImagePixels(image) == MagickFalse)
      break;
    if (QuantumTick(y,image->rows) != 0)
      {
        status=MagickMonitor(RaiseImageTag,y,image->rows,&image->exception);
        if (status == MagickFalse)
          break;
      }
  }
  return(MagickTrue);
}
