# (c) 2008 Canonical Ltd.
# Author: Martin Pitt <martin.pitt@ubuntu.com>
# License: GPL v2 or later

from jockey.handlers import KernelModuleHandler
from jockey.xorg_driver import XorgDriverHandler
from jockey.oslib import OSLib

class NvidiaDriver(XorgDriverHandler):
    def __init__(self, ui, module = 'nvidia', driver_package='nvidia-glx', 
            extra_conf_options = {'NoLogo': 'True', 
                'AddARGBGLXVisuals': 'True'}):
        XorgDriverHandler.__init__(self, ui, module, driver_package,
            'nvidia', 'nv', extra_conf_options, 
            add_modules=['glx'], remove_modules=['dri', 'GLcore'],
            name=ui._('NVIDIA accelerated graphics driver'),
            description=ui._('3D-accelerated proprietary graphics driver for '
                'NVIDIA cards.'),
            rationale=ui._('This driver is required to fully utilise '
                'the 3D potential of NVIDIA graphics cards, as well as provide '
                '2D acceleration of newer cards.\n\n'
                'If you wish to enable desktop effects, this driver is '
                'required.\n\n'
                'If this driver is not enabled, you will not be able to '
                'enable desktop effects and will not be able to run software '
                'that requires 3D acceleration, such as some games.'))

    def used(self):
        # if you modprobe nvidia_{new,legacy} they will still appear as
        # 'nvidia' in /proc/modules for some weird reason, so we need to
        # account for that
        return KernelModuleHandler.module_loaded('nvidia') and \
            OSLib.inst.package_installed(self.package)

    def enable_config_hook(self):
        # X.org does not work otherwise
        if len(self.xorg_conf.getSections('Screen')) == 0:
            self.xorg_conf.append(self.xorg_conf.makeSection(None, 
                ['Section', 'Screen']))
        self.xorg_conf.getSections('Screen')[0].defaultdepth = 24

    def supports_composite(self):
        '''Return whether this driver supports the composite extension.'''

        return True

class LegacyNvidiaDriver(NvidiaDriver):
    def __init__(self, ui):
        NvidiaDriver.__init__(self, ui, 'nvidia_legacy', 'nvidia-glx-legacy', 
            {'AllowGLXWithComposite': 'True', 'AddARGBVisuals': 'True',
                'AddARGBGLXVisuals': 'True', 'UseEdidFreqs': 'True',
                'NoLogo': 'True'})

    def name(self):
        return self.ui._('NVIDIA accelerated graphics driver (legacy cards)')

class NewNvidiaDriver(NvidiaDriver):
    def __init__(self, ui):
        NvidiaDriver.__init__(self, ui, 'nvidia_new', 'nvidia-glx-new', 
            {'NoLogo': 'True'})

    def name(self):
        return self.ui._('NVIDIA accelerated graphics driver (latest cards)')
