#!/usr/bin/python
###########################################################################
# xorgconfig.py - description                                             #
# ------------------------------                                          #
# begin     : Wed Feb 9 2004                                              #
# copyright : (C) 2005 by Simon Edwards                                   #
# email     : simon@simonzone.com                                         #
#                                                                         #
###########################################################################
#                                                                         #
#   This program is free software; you can redistribute it and/or modify  #
#   it under the terms of the GNU General Public License as published by  #
#   the Free Software Foundation; either version 2 of the License, or     #
#   (at your option) any later version.                                   #
#                                                                         #
###########################################################################
import csv
"""
General usage:

    import xorgconfig
    config = readConfig("/etc/X11/xorg.conf")

    input_devices = config.getSections("InputDevice")
    print input_devices[0].driver
    options = input_devices[0].options
    for option in options:
        # option is of type OptionLine.
        print option._row[0],
        if len(option._row)>=2:
            print "=>",option._row[1]
    
    # Add line: Option "XkbModel" "pc105"
    options.append( options.makeLine("Comment text",["XkbModel" "pc105"]) )
    
"""
############################################################################
class ConfigLine(object):
    """Represents one line from the Xorg.conf file.
    
    Each part of the line is printed without quotes.
    """
    def __init__(self,comment,row):
        self._row = [item for item in row if item!='']
        self._comment = comment
        
    def toString(self,depth=0):
        string = (' ' * depth) + ' '.join([str(item) for item in self._row])
        if self._comment is not None:
            string += '#' + self._comment
        return string + '\n'

############################################################################
class ConfigLineQuote(ConfigLine):
    """Represents one line from the Xorg.conf file.
    
    The first item in the line is not quoted, but the remaining items are.
    """
    def toString(self,depth=0):
        string = (' ' * depth)
        if len(self._row)!=0:
            string += self._row[0]
        if len(self._row)>1:
            string += ' "' + '" "'.join([str(item) for item in self._row[1:]]) + '"'
        if self._comment is not None:
            string += '#' + self._comment
        return string + '\n'

############################################################################
class OptionLine(ConfigLineQuote):
    def __init__(self,comment,row):
        arg = ['option']
        arg.extend(row)
        ConfigLineQuote.__init__(self,comment,arg)

############################################################################
class ConfigList(list):
    def toString(self,depth=0):
        result = ""
        for item in self:
            result += item.toString(depth)
        return result

############################################################################
class OptionList(ConfigList):
    name = "option"
    def __setitem__(self,key,value):
        print value # FIXME check argument class
        list.__setitem__(self,key,value)
        
    def makeLine(self,comment,row):
        return OptionLine(comment,row)
    
############################################################################
class ScreenConfigLine(ConfigLine):
    def __init__(self,comment,row):
        arg = ["screen"]
        arg.extend(row)
        ConfigLine.__init__(self,comment,arg)
        
    def toString(self,depth=0):
        str = (' ' * depth)
        
        try: # Keep on building up the string until the IndexError is thrown.
            str += self._row[0]
            i = 1
            if self._row[i].isdigit():
                str += ' ' + self._row[i]
                i += 1
            str += ' "' + self._row[i] + '"'
            i += 1
            while True:
                item = self._row[i].lower()
                if item in ['rightof','leftof','above','below']:
                    str += ' %s "%s"' % (item, self._row[i+1])
                    i += 1
                elif item=='absolute':
                    str += ' %s %d %d' % (item, self._row[i+1], self._row[i+2])
                    i += 2
                elif item.isdigit():
                    i += 1
                    str += ' %s %s' % (item,self._row[i])
                i += 1
        except IndexError: pass
        
        if self._comment is not None:
            str += ' #' + self._comment
        return str + '\n'

############################################################################
class ScreenConfigList(ConfigList):
    name = "screen"
    def __setitem__(self,key,value):
        print value # FIXME check argument class
        list.__setitem__(self,key,value)
        
    def makeLine(self,comment,row):
        return ScreenConfigLine(comment,row)

############################################################################
class ConfigContainer(object):
    """Acts as a container for ConfigLines and other ConfigContainers.
    Is used for representing things like the whole config file, sections
    and subsections inside the file.
    
    """
    def __init__(self):
        self._contents = []
        
    def append(self,item):
        assert (item is not None)
        self._contents.append(item)
        
    def remove(self,item):
        self._contents.remove(item)
        
    def toString(self,depth=0):
        str = ''
        for item in self._contents:
            str += item.toString(depth+1)
        return str
        
    def makeSection(self,comment,name):
        return Section(comment,name)
        
    def isSection(self,name):
        lname = name.lower()
        return lname=='section'
        
    def isEndSection(self,name):
        return False
        
    def makeLine(self,comment,row):
        return ConfigLine(comment,row)
        
    def isListAttr(self,name):
        lname = name.lower()
        return lname in self._listattr
        
    def makeListAttr(self,comment,row):
        listobj = self.__getattr__(row[0].lower())
        listobj.append( listobj.makeLine(comment,row[1:]) )
    
    def getSections(self,name):
        """Get all sections having the given name.
        
        Returns a list of ConfigContainer objects.
        """
        name = name.lower()
        sections = []
        for item in self._contents:
            try:
                if isinstance(item,ConfigContainer) and item._name.lower()==name:
                    sections.append(item)
            except IndexError: pass
        print "Section: ", name
        print sections
        return sections
        
    def __getattr__(self,name):
        if not name.startswith("_"):
            lname = name.lower()
            if lname in self._listattr:
                # Lookup list attributes.
                for item in self._contents:
                    if isinstance(item,ConfigList) and item.name==lname:
                        return item
                else:
                    listitem = self._listattr[lname]()
                    self._contents.append(listitem)
                    return listitem
            else:
                for item in self._contents:
                    try:
                        if isinstance(item,ConfigLine) and item._row[0].lower()==lname:
                            return item._row[1]
                    except IndexError: pass
                if lname in self._attr or lname in self._quoteattr:
                    return None
        raise AttributeError, name
        
    def __setattr__(self,name,value):
        if name.startswith('_'):
            return super(ConfigContainer,self).__setattr__(name,value)

        lname = name.lower()
        for item in self._contents:
            try:
                if isinstance(item,ConfigLine) and item._row[0].lower()==lname:
                    item._row[1] = value
                    break
            except IndexError: pass
        else:
            if lname in self._attr or lname in self._quoteattr:
                line = self.makeLine(None,[name,value])
                self.append(line)
            else:
                raise AttributeError, name
                
    def clear(self):
        self._contents = []
        
    def getRow(self,name):
        if not name.startswith("_"):
            lname = name.lower()
            for item in self._contents:
                try:
                    if isinstance(item,ConfigLine) and item._row[0].lower()==lname:
                        return item._row[1:]
                except IndexError: pass
        raise AttributeError, name
    
############################################################################
class Section(ConfigContainer):
    """Represents a Section in the config file.
    
    """
    
    # List of config line types allowed inside this section.
    # A list of strings naming lines that need to be stored in ConfigLine objects.
    _attr = []
    
    # A list of strings naming the lines that need to be stored in ConfigLineQuote objects.
    # This is often overridden in subclasses.
    _quoteattr = []
    
    _listattr = {}
    
    def __init__(self,comment,name):
        ConfigContainer.__init__(self)
        self._name = name
        self._comment = comment
    
    def __show__(self):
        """ For debugging """
        for a in self._attr:
            print self._name, "Attribute:", a
        for a in self._quoteattr:
            print self._name, "QuoteAttribute:", a
        for a in self._listattr:
            print self._name, "ListAttr:", a
    
    def isSection(self,name):
        return name.lower()=='subsection'
        
    def isEndSection(self,name):
        return name.lower()=='endsection'
        
    def makeLine(self,comment,row):
        try:
            lname = row[0].lower()
            if lname in self._quoteattr:
                return ConfigLineQuote(comment,row)
            if lname in self._attr:
                return ConfigLine(comment,row)
            return None
        except IndexError:
            pass
        return ConfigContainer.makeLine(self,comment,row)
        
    def toString(self,depth=0):
        if self._comment is None:
            return '%sSection "%s"\n%s%sEndSection\n' % \
                (' ' * depth, self._name, ConfigContainer.toString(self,depth+1), ' ' * depth)
        else:
            return '%sSection "%s" # %s\n%s%sEndSection\n' % \
                (' ' * depth, self._name, self._comment, ConfigContainer.toString(self,depth+1), ' ' * depth)

############################################################################
class SubSection(Section):
    def isSection(self,name):
        return False
        
    def isEndSection(self,name):
        return name.lower()=='endsubsection'
        
    def toString(self,depth=0):
        return '%sSubSection "%s"\n%s%sEndSubSection\n' % \
            (' ' * depth, self._name, ConfigContainer.toString(self,depth+1), ' ' * depth)

        
############################################################################
class DeviceSection(Section):
    _attr = ["endsection","dacspeed","clocks","videoram","biosbase","membase", \
        "iobase","chipid","chiprev","textclockfreq","irq","screen"]

    _quoteattr = ["identifier","vendorname","boardname","chipset","ramdac", \
        "clockchip","card","driver","busid"]
        
    _listattr = {"option" : OptionList}
    
    # If  there's only one GraphicsCard, set BusID to standard, see man xorg.conf / XF86Config-4
    busid = "PCI:1:0:0"
############################################################################
class DriSection(Section):
    _attr = ["group","buffers","mode"]
    def makeLine(self,comment,row):
        try:
            lname = row[0].lower()
            if lname=="group" and not row[1].isdigit():
                return ConfigLineQuote(comment,row)
        except IndexError:
            pass
        return Section.makeLine(self,comment,row)

############################################################################
class ExtensionsSection(Section):
    _listattr = {"option" : OptionList}

############################################################################
class FilesSection(Section):
    _quoteattr = ["fontpath","rgbpath","modulepath","inputdevices","logfile"]
    def makeLine(self,comment,row):
        return ConfigLineQuote(comment,row)
        
############################################################################
class ModuleSection(Section):
    _quoteattr = ["load","loaddriver"]
    
    def makeSection(self,comment,name):
        return ModuleSubSection(comment,name)
        
    def removeModule(self,modname):
        killlist = []
        for item in self._contents:
            try:
                if isinstance(item,ConfigLineQuote) \
                        and item._row[0].lower()=='load' \
                        and item._row[1]==modname:
                    killlist.append(item)
            except IndexError: pass
        
        for item in killlist:
            self._contents.remove(item)
            
    def addModule(self,modname):
        self.removeModule(modname)
        self._contents.append(ConfigLineQuote(None,['load',modname]))
        
############################################################################
class ModuleSubSection(SubSection):
    _listattr = {"option" : OptionList}

############################################################################
class ModeSection(Section):
    _attr = ["dotclock","htimings","vtimings","hskew","bcast","vscan"]
    _quoteattr = ["flags"]
    def toString(self,depth=0):
        if self._comment is None:
            return '%sMode "%s"\n%s%sEndMode\n' % \
                (' ' * depth, self._name, ConfigContainer.toString(self,depth+1), ' ' * depth)
        else:
            return '%sMode "%s" # %s\n%s%sEndMode\n' % \
                (' ' * depth, self._name, self._comment, ConfigContainer.toString(self,depth+1), ' ' * depth)
            
############################################################################
class ModeList(ConfigList):
    name = "mode"
    def __setitem__(self,key,value):
        print value # FIXME check argument class
        list.__setitem__(self,key,value)
        
    def makeLine(self,comment,row):
        return ModeLine(comment,row)
    
############################################################################
class MonitorSection(Section):
    _attr = ["modeline","displaysize","horizsync","vertrefresh","gamma"]
    _quoteattr = ["identifier","vendorname","modelname","usemodes"]
    _listattr = {"option" : OptionList, "mode" : ModeList}
    def makeLine(self,comment,row):
        try:
            if row[0].lower()=='modeline':
                return ModeLineConfigLine(comment,row)
        except IndexError:
            pass
        return Section.makeLine(self,comment,row)
        
    def isSection(self,name):
        lname = name.lower()
        return lname=='mode'
        
    def isEndSection(self,name):
        return name.lower()=='endsection'
        
    def makeSection(self,comment,name):
        if name.lower()=='mode':
            return ModeSection(comment,name)
        else:            
            return Section.makeSection(self,comment,name)
            
############################################################################
class ModeLineConfigLine(ConfigLine):
    def toString(self,depth=0):
        str = (' ' * depth)
        if len(self._row)!=0:
            str += self._row[0]
        if len(self._row)>1:
            str += ' "' + self._row[1] + '"'
        if len(self._row)>2:
            str +=  ' ' + ' '.join([str(item) for item in self._row[2:]])
        if self._comment is not None:
            str += '#' + self._comment
        return str + '\n'

############################################################################
class ModesSection(MonitorSection):
    # Like a MonitorSection, only smaller.
    _attr = ["modeline"]
    _quoteattr = ["identifier"]
    
############################################################################
class PointerSection(Section):
    _attr = ["emulate3timeout","baudrate","samplerate","resolution",\
        "devicename","buttons"]
    _quoteattr = ["protocol","device","port","emulate3buttons","chordmiddle",\
        "cleardtr","clearrts","zaxismapping","alwayscore"]

############################################################################
class ScreenSection(Section):
    _attr = ["screenno","defaultcolordepth","defaultdepth","defaultbpp","defaultfbbpp"]
    _quoteattr = ["identifier","driver","device","monitor","videoadaptor","option"]
    _listattr = {"option" : OptionList}
    def makeSection(self,comment,name):
        if name.lower()=='display':
            return DisplaySubSection(comment,name)
        return SubSection(comment,name)

############################################################################
class DisplaySubSection(SubSection):
    _attr = ["viewport","virtual","black","white","depth","fbbpp","weight"]
    _quoteattr = ["modes","visual","option"]
    _listattr = {"option" : OptionList}
############################################################################
class ServerFlagsSection(Section):
    _quoteattr = ["notrapsignals","dontzap","dontzoom","disablevidmodeextension",\
        "allownonlocalxvidtune","disablemodindev","allownonlocalmodindev","allowmouseopenfail", \
        "blanktime","standbytime","suspendtime","offtime","defaultserverlayout"]
    _listattr = {"option" : OptionList}
    
############################################################################
class ServerLayoutSection(Section):
    _attr = []
    _quoteattr = ["identifier","inactive","inputdevice","option"]
    _listattr = {"option" : OptionList, "screen" : ScreenConfigList}
    
############################################################################
class InputDeviceSection(Section):
    _quoteattr = ["identifier","driver"]
    _listattr = {"option" : OptionList}
############################################################################
class KeyboardSection(Section):
    _attr = ["autorepeat","xleds"]
    _quoteattr = ["protocol","panix106","xkbkeymap","xkbcompat","xkbtypes",\
        "xkbkeycodes","xkbgeometry","xkbsymbols","xkbdisable","xkbrules",\
        "xkbmodel","xkblayout","xkbvariant","xkboptions","vtinit","vtsysreq",\
        "servernumlock","leftalt","rightalt","altgr","scrolllock","rightctl"]

############################################################################
class VendorSection(Section):
    _attr = []
    _quoteattr = ["identifier"]
    _listattr = {"option" : OptionList}
    def isSection(self,name): return False
    
############################################################################
class VideoAdaptorSection(Section):
    _attr = []
    _quoteattr = ["identifier","vendorname","boardname","busid","driver"]
    _listattr = {"option" : OptionList}
    def makeSection(self,comment,name): 
        return VideoPortSection(comment,name)

############################################################################
class VideoPortSection(SubSection):
    _attr = []
    _quoteattr = ["identifier"]
    _listattr = {"option" : OptionList}
############################################################################
class XorgConfig(ConfigContainer):
    _sectiontypes = { \
        'device': DeviceSection,
        'dri': DriSection,
        'extensions': ExtensionsSection,
        'files': FilesSection,
        'inputdevice': InputDeviceSection,
        'keyboard': KeyboardSection,
        'modes': ModesSection,
        'monitor': MonitorSection,
        'module': ModuleSection,
        'pointer': PointerSection,
        'serverflags': ServerFlagsSection,
        'serverlayout': ServerLayoutSection,
        'screen': ScreenSection,
        'videoadaptor': VideoAdaptorSection}
            
    def makeSection(self,comment,name):
        lname = name.lower()
        try:
            return self._sectiontypes[lname](comment,name)
        except KeyError:
            return ConfigContainer.makeSection(self,comment,name)

    def toString(self,depth=-1):
        return ConfigContainer.toString(self,depth)
        
    def writeConfig(self,filename):
        fhandle = open(filename,'w')
        fhandle.write(self.toString())
        fhandle.close()
        
############################################################################
def readConfig(filename):
    
    # This class is just for skipping comment lines in the database file.
    # This whole class is just an iterator wrapper that we put around our file iterator.
    class xorgconfigiterator(object):
        def __init__(self,fhandle):
            self.fhandle = iter(fhandle)
        def __iter__(self):
            return self
        def next(self):
            line = self.fhandle.next()
            self.comment = None
            try:
                hash = line.index('#')
                self.comment = line[hash+1:].rstrip()
                line = line[:hash]
            except ValueError:
                pass
            return line.expandtabs(1).strip()
    ########
    print "Filename:", filename
    fhandle = open(filename)
    configiter = xorgconfigiterator(fhandle)
    context = XorgConfig()
    stack = []
    line = 1
    for row in csv.reader(configiter,delimiter=' ',skipinitialspace=True):
        try:
            first = row[0].lower()
            if context.isSection(first):
                section = context.makeSection(configiter.comment,row[1])
                context.append(section)
                stack.append(context)
                context = section
            elif context.isEndSection(first):
                context = stack.pop()
            elif context.isListAttr(first):
                context.makeListAttr(configiter.comment,row)
            else:
                newline = context.makeLine(configiter.comment,row)
                if newline is None:
                    raise ParseException,"Unknown line type '%s' on line %i" % (first,line)
                context.append(newline)
        except IndexError:
            context.append(ConfigLine(configiter.comment,row))
        line += 1
    fhandle.close()
    if len(stack)!=0:
        raise ParseException,"Unexpected end of file on line %i" % line
    return context

############################################################################
class ParseException(Exception):
    def __init__(self,*args):
        Exception.__init__(self,*args)
        
############################################################################
if __name__=='__main__':    
    import sys
    if len(sys.argv)==2:
        filename = sys.argv[1]
    else:
        filename = "/etc/X11/xorg.conf"
    print "Reading",filename
    c = readConfig(filename)
    print c.toString()
    
