/***************************************************************************
 *   Copyright (C) 2003 by Sylvain Joyeux                                  *
 *   sylvain.joyeux@m4x.org                                                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/

#include "parsers.h"
#include "../apt.h"

#include <klocale.h>
#include <kdebug.h>
#include <qhtmlstream.h>
#include <qregexp.h>

static void	policy_begin(QHtmlStream& stream)
{
  stream
    << block("div") << endl
    << block("table", "policy") << endl
    	<< block("tr") << block("td") << endl
    		<< block("table", "curver") << endl
    			<< block("tbody");
}

static void add_button(QHtmlStream& stream, const QString& mode, const QString& text, const QString& package)
{
  stream
    << block("form")
        << param("action") << "apt:/"
        << param("method") << "get" << endl
      << block("p") << endl
        << tag("input")
          << param("type") << "hidden"
          << param("name") << "get"
          << param("value") << mode << endl
        << tag("input")
          << param("type") << "hidden"
          << param("name") << "package"
          << param("value") << package << endl
        << tag("input")
          << param("type") << "submit"
          << param("value") << text << endl
      << close()
    << close() << endl;
}

static void add_version_link(QHtmlStream& stream, AptProtocol* slave, const QString& package, const QString& version)
{
  KURL url(slave->buildURL("show", package));
  url.setHTMLRef(Parsers::mangle_version(version));

  stream
    << block("a", "vtable-version")
        << param("href") << url.htmlURL()
      << data() << version
    << close();
}

namespace Parsers
{

Policy::Policy(const QString& package, bool act)
  : m_package(package), m_act(false)
{
  m_act = hasPkgManager() && act;
}

void Policy::operator() (AptProtocol* slave, const QString& type, const QString& value)
{
  static bool first_version = false, received_sth = false;
  static QString buffer;
  static QHtmlStream* stream;

  static QRegExp rx_notinstalled("(none)");

  kdDebug() << "policy : " << type << " " << value << endl;

  if (type == "begin")
  {
    stream = new QHtmlStream(&buffer);
  	policy_begin(*stream);
  }
  else if (type == "installed")
  {
  	received_sth = true;

    attribute_begin(*stream, i18n("Installed"));
    if (rx_notinstalled.match(value) >= 0)
    {
    	m_installed = QString::null;
      *stream << i18n("no");
    }
    else
  	{
    	m_installed = value.stripWhiteSpace();
      add_version_link(*stream, slave, m_package, m_installed);
  	}
    attribute_end(*stream);
 }
  else if (type == "candidate")
  {
  	received_sth = true;

    bool has_candidate = (rx_notinstalled.match(value) == -1);

    if (m_act && has_candidate)
    {
      *stream
        << block("tr") << endl
          << block("td") << param("colspan") << 2 << endl;

      if (m_installed.isEmpty())
        add_button(*stream, "install", i18n("Install"), m_package);
      else
        add_button(*stream, "remove", i18n("Remove"), m_package);
      *stream << close() << close() << endl;
    }

    attribute_begin(*stream, i18n("Candidate"));
    if (has_candidate)
      add_version_link(*stream, slave, m_package, value);
    else
      *stream << i18n("none");
    attribute_end(*stream);

    if (m_act && has_candidate
      && !m_installed.isEmpty() && m_installed != value)
    {
      *stream
        << block("tr") << endl
          << block("td") << param("colspan") << 2 << endl;
      add_button(*stream, "install", i18n("Upgrade"), m_package);
      *stream << close() << close() << endl;
    }
  }
  else if (type == "version_table")
  {
  	received_sth = true;

    *stream
    	<< close() << close() << endl // tbody, table, form
      << close() << endl; // td

  	first_version = true;
  }
  else if (type == "version")
  {
    QString version = value.section(' ', 0, 0);
    QString pin = value.section(' ', 1, 1);

    if (first_version)
    {
      *stream
        << block("td") << endl
          << block("div", "vtable") << endl
            << block("div", "header") << i18n("Version Table") << close() << endl
            << block("div", "versions") << endl;
    } /*else {
      *stream << close() << close();
    }*/

    QString version_link;
    version_link = "<a href=\"apt:/show?" + m_package + "#" + mangle_version(version) + "\">"
       + version + "</a>";

    *stream << tag("br") << endl;
    add_version_link(*stream, slave, m_package, version);
    *stream << "[Pin " << block("span", "version-pin") << pin << close() << "]";

    first_version = false;
  }
  else if (type == "location")
  {
    QStringList sections = QStringList::split(' ', value);
    QString pin = sections.first();
    sections.pop_front();
    // remove the "Packages" field if it is here
    if (sections.last() == "Packages")
      sections.pop_back();

    *stream << tag("br") << endl;
    *stream << block("span", "location-pin") << pin << close() << sections.join(" ") << endl;
  }
  else if (type == "end")
  {
  	if (received_sth)
    {
      *stream << close_all() << endl;
      *slave << buffer;
   	}

    buffer = QString::null;
    received_sth = false;
    delete stream;
  }
}

}
