/***************************************************************************
                          knetworkconf.cpp  -  description
                             -------------------
    begin                : Sun Jan 12 8:54:19 UTC 2003
    copyright            : (C) 2003 by Juan Luis Baptiste
    email                : jbaptiste@merlinux.org
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "knetworkconf.h"

KNetworkConf::KNetworkConf(QWidget *parent, const char *name) : KNetworkConfDlg(parent, name), DCOPObject("KNetworkConfIface")
{
  makeButtonsResizeable();
  config = new KNetworkConfigParser();
  klvCardList->setAllColumnsShowFocus(true);
  klvKnownHosts->setAllColumnsShowFocus(true);
  
  //Connect signals emmitted by the backend to know when data is ready to be painted.
  connect(config,SIGNAL(readyLoadingNetworkInfo()),this,SLOT(getNetworkInfoSlot()));
  connect(config,SIGNAL(readyLoadingNetworkInfo()),this,SLOT(showMainWindow()));
  connect(config,SIGNAL(readyLoadingNetworkInfo()),this,SLOT(enableSignals()));
  connect(config, SIGNAL(setReadOnly(bool)),this,SLOT(setReadOnlySlot(bool)));
  connect(klvCardList, 
          SIGNAL(contextMenu(KListView*,QListViewItem*,const QPoint&)),
          this,
          SLOT(showInterfaceContextMenuSlot(KListView*,QListViewItem*, const QPoint&)));  
  
  // Register with DCOP - No longer needed as now we are a kcontrol module?
  if ( !kapp->dcopClient()->isRegistered() ) {
    kapp->dcopClient()->registerAs( "knetworkconf" );
    kapp->dcopClient()->setDefaultObject( objId() );
  }

  //Temporarly added while gst supports routing option.
//  cbEnableRouting->hide();
}

KNetworkConf::~KNetworkConf()
{
}

void KNetworkConf::getNetworkInfoSlot()
{
  netInfo = config->getNetworkInfo();
  if (netInfo == NULL)
  {
    KMessageBox::error(this,
                        i18n("Couldn't load Network Info."),
                        i18n("Error reading Configuration File"));
    //kapp->quit();
    //exit(1);
  }
  else
  {
    deviceList.clear();
    deviceList = netInfo->getDeviceList();
    routingInfo = netInfo->getRoutingInfo();
    dnsInfo = netInfo->getDNSInfo();

    loadNetworkDevicesInfo();
    loadRoutingAndDNSInfo();
    loadDNSInfoTab();
    nameServersModified = false;
    devicesModified = false;
    modified = false;
//    kpbApply->setEnabled(false);
  }

}

/**
  Fill the Listview with the info of the network interfaces.
*/
void KNetworkConf::loadNetworkDevicesInfo()
{
  KNetworkInterface *device;
  QPixmap activeEthernetDeviceImg(locate("icon","default.kde/22x22/actions/network_connected_lan.png"));
  QPixmap inactiveEthernetDeviceImg(locate("icon","default.kde/22x22/actions/network_disconnected_lan.png"));  
  QPixmap activeWirelessDeviceImg(locate("icon","default.kde/22x22/actions/network_traffic_wlan.png"));
  QPixmap inactiveWirelessDeviceImg(locate("icon","default.kde/22x22/actions/network_disconnected_wlan.png"));
  
  klvCardList->clear();
  for (device = deviceList.first(); device; device = deviceList.next())
  {
    if ( device->getType() != "loopback" )
    {
      if (klvCardList->findItem(device->getDeviceName(),0,CaseSensitive|ExactMatch) == 0)
      {
        QListViewItem * item = new QListViewItem( klvCardList, 0 );

        if (device->isActive())
        {
          if (device->getType() == "ethernet")
            item->setPixmap(0,activeEthernetDeviceImg);
          else if (device->getType() == "wireless")
            item->setPixmap(0,activeWirelessDeviceImg);          
          
          item->setText(3,i18n("Enabled"));
          item->setPixmap(3,locate("icon","default.kde/16x16/actions/ok.png"));        
        }
        else
        {
          if (device->getType() == "ethernet")
            item->setPixmap(0,inactiveEthernetDeviceImg);
          else if (device->getType() == "wireless")
            item->setPixmap(0,inactiveWirelessDeviceImg);          
        
          item->setText(3,i18n("Disabled"));
          item->setPixmap(3,locate("icon","default.kde/16x16/actions/stop.png"));        
          if (device->getBootProto().lower() == "dhcp")
            item->setText(1,"");
        }
            
        item->setText(0,device->getDeviceName());
        item->setText(1,device->getIpAddress());
        if (device->getBootProto() == "none")
          item->setText(2,i18n("Manual"));
        else 
          item->setText(2,device->getBootProto());
        item->setText(4,device->getDescription());
        QStringList l = deviceNamesList.grep(device->getDeviceName());
        if (l.size() == 0)
          deviceNamesList.append(device->getDeviceName());
      }
    }  
  }
}


/** Terminates the application
 */
void KNetworkConf::quitSlot(){
  int code = 0;
  connect( config, SIGNAL(readyLoadingNetworkInfo()), this, SLOT(quitSlot()) );
  if (modified)
  {
    code = KMessageBox::warningYesNoCancel(this,
                          i18n("The new configuration hasn't been saved.\n Do you want to apply changes before quitting?"),
                          i18n("New configuration not saved."));
    if (code == KMessageBox::Yes)
      saveInfoSlot();
    else if (code == KMessageBox::No)
      kapp->quit();
  }
  else
    kapp->quit();
}
/** Enables the configure and remove buttons. */
void KNetworkConf::enableButtonsSlot(){
  if (!readOnly)
  {
     kpbConfigureNetworkInterface->setEnabled(true);
    QListViewItem *item = klvCardList->currentItem();
    QString currentDevice = item->text(0);
    KNetworkInterface *dev = getDeviceInfo(currentDevice);
     
     if (dev->isActive())
     {
       kpbUpButton->setEnabled(false);
       kpbDownButton->setEnabled(true);     
     }
     else
     {
       kpbUpButton->setEnabled(true);
       kpbDownButton->setEnabled(false);     
     }     
  }
}
/** opens the add server dialog. */
void KNetworkConf::addServerSlot(){
  KAddDNSServerDlg addDlg(this, 0);
  addDlg.exec();
  if(addDlg.modified())
  {
    klbDomainServerList->insertItem(addDlg.kleNewServer->text());
    nameServersModified = true;
    enableApplyButtonSlot();
  }
}
/** opens the edit server dialog. */
void KNetworkConf::editServerSlot(){
  KAddDNSServerDlg dlg(this, 0);
  if (klbDomainServerList->currentItem() >= 0)
  {
    int currentPos = klbDomainServerList->currentItem();
    dlg.setCaption(i18n("Edit Server"));
    QListBoxItem *item = klbDomainServerList->item(currentPos);
    QString currentText = item->text();
    dlg.kleNewServer->setText(currentText);
    dlg.kpbAddServer->setText(i18n("&Ok"));
    dlg.exec();
  
    if(dlg.modified())
    {
      klbDomainServerList->changeItem(dlg.kleNewServer->text(),currentPos);
      nameServersModified = true;
      enableApplyButtonSlot();
    }
  }  
}

/** Pops up the window for adding a new interface. */
void KNetworkConf::configureDeviceSlot(){
  KAddDeviceDlg addDlg(this, 0);
  QListViewItem *item = klvCardList->currentItem();
  
  if (item != NULL)
  {
    QString currentDevice = item->text(0);
    addDlg.setCaption(i18n("Configure Device %1").arg(currentDevice));
    KNetworkInterface *dev = getDeviceInfo(currentDevice);
    KAddDeviceDlgExtension *advancedOptions = (KAddDeviceDlgExtension *)addDlg.extension();

    if (dev->getBootProto() == "none")
    {
      addDlg.rbBootProtoManual->setChecked(true);
      addDlg.rbBootProtoAuto->setChecked(false);
    }
    else if (dev->getBootProto().lower() == "dhcp")
    {
      addDlg.kcbAutoBootProto->setCurrentItem(0);
  //    addDlg.gbBasicDeviceInfo->setEnabled(false);
      addDlg.rbBootProtoManual->setChecked(false);
      addDlg.rbBootProtoAuto->setChecked(true);
      addDlg.kleIPAddress->setEnabled(false);
      addDlg.kcbNetmask->setEnabled(false);
      addDlg.kpbAdvanced->setEnabled(false);
    }
   else if (dev->getBootProto().lower() == "bootp")
    {
      addDlg.kcbAutoBootProto->setCurrentItem(1);
      addDlg.rbBootProtoManual->setChecked(false);
      addDlg.rbBootProtoAuto->setChecked(true);    
  //    addDlg.gbBasicDeviceInfo->setEnabled(false);
      addDlg.kpbAdvanced->setEnabled(false);
    }
    if (dev->getOnBoot() == "yes")
      addDlg.kcbstartAtBoot->setChecked(true);
    else
    {
      addDlg.kcbstartAtBoot->setChecked(false);
    }
    addDlg.kleIPAddress->setText(dev->getIpAddress());
    advancedOptions->kleDescription->setText(dev->getDescription());
    advancedOptions->kleBroadcast->setText(dev->getBroadcast());

    if (!dev->getNetmask().isEmpty())
      addDlg.kcbNetmask->setCurrentText(dev->getNetmask());

    if (readOnly)
    {
     addDlg.kcbAutoBootProto->setEnabled(false);
      addDlg.kcbstartAtBoot->setEnabled(false);
  //    addDlg.gbBasicDeviceInfo->setEnabled(false);
      advancedOptions->gbAdvancedDeviceInfo->setEnabled(false);
    }
    addDlg.kpbApply->setEnabled(false);

    addDlg.exec(); //show window

    if (addDlg.modified())
    {
      item->setText(1,addDlg.kleIPAddress->text());
    
      item->setText(4,advancedOptions->kleDescription->text());

      //If the selected boot protocol is dhcp or bootp (Auto), then we don't need the
      //past IP address, netmask, network and broadcast.    
      if (addDlg.rbBootProtoAuto->isChecked())
      {
        item->setText(2,addDlg.kcbAutoBootProto->currentText());
        dev->setBootProto(addDlg.kcbAutoBootProto->currentText());
        dev->setIpAddress("");
        dev->setNetmask("");
        dev->setNetwork("");
        dev->setBroadcast("");
        //item->setText(2,"");
        //item->setText(3,"");
        enableApplyButtonSlot();
      }
      else
      { //Only update the device object and set modified=true if the values changed
        item->setText(2,i18n("Manual"));
        dev->setBootProto("none");
        if (valuesChanged(dev,
                        addDlg.kcbAutoBootProto->currentText(),
                        addDlg.kcbNetmask->currentText(),
                        addDlg.kleIPAddress->text(),
                        addDlg.kcbstartAtBoot->isChecked(),
	 		  advancedOptions->kleDescription->text(),
		  	advancedOptions->kleBroadcast->text()))
        {
          dev->setIpAddress(addDlg.kleIPAddress->text().stripWhiteSpace());
          dev->setNetmask(addDlg.kcbNetmask->currentText().stripWhiteSpace());
          QString network = KAddressValidator::calculateNetwork(dev->getIpAddress().stripWhiteSpace(),dev->getNetmask().stripWhiteSpace());
          dev->setNetwork(network);
          QString broadcast;
      	 if (addDlg.advanced())
	          broadcast = advancedOptions->kleBroadcast->text().stripWhiteSpace();
  	      else
            broadcast = KAddressValidator::calculateBroadcast(dev->getIpAddress().stripWhiteSpace(),dev->getNetmask().stripWhiteSpace());
          dev->setBroadcast(broadcast);
          dev->setDescription(advancedOptions->kleDescription->text());
  
          if (addDlg.kcbstartAtBoot->isChecked())
            dev->setOnBoot("yes");
          else
            dev->setOnBoot("no");
        }
        devicesModified = true;
        enableApplyButtonSlot();
      }
    }  
  }
}

/**Returns the info of the network device 'device or NULL if not found.'*/
KNetworkInterface * KNetworkConf::getDeviceInfo(QString device){
  QPtrListIterator<KNetworkInterface>  i(deviceList);
  KNetworkInterface *temp;
  while ((temp = i.current()) != 0)
  {
    if (temp->getDeviceName() == device)
    {
      return temp;
    }
    ++i;
  }
  return NULL;
}

/**Returns the name of the network device that corresponds to the IP address 'ipAddr' or NULL if not found.'*/
QString KNetworkConf::getDeviceName(QString ipAddr){
  QPtrListIterator<KNetworkInterface>  i(deviceList);
  KNetworkInterface *temp;
  while ((temp = i.current()) != 0)
  {
    if (temp->getIpAddress().compare(ipAddr) == 0)
    {
      return temp->getDeviceName();
    }
    ++i;
  }
  return NULL;
}


/** Looks in the output returned by ifconfig to see if there are the devices up or down.*/
void KNetworkConf::readFromStdout(){
  commandOutput = "";
  commandOutput += procUpdateDevice->readStdout();
}

/** Loads the info about the default gateway and host and domain names. */
void KNetworkConf::loadRoutingAndDNSInfo(){
  //routingInfo = config->getNetworkInfoSlot();
  if (routingInfo == NULL)
    KMessageBox::error(this,i18n("Couldn't open file '/etc/sysconfig/entwork' for reading!"),
                        i18n("There was an error loading the config files"));
  else
  {
    kleDefaultRoute->setText(routingInfo->getGateway());
    kcbGwDevice->clear();
    kcbGwDevice->insertStringList(deviceNamesList);
    if (!routingInfo->getGatewayDevice().isEmpty())
      kcbGwDevice->setCurrentText(routingInfo->getGatewayDevice());
/*    if (routingInfo->isForwardIPv4Enabled().compare("yes") == 0)
      cbEnableRouting->setChecked(true);
    else
      cbEnableRouting->setChecked(false);*/
  }
}

void KNetworkConf::loadDNSInfoTab(){
  QStringList nameServers;
  if (dnsInfo == NULL)
    KMessageBox::error(this,i18n("Couldn't open file '/etc/resolv.conf' for reading!"),
                        i18n("There was an error loading the config files"));
  else
  {
    kleHostName->setText(dnsInfo->getMachineName());
    kleDomainName->setText(dnsInfo->getDomainName());
    klbDomainServerList->clear();
    nameServers = dnsInfo->getNameServers();
    for ( QStringList::Iterator it = nameServers.begin(); it != nameServers.end(); ++it)
    {
      klbDomainServerList->insertItem(*it);
    }
    klvKnownHosts->clear();
    knownHostsList = dnsInfo->getKnownHostsList();
    QPtrListIterator<KKnownHostInfo> it(knownHostsList);
    KKnownHostInfo *host;
    while ((host = it.current()) != 0)
    {
      ++it;
      if (host->getIpAddress() != "")
      {
        QListViewItem * item = new QListViewItem( klvKnownHosts, 0 );
        item->setText(0,host->getIpAddress());
        QStringList aliasesList = host->getAliases();
        QString aliases;
        for ( QStringList::Iterator it = aliasesList.begin(); it != aliasesList.end(); ++it ) 
        {
          aliases += *it+" ";
        }
        item->setText(1,aliases);
      }
    }
  }
}

/** Shows the help browser. Hopefully some day it will be one :-). */
void KNetworkConf::helpSlot(){
  kapp->invokeHelp();
}

/** No descriptions */
void KNetworkConf::aboutSlot(){
  KAboutApplication *about = new KAboutApplication(kapp->aboutData(),0);

 // about->setLogo(locate("icon","knetworkconf.png"));
  //qDebug("locate icon= %s",locate("icon","knetworkconf.png").latin1());

  about->show();
}
/** No descriptions */
void KNetworkConf::enableApplyButtonSlot(){
 //if (!readOnly)
    //kpbApply->setEnabled(true);
  modified = true;
  emit networkStateChanged(true);
}
/** Puts the application in read-only mode. This happens when the user runing
the application t root. */
void KNetworkConf::setReadOnly(bool state){
  KNetworkConf::readOnly = state;
}
/** No descriptions */
void KNetworkConf::enableApplyButtonSlot(const QString &text){
  enableApplyButtonSlot();
}
/** No descriptions */
void KNetworkConf::enableApplyButtonSlot(bool){
  enableApplyButtonSlot();
}
/** No descriptions */
void KNetworkConf::removeServerSlot(){
  if (klbDomainServerList->currentItem() >= 0)
  {  
    klbDomainServerList->removeItem(klbDomainServerList->currentItem());
    enableApplyButtonSlot();
  }  
}
void KNetworkConf::moveUpServerSlot(){
  int curPos = klbDomainServerList->currentItem();
  int antPos = klbDomainServerList->currentItem() - 1;

  if (antPos >= 0)
  {
    QListBoxItem *current = klbDomainServerList->item(curPos);
    QListBoxItem *ant = current->prev();
    QString antText = ant->text();
    klbDomainServerList->removeItem(antPos);
    klbDomainServerList->insertItem(antText,curPos);
    enableApplyButtonSlot();
  }
}
void KNetworkConf::moveDownServerSlot(){
  unsigned curPos = klbDomainServerList->currentItem();
  unsigned nextPos = klbDomainServerList->currentItem() + 1;

  if (curPos != -1)
  {
    if (klbDomainServerList->count() >= nextPos)
    {
      QListBoxItem *current = klbDomainServerList->item(curPos);
      QString curText = current->text();
      klbDomainServerList->removeItem(curPos);
      klbDomainServerList->insertItem(curText,nextPos);
      klbDomainServerList->setSelected(nextPos,true);
     enableApplyButtonSlot();
    }
  }    
}
/** Disables all buttons a line edit widgets when the user has read only access. */
void KNetworkConf::disableAll(){
  kleHostName->setReadOnly(true);
  kleDomainName->setReadOnly(true);
  tlDomainName->setEnabled(false);
  tlHostName->setEnabled(false);
  disconnect(klvCardList,SIGNAL(doubleClicked(QListViewItem *)),this,SLOT(configureDeviceSlot()));
  klvCardList->setEnabled(false);
  kpbUpButton->setEnabled(false);
  kpbDownButton->setEnabled(false);
  kpbConfigureNetworkInterface->setEnabled(false);
  gbDefaultGateway->setEnabled(false);
  gbDNSServersList->setEnabled(false);
  gbKnownHostsList->setEnabled(false);
//  gbNetworkOptions->setEnabled(false);
}

/** Saves all the modified info of devices, routes,etc. */
void KNetworkConf::saveInfoSlot(){
  config->setProgramVersion(getVersion());

  if (!KAddressValidator::isValidIPAddress(kleDefaultRoute->text()) && (kleDefaultRoute->text() != ""))
  {
    KMessageBox::error(this,i18n("The default Gateway IP address is invalid."),i18n(" Invalid IP address"));
  }
  else
  {
    //Update DNS info
    routingInfo->setDomainName(kleDomainName->text());
    routingInfo->setHostName(kleHostName->text());
    dnsInfo->setDomainName(kleDomainName->text());
    dnsInfo->setMachineName(kleHostName->text());
    dnsInfo->setNameServers(getNamserversList(klbDomainServerList));
    dnsInfo->setKnownHostsList(getKnownHostsList(klvKnownHosts));
    
    //Update routing info
    routingInfo->setGateway(kleDefaultRoute->text());
    if (routingInfo->getGateway().isEmpty())
      routingInfo->setGatewayDevice("");



    if (kleDefaultRoute->text() != "")
      routingInfo->setGatewayDevice(kcbGwDevice->currentText());


    //Save all info
    netInfo->setDeviceList(deviceList);
    netInfo->setRoutingInfo(routingInfo);
    netInfo->setDNSInfo(dnsInfo);
    
    config->saveNetworkInfo(netInfo);
    modified = false;
  }
}
/** Creates a QStringList with the IP addresses contained in the QListBox of name servers. */
QStringList KNetworkConf::getNamserversList(KListBox * serverList){
  QStringList list;
  for (unsigned i = 0; i < serverList->count(); i++)
  {
    list.append(serverList->text(i));
  }
  return list;  
}
/** Creates a QPtrList<KKownHostInfo> with the info contained in the KListView of name servers. */
QPtrList<KKnownHostInfo> KNetworkConf::getKnownHostsList(KListView * hostsList){
  QPtrList<KKnownHostInfo> list;
  QListViewItem *it = hostsList->firstChild();
  for (int i = 0; i < hostsList->childCount(); i++)
  {
    KKnownHostInfo *host = new KKnownHostInfo();
    
    if (it->text(0) != "")
    {
      host->setIpAddress(it->text(0));
//      host->setHostName(it->text(1));
      host->setAliases(QStringList::split(" ",it->text(1)));
      it = it->nextSibling();
      list.append(host);
    }
  }
  return list;
}

QString KNetworkConf::getVersion(){
  return version;
}
void KNetworkConf::setVersion(QString ver){
  KNetworkConf::version = ver;
}

/** Changes the state of device 'dev' to DEVICE_UP or DEVICE_DOWN.
Return true on success, false on failure.  */
void KNetworkConf::changeDeviceState(QString dev, int state){
  KInterfaceUpDownDlg* dialog = new KInterfaceUpDownDlg(this,i18n("Changing device state"));
  if (state == DEVICE_UP)
    dialog->label->setText(i18n("Enabling interface %1" + dev + "%2...").arg("<b>").arg("</b>"));
  else
    dialog->label->setText(i18n("Disabling interface %1" + dev + "%2...").arg("<b>").arg("</b>"));
  
  dialog->setModal(true);  
  dialog->show();

  procDeviceState = new QProcess(this);
  QString cmd;
  procDeviceState->addArgument( locate("data",BACKEND_PATH) );

  //If the platform couldn't be autodetected specify it manually
  if (netInfo->getPlatformName() != QString::null)
  {
    procDeviceState->addArgument( "--platform" );
    procDeviceState->addArgument( netInfo->getPlatformName() );
  }
  procDeviceState->addArgument( "-d" );

  if (state == DEVICE_UP)
    procDeviceState->addArgument("enable_iface::"+dev+"::1" );
  else if (state == DEVICE_DOWN)
    procDeviceState->addArgument("enable_iface::"+dev+"::0" );

  connect( procDeviceState, SIGNAL(readyReadStdout()),this, SLOT(readFromStdoutUpDown()) );
  connect( procDeviceState, SIGNAL(readyReadStderr()),this, SLOT(readFromStdErrUpDown()) );
  connect( procDeviceState, SIGNAL(processExited()),this, SLOT(verifyDeviceStateChanged()) );
  connect( procDeviceState, SIGNAL(processExited()), dialog, SLOT(close()) );
  
  currentDevice = dev;
  commandOutput = "";

  if ( !procDeviceState->start() )
  {
    // error handling
    KMessageBox::error(this,
                        i18n("couldn't launch backend to change network device state. You will have to do it manually. "),
                        i18n("Error "));
    dialog->close();
  }

}
void KNetworkConf::readFromStdoutUpDown(){
  commandOutput.append(procDeviceState->readStdout());
}

void KNetworkConf::verifyDeviceStateChanged(){
  KNetworkInterface *dev;
  QPixmap activeEthernetDeviceImg(locate("icon","default.kde/22x22/actions/network_connected_lan.png"));
  QPixmap inactiveEthernetDeviceImg(locate("icon","default.kde/22x22/actions/network_disconnected_lan.png"));  
  QPixmap activeWirelessDeviceImg(locate("icon","default.kde/22x22/actions/network_traffic_wlan.png"));
  QPixmap inactiveWirelessDeviceImg(locate("icon","default.kde/22x22/actions/network_disconnected_wlan.png"));

  commandOutput = commandOutput.section('\n',1);
  if (commandErrOutput.length() > 0)
  {
        KMessageBox::error(this,
                            i18n("There was an error changing the device's state. You will have to do it manually."),
                            i18n("Couldn't change device state"));

  }
  else if (commandOutput == "\n<!-- GST: end of request -->")
  {
    QListViewItem *item = klvCardList->findItem(currentDevice,0,ExactMatch);
    if (item != NULL)
    {
      dev = getDeviceInfo(currentDevice);
      if (!dev->isActive())
      {
        dev->setActive(true);
        if (dev->getType() == "ethernet")
          item->setPixmap(0,activeEthernetDeviceImg);
        else if (dev->getType() == "wireless")
          item->setPixmap(0,activeWirelessDeviceImg);          
          
        item->setText(3,i18n("Enabled"));
        item->setPixmap(3,locate("icon","default.kde/16x16/actions/ok.png"));        
//        config->runDetectionScript(netInfo->getPlatformName());
        config->listIfaces(netInfo->getPlatformName());
//        item->setText(1,dev->getIpAddress());
      }
      else
      {
        dev->setActive(false);
        if (dev->getType() == "ethernet")
          item->setPixmap(0,inactiveEthernetDeviceImg);
        else if (dev->getType() == "wireless")
          item->setPixmap(0,inactiveWirelessDeviceImg);          
        
        item->setText(3,i18n("Disabled"));
        item->setPixmap(3,locate("icon","default.kde/16x16/actions/stop.png"));        
        if (dev->getBootProto().lower() == "dhcp")
          item->setText(1,"");
      }
      enableButtonsSlot();
    }
  }
}
/** Returns a list of strings of all the configured devices. */
QStringList KNetworkConf::getDeviceList(){
  QStringList list;
  KNetworkInterface * device;
  for (device = deviceList.first(); device; device = deviceList.next())
  {
    list.append(device->getDeviceName());
  }
  return list;
}
/** No descriptions */
bool KNetworkConf::valuesChanged(KNetworkInterface *dev,
                                  QString bootProto,
                                  QString netmask,
                                  QString ipAddr,
                                  bool onBoot,
                                  QString desc,
				  QString broadcast){
  if ((dev->getBootProto() != bootProto) ||
      (dev->getNetmask() != netmask) ||
      (dev->getIpAddress() != ipAddr) ||
      ((dev->getOnBoot() == "yes") && !(onBoot)) ||
      ((dev->getOnBoot() == "no") && (onBoot)) ||
      (dev->getDescription() != desc) ||
      (dev->getBroadcast() != broadcast))
    return true;
  else
    return false;
}

/** Sets the QPushButton::autoResize() in true for all buttons. */
void KNetworkConf::makeButtonsResizeable(){
  kpbConfigureNetworkInterface->setAutoResize(true);
  kcbGwDevice->setAutoResize(true);
  kpbAddDomainServer->setAutoResize(true);
  kpbEditDomainServer->setAutoResize(true);
  kpbRemoveDomainServer->setAutoResize(true);
  kpbUpButton->setAutoResize(true);
  kpbDownButton->setAutoResize(true);  
  kpbAddKnownHost->setAutoResize(true);
  kpbEditKnownHost->setAutoResize(true);
  kpbRemoveKnownHost->setAutoResize(true);
}

void KNetworkConf::enableInterfaceSlot()
{
  KNetworkInterface *dev = getDeviceInfo(klvCardList->currentItem()->text(0));
  if (dev->isActive())
    changeDeviceState(dev->getDeviceName(),DEVICE_DOWN);
  else
    changeDeviceState(dev->getDeviceName(),DEVICE_UP);
}

void KNetworkConf::disableInterfaceSlot()
{
  KNetworkInterface *dev = getDeviceInfo(klvCardList->currentItem()->text(0));
  if (dev->isActive())
    changeDeviceState(dev->getDeviceName(),DEVICE_DOWN);
  else
    changeDeviceState(dev->getDeviceName(),DEVICE_UP);
}

/** Adds a new host to the KListView that has the known hosts. */
void KNetworkConf::addKnownHostSlot(){
  KAddKnownHostDlg dlg(this,0);
  dlg.setCaption(i18n("Add a new Static Host"));
  QString aliases;
  
  dlg.exec();
  
  if (!dlg.kleIpAddress->text().isEmpty() && dlg.klbAliases->firstItem() > 0 )
  {
    QListViewItem * item = new QListViewItem( klvKnownHosts, 0 );

    item->setText(0,dlg.kleIpAddress->text());
    
    for ( int i = 0; i < dlg.klbAliases->count(); i++ ) 
      aliases += dlg.klbAliases->text(i) + " ";
    
    item->setText(1,aliases);
    enableApplyButtonSlot();
  }
}

/** Removes a known host from the list view */
void KNetworkConf::removeKnownHostSlot()
{
  if (klvKnownHosts->currentItem() != 0)
  {
    klvKnownHosts->removeItem(klvKnownHosts->currentItem());
    enableApplyButtonSlot();
  }
}

/** Edits the info about a known host. */
void KNetworkConf::editKnownHostSlot()
{
  KAddKnownHostDlg dlg(this,0);
  dlg.setCaption(i18n("Edit Static Host"));
  QListViewItem *item = klvKnownHosts->currentItem();
  dlg.kleIpAddress->setText(item->text(0));
  
  QStringList aliases = QStringList::split( " ", item->text(1) );
  //dlg.klbAliases->insertStringList(aliases);
  
  qDebug("->" + aliases.join(""));
  int n = 0;
  for ( QStringList::Iterator it = aliases.begin(); it != aliases.end(); ++it, n++ ) 
  {
    QString alias = *it;
    dlg.klbAliases->insertItem(alias,n);
  }      
  
  dlg.exec();
  
  QString _aliases;
  if (!dlg.kleIpAddress->text().isEmpty() && dlg.klbAliases->firstItem() > 0 )
  {
    QListViewItem * item = klvKnownHosts->currentItem();

    item->setText(0,dlg.kleIpAddress->text());
    
    for ( int i = 0; i < dlg.klbAliases->count(); i++ ) 
      _aliases += dlg.klbAliases->text(i) + " ";
    
    item->setText(1,_aliases);
    enableApplyButtonSlot();
  }
  
}

/** Shows the main window after the network info has been loaded. */
void KNetworkConf::showMainWindow()
{
  this->show();
}
/** No descriptions */
void KNetworkConf::readFromStdErrUpDown()
{
  commandErrOutput.append(procDeviceState->readStderr());
}
/** Sees if a device is active or not in the ifconfig output. Not very nice, but it works. Inthe future, this has to be managed by gst. */
bool KNetworkConf::isDeviceActive(QString device, QString ifconfigOutput){
    QString temp = ifconfigOutput.section(device,1,1);
    if (temp == "")
      return false;
    else
    {  
      QString temp2 = temp.section("UP",0,0); //two firts lines of the device info.
      QString temp3 = temp2.section("\n",0,0); //Link encap:Ethernet  HWaddr 00:00:21:C5:99:A0
      QString temp4 = temp2.section("\n",1,1); //inet addr:192.168.1.1  Bcast:192.255.255.255  Mask:255.0.0.0
      temp3 = temp3.stripWhiteSpace();
      temp4 = temp4.stripWhiteSpace();
      QString temp5 = temp3.section(" ",4,4); //00:00:21:C5:99:A0
      QString temp6 = temp4.section(" ",1,1); // addr:192.168.1.1
      temp6 = temp6.section(":",1,1); //192.168.1.1
      QString temp7 = temp4.section(" ",3,3); //Bcast:192.255.255.255
      temp7 = temp7.section(":",1,1); //192.255.255.255
      QString temp8 = temp4.section(" ",5,5); // Mask:255.0.0.0
      temp8 = temp8.section(":",1,1); //255.0.0.0

      //If the ip address is empty it must be a dhcp interface, so fill these fields:
      if (temp6.isEmpty())
        return false;
    }
    return true;
}
void KNetworkConf::setReadOnlySlot(bool state)
{
  state = !state;
  gbDefaultGateway->setEnabled(state);
  kleDomainName->setEnabled(state);
  kleHostName->setEnabled(state);
  gbDNSServersList->setEnabled(state);
  gbKnownHostsList->setEnabled(state);
  klvCardList->setEnabled(state);
  kpbUpButton->setEnabled(state);
  kpbDownButton->setEnabled(state);  
  kpbConfigureNetworkInterface->setEnabled(state);
}

/*Shows a context menu when right-clicking in the interface list*/
  void KNetworkConf::showInterfaceContextMenuSlot(KListView* lv, QListViewItem* lvi, const QPoint& pt)
  {

    KPopupMenu *context = new KPopupMenu( this );
    Q_CHECK_PTR( context );
    context->insertItem( "&Enable Interface", this, SLOT(enableInterfaceSlot()));
    context->insertItem( "&Disable Interface", this, SLOT(disableInterfaceSlot()));
    QListViewItem *item = klvCardList->currentItem();
    QString currentDevice = item->text(0);
    KNetworkInterface *dev = getDeviceInfo(currentDevice);
     
     if (dev->isActive())
     {
       context->setItemEnabled(0,false);          
       context->setItemEnabled(1,true);
     }
     else
     {
       context->setItemEnabled(0,true);     
       context->setItemEnabled(1,false);            
     }         
    context->insertSeparator(2);
    context->insertItem( "&Configure Interface", this, SLOT(configureDeviceSlot()));
    context->popup(pt);
    //context->insertItem( "About &Qt", this, SLOT(aboutQt()) );
  }
  
   void KNetworkConf::enableSignals()
   {
     connect(kleDefaultRoute,SIGNAL(textChanged(const QString&)),this,SLOT(enableApplyButtonSlot(const QString&)));
     connect(kleDomainName,SIGNAL(textChanged(const QString&)),this,SLOT(enableApplyButtonSlot(const QString&)));
     connect(kleHostName,SIGNAL(textChanged(const QString&)),this,SLOT(enableApplyButtonSlot(const QString&)));
   }