/***************************************************************************
 *
 * knetworkmanager_dbus-dispatch.cpp - A NetworkManager frontend for KDE 
 *
 * Copyright (C) 2005, 2006 Novell, Inc.
 *
 * Author: Timo Hoenig <thoenig@suse.de>, <thoenig@nouse.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 **************************************************************************/
#include <kdebug.h>

#include "knetworkmanager.h"
#include "knetworkmanager-dbus.h"
#include "knetworkmanager-devicestore_dbus.h"
#include "knetworkmanager-nminfo_dbus.h"
#include "knetworkmanager-vpn_dbus.h"
#include "knetworkmanager-state_dbus.h"

const char * UPDATE_DEVICE_SIGNALS[] = { "DeviceStrengthChanged", "DeviceCarrierOn", "DeviceCarrierOff", "DeviceAdded",  "DeviceNoLongerActive", "DeviceNowActive", "DeviceActivating" };
const int UPDATE_DEVICE_SIGNAL_COUNT = 7;

bool handleUpdateDeviceSignal( DBusMessage * msg );

DBusHandlerResult
DBusConnection::networkManagerInfoMessageHandler (DBusConnection* con, DBusMessage* msg, void* /*data*/)
{
//	const char*  objectpath = dbus_message_get_path   (msg);
	const char*  member     = dbus_message_get_member (msg);
	DBusMessage* reply      = NULL; 
	bool         handled    = true;

//	printf ("NMI>> method %s, path %s\n", member, objectpath);
	kdDebug() << k_funcinfo << member << endl;

	if (strcmp ("getKeyForNetwork", member) == 0) {
		reply = NetworkManagerInfoDBus::getKeyForNetwork (msg);
	} else if (strcmp ("cancelGetKeyForNetwork", member) == 0) {
		printf ("networkManagerInfoMessageHandler: cancelGetKeyForNetwork\n");
		/* FIXME nothing to do here? */
	} else if (strcmp ("getNetworks", member) == 0) {
		reply = NetworkManagerInfoDBus::getNetworksMessage (msg);
	} else if (strcmp ("getNetworkProperties", member) == 0) {
		reply = NetworkManagerInfoDBus::getNetworkProperties (msg);
	} else if (strcmp ("updateNetworkInfo", member) == 0) {
		NetworkManagerInfoDBus::updateNetworkInfo (msg);
	} else if (strcmp ("getVPNConnections", member) == 0) {
		reply = NetworkManagerInfoDBus::getVPNConnections (msg);
	} else if (strcmp ("getVPNConnectionProperties", member) == 0) {
		reply = NetworkManagerInfoDBus::getVPNConnectionProperties (msg);
	} else if (strcmp ("getVPNConnectionVPNData", member) == 0) {
		reply = NetworkManagerInfoDBus::getVPNConnectionData (msg);
	} else if (strcmp ("getVPNConnectionRoutes", member) == 0) {
		reply = NetworkManagerInfoDBus::getVPNConnectionRoutes (msg);
	} else {
		handled = false;
	}

	if (reply) {
		dbus_connection_send (con, reply, NULL);
		dbus_message_unref (reply);
	}

	return (handled ? DBUS_HANDLER_RESULT_HANDLED : DBUS_HANDLER_RESULT_NOT_YET_HANDLED);
}

DBusHandlerResult
DBusConnection::filterFunction (DBusConnection* /*con*/, DBusMessage* msg, void* /*data*/)
{
//	const char* objectpath  = dbus_message_get_path      (msg);
//	const char* member      = dbus_message_get_member    (msg);
//	const char* interface   = dbus_message_get_interface (msg);
	bool        handled     = true;

	if (dbus_message_is_signal (msg, DBUS_INTERFACE_LOCAL, "Disconnected")) {
		DBusConnection::triggerReconnect ();
	}
	/* Signal indicating that NetworkManager's D-Bus interface owner has changed 
	 *
	 * D-Bus signal: NameOwnerChanged
	 */
	else if (dbus_message_is_signal (msg, DBUS_INTERFACE_DBUS, "NameOwnerChanged")) {
		char* service;
		char* old_owner;
		char* new_owner;

		if (dbus_message_get_args (msg, NULL, DBUS_TYPE_STRING, &service,
						      DBUS_TYPE_STRING, &old_owner,
						      DBUS_TYPE_STRING, &new_owner, DBUS_TYPE_INVALID)) {
			if (strcmp (service, NM_DBUS_SERVICE) == 0) {
				bool old_owner_good = (old_owner && (strlen (old_owner) > 0));
				bool new_owner_good = (new_owner && (strlen (new_owner) > 0));

				if (!old_owner_good && new_owner_good)	{
					DeviceStoreDBus::populateStore ();
				} else if (old_owner_good && !new_owner_good) {
					DeviceStoreDBus::clearStore ();
				}
			}
		}
	}

	/* Signal indicating that the state of NetworkManager has changed.
	 *
	 * D-Bus signal: NM_DBUS_SIGNAL_STATE_CHANGE
	 */

	else if (dbus_message_is_signal (msg, NM_DBUS_INTERFACE, NM_DBUS_SIGNAL_STATE_CHANGE)) {
		NMState state = NM_STATE_UNKNOWN;

		if (dbus_message_get_args (msg, NULL, DBUS_TYPE_UINT32, &state, DBUS_TYPE_INVALID)) {
			StateDBus::setState (state);
		} 
	}

	/* Signals regarding device changes.
	 *
	 * D-Bus signals: DeviceActivating,
	 *		  DeviceAdded,
	 *		  DeviceCarrierOff,
	 *		  DeviceCarrierOn,
	 *		  DeviceNoLongerActive,
	 *		  DeviceNowActive,
	 *		  DeviceStrengthChanged
	 */
	else if ( handleUpdateDeviceSignal( msg ) ) {
		;
	}

	/* Signal regarding removal of devices.
	 *
	 * D-Bus signal: DeviceRemoved
	 *
	 */

	else if (dbus_message_is_signal (msg, NM_DBUS_INTERFACE, "DeviceRemoved")) {
		char* obj_path = NULL;
		if (dbus_message_get_args (msg, NULL, DBUS_TYPE_OBJECT_PATH, &obj_path, DBUS_TYPE_INVALID)) {
			DeviceStoreDBus::removeDevice (obj_path);	
		}
	}

	/* Signals regarding wireless devices.
	 *
	 * D-Bus signals: WirelessNetworkAppeared
	 */
	else if (dbus_message_is_signal (msg, NM_DBUS_INTERFACE, "WirelessNetworkAppeared")) {
		char* obj_path = NULL;
		char* net_path = NULL;

		if (dbus_message_get_args (msg, NULL, DBUS_TYPE_OBJECT_PATH, &obj_path,
						      DBUS_TYPE_OBJECT_PATH, &net_path, DBUS_TYPE_INVALID)) {
			DeviceStoreDBus::updateNetwork (obj_path, net_path, NULL, "WirelessNetworkAppeared");
		}
	}

	/* Signals regarding wireless devices.
	 *
	 * D-Bus signals: WirelessNetworkDisappeared
	 */
	else if (dbus_message_is_signal (msg, NM_DBUS_INTERFACE, "WirelessNetworkDisappeared")) {
		char* obj_path = NULL;
		char* net_path = NULL;

		if (dbus_message_get_args (msg, NULL, DBUS_TYPE_OBJECT_PATH, &obj_path,
						      DBUS_TYPE_OBJECT_PATH, &net_path, DBUS_TYPE_INVALID)) {
			DeviceStoreDBus::removeNetwork (obj_path, net_path);
		}

	}

	/* Signal for signal strength change of wireless devices.
	 *
	 * D-Bus signal: WirelessNetworkStrengthChanged
	 */
	else if (dbus_message_is_signal (msg, NM_DBUS_INTERFACE, "WirelessNetworkStrengthChanged")) {
		char* obj_path = NULL;
		char* net_path = NULL;
		int   strength = -1;
		
		if (dbus_message_get_args (msg, NULL, DBUS_TYPE_OBJECT_PATH, &obj_path,
						      DBUS_TYPE_OBJECT_PATH, &net_path,
						      DBUS_TYPE_INT32,       &strength, DBUS_TYPE_INVALID)) {
			DeviceStoreDBus::updateNetworkStrength (obj_path, net_path, strength);
		}
	}

	/* Signal for activation stage of a connection attempt.
	 *
	 * D-Bus signal: DeviceActivationStage
	 */
	else if (dbus_message_is_signal (msg, NM_DBUS_INTERFACE, "DeviceActivationStage")) {
		char*      obj_path  = NULL;
		NMActStage act_stage = NM_ACT_STAGE_UNKNOWN;

		if (dbus_message_get_args (msg, NULL, DBUS_TYPE_OBJECT_PATH, &obj_path,
				                      DBUS_TYPE_UINT32,      &act_stage, DBUS_TYPE_INVALID)) {
			DeviceStoreDBus::updateActivationStage (obj_path, act_stage);
		}
						      
	}
	
	/* Signal emitted if the activation of a device failed.
	 *
	 * D-Bus signal: DeviceActivationFailed
	 */
	else if (dbus_message_is_signal (msg, NM_DBUS_INTERFACE, "DeviceActivationFailed")) {
		char* dev_path = NULL;
		char* net_path = NULL;

		/* NM FIXME: NetworkManager does not invalidate the devices by asking us to update the device
		 * and/or network which failed.  Hence, the 'active' flag for devices and networks is out of sync.
		 */

		if (dbus_message_get_args (msg, NULL, DBUS_TYPE_OBJECT_PATH, &dev_path,
				                      DBUS_TYPE_OBJECT_PATH, &net_path, DBUS_TYPE_INVALID)) {
			DeviceStoreDBus::updateNetwork (dev_path, net_path, ""); //TODO Do we need this? It will be called from updateDevice
			DeviceStoreDBus::updateDevice (dev_path);
		} else if (dbus_message_get_args (msg, NULL, DBUS_TYPE_OBJECT_PATH, &dev_path, DBUS_TYPE_INVALID)) {
			DeviceStoreDBus::updateDevice (dev_path);
		}
	}

	else if (dbus_message_is_signal (msg, NM_DBUS_INTERFACE_VPN, "VPNConnectionAdded") ||
	         dbus_message_is_signal (msg, NM_DBUS_INTERFACE_VPN, "VPNConnectionUpdate")) {
			const char* name = NULL;
			if (dbus_message_get_args (msg, NULL, DBUS_TYPE_STRING, &name, DBUS_TYPE_INVALID)) {
				VPNDBus::updateVPNConnection (name);
			}
	}

	else if (dbus_message_is_signal (msg, NM_DBUS_INTERFACE_VPN, "VPNConnectionStateChange")) {
			const char*   name = NULL;
			NMVPNActStage act_stage;
			if (dbus_message_get_args (msg, NULL, DBUS_TYPE_STRING, &name,
							      DBUS_TYPE_UINT32, &act_stage, DBUS_TYPE_INVALID)) {
				VPNDBus::updateVPNActivationStage (name, act_stage);
			}
	}

	/* FIXME dbus_request_name does not work during the initialization, workaround */
	else if (dbus_message_is_signal (msg, DBUS_INTERFACE_DBUS, "NameAcquired")) {
		const char* name = NULL;
		if (dbus_message_get_args (msg, NULL, DBUS_TYPE_STRING, &name, DBUS_TYPE_INVALID) && (strcmp (name, NMI_DBUS_SERVICE) != 0)) {
			NetworkManagerInfoDBus::requestName (msg);
		}
	}

	else {

//		printf ("Unhandled message: objectpath='%s' member='%s' interface='%s'\n", objectpath, member, interface);
		handled = false;
	}

	return (handled ? DBUS_HANDLER_RESULT_HANDLED : DBUS_HANDLER_RESULT_NOT_YET_HANDLED);
}

bool handleUpdateDeviceSignal( DBusMessage * msg )
{
	bool matched = false;
	char * obj_path = 0;
	for ( int i = 0; !matched && (i < UPDATE_DEVICE_SIGNAL_COUNT ); ++i )
	{
		if (dbus_message_is_signal (msg, NM_DBUS_INTERFACE, UPDATE_DEVICE_SIGNALS[i] ) ) {
			matched = true;
			if (dbus_message_get_args (msg, NULL, DBUS_TYPE_OBJECT_PATH, &obj_path, DBUS_TYPE_INVALID)) {
				DeviceStoreDBus::updateDevice (obj_path, UPDATE_DEVICE_SIGNALS[i]);
			}
		}
	}
	return matched;
}
