/***************************************************************************
 *
 * knetworkmanager-connection.cpp - A NetworkManager frontend for KDE
 *
 * Copyright (C) 2005, 2006 Novell, Inc.
 *
 * Author: Helmut Schaa <hschaa@suse.de>, <helmut.schaa@gmx.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 **************************************************************************/

/* qt headers */
#include <qvaluelist.h>

/* kde headers */
#include <kdebug.h>
#include <klocale.h>

/* QDbus headers */
#include <dbus/qdbusconnection.h>
#include <dbus/qdbusobjectpath.h>
#include <dbus/qdbusdata.h>
#include <dbus/qdbusdatamap.h>
#include <dbus/qdbusvariant.h>

/* NM headers */
#include <NetworkManager.h>

/* knetworkmanager headers */
#include "knetworkmanager.h"
#include "knetworkmanager-connection.h"
#include "knetworkmanager-connection_dbus.h"
#include "knetworkmanager-connection_secrets_dbus.h"
#include "knetworkmanager-connection_setting.h"
#include "knetworkmanager-nmsettings.h"

using namespace ConnectionSettings;

namespace ConnectionSettings
{

class ConnectionPrivate
{
	public:
		ConnectionPrivate(Connection* parent)
    {
			conn_dbus = new ConnectionDBus(parent);
			conn_secrets_dbus = new ConnectionSecretsDBus(parent);
		}
		~ConnectionPrivate() {}

		QDBusObjectPath        obj_path;
		ConnectionDBus*        conn_dbus;
		ConnectionSecretsDBus* conn_secrets_dbus;
		QValueList<ConnectionSetting*> settings;
		QString                specific_object;
};

}

/*
	class Connection
*/
Connection::Connection()
{
	d = new ConnectionPrivate(this);

	NMSettings* nmSettings = NMSettings::getInstance();
	d->obj_path = nmSettings->getObjPathForConnection();

	QDBusConnection conn = QDBusConnection::systemBus();

	if (!registerObject(conn, objectPath()))
		kdError() << "registerobjectpath failed" << endl;

	// get notified whenever NM needs a secret
	connect(d->conn_secrets_dbus, SIGNAL(SecretsNeeded(const QString&, const QStringList&, bool)), this, SLOT(slotSecretsNeeded(const QString&, const QStringList&, bool)));
}

Connection::~Connection()
{
	for (QValueList<ConnectionSetting*>::Iterator it= d->settings.begin(); it != d->settings.end(); ++it)
	{
		delete (*it);
		*it = NULL;
	}
	delete d;
}

ConnectionSetting*
Connection::getSetting(const QString& type) const
{
	// find a setting by its type
	for (QValueList<ConnectionSetting*>::ConstIterator it = d->settings.begin(); it != d->settings.end(); ++it)
	{
		if ((*it)->getType() == type)
			return (*it);
	}
	return NULL;
}

QValueList<ConnectionSetting*>
Connection::getSettings() const
{
	return d->settings;
}

void
Connection::appendSetting(ConnectionSetting* setting)
{
	// that's our setting now :)
	d->settings.append(setting);
	connect(setting, SIGNAL(validityChanged()), this, SLOT(slotSettingValidityChanged()));
}

void
Connection::setSpecificObject(const QString& obj_path)
{
	d->specific_object = obj_path;
}

QString
Connection::getSpecificObject() const
{
	return d->specific_object;
}

QDBusObjectPath
Connection::getObjectPath() const
{
	return d->obj_path;
}

QString
Connection::objectPath() const
{
	return d->obj_path;
}

bool
Connection::isValid() const
{
	bool retval = true;
	// check if every enabled setting is valid
	for (QValueList<ConnectionSetting*>::ConstIterator it = d->settings.begin(); it != d->settings.end(); ++it)
	{
		if ((*it)->getEnabled())
			retval &= (*it)->isValid();
	}
	return retval;
}

void
Connection::slotSecretsNeeded(const QString& setting_name, const QStringList& hints, bool request_new)
{
	printf("Connection::slotSecretsNeeded %s, new: %s\n", setting_name.ascii(), request_new ? "yes" : "no");
	ConnectionSetting* setting = getSetting(setting_name);

	if (!setting)
	{
		// send an error to NM
		d->conn_secrets_dbus->SendGetSecretsReply(NULL);
	}
	else
	{
		if (!request_new && setting->hasSecrets())
		{
			// if the setting has secrets just send them
			d->conn_secrets_dbus->SendGetSecretsReply(setting);
		}
		else
		{
			// NetworkManager asks for new secrets, ask user for new secrets/retry
			emit SecretsNeeded(this, setting, hints, request_new);
		}
	}
}

void
Connection::slotSecretsProvided(ConnectionSetting* setting)
{
	if (!setting)
	{
		// send an error to NM
		d->conn_secrets_dbus->SendGetSecretsError();
	}
	else
	{
		// if we have the secrets already send them to NM
		d->conn_secrets_dbus->SendGetSecretsReply(setting);
	}

}

void
Connection::slotSecretsError()
{
	d->conn_secrets_dbus->SendGetSecretsError();
}

QDBusObjectBase*
Connection::createInterface(const QString& interfaceName)
{
	// the interfaces are already created, just return the right one
	if (interfaceName == NM_DBUS_IFACE_SETTINGS_CONNECTION)
		return d->conn_dbus;

	if (interfaceName == NM_DBUS_IFACE_SETTINGS_CONNECTION_SECRETS)
		return d->conn_secrets_dbus;

	return NULL;
}

QString
Connection::getType()
{
	return QString::null;
}

void
Connection::slotSettingValidityChanged()
{
	emit validityChanged();
}

void
Connection::slotAboutToBeRemoved()
{
	d->conn_dbus->slotAboutToBeRemoved();
}
void
Connection::slotUpdated()
{
	d->conn_dbus->slotUpdated();
}

void
Connection::updateSettings(Connection* conn)
{
	QValueList<ConnectionSetting*> settings = conn->getSettings();
	// copy all settings over to the new connection
	for (QValueList<ConnectionSetting*>::Iterator it = settings.begin(); it != settings.end(); ++it)
	{
		ConnectionSetting* other_setting = *it;
		ConnectionSetting* my_setting = getSetting(other_setting->getType());
		if (my_setting)
		{
			my_setting->fromMap(other_setting->toMap());
			my_setting->fromSecretsMap(other_setting->toSecretsMap(false));
		}
		else
		{
			// should not happen
		}
	}
}

#include "knetworkmanager-connection.moc"
