/***************************************************************************
 *
 * knetworkmanager-device.cpp - A NetworkManager frontend for KDE 
 *
 * Copyright (C) 2005, 2006 Novell, Inc.
 *
 * Author: Timo Hoenig     <thoenig@suse.de>, <thoenig@nouse.net>
 *         Will Stephenson <wstephenson@suse.de>, <wstephenson@kde.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 **************************************************************************/

// NM includes
#include <NetworkManager.h>

// KNM includes
#include "knetworkmanager.h"
#include "knetworkmanager-wireless_device.h"
#include "dbus/accesspointproxy.h"
#include "dbus/wirelessproxy.h"
#include "knetworkmanager-accesspoint.h"

// KDE includes
#include <kdebug.h>

// Qt includes
#include <qhostaddress.h>
#include <qmap.h>
#include <qvaluelist.h>

// Qt-Dbus includes
#include <dbus/qdbusconnection.h>
#include <dbus/qdbuserror.h>
#include <dbus/qdbusobjectpath.h>

class WirelessDevicePrivate
{
public:
	WirelessDevicePrivate(QString service, QString obj_path)
		: nmWireless(service, obj_path)
	{}

	~WirelessDevicePrivate()
	{
		// cleanup the AP-list
		for (QMap<QDBusObjectPath, AccessPoint*>::Iterator it = aps.begin(); it != aps.end(); ++it)
		{
			AccessPoint* p = it.data();
			delete p;
		}
	}

	DBus::WirelessDeviceProxy nmWireless;
	QMap<QDBusObjectPath, AccessPoint *> aps;
};

WirelessDevice::WirelessDevice (const QString & obj_path)
	: Device(obj_path)
{
	d = new WirelessDevicePrivate(NM_DBUS_SERVICE, obj_path);
	d->nmWireless.setConnection(QDBusConnection::systemBus());

	// get notified when the devices properties change
	connect(&(d->nmWireless), SIGNAL(PropertiesChanged(const QMap<QString, QDBusVariant>&)), this, SLOT(slotPropertiesChanged(const QMap<QString, QDBusVariant>&)));
	connect(&(d->nmWireless), SIGNAL(AccessPointAdded(const QDBusObjectPath&)), this, SLOT(slotAccessPointAdded(const QDBusObjectPath&)));
	connect(&(d->nmWireless), SIGNAL(AccessPointRemoved(const QDBusObjectPath&)), this, SLOT(slotAccessPointRemoved(const QDBusObjectPath&)));
}

WirelessDevice::~WirelessDevice ()
{
	delete d;
}

void WirelessDevice::slotPropertiesChanged(const QMap<QString, QDBusVariant>& properties)
{
	// TODO: optimization: emit different signals for each property
	emit propertiesChanged();
}

void WirelessDevice::slotAccessPointAdded(const QDBusObjectPath& obj_path)
{
	AccessPoint* ap = 0;

	if ( !d->aps.contains(obj_path)) {
		// we do not know this AP yet, add it to the list
		ap = new AccessPoint(obj_path, this, "access_point_object");
		d->aps.insert(obj_path, ap);
	}
	else
		ap = d->aps[obj_path];

	// notify about the new accesspoint
	emit accessPointAdded(ap);
}

void WirelessDevice::slotAccessPointRemoved(const QDBusObjectPath& obj_path)
{
	AccessPoint* ap = 0;

	if ( d->aps.contains(obj_path)) {
		ap = d->aps[obj_path];

		// notify about the AP removal
		emit accessPointRemoved(obj_path);

		// remove the appropriate AP from our list
		d->aps.remove(obj_path);
		delete ap;
	}
	else
	{
		// nothing we can do here
		kdDebug() << k_funcinfo << "NM requests removal of unknown AP " << obj_path << endl;
	}
}

QValueList<AccessPoint*> WirelessDevice::accessPoints()
{
	// update our AP list
	if (d->aps.isEmpty()) {
		updateAPList();
	}

	return d->aps.values();
}

void WirelessDevice::updateAPList()
{
	QDBusError err;
	QValueList<QDBusObjectPath> aps;

	// get the APs from NM
	if (d->nmWireless.GetAccessPoints(aps, err))
	{
        // for each AP
		for (QValueList<QDBusObjectPath>::iterator it = aps.begin(); it != aps.end(); ++it)
		{
			// create an AP-object for each objectpath
			if (d->aps.contains(*it)) {
				continue;
			}
			AccessPoint* ap = new AccessPoint(*it, this, "access_point_object");
			d->aps.insert(*it, ap);
		}
	}
	else
		kdWarning() << k_funcinfo << "Could not get a list of wireless accesspoints over DBus." << endl;

}

QValueList<AccessPoint*> WirelessDevice::accessPointsForEssid( QByteArray essid)
{
	QValueList<AccessPoint*> aps;
	// the DBus proxy is shared
	for (QMap<QDBusObjectPath, AccessPoint*>::Iterator it = d->aps.begin(); it != d->aps.end(); ++it)
	{
		AccessPoint * ap = it.data();
		if (essid.isNull() || (ap && ap->getSsidByteArray() == essid))
			aps.append(ap);
	}
	return aps;
}

AccessPoint * WirelessDevice::getActiveAccessPoint()
{
	QDBusError err;
	QDBusObjectPath obj_path;

	AccessPoint * ap = 0;
	//fixme, listen to signal and use cached value
	obj_path = d->nmWireless.getActiveAccessPoint(err);
	if (!obj_path.isEmpty()) {
		if (d->aps.contains(obj_path)) {
			ap = d->aps[obj_path];
		} else {
			kdWarning() << k_funcinfo << "No object for active access point found!" << endl;
		}
	}
	return ap;
}

Q_UINT32 WirelessDevice::getWirelessCapabilities() const
{
	QDBusError err;
	return d->nmWireless.getWirelessCapabilities(err);
}

Q_UINT32 WirelessDevice::getBitrate() const
{
	QDBusError err;
	return d->nmWireless.getBitrate(err);
}

#include "knetworkmanager-wireless_device.moc"
