/*
 * Copyright (c) 2001-2003 The Trustees of Indiana University.  
 *                         All rights reserved.
 * Copyright (c) 1998-2001 University of Notre Dame. 
 *                         All rights reserved.
 * Copyright (c) 1994-1998 The Ohio State University.  
 *                         All rights reserved.
 * 
 * This file is part of the LAM/MPI software package.  For license
 * information, see the LICENSE file in the top level directory of the
 * LAM/MPI source distribution.
 * 
 * $HEADER$
 *
 * $Id: ssi_boot_rsh_inetexec.c,v 1.10.2.3 2003/10/11 19:04:03 brbarret Exp $
 * 
 */

#include <lam_config.h>
#include <lam-ssi-boot-rsh-config.h>

#include <errno.h>
#include <stdio.h>
#include <string.h>

#if defined(HAVE_STRINGS_H) && HAVE_STRINGS_H
#include <strings.h>
#endif

#include <stdlib.h>
#include <ctype.h>
#include <pwd.h>
#include <unistd.h>
#include <sys/time.h>
#include <sys/types.h>

#if defined(HAVE_SYS_SELECT_H) && HAVE_SYS_SELECT_H
#include <sys/select.h>
#endif

#include <sys/wait.h>
#include <sys/param.h>

#include <args.h>
#include <all_opt.h>
#include <etc_misc.h>
#include <typical.h>
#include <sfh.h>
#include <lamdebug.h>
#include <lam-ssi-boot-rsh.h>


int
lam_ssi_boot_rsh_inetexec(struct lamnode *node, char **argv)
{
  char **cmdv;			/* rsh command argv */
  int cmdc;			/* rsh command argc */
  int fl_csh;			/* csh-flavoured flag */
  int fl_bash;			/* bash-flavoured flag */
  int i;			/* favorite counter */
  char printable[BUFSIZ];	/* command to exec */
  char remote_host[BUFSIZ];	/* username@hostname */
  char *cmdv0;			/* Copy of cmdv[0] */
  char shellpath[BUFSIZ];	/* return of 'echo $SHELL' */
  struct passwd *p;
  int fl_fast = ao_taken(lam_ssi_boot_optd, "b");
  int fl_no_n = 0;
  int fl_no_profile = 0;
  char *tmp;

  /* Special command line options */

  if (getenv("LAM_MPI_SSI_boot_base_no_n") != NULL)
    fl_no_n = 1;
  if (getenv("LAM_MPI_SSI_boot_base_no_profile") != NULL)
    fl_no_profile = 1;
  if (getenv("LAM_MPI_SSI_boot_rsh_fast") != NULL)
    fl_fast = 1;

  /* Get the user's shell by executing 'echo $SHELL' on remote
     machine, but only if they asked for it via not specifying
     fl_fast.  Since *only* the bourne and korn shells require this,
     and *most* users don't use plain old bourne/korn (bash does *not*
     require explicitly running .profile), only do this if
     specifically requested, 'cause we can ditch one of the 2 rsh's,
     and double the speed of things like lamboot, recon, and wipe.  */

  /* Build the username/hostname to pass to the error routine */

  memset(remote_host, 0, BUFSIZ);
  if (node->lnd_uname != NULL) 
    snprintf(remote_host, BUFSIZ, "%s@%s", node->lnd_uname, node->lnd_hname);
  else
    snprintf(remote_host, BUFSIZ, "%s", node->lnd_hname);
  remote_host[BUFSIZ - 1] = '\0';

  /* If we don't need to look for the .profile, it's a lot easier */

  if (fl_no_profile) {
    fl_csh = 1;
    fl_bash = 0;
  } 

  /* If we didn't specify to be "fast", look for .profile */

  else if (!fl_fast) {
    cmdv = sfh_argv_dup(lam_ssi_boot_rsh_agent);
    cmdc = sfh_argv_count(lam_ssi_boot_rsh_agent);
    sfh_argv_add(&cmdc, &cmdv, node->lnd_hname);
    if (!fl_no_n)
      sfh_argv_add(&cmdc, &cmdv, "-n");
    if (node->lnd_uname != NULL) {
      sfh_argv_add(&cmdc, &cmdv, "-l");
      sfh_argv_add(&cmdc, &cmdv, node->lnd_uname);
    }
    sfh_argv_add(&cmdc, &cmdv, "echo $SHELL");

    /* Assemble a string for possible error usage -- argv gets freed
       in ioexecvp */

    printable[0] = '\0';
    for (i = 0; i < cmdc; i++) {
      strncat(printable, cmdv[i], BUFSIZ);
      strncat(printable, " ", BUFSIZ);
    }

    if (lam_ssi_boot_verbose >= 1) {
      tmp = sfh_argv_glue(cmdv, ' ', 0);
      lam_debug(lam_ssi_boot_did,
		"rsh: attempting to execute \"%s\"", tmp);
      free(tmp);
    }

    cmdv0 = strdup(cmdv[0]);
    shellpath[sizeof(shellpath) - 1] = '\0';
    if (lam_ssi_boot_base_ioexecvp(cmdv, 0, shellpath,
				   sizeof(shellpath) - 1)) {
      if (errno == EFAULT)
	show_help("boot", "remote-stderr", remote_host, cmdv0,
		  "echo $SHELL", printable, NULL);
      else
	show_help("boot", "remote-shell-fail", remote_host, cmdv0,
		  "echo $SHELL", printable, NULL);

      free(cmdv0);
      return (LAMERROR);
    }

    /* Did we get valid output? */
    
    if (strlen(shellpath) == 0) {
      show_help("boot", "no-shell", remote_host, cmdv0, "echo $SHELL",
		printable, NULL);
      free(cmdv0);
      return (LAMERROR);
    }
    free(cmdv0);
    
    /* Perl chomp */
    
    if (shellpath[strlen(shellpath) - 1] == '\n')
	shellpath[strlen(shellpath) - 1] = '\0';
    if (lam_ssi_boot_verbose >= 1)
      lam_debug(lam_ssi_boot_did, "rsh: remote shell %s", shellpath);
    fl_csh = (strstr(shellpath, "csh") != 0) ? TRUE : FALSE;
    fl_bash = (strstr(shellpath, "bash") != 0) ? TRUE : FALSE;
  }

  /* If we didn't ask for the bourne shell, look up the user's local
     shell and assume they have the same shell on the remote hoted;
     fill in values for fl_csh and fl_bash */

  else {
    p = getpwuid(getuid());
    if (p == NULL)
      return LAMERROR;
    if (lam_ssi_boot_verbose >= 1) {
      lam_debug(lam_ssi_boot_did, 
		"rsh: -b used, assuming same shell on remote nodes");
      lam_debug(lam_ssi_boot_did, "rsh: got local shell %s", 
		p->pw_shell);
    }

    fl_csh = (strstr(p->pw_shell, "csh") != 0) ? TRUE : FALSE;
    fl_bash = (strstr(p->pw_shell, "bash") != 0) ? TRUE : FALSE;
  }

  /* Remotely execute the command */

  cmdv = sfh_argv_dup(lam_ssi_boot_rsh_agent);
  cmdc = sfh_argv_count(lam_ssi_boot_rsh_agent);
  sfh_argv_add(&cmdc, &cmdv, node->lnd_hname);
  if (!fl_no_n)
    sfh_argv_add(&cmdc, &cmdv, "-n");
  if (node->lnd_uname != NULL) {
    sfh_argv_add(&cmdc, &cmdv, "-l");
    sfh_argv_add(&cmdc, &cmdv, node->lnd_uname);
  }

  /* If the user's shell is not based on "csh" or "bash", force the
     interpretation of the user's .profile script in order to
     initialize the paths. This works for "sh" and "ksh".  */

  if (!(fl_csh || fl_bash))
    sfh_argv_add(&cmdc, &cmdv, "(. ./.profile;");
  for (i = 0; argv[i]; ++i)
    sfh_argv_add(&cmdc, &cmdv, argv[i]);
  if (!(fl_csh || fl_bash))
    sfh_argv_add(&cmdc, &cmdv, ")");

  if (lam_ssi_boot_verbose >= 1) {
    tmp = sfh_argv_glue(cmdv, ' ', 0);
    lam_debug(lam_ssi_boot_did,
	      "rsh: attempting to execute \"%s\"", tmp);
    free(tmp);
  }

  /* Build the sample command to pass to the error routine.  Have to
     do this ahead of time, because ioexecvp frees the argv array. */

  printable[0] = '\0';
  for (i = 0; i < cmdc; i++) {
    strncat(printable, cmdv[i], BUFSIZ);
    strncat(printable, " ", BUFSIZ);
  }

  cmdv0 = strdup(cmdv[0]);
  i = lam_ssi_boot_base_ioexecvp(cmdv, 1, (char *) 0, 0);

  /* Do we need to print an error message? */

  if (i) {

    if (errno == EFAULT)
      show_help("boot", "remote-stderr", remote_host, cmdv0,
		argv[0], printable, NULL);
    else
      show_help("boot", "remote-boot-fail", remote_host, cmdv0,
		argv[0], printable, NULL);

    free(cmdv0);
    return (LAMERROR);
  }

  free(cmdv0);
  return 0;
}
