/*
 * Copyright (c) 2001-2003 The Trustees of Indiana University.  
 *                         All rights reserved.
 * Copyright (c) 1998-2001 University of Notre Dame. 
 *                         All rights reserved.
 * Copyright (c) 1994-1998 The Ohio State University.  
 *                         All rights reserved.
 * 
 * This file is part of the LAM/MPI software package.  For license
 * information, see the LICENSE file in the top level directory of the
 * LAM/MPI source distribution.
 * 
 * $HEADER$
 *
 * $Id: ssi_rpi_open.c,v 1.19 2003/04/11 13:47:38 jsquyres Exp $
 *
 */

#include <lam_config.h>

#include <string.h>
#include <stdlib.h>

#include <lam-ssi.h>
#include <lam-ssi-rpi.h>
#include <etc_misc.h>
#include <lamdebug.h>
#include <all_list.h>


/*
 * The following file was created by configure.  It contains extern
 * statements and the definition of an array of pointers to each
 * module's public lam_ssi_rpi_t struct.
 */

#include "lam-ssi-rpi-modules.h"


/*
 * Global variables
 */

int lam_ssi_rpi_verbose = -1;
int lam_ssi_rpi_did = -1;
lam_ssi_rpi_actions_t lam_ssi_rpi;
lam_ssi_rpi_t lam_ssi_rpi_base_module;
LIST *lam_ssi_rpi_base_available = NULL;


/*
 * Local variables
 */

static lam_debug_stream_info_t lds = {
  0, -1, -1, NULL, "rpi:", -1, -1, -1, -1, "ssi-rpi.txt"
};


/*
 * Local functions
 */

static int check_specific_module(OPT *aod, char *name);
static int check_all_modules(OPT *aod);


int
lam_ssi_rpi_base_open(OPT *aod)
{
  int i;
  char *env;

  /* Call to base SSI open */

  lam_ssi_base_open(aod);

  /* Setup verbosity for this kind */

  lam_ssi_base_set_verbose("LAM_MPI_SSI_rpi_verbose", &lds, aod,
			   &lam_ssi_rpi_verbose, &lam_ssi_rpi_did);
  if (lam_ssi_rpi_verbose >= 1)
    lam_debug(lam_ssi_rpi_did, " Opening, verbosity:%d", lam_ssi_rpi_verbose);

  /* Initialize the list */

  lam_ssi_rpi_base_available = 
    al_init(sizeof(lam_ssi_module_t), lam_ssi_base_module_compare);
  if (lam_ssi_rpi_base_available == NULL) {
    show_help(NULL, "system-call-failed", "malloc", NULL);
    return LAMERROR;
  }

  /* See if a specific boot module was requested */
  
  if ((env = getenv("LAM_MPI_SSI_rpi")) != NULL)
    i = check_specific_module(aod, env);

  /* Nope -- a specific one was not selected.  Go choose one. */
  
  else
    i = check_all_modules(aod);
  
  if (i == LAMERROR) {
    al_free(lam_ssi_rpi_base_available);
    lam_ssi_rpi_base_available = NULL;
    return LAMERROR;
  }
  
  /* Now that that's all done, setup unexpected message buffering */

  if (lam_ssi_rpi_cbuf_init() != 0) {
    al_free(lam_ssi_rpi_base_available);
    lam_ssi_rpi_base_available = NULL;
    return LAMERROR;
  }

  return 0;
}


/*
 * A specific module was selected on the command line.  If a module by
 * that name is found, call its open (if it exists) and query
 * functions.  If they both return happiness, add the module and its
 * meta info to a list for use in later selection.  Otherwise, call
 * its close function and return an error (i.e., don't try to find any
 * other available modules).
 */
static int
check_specific_module(OPT *aod, char *name)
{
  int i;
  int priority, thread_min, thread_max;
  const lam_ssi_t *ls;
  lam_ssi_module_t entry;

  /* In 64 bit mode, this struct can have empty padding */

  LAM_ZERO_ME(entry);

  if (lam_ssi_rpi_verbose > 10)
    lam_debug(lam_ssi_rpi_did, " looking for module named %s", name);
  
  for (i = 0; lam_ssi_rpi_modules[i] != NULL; ++i) {
    ls = &(lam_ssi_rpi_modules[i]->lsr_meta_info);
    if (strcmp(name, ls->ssi_module_name) == 0) {
      if (lam_ssi_rpi_verbose > 10)
	lam_debug(lam_ssi_rpi_did, " opening module %s", ls->ssi_module_name);
      if (ls->ssi_open_module == NULL ||
	  (ls->ssi_open_module != NULL &&
	   ls->ssi_open_module(aod) == 1)) {
	
	/* If it's available, run its init function and see if it
	   wants to run.  And if so, ignore the priority but get its
	   function list. */
	
	if (lam_ssi_rpi_verbose > 10)
	  lam_debug(lam_ssi_rpi_did, " querying module %s", 
		    ls->ssi_module_name);
	if (lam_ssi_rpi_modules[i]->lsr_query(&priority, &thread_min,
                                              &thread_max) == 0) {
          
          /* If we got 0 back, then this module does not want to run.
             Give up in despair. */
	
	  if (ls->ssi_close_module != NULL)
	    ls->ssi_close_module();
	  
	  show_help("rpi-ssi", "selected-module-unavailable", name, NULL);
	  return LAMERROR;
        }
	
	/* Otherwise, we're found the module, and it wants to run.
	   We're done! */

	entry.lsm_priority = priority;
        entry.lsm_thread_min = thread_min;
        entry.lsm_thread_max = thread_max;
        entry.lsm_module = (lam_ssi_t*) lam_ssi_rpi_modules[i];
	break;
      }

      /* Otherwise, this module's init function did not want to run.
	 Abort. */

      else {
	show_help("rpi-ssi", "selected-module-unavailable", name, NULL);
	return LAMERROR;
      }
    }
  }

  /* Did we find one? */

  if (lam_ssi_rpi_modules[i] == NULL) {
    show_help("ssi", "module-not-found", "rpi", name, NULL);
    return LAMERROR;
  }

  /* We found the desired module, and it wanted to be considered.  Add
     it to the list.  It'll be the only entry, but that's what the
     user asked for.  :-) */

  al_insert(lam_ssi_rpi_base_available, &entry);

  return 0;
}


/*
 * Call open on all the available modules (if any exist).  If open
 * returns happiness (or doesn't exist), call the query function to
 * see if that module wants to run.  If it does, add it to the
 * priority list for secondary selection.
 *
 * Return LAMERROR if there are no modules available.
 */
static int
check_all_modules(OPT *aod)
{
  int i, found;
  const lam_ssi_t *ls;
  int priority, thread_min, thread_max;
  lam_ssi_module_t entry;

  /* In 64 bit mode, this struct can have empty padding */

  LAM_ZERO_ME(entry);

  /* Call the open/query functions in every module and see if they
     want to run */

  for (found = i = 0; lam_ssi_rpi_modules[i] != NULL; ++i) {
    ls = &(lam_ssi_rpi_modules[i]->lsr_meta_info);
    if (lam_ssi_rpi_verbose > 0)
      lam_debug(lam_ssi_rpi_did, " opening module %s", ls->ssi_module_name);
    if (ls->ssi_open_module == NULL ||
	(ls->ssi_open_module != NULL &&
	 ls->ssi_open_module(aod) == 1)) {
      
      /* If it's available, run its init function and see if it wants
	 to run.  And if so, what's its priority and function list. */

      if (lam_ssi_rpi_verbose > 0)
	lam_debug(lam_ssi_rpi_did, " querying module %s", 
		  ls->ssi_module_name);
      if (lam_ssi_rpi_modules[i]->lsr_query(&priority, &thread_min, 
                                            &thread_max) == 1) {
	if (lam_ssi_rpi_verbose >= 10)
	  lam_debug(lam_ssi_rpi_did, 
		    " module available: %s, priority: %d", 
		    ls->ssi_module_name, priority);

        /* This module wants to be considered */

        found = 1;
	entry.lsm_priority = priority;
        entry.lsm_thread_min = thread_min;
        entry.lsm_thread_max = thread_max;
        entry.lsm_module = (lam_ssi_t*) lam_ssi_rpi_modules[i];
        al_insert(lam_ssi_rpi_base_available, &entry);
      } else {
	if (lam_ssi_rpi_verbose >= 10)
	  lam_debug(lam_ssi_rpi_did, " module not available: %s",
		    ls->ssi_module_name);
      }
    }
  }

  /* If there were no modules available, it's an error. */

  if (found == 0) {
    if (lam_ssi_rpi_verbose > 1)
      lam_debug(lam_ssi_rpi_did, " No rpi moduless available!");
    show_help("ssi-rpi", "none-available", NULL);
    return LAMERROR;
  }

  return 0;
}
