import re
from linda.debug import dprint
from linda.funcs import run_external_cmd, ExtCmdException
from linda.parser.unixperm import UnixPermParser

class Collector:
    def __init__(self, type, lab, files):
        self.files = {'files': {}, 'dirs': {}, 'elf': []}
        self.output = {'file': {}, 'objdump': {}}
        self.lab = lab
        self._files = files
        self.elfre = re.compile(r'ELF.*(executable|shared object|relocatable)')
        self.bpr = re.compile(r'^(?P<perm>[^ ]+) (?P<owner>[^ ]+).+\d{2}:\d{2}[:\d{2}]? \.?(?P<file>.+?)( (->|link to) .+)?$')
        getattr(self, '%s_parse' % type)()
        self.run_file()
        if type == 'bin':
            self.determine_elf()
            self.run_objdump()

    def bin_parse(self):
        for line in filter(None, self._files.split('\n')):
            bits = re.match(self.bpr, line).groupdict()
            if bits['file'] == '/':
                continue
            if bits['file'].endswith('/') and bits['file'][1:-1]:
                self.files['dirs'][bits['file'][:-1]] = \
                    [UnixPermParser(bits['perm']), bits['owner']]
            else:
                self.files['files'][bits['file']] = UnixPermParser(bits['perm'])

    def src_parse(self):
        src_files = self.files['files'].fromkeys(self._files)
        self.files['files'] = {}
        for file in src_files.keys():
            cur_file = file
            if file.startswith('./'):
                cur_file = file[len('./'):]
            self.files['files'][cur_file] = src_files[file]
                
    def run_file(self):
        lists_to_run = ['']
        for file in map(lambda x: '%s/%s' % (self.lab, x), \
            self.files['files'].keys()):
            if len(lists_to_run[-1]) + len(file) < 65535:
                lists_to_run[-1] += '"%s" ' % file
            else:
                lists_to_run.append('"%s" ' % file)
        file_output = []
        for list in filter(None, lists_to_run):
            try:
                file_output.append(run_external_cmd('file %s' % list))
            except ExtCmdException:
                dprint(_('Failed to run %s') % list, 2)
        for output in file_output:
            for line in output.split('\n'):
                split_line = re.split(r': +', line)
                cur_file = re.sub('%s/' % self.lab, '', split_line[0])
                if len(split_line) == 2:
                    self.output['file'][cur_file] = split_line[1]
                else:
                    self.output['file'][cur_file] = \
                        ': '.join(split_line[1:])
    
    def determine_elf(self):
        for file in self.output['file'].keys():
            if re.search(self.elfre, self.output['file'][file]):
                self.files['elf'].append(file)
    
    def run_objdump(self):
        for file in self.files['elf']:
            self.output['objdump'][file] = \
                run_external_cmd('LC_ALL=C objdump -hpT "%s/%s"' % \
                (self.lab, file), 1)

    def __call__(self, kind, key):
        if hasattr(self, kind):
            if getattr(self, kind).has_key(key):
                return getattr(self, kind)[key]

    def set(self, kind, key, data):
        if hasattr(self, kind):
            getattr(self, kind)[key] = data
        else:
            raise KeyError

