import unittest, os, linda, tempfile, shutil, gzip
from po_count import POFileParser
from linda.collector import Collector
from linda.debug import dprint
from linda.funcs import run_external_cmd
from linda.parser.datavalidator import DataValidator
from linda.parser.unixperm import UnixPermParser
from linda.libchecks import Checks
from sets import Set

linda.checks = Checks()

class FilesCheckBinaryTest(unittest.TestCase):
    def setUp(self):
        execfile('checks/files.py', {'_': __builtins__['_'], 'dprint': \
            dprint, 'DataValidator': DataValidator, 'os': os})
        for klass in linda.checks.registry['binary'][2]:
            self.instance = klass('/tmp', '/tmp/test_0.0.1-1_all.deb', {})
        self.instance.information['dir'] = tempfile.mkdtemp()
        run_external_cmd('tar zxCf %s tests/binary_files.tar.gz' % \
            self.instance.information['dir'])
        self.po_keys = POFileParser('po/en.po')
        
    def tearDown(self):
        shutil.rmtree(self.instance.information['dir'])

    def test_directory_checks(self):
        "Check directory_checks() method of FilesCheck"
        method = getattr(self.instance, 'directory_checks')
        self.instance.information['collector'] = Collector('bin', '/tmp', '')
        os.makedirs(os.path.join(self.instance.information['dir'], \
            'unpacked', 'usr', 'doc'))
        method()
        self.assert_('usr-doc-no-usr-share-doc' in \
            self.instance.err_dict.keys())
        self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
            'usr-doc-no-usr-share-doc'), True)
        shutil.rmtree(os.path.join(self.instance.information['dir'], \
            'unpacked', 'usr', 'doc'))
        self.instance.information['collector'].set('files', 'dirs', \
            {'/usr/share/lala': ['root/root', UnixPermParser('-rwxr-xr-x')]})
        os.makedirs(os.path.join(self.instance.information['dir'], \
            'unpacked', 'usr', 'share', 'lala'))
        method()
        self.assert_('empty-directory-in-pkg' in self.instance.err_dict.keys())
        self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
            'empty-directory-in-pkg'), True)
        shutil.rmtree(os.path.join(self.instance.information['dir'], \
            'unpacked', 'usr', 'share', 'lala'))
        self.instance.information['collector'].set('files', 'dirs', \
            {'/usr/share/doc/.svn/.xvpics': [UnixPermParser('-rwxr-xr-x'), \
            'root/root']})
        os.makedirs(os.path.join(self.instance.information['dir'], \
            'unpacked', 'usr', 'share', 'doc', '.svn', '.xvpics'))
        method()
        expected_errs = Set(('vercon-dir-in-package', 'dot-xvpics-in-package'))
        self.assertEqual(expected_errs.intersection(self.instance.err_dict.keys()), expected_errs)
        for key in expected_errs:
            self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
                key), True)
        shutil.rmtree(os.path.join(self.instance.information['dir'], \
            'unpacked', 'usr', 'share', 'doc', '.svn', '.xvpics'))
        self.instance.err_dict.clear()
        self.instance.information['collector'].set('files', 'dirs', \
            {'/usr/share/doc/lala': [UnixPermParser('-rwxr-xr-x'), \
            'root/root']})
        os.makedirs(os.path.join(self.instance.information['dir'], \
            'unpacked', 'usr', 'share', 'doc', 'lala'))
        f = open(os.path.join(self.instance.information['dir'], 'unpacked', \
            'usr', 'share', 'doc', 'lala', 'a'), 'w')
        f.close()
        method()
        self.assertEqual(self.instance.err_dict.keys(), [])
        self.instance.information['collector'].set('files', 'dirs', \
            {'/usr/share/doc/lala': ['root/root', \
            UnixPermParser('-rwx-w--w-')]})
        method()
        self.assert_('non-standard-dir-perm' in self.instance.err_dict.keys())
        self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
            'non-standard-dir-perm'), True)
        shutil.rmtree(os.path.join(self.instance.information['dir'], \
            'unpacked', 'usr', 'share', 'doc', 'lala'))
        self.instance.information['collector'].set('files', 'dirs', \
            {'/var/named': [UnixPermParser('-rwxr-xr-x'), 'root/root']})
        os.makedirs(os.path.join(self.instance.information['dir'], \
            'unpacked', 'var', 'named'))
        method()
        self.assert_('FSSTND-dir-found' in self.instance.err_dict.keys())
        self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
            'FSSTND-dir-found'), True)
        shutil.rmtree(os.path.join(self.instance.information['dir'], \
            'unpacked', 'var', 'named'))
        self.instance.information['collector'].set('files', 'dirs', \
            {'/var/lala': [UnixPermParser('-rwxr-xr-x'), 'root/root']})
        os.makedirs(os.path.join(self.instance.information['dir'], \
            'unpacked', 'var', 'lala'))
        method()
        self.assert_('non-standard-dir-in-var' in \
            self.instance.err_dict.keys())
        self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
            'non-standard-dir-in-var'), True)
        shutil.rmtree(os.path.join(self.instance.information['dir'], \
            'unpacked', 'var', 'lala'))
        self.instance.err_dict.clear()
        self.instance.information['collector'].set('files', 'dirs', \
            {'/var/lib/games': [UnixPermParser('-rwxr-xr-x'), 'root/root']})
        os.makedirs(os.path.join(self.instance.information['dir'], \
            'unpacked', 'var', 'lib', 'games'))
        method()
        self.assert_('non-standard-dir-in-var' in \
            self.instance.err_dict.keys())
        shutil.rmtree(os.path.join(self.instance.information['dir'], \
            'unpacked', 'var', 'lib', 'games'))
        self.instance.information['collector'].set('files', 'dirs', \
            {'/usr/dict': [UnixPermParser('-rwxr-xr-x'), 'root/root']})
        os.makedirs(os.path.join(self.instance.information['dir'], \
            'unpacked', 'usr', 'dict'))
        method()
        self.assert_('FSSTND-dir-found' in self.instance.err_dict.keys())
        shutil.rmtree(os.path.join(self.instance.information['dir'], \
            'unpacked', 'usr', 'dict'))
        self.instance.information['collector'].set('files', 'dirs', \
            {'/usr/lala': [UnixPermParser('-rwxr-xr-x'), 'root/root']})
        os.makedirs(os.path.join(self.instance.information['dir'], \
            'unpacked', 'usr', 'lala'))
        method()
        self.assert_('non-standard-dir-in-usr' in \
            self.instance.err_dict.keys())
        self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
            'non-standard-dir-in-usr'), True)
        shutil.rmtree(os.path.join(self.instance.information['dir'], \
            'unpacked', 'usr', 'lala'))

    def test_file_check(self):
        "Check file_checks() method of FilesCheck"
        method = getattr(self.instance, 'file_checks')
        self.instance.information['collector'] = Collector('bin', '/tmp', '')
        self.instance.information['collector'].set('files', 'files', \
            {'/opt/kde/lala': UnixPermParser('-rwxr-xr-x')})
        self.instance.information['control'] = {'self': [{}]}
        method()
        self.assert_('file-in-opt' in self.instance.err_dict.keys())
        self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
            'file-in-opt'), True)
        self.instance.information['collector'] = Collector('bin', '/tmp', '')
        self.instance.information['collector'].set('files', 'files', \
            {'/usr/local/bin/lala': UnixPermParser('-rwxr-xr-x')})
        method()
        self.assert_('file-in-usr-local' in self.instance.err_dict.keys())
        self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
            'file-in-usr-local'), True)
        self.instance.information['collector'].set('files', 'files', \
            {'/usr/bin/foo.rej': UnixPermParser('-rw-r--r--')})
        method()
        self.assert_('failed-diff' in self.instance.err_dict.keys())
        self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
            'failed-diff'), True)
        self.instance.information['collector'].set('files', 'files', \
            {'/usr/share/doc/lala/examples/.cvsignore': \
            UnixPermParser('-rw-r--r--')})
        method()
        self.assert_('cvsignore-in-package' in self.instance.err_dict.keys())
        self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
            'cvsignore-in-package'), True)
        self.instance.information['collector'].set('files', 'files', \
            {'/usr/share/examples': UnixPermParser('-rwxr-xr-x')})
        method()
        self.assert_('file-directly-in-u-s' in self.instance.err_dict.keys())
        self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
            'file-directly-in-u-s'), True)
        self.instance.err_dict.clear()
        self.instance.information['collector'].set('files', 'files', \
            {'/usr/games/foo': UnixPermParser('-rwxr-xr-x')})
        method()
        self.assertEqual(self.instance.err_dict.keys(), [])
        self.instance.information['collector'].set('files', 'files', \
            {'/usr/bin/lala': UnixPermParser('lrwxrwxrwx')})
        method()
        self.assertEqual(self.instance.err_dict.keys(), [])
        self.instance.information['collector'].set('files', 'files', \
            {'/usr/bin/foobar': UnixPermParser('-rw-r--r--')})
        method()
        self.assert_('incorrect-file-perms' in self.instance.err_dict.keys())
        self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
            'incorrect-file-perms'), True)
        self.instance.err_dict.clear()
        self.instance.information['collector'].set('files', 'files', \
            {'/usr/games/foo': UnixPermParser('-rwxr-sr-x')})
        method()
        self.assertEqual(self.instance.err_dict.keys(), [])
        self.instance.information['collector'].set('files', 'files', \
            {'/usr/lib/ali/test.ali': UnixPermParser('-r--r--r--')})
        method()
        self.assertEqual(self.instance.err_dict.keys(), [])
        self.instance.information['collector'].set('files', 'files', \
            {'/usr/lib/ali/test.ali': UnixPermParser('-rwxr-xr-x')})
        method()
        self.assert_('ali-file-not-444' in self.instance.err_dict.keys())
        self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
            'ali-file-not-444'), True)
        self.instance.information['collector'].set('files', 'files', \
            {'/usr/share/doc/.#examples.333': UnixPermParser('-rw-r--r--')})
        method()
        self.assert_('cvs-bu-in-pkg' in self.instance.err_dict.keys())
        self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
            'cvs-bu-in-pkg'), True)
        self.instance.err_dict.clear()
        self.instance.information['collector'].set('files', 'files', \
            {'/etc/init.d/lala': UnixPermParser('-rwxr-xr-x')})
        method()
        self.assertEqual(self.instance.err_dict.keys(), [])
        self.instance.information['collector'].set('files', 'files', \
            {'/etc/init.d/foo': UnixPermParser('-rw-r--r--')})
        method()
        self.assert_('bad-perms-for-initd' in self.instance.err_dict.keys())
        self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
            'bad-perms-for-initd'), True)
        self.instance.information['collector'].set('files', 'files', \
            {'/etc/rc.boot/foo': UnixPermParser('-rwxr-xr-x')})
        method()
        self.assert_('package-install-bad-dir' in \
            self.instance.err_dict.keys())
        self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
            'package-install-bad-dir'), True)
        self.instance.information['collector'].set('files', 'files', \
            {'/usr/X11R6/lib/X11/app-defaults/foo': \
            UnixPermParser('-rwxr-xr-x')})
        method()
        self.assert_('pkg-has-old-app-defaults' in \
            self.instance.err_dict.keys())
        self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
            'pkg-has-old-app-defaults'), True)
        self.instance.err_dict.clear()
        self.instance.information['collector'].set('files', 'files', \
            {'/usr/share/doc/copying.html': UnixPermParser('-rwxr-xr-x')})
        method()
        self.assertEqual(self.instance.err_dict.keys(), [])
        self.instance.information['collector'].set('files', 'files', \
            {'/usr/share/doc/copying': UnixPermParser('-rwxr-xr-x')})
        method()
        self.assert_('extra-license-file' in self.instance.err_dict.keys())
        self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
            'extra-license-file'), True)
        self.instance.information['collector'].set('files', 'files', \
            {'/usr/lib/mime/foo': UnixPermParser('-rwxr-xr-x')})
        self.instance.information['control']['self'][0]['architecture'] = 'all'
        method()
        self.assert_('usr-lib-in-arch-all' not in \
            self.instance.err_dict.keys())
        self.instance.information['collector'].set('files', 'files', \
            {'/usr/lib/ruby/1.8/dbi/trace.rb': UnixPermParser('-rw-r--r--')})
        method()
        self.assert_('usr-lib-in-arch-all' not in \
            self.instance.err_dict.keys())
        self.instance.information['collector'].set('files', 'files', \
            {'/usr/lib/foo': UnixPermParser('-rwxr-xr-x')})
        method()
        self.assert_('usr-lib-in-arch-all' in self.instance.err_dict.keys())
        self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
            'usr-lib-in-arch-all'), True)
        os.makedirs('%s/unpacked/usr/share/doc/test' % \
            self.instance.information['dir'])
        os.symlink('%s/unpacked/usr/share/doc/test-doc' % \
            self.instance.information['dir'], \
            '%s/unpacked/usr/share/doc/test/html' % \
            self.instance.information['dir'])
        self.instance.information['collector'].set('files', 'files', \
            {'/usr/share/doc/test/html': UnixPermParser('lrwxrwxrwx')})
        self.instance.information['collector'].set('output', 'file', \
            {'/usr/share/doc/test/html': 'symbolic link'})
        method()
        self.assert_('zero-length-file' not in self.instance.err_dict.keys())
        f = open('%s/unpacked/usr/share/doc/test/a' % \
            self.instance.information['dir'], 'w')
        f.close()
        self.instance.information['collector'].set('files', 'files', \
            {'/usr/share/doc/test/a': UnixPermParser('-r--------')})
        self.instance.information['collector'].set('output', 'file', \
            {'/usr/share/doc/test/a': 'ASCII English text'})
        method()
        self.assert_('zero-length-file' in self.instance.err_dict.keys())
        self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
            'zero-length-file'), True)
        self.instance.err_dict.clear()
        f = gzip.open('%s/unpacked/usr/share/doc/test/b.gz' % \
            self.instance.information['dir'], 'w')
        f.close()
        self.instance.information['collector'].set('files', 'files', \
            {'/usr/share/doc/test/b.gz': UnixPermParser('-r--------')})
        self.instance.information['collector'].set('output', 'file', \
            {'/usr/share/doc/test/b.gz': 'gzip compressed data'})
        method()
        self.assert_('zero-length-file' in self.instance.err_dict.keys())
        shutil.rmtree('%s/unpacked/usr/share/doc/test' % \
            self.instance.information['dir'])
        self.instance.err_dict.clear()
        self.instance.information['collector'].set('files', 'files', \
            {'/usr/bin/lala': UnixPermParser('-rwxr-xr-x')})
        self.instance.information['collector'].set('output', 'file', \
            {'/usr/bin/lala': 'Bourne shell script text executable'})
        method()
        self.assertEqual(self.instance.err_dict.keys(), [])
        self.instance.information['collector'].set('output', 'file', \
            {'/usr/bin/lala': 'MS-DOS executable PE  for MS Windows (console) Intel 80386 32-bit'})
        method()
        self.assertEqual(self.instance.err_dict.keys(), [])
        self.instance.information['collector'].set('files', 'files', \
            {'/usr/bin/lala': UnixPermParser('-rwxr-xr-x')})
        self.instance.information['collector'].set('output', 'file', \
            {'/usr/bin/lala': 'gzip compressed data'})
        method()
        self.assert_('exec-not-elf-or-script' in \
            self.instance.err_dict.keys())
        self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
            'exec-not-elf-or-script'), True)
        self.instance.err_dict.clear()
        self.instance.information['collector'].set('files', 'files', \
            {'/usr/lib/foo/lala.png': UnixPermParser('-r--------')})
        self.instance.information['collector'].set('output', 'file', \
            {'/usr/lib/foo/lala.png': 'PNG image data'})
        method()
        self.assert_('image-in-usr-lib' in self.instance.err_dict.keys())
        self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
            'image-in-usr-lib'), True)

    def test_stat_checks(self):
        "Check stat_checks() method of FilesCheck"
        method = getattr(self.instance, 'stat_checks')
        self.instance.information['collector'] = Collector('bin', '/tmp', '')
        os.makedirs(os.path.join(self.instance.information['dir'], \
            'unpacked/usr'))
        os.symlink(os.path.join(self.instance.information['dir'], 'unpacked/usr', \
            'sh'), os.path.join(self.instance.information['dir'], 'unpacked/usr', \
            'foo'))
        self.instance.information['collector'].set('files', 'files', \
            {'/usr/foo': UnixPermParser('-rwxr-xr-x')})
        method()
        self.assertEqual(self.instance.err_dict.keys(), [])
        shutil.move(os.path.join(self.instance.information['dir'], \
            'lala'), os.path.join(self.instance.information['dir'], \
            'unpacked/usr/lala'))
        self.instance.information['collector'].set('files', 'files', \
             {'/usr/lala': UnixPermParser('-rwxr-xr-x')})
        method()
        self.assert_('too-old-file' in self.instance.err_dict.keys())
        self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
            'too-old-file'), True)
        shutil.rmtree(os.path.join(self.instance.information['dir'], \
            'unpacked/usr'))
        
    def test_readme_debian(self):
        "Check readme_debian() method of FilesCheck"
        method = getattr(self.instance, 'readme_debian')
        os.makedirs(os.path.join(self.instance.information['dir'], \
            'unpacked/usr/share/doc/test'))
        shutil.move(os.path.join(self.instance.information['dir'], \
            'README.Debian'), os.path.join(self.instance.information['dir'], \
            'unpacked/usr/share/doc/test/README.Debian'))
        method()
        self.assert_('possible-readme-debian-template' in \
            self.instance.err_dict.keys())
        self.assertEqual(self.po_keys.compare(self.instance.err_dict, \
            'possible-readme-debian-template'), True)
        shutil.rmtree(os.path.join(self.instance.information['dir'], \
            'unpacked'))

