import os, unittest, shutil, tempfile
from linda import clparser
from linda.funcs import run_external_cmd
from linda.libchecks import ErrDict
from linda.overrides import Overrides
from linda.parser.unixperm import UnixPermParser

class OverridesBaseTest(unittest.TestCase):
    def setUp(self):
        self.overrides = Overrides('/tmp/lala_0.0.1-1_all.deb', '/tmp')
        self.dir = tempfile.mkdtemp()
        run_external_cmd('tar zxCf %s tests/base_overrides.tar.gz' % \
            self.dir)

    def tearDown(self):
        self.overrides.data = {}
        shutil.rmtree(self.dir)
        
    def test_empty_parse(self):
        "Check the parse() method doesn't die with no files"
        clparser['more-overrides'] = ''
        self.overrides.parse()
        self.assertEqual(self.overrides.data, {})

    def test_parse(self):
        "Check the parse() method with files"
        clparser['more-overrides'] = os.path.join(self.dir, 'overrides')
        self.overrides.parse()
        self.assertEqual(self.overrides.data.keys(), \
            ['usd-symlink-no-depends'])
        self.assertEqual(self.overrides.data['usd-symlink-no-depends'], {})

    def test_check_empty(self):
        "Check the check() method of Overrides with a general override"
        clparser['more-overrides'] = os.path.join(self.dir, 'overrides')
        self.overrides.parse()
        err_dicts = {'data/test.data': ErrDict()}
        for err in ('usd-symlink-no-depends', 'no-package-line'):
            err_dicts['data/test.data'].add_error(err)
        data_files = {}
        new_err_dicts, new_data_files = self.overrides.check(err_dicts, \
            data_files)
        self.assertEqual(new_err_dicts['data/test.data'].keys(), \
            ['no-package-line'])
        self.assertEqual(new_err_dicts['data/test.data']['no-package-line'], \
            [[]])
        
    def test_check_type(self):
        "Check the check() method of Overrides with a Type override"
        clparser['more-overrides'] = os.path.join(self.dir, 'type')
        self.overrides.parse()
        err_dicts = {'data/test.data': ErrDict()}
        for err in ('usd-symlink-no-depends', 'no-package-line'):
            err_dicts['data/test.data'].add_error(err)
        data_files = {'data/test.data': {'usd-symlink-no-depends': {'Type': \
            ''}}}
        new_err_dicts, new_data_files = self.overrides.check(err_dicts, \
            data_files)
        self.assertEqual(new_data_files['data/test.data']['usd-symlink-no-depends']['Type'], 'Informational')
        
    def test_check_data(self):
        "Check the check() method of Overrides with a Data override"
        clparser['more-overrides'] = os.path.join(self.dir, 'data')
        self.overrides.parse()
        err_dicts = {'data/test.data': ErrDict()}
        err_dicts['data/test.data'].add_error('binary-note-or-comment', \
            ['bin/foobar'])
        err_dicts['data/test.data'].add_error('binary-in-etc', \
            ['bin/foo'])
        err_dicts['data/test.data'].add_error('binary-in-etc', \
            ['bin/bar'])
        data_files = {}
        self.assertEqual(err_dicts['data/test.data']['binary-in-etc'], \
            [['bin/foo'], ['bin/bar']])
        new_err_dicts, new_data_files = self.overrides.check(err_dicts, \
            data_files)
        self.assertEqual(new_err_dicts['data/test.data']['binary-in-etc'], \
            [['bin/bar']])

    def test_check_non_string_data(self):
        "Check the check() method of Overrides with non-strings in Data"
        clparser['more-overrides'] = os.path.join(self.dir, 'data')
        self.overrides.parse()
        err_dicts = {'data/test.data': ErrDict()}
        err_dicts['data/test.data'].add_error('binary-in-etc', \
            ['bar', UnixPermParser('-rwxr-xr-x')])
        data_files = {}
        new_err_dicts, new_data_files = self.overrides.check(err_dicts, \
            data_files)

    def test_check_non_data_tag(self):
        "Check the check() method of Overrides with a tag with no data"
        clparser['more-overrides'] = os.path.join(self.dir, 'non-data')
        self.overrides.parse()
        err_dicts = {'data/test.data': ErrDict()}
        err_dicts['data/test.data'].add_error('package-not-start-lib')
        data_files = {}
        new_err_dicts, new_data_files = self.overrides.check(err_dicts, \
            data_files)
        self.assertEqual(new_err_dicts['data/test.data'].keys(), [])

