import os, unittest, tempfile, shutil, pwd, commands
from linda.funcs import run_external_cmd
from sets import Set

class EndToEndTest(unittest.TestCase):
    def setUp(self):
        self.dir = tempfile.mkdtemp()
        self.opts = '-SDq'
        run_external_cmd('tar zxCf %s tests/endtoend.tar.gz' % \
            self.dir)
        os.environ['LINDA_ROOT'] = os.getcwd()
        os.environ['LANG'] = 'C'

    def tearDown(self):
        shutil.rmtree(self.dir)
        
    def test_nonbrokendeb(self):
        "Check a non-broken deb"
        output = run_external_cmd('./linda %s -c /dev/null %s' % \
            (self.opts, os.path.join(self.dir, 'testcontrol_0.0.1_all.deb')))
        self.assertEqual(output, '')
        
    def test_brokendeb(self):
        "Check a broken deb"
        output = run_external_cmd('./linda %s -t E,W,X -c /dev/null %s' % \
            (self.opts, os.path.join(self.dir, \
            'testunpacked_0.0.1_all.deb')), 1).split('\n')
        ex_output = Set(('W: testunpacked; Package Architecture: all, but contains arch-dep binary objects.', 'E: testunpacked; Package does not contain a copyright file.', "X: testunpacked; Package name doesn't contain one of the SONAMEs."))
        self.assertEqual(ex_output.intersection(output), ex_output)
        
    def test_brokendeboverrides(self):
        "Check a broken deb with overrides"
        output = run_external_cmd('./linda %s -c /dev/null %s' % \
            (self.opts, os.path.join(self.dir, \
            'testunpackedover_0.0.1_all.deb')), 1).split('\n')
        ex_output = Set(('W: testunpackedover; Package Architecture: all, but contains arch-dep binary objects.', 'E: testunpackedover; Package does not contain a copyright file.'))
        self.assertEqual(ex_output.intersection(output), ex_output)

    def test_nonbrokendsc(self):
        "Check a non-broken dsc"
        output = run_external_cmd('./linda %s %s' % (self.opts, \
            os.path.join(self.dir, 'testcontrol_0.0.1.dsc')))
        self.assertEqual(output, '')

    def test_brokendsc(self):
        "Check a broken dsc"
        output = run_external_cmd('./linda %s -c /dev/null %s' % \
            (self.opts, os.path.join(self.dir, \
            'testunpacked_0.0.1-1.dsc')), 1).split('\n')
        self.assertEqual(output, ['E: testunpacked; Package contains debian/files.'])

    def test_brokendscinfo(self):
        "Check a broken dsc with more information (-i)"
        output = run_external_cmd('./linda %s -i -c /dev/null %s' % \
            (self.opts, os.path.join(self.dir, \
            'testunpacked_0.0.1-1.dsc')), 1).split('\n')
        expected = Set(('E: testunpacked; Package contains debian/files.', ' This package contains debian/files, which should not be installed, and', ' removed within the clean target.'))
        self.assertEqual(expected.intersection(output), expected)

    def test_brokenindeb(self):
        "Check a broken dsc with in-deb overrides being a directory"
        output = run_external_cmd('./linda %s %s' % (self.opts, \
            os.path.join(self.dir, 'testbido_0.0.1.dsc')))
        self.assertEqual(output, '')

