import re, gzip, stat
from linda import libchecks, checks
from linda.funcs import run_external_cmd, explode_path

class FilesCheck(libchecks.LindaChecker):
    'Runs checks over binary and source files in packages.'
    def check_binary_2(self):
        self.directory_checks()
        self.file_checks()
        self.stat_checks()
        self.readme_debian()
        
    def check_source_2(self):
        self.debian_rules()
        self.autoconf_timestamp()
        
    def directory_checks(self):
        if os.path.isdir(os.path.join(self.information['dir'], 'unpacked', \
            'usr/doc')) and not \
            os.path.isdir(os.path.join(self.information['dir'], 'unpacked', \
            'usr/share/doc')):
            self.signal_error('usr-doc-no-usr-share-doc')
        dirs_found = {'dotxvpics': 0, 'versioncontrol': 0}
        for directory in self.information['collector']('files', 'dirs').keys():
            exploded_path = explode_path(directory)
            if not os.path.islink(os.path.join(self.information['dir'], \
                'unpacked', directory[1:])) and not \
                os.listdir(os.path.join(self.information['dir'], 'unpacked', \
                directory[1:])) and exploded_path[:2] == ['usr', 'share']:
                self.signal_error('empty-directory-in-pkg', [directory])
            for component in exploded_path:
                if not dirs_found['versioncontrol']:
                    if component in ('CVS', '.svn', '{arch}'):
                        self.signal_error('vercon-dir-in-package')
                        dirs_found['versioncontrol'] = 1
                if not dirs_found['dotxvpics']:
                    if component == '.xvpics':
                        self.signal_error('dot-xvpics-in-package')
                        dirs_found['dotxvpics'] = 1
            if self.information['collector']('files', 'dirs')[directory] != \
                ['-rwxr-xr-x', 'root/root']:
                self.signal_error('non-standard-dir-perm', \
                    [directory] + self.information['collector']('files', 'dirs')[directory])
            if exploded_path[0] == 'var':
                if exploded_path[1] in ('adm', 'catman', 'named', 'nis', \
                    'preserve'):
                    self.signal_error('FSSTND-dir-found', [directory, '/var'])
                else:
                    if exploded_path[1] not in ('account', 'lib', 'cache', \
                        'crash', 'games', 'lock', 'log', 'opt', 'run', \
                        'spool', 'state', 'tmp', 'www', 'yp'):
                        self.signal_error('non-standard-dir-in-var', \
                            [directory])
                    elif exploded_path[1:3] == ['lib', 'games']:
                        self.signal_error('non-standard-dir-in-var', \
                            [directory])
            elif exploded_path[0] == 'usr':
                if exploded_path[1] in ('dict', 'doc', 'etc', 'info', 'man', \
                    'adm', 'preserve'):
                    self.signal_error('FSSTND-dir-found', [directory, '/usr'])
                elif exploded_path[1] not in ('X11R6', 'X386', 'bin', \
                        'games', 'include', 'lib', 'local', 'sbin', 'share', \
                        'src', 'spool', 'tmp'):
                        self.signal_error('non-standard-dir-in-usr', \
                            [directory])
    def file_checks(self):
        files_found = {'cvsignore': 0, 'opt': 0, 'usr_local': 0, \
            'usr_share': 0, 'cvs_backup': 0, 'old-a-d': 0, 'arch_all_lib': 0}
        for file in self.information['collector']('files', 'files').keys():
            path, filename = os.path.split(file)
            extension = os.path.splitext(filename)[1]
            if path.startswith('/opt') and not files_found['opt']:
                self.signal_error('file-in-opt')
                files_found['opt'] = 1
            if path.startswith('/usr/local') and not files_found['usr_local']:
                self.signal_error('file-in-usr-local')
                files_found['usr_local'] = 1
            if extension in ('.rej', '.orig'):
                self.signal_error('failed-diff', [file])
            if extension == '.ali':
                if self.information['collector']('files', 'files')[file] != \
                    '-r--r--r--':
                    self.signal_error('ali-file-not-444', [file])
            if filename == '.cvsignore' and not files_found['cvsignore']:
                self.signal_error('cvsignore-in-package')
                files_found['cvsignore'] = 1
            if path == '/usr/share' and not files_found['usr_share']:
                self.signal_error('file-directly-in-u-s', [file])
                files_found['usr_share'] = 1
            if path in ('/bin', '/sbin', '/usr/bin', '/usr/sbin', \
                '/usr/X11R6/bin', '/usr/games'):
                if self.information['collector']('files', 'files')[file] != \
                    '-rwxr-xr-x' and not \
                    self.information['collector']('files', 'files')[file].is_link():
                    self.signal_error('incorrect-file-perms', [file, \
                        self.information['collector']('files', 'files')[file]])
            if re.match(r'^\.#.*[0-9\.]{3}', filename) and not \
                files_found['cvs_backup']:
                self.signal_error('cvs-bu-in-pkg', [file])
                files_found['cvs_backup'] = 1
            if path == '/etc/init.d' and self.information['collector']('files', 'files')[file] != '-rwxr-xr-x':
                self.signal_error('bad-perms-for-initd', [file, \
                    self.information['collector']('files', 'files')[file]])
            if path in ('/etc/rc.d', '/etc/rc.boot'):
                self.signal_error('package-install-bad-dir', [path])
            if path == '/usr/X11R6/lib/X11/app-defaults' and not \
                files_found['old-a-d']:
                self.signal_error('pkg-has-old-app-defaults', [file])
            license_re = re.compile(r'(copying|licen[cs]e)', re.IGNORECASE)
            if re.search(license_re, filename):
                if extension not in ('.html', '.el', '.xpm'):
                    self.signal_error('extra-license-file', [file])
            if self.information['control']['self'][0].has_key('architecture'):
                if path.startswith('/usr/lib') and \
                    self.information['control']['self'][0]['architecture'] == 'all' \
                    and not files_found['arch_all_lib'] and not \
                    self.usr_lib_exceptions(path):
                    files_found['arch_all_lib'] = 1
                    self.signal_error('usr-lib-in-arch-all', [file])
            if path in ('/usr/doc/%s' % self.pkg_name, '/usr/share/doc/%s' % \
                self.pkg_name):
                f = None
                if self.information['collector']('output', 'file')[file].find('compressed') != -1:
                    f = gzip.open(os.path.join(self.information['dir'], \
                        'unpacked', file[1:]))
                elif self.information['collector']('output', 'file')[file].find('symbolic link') == -1:
                    f = open(os.path.join(self.information['dir'], \
                        'unpacked', file[1:]))
                if f:
                    if not f.readlines():
                        self.signal_error('zero-length-file', [file])
        for file in self.information['collector']('output', 'file').keys():
            path, filename = os.path.split(file)
            if self.information['collector']('files', 'files')[file].is_executable() and not self.information['collector']('files', 'files')[file].is_link():
                if file not in self.information['collector']('files', 'elf') \
                    and not re.search(r'(script|makefile)', \
                    self.information['collector']('output', 'file')[file], \
                    re.IGNORECASE):
                    self.signal_error('exec-not-elf-or-script', [file, \
                        self.information['collector']('files', \
                        'files')[file]])
            if path.startswith('/usr/lib') and \
                re.search(r'image (data|text)', \
                    self.information['collector']('output', 'file')[file]):
                self.signal_error('image-in-usr-lib', [file])
    def stat_checks(self):
        for file in self.information['collector']('files', 'files').keys():
            if os.path.islink(os.path.join(self.information['dir'], 'unpacked', \
                file[1:])):
                continue
            if os.stat(os.path.join(self.information['dir'], 'unpacked', \
                file[1:]))[stat.ST_MTIME] < 0:
                self.signal_error('too-old-file', [file])
    def readme_debian(self):
        f = 0
        for rd in ('README.Debian', 'README.debian'):
            if os.path.exists(os.path.join(self.information['dir'], \
                'unpacked/usr/share/doc', self.pkg_name, rd)):
                if f:
                    continue
                if os.path.islink(os.path.join(self.information['dir'], \
                    'unpacked/usr/share/doc', self.pkg_name, rd)):
                    if not os.path.exists(os.readlink(os.path.join(self.information['dir'], \
                        'unpacked/usr/share/doc/', self.pkg_name, rd))):
                        continue
                f = open(os.path.join(self.information['dir'], \
                    'unpacked/usr/share/doc', self.pkg_name, rd))
        if f:
            for k in f:
                if k.find("<possible notes regarding this package - if " +\
                "none, delete this file>") != -1:
                    self.signal_error('possible-readme-debian-template')
            f.close()

    def usr_lib_exceptions(self, path):
        exceptions = ('menu', 'python', 'emacsen-common', 'base-config', \
            'dpkg', 'sgml', 'site-python', 'debian-test', 'games', 'cgi-bin')
        for x in exceptions:
            if path.startswith('/usr/lib/%s' % x):
                return 1
        return 0

    def debian_rules(self):
        try:
            f = open(os.path.join(self.information['dir'], 'debian/rules'))
        except IOError:
            f = 0
        if f:
            k = f.readline()[:-1]
            f.close()
            if k.find('/usr/bin/make -f') == -1:
                self.signal_error('debian-rules-not-makefile')
            if k.endswith(' '):
                self.signal_error('debian-rules-hashbang-space')

    def autoconf_timestamp(self):
        config_to_check = []
        for extension in ('guess', 'sub'):
            if os.path.exists(os.path.join(self.information['dir'], \
                'config.%s' % extension)):
                config_to_check.append('config.%s' % extension)
        for file in config_to_check:
            timestamp = 0
            f = open(os.path.join(self.information['dir'], file))
            for k in f:
                if k.startswith('timestamp'):
                    timestamp = k.split("='")[1].split('-')[0]
                    dprint(_("Setting timestamp of %s to %s.") % \
                        (file, timestamp), 2)
            if int(timestamp) < 2002:
                self.signal_error('config-too-old', [file])

checks.register(FilesCheck)


