/**
 * ntfsdecrypt - Decrypt ntfs encrypted files.  Part of the Linux-NTFS project.
 *
 * Copyright (c) 2005 Yuval Fledel
 * Copyright (c) 2005 Anton Altaparmakov
 *
 * This utility will decrypt files and print the decrypted data on the standard
 * output.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program (in the main directory of the Linux-NTFS
 * distribution in the file COPYING); if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "config.h"

#if !defined(HAVE_GCRYPT_H) || !defined(HAVE_GNUTLS_PKCS12_H)
#error A required header file is missing. Aborting.
#endif

#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif
#ifdef HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif
#ifdef HAVE_FCNTL_H
#include <fcntl.h>
#endif
#ifdef HAVE_STDIO_H
#include <stdio.h>
#endif
#ifdef HAVE_GETOPT_H
#include <getopt.h>
#endif
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif
#ifdef HAVE_STRING_H
#include <string.h>
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#ifdef HAVE_ERRNO_H
#include <errno.h>
#endif
#ifdef HAVE_GCRYPT_H
#include <gcrypt.h>
#endif
#ifdef HAVE_GNUTLS_PKCS12_H
#include <gnutls/pkcs12.h>
#endif

#include "types.h"
#include "attrib.h"
#include "utils.h"
#include "volume.h"
#include "debug.h"
#include "dir.h"
#include "layout.h"
#include "version.h"

typedef gcry_sexp_t ntfs_rsa_private_key;

typedef struct {
	u8 *key_data;
	u32 alg_id;
	gcry_cipher_hd_t gcry_cipher_hd;
	gcry_cipher_hd_t *des_gcry_cipher_hd_ptr;
} ntfs_fek;

#define CALG_DES (0x6601)
/* If not one of the below three, fall back to standard Des. */
#define CALG_3DES (0x6603)
#define CALG_DESX (0x6604)
#define CALG_AES_256 (0x6610)

/* DESX-MS128 implementation for libgcrypt. */
static gcry_module_t ntfs_desx_module;
static int ntfs_desx_algorithm_id = -1;

typedef struct {
	u64 in_whitening, out_whitening;
	gcry_cipher_hd_t gcry_cipher_hd;
} ntfs_desx_ctx;

struct options {
	char *keyfile;	/* .pfx file containing the user's private key. */
	char *device;		/* Device/File to work with */
	char *file;		/* File to display */
	s64 inode;		/* Inode to work with */
	ATTR_TYPES attr;	/* Attribute type to display */
	int force;		/* Override common sense */
	int quiet;		/* Less output */
	int verbose;		/* Extra output */
};

static const char *EXEC_NAME = "ntfsdecrypt";
static struct options opts;

GEN_PRINTF(Eprintf, stderr, NULL, FALSE)
GEN_PRINTF(Vprintf, stderr, &opts.verbose, TRUE)
GEN_PRINTF(Qprintf, stderr, &opts.quiet, FALSE)
static GEN_PRINTF(Printf, stderr, NULL, FALSE)

static ntfschar EFS[5] = {
	const_cpu_to_le16('$'), const_cpu_to_le16('E'), const_cpu_to_le16('F'),
	const_cpu_to_le16('S'), const_cpu_to_le16('\0')
};

/**
 * version - Print version information about the program
 *
 * Print a copyright statement and a brief description of the program.
 *
 * Return:  none
 */
static void version(void)
{
	Printf("\n%s v%s (libntfs %s) - Decrypt files and print on the "
			"standard output.\n\n", EXEC_NAME, VERSION,
			ntfs_libntfs_version());
	Printf("Copyright (c) 2005 Yuval Fledel\n");
	Printf("Copyright (c) 2005 Anton Altaparmakov\n");
	Printf("\n%s\n%s%s\n", ntfs_gpl, ntfs_bugs, ntfs_home);
}

/**
 * usage - Print a list of the parameters to the program
 *
 * Print a list of the parameters and options for the program.
 *
 * Return:  none
 */
static void usage(void)
{
	Printf("\nUsage: %s [options] -k name.pfx device [file]\n\n"
	       "    -i, --inode num         Display this inode\n\n"
	       "    -k  --keyfile name.pfx  Use file name as the user's private key file.\n"
	       "    -f  --force             Use less caution\n"
	       "    -h  --help              Print this help\n"
	       "    -q  --quiet             Less output\n"
	       "    -V  --version           Version information\n"
	       "    -v  --verbose           More output\n\n",
	       EXEC_NAME);
	Printf("%s%s\n", ntfs_bugs, ntfs_home);
}

/**
 * parse_options - Read and validate the programs command line
 *
 * Read the command line, verify the syntax and parse the options.
 * This function is very long, but quite simple.
 *
 * Return:  1 Success
 *	    0 Error, one or more problems
 */
static int parse_options(int argc, char **argv)
{
	static const char *sopt = "-fh?i:k:qVv";
	static const struct option lopt[] = {
		{"force", no_argument, NULL, 'f'},
		{"help", no_argument, NULL, 'h'},
		{"inode", required_argument, NULL, 'i'},
		{"keyfile", required_argument, NULL, 'k'},
		{"quiet", no_argument, NULL, 'q'},
		{"version", no_argument, NULL, 'V'},
		{"verbose", no_argument, NULL, 'v'},
		{NULL, 0, NULL, 0}
	};

	char c = -1;
	int err = 0;
	int ver = 0;
	int help = 0;

	opterr = 0;		/* We'll handle the errors, thank you. */

	opts.inode = -1;

	while ((c = getopt_long(argc, argv, sopt, lopt, NULL)) != (char)-1) {
		switch (c) {
		case 1:	/* A non-option argument */
			if (!opts.device)
				opts.device = argv[optind - 1];
			else if (!opts.file)
				opts.file = argv[optind - 1];
			else {
				Eprintf("You must specify exactly one file.\n");
				err++;
			}
			break;
		case 'f':
			opts.force++;
			break;
		case 'h':
		case '?':
			help++;
			break;
		case 'k':
			if (!opts.keyfile)
				opts.keyfile = argv[optind - 1];
			else {
				Eprintf("You must specify exactly one "
						"key file.\n");
				err++;
			}
			break;
		case 'i':
			if (opts.inode != -1)
				Eprintf("You must specify exactly one "
						"inode.\n");
			else if (utils_parse_size(optarg, &opts.inode, FALSE))
				break;
			else
				Eprintf("Couldn't parse inode number.\n");
			err++;
			break;
		case 'q':
			opts.quiet++;
			break;
		case 'V':
			ver++;
			break;
		case 'v':
			opts.verbose++;
			break;
		default:
			Eprintf("Unknown option '%s'.\n", argv[optind - 1]);
			err++;
			break;
		}
	}

	if (help || ver) {
		opts.quiet = 0;
	} else {
		if (!opts.keyfile) {
			Eprintf("You must specify a key file.\n");
			err++;
		} else if (opts.device == NULL) {
			Eprintf("You must specify a device.\n");
			err++;
		} else if (opts.file == NULL && opts.inode == -1) {
			Eprintf("You must specify a file or inode with the -i "
					"option.\n");
			err++;
		} else if (opts.file != NULL && opts.inode != -1) {
			Eprintf("You can't specify both a file and inode.\n");
			err++;
		}
		if (opts.quiet && opts.verbose) {
			Eprintf("You may not use --quiet and --verbose at the "
					"same time.\n");
			err++;
		}
	}

	if (ver)
		version();
	if (help || err)
		usage();

	return (!err && !help && !ver);
}

static int ntfs_pkcs12_load_pfxfile(const char *keyfile, u8 **pfx,
		unsigned *pfx_size)
{
	int f, to_read, total, attempts, br;
	struct stat key_stat;

	if (!keyfile || !pfx || !pfx_size) {
		fprintf(stderr, "You have to specify the key file, a pointer "
				"to hold the key file contents, and a pointer "
				"to hold the size of the key file contents.");
		return -1;
	}
	f = open(keyfile, O_RDONLY);
	if (f == -1) {
		perror("Failed to open key file");
		return -1;
	}
	if (fstat(f, &key_stat) == -1) {
		perror("Failed to stat key file");
		goto file_out;
	}
	if (!S_ISREG(key_stat.st_mode)) {
		fprintf(stderr, "Key file is not a regular file, cannot read "
				"it.");
		goto file_out;
	}
	if (!key_stat.st_size) {
		fprintf(stderr, "Key file has zero size.");
		goto file_out;
	}
	*pfx = malloc(key_stat.st_size + 1);
	if (!*pfx) {
		perror("Failed to allocate buffer for key file contents");
		goto file_out;
	}
	to_read = key_stat.st_size;
	total = attempts = 0;
	do {
		br = read(f, *pfx + total, to_read);
		if (br == -1) {
			perror("Failed to read from key file");
			goto free_out;
		}
		if (!br)
			attempts++;
		to_read -= br;
		total += br;
	} while (to_read > 0 && attempts < 3);
	close(f);
	/* Make sure it is zero terminated. */
	(*pfx)[key_stat.st_size] = 0;
	*pfx_size = key_stat.st_size;
	return 0;
free_out:
	free(*pfx);
file_out:
	close(f);
	return -1;
}

static int ntfs_crypto_init(void)
{
	int err;

	/* Initialize gcrypt library.  Note: Must come before GNU TLS init. */
	if (gcry_control(GCRYCTL_DISABLE_SECMEM, 0) != GPG_ERR_NO_ERROR) {
		fprintf(stderr, "Failed to initialize the gcrypt library.\n");
		return -1;
	}
	/* Initialize GNU TLS library.  Note: Must come after libgcrypt init. */
	err = gnutls_global_init();
	if (err < 0) {
		fprintf(stderr, "Failed to initialize GNU TLS library: %s\n",
				gnutls_strerror(err));
		return -1;
	}
	return 0;
}

static void ntfs_crypto_deinit(void)
{
	gnutls_global_deinit();
	if (ntfs_desx_module) {
		gcry_cipher_unregister(ntfs_desx_module);
		ntfs_desx_module = NULL;
		ntfs_desx_algorithm_id = -1;
	}
}

static ntfs_rsa_private_key ntfs_rsa_private_key_import_from_gnutls(
		gnutls_x509_privkey_t priv_key)
{
	int i, j, tmp_size;
	gnutls_datum_t rd[6];
	gcry_mpi_t rm[6];
	gcry_sexp_t rsa_key;

	/* Extract the RSA parameters from the GNU TLS private key. */
	if (gnutls_x509_privkey_export_rsa_raw(priv_key, &rd[0], &rd[1],
			&rd[2], &rd[3], &rd[4], &rd[5])) {
		fprintf(stderr, "Failed to export rsa parameters.  (Is the "
				"key an RSA private key?)\n");
		return NULL;
	}
	/* Convert each RSA parameter to mpi format. */
	for (i = 0; i < 6; i++) {
		if (gcry_mpi_scan(&rm[i], GCRYMPI_FMT_USG, rd[i].data,
				rd[i].size, &tmp_size) != GPG_ERR_NO_ERROR) {
			fprintf(stderr, "Failed to convert RSA parameter %i "
					"to mpi format (size %d)\n", i,
					rd[i].size);
			rsa_key = NULL;
			break;
		}
	}
	/* Release the no longer needed datum values. */
	for (j = 0; j < 6; j++) {
		/*
		 * FIXME: _gnutls_free_datum() is not exported from libgnutls
		 * so we do it by hand...  )-:  Let us just hope the
		 * gnutls_datum_t structure does not change across versions of
		 * the gnutls library.
		 */
#if 0
		_gnutls_free_datum(&rd[j]);
#else
		if (rd[j].data && rd[j].size)
			gnutls_free(rd[j].data);
#endif
	}
	/*
	 * Build the gcrypt private key, note libgcrypt uses p and q inversed
	 * to what gnutls uses.
	 */
	if (i == 6 && gcry_sexp_build(&rsa_key, NULL,
			"(private-key(rsa(n%m)(e%m)(d%m)(p%m)(q%m)(u%m)))",
			rm[0], rm[1], rm[2], rm[4], rm[3], rm[5]) !=
			GPG_ERR_NO_ERROR) {
		fprintf(stderr, "Failed to build RSA private key s-exp.\n");
		rsa_key = NULL;
	}
	/* Release the no longer needed mpi values. */
	for (j = 0; j < i; j++)
		gcry_mpi_release(rm[j]);
	return (ntfs_rsa_private_key)rsa_key;
}

static ntfs_rsa_private_key ntfs_pkcs12_extract_rsa_key(u8 *pfx, int pfx_size,
		char *password)
{
	int err, bag_index, flags;
	gnutls_datum_t dpfx, dkey;
	gnutls_pkcs12_t pkcs12;
	gnutls_pkcs12_bag_t bag;
	gnutls_x509_privkey_t pkey;
	ntfs_rsa_private_key rsa_key = NULL;

	/* Create a pkcs12 structure. */
	err = gnutls_pkcs12_init(&pkcs12);
	if (err) {
		fprintf(stderr, "Failed to initialize PKCS#12 structure: %s\n",
				gnutls_strerror(err));
		return NULL;
	}
	/* Convert the PFX file (DER format) to native pkcs12 format. */
	dpfx.data = pfx;
	dpfx.size = pfx_size;
	err = gnutls_pkcs12_import(pkcs12, &dpfx, GNUTLS_X509_FMT_DER, 0);
	if (err) {
		fprintf(stderr, "Failed to convert the PFX file from DER to "
				"native PKCS#12 format: %s\n",
				gnutls_strerror(err));
		goto out;
	}
	/*
	 * Verify that the password is correct and that the key file has not
	 * been tampered with.  Note if the password has zero length and the
	 * verification fails, retry with password set to NULL.  This is needed
	 * to get passwordless .pfx files generated with Windows XP SP1 (and
	 * probably earlier versions of Windows) to work.
	 */
retry_verify:
	err = gnutls_pkcs12_verify_mac(pkcs12, password);
	if (err) {
		if (err == GNUTLS_E_MAC_VERIFY_FAILED &&
				password && !strlen(password)) {
			password = NULL;
			goto retry_verify;
		}
		fprintf(stderr, "Failed to verify the MAC (%s).  Is the "
				"password correct?\n", gnutls_strerror(err));
		goto out;
	}
	for (bag_index = 0; ; bag_index++) {
		err = gnutls_pkcs12_bag_init(&bag);
		if (err) {
			fprintf(stderr, "Failed to initialize PKCS#12 Bag "
					"structure: %s\n",
					gnutls_strerror(err));
			goto out;
		}
		err = gnutls_pkcs12_get_bag(pkcs12, bag_index, bag);
		if (err) {
			if (err == GNUTLS_E_REQUESTED_DATA_NOT_AVAILABLE)
				break;
			fprintf(stderr, "Failed to obtain Bag from PKCS#12 "
					"structure: %s\n",
					gnutls_strerror(err));
			goto bag_out;
		}
check_again:
		err = gnutls_pkcs12_bag_get_count(bag);
		if (err < 0) {
			fprintf(stderr, "Failed to obtain Bag count: %s\n",
					gnutls_strerror(err));
			goto bag_out;
		}
		err = gnutls_pkcs12_bag_get_type(bag, 0);
		if (err < 0) {
			fprintf(stderr, "Failed to determine Bag type: %s\n",
					gnutls_strerror(err));
			goto bag_out;
		}
		flags = 0;
		switch (err) {
		case GNUTLS_BAG_PKCS8_KEY:
			flags = GNUTLS_PKCS_PLAIN;
		case GNUTLS_BAG_PKCS8_ENCRYPTED_KEY:
			err = gnutls_pkcs12_bag_get_data(bag, 0, &dkey);
			if (err < 0) {
				fprintf(stderr, "Failed to obtain Bag data: "
						"%s\n", gnutls_strerror(err));
				goto bag_out;
			}
			err = gnutls_x509_privkey_init(&pkey);
			if (err) {
				fprintf(stderr, "Failed to initialized "
						"private key structure: %s\n",
						gnutls_strerror(err));
				goto bag_out;
			}
			/* Decrypt the private key into GNU TLS format. */
			err = gnutls_x509_privkey_import_pkcs8(pkey, &dkey,
					GNUTLS_X509_FMT_DER, password, flags);
			if (err) {
				fprintf(stderr, "Failed to convert private "
						"key from DER to GNU TLS "
						"format: %s\n",
						gnutls_strerror(err));
				goto key_out;
			}
			/* Convert the private key to our internal format. */
			rsa_key = ntfs_rsa_private_key_import_from_gnutls(pkey);
			goto key_out;
		case GNUTLS_BAG_ENCRYPTED:
			err = gnutls_pkcs12_bag_decrypt(bag, password);
			if (err) {
				fprintf(stderr, "Failed to decrypt Bag: %s\n",
						gnutls_strerror(err));
				goto bag_out;
			}
			goto check_again;
		default:
			/* We do not care about other types. */
			break;
		}
		gnutls_pkcs12_bag_deinit(bag);
	}
key_out:
	gnutls_x509_privkey_deinit(pkey);
bag_out:
	gnutls_pkcs12_bag_deinit(bag);
out:
	gnutls_pkcs12_deinit(pkcs12);
	return rsa_key;
}

static void ntfs_rsa_private_key_release(ntfs_rsa_private_key rsa_key)
{
	gcry_sexp_release((gcry_sexp_t)rsa_key);
}

/**
 * ntfs_buffer_reverse -
 *
 * This is a utility function for reversing the order of a buffer in place.
 * Users of this function should be very careful not to sweep byte order
 * problems under the rug.
 */
static inline void ntfs_buffer_reverse(u8 *buf, unsigned buf_size)
{
	unsigned i;
	u8 t;

	for (i = 0; i < buf_size / 2; i++) {
		t = buf[i];
		buf[i] = buf[buf_size - i - 1];
		buf[buf_size - i - 1] = t;
	}
}

/**
 * ntfs_raw_fek_decrypt -
 *
 * Note: decrypting into the input buffer.
 */
static unsigned ntfs_raw_fek_decrypt(u8 *fek, u32 fek_size,
		ntfs_rsa_private_key rsa_key)
{
	gcry_mpi_t fek_mpi;
	gcry_sexp_t fek_sexp, fek_sexp2;
	gcry_error_t err;
	size_t size, padding;

	/* Reverse the raw FEK. */
	ntfs_buffer_reverse(fek, fek_size);
	/* Convert the FEK to internal MPI format. */
	err = gcry_mpi_scan(&fek_mpi, GCRYMPI_FMT_USG, fek, fek_size, NULL);
	if (err != GPG_ERR_NO_ERROR) {
		fprintf(stderr, "Failed to convert file encryption key to "
				"internal MPI format: %s\n",
				gcry_strerror(err));
		return 0;
	}
	/* Create an internal S-expression from the FEK. */
	err = gcry_sexp_build(&fek_sexp, NULL,
			"(enc-val (flags) (rsa (a %m)))", fek_mpi);
	gcry_mpi_release(fek_mpi);
	if (err != GPG_ERR_NO_ERROR) {
		fprintf(stderr, "Failed to create internal S-expression of "
				"the file encryption key: %s\n",
				gcry_strerror(err));
		return 0;
	}
	/* Decrypt the FEK. */
	err = gcry_pk_decrypt(&fek_sexp2, fek_sexp, (gcry_sexp_t)rsa_key);
	gcry_sexp_release(fek_sexp);
	if (err != GPG_ERR_NO_ERROR) {
		fprintf(stderr, "Failed to decrypt the file encryption key: "
				"%s\n", gcry_strerror(err));
		return 0;
	}
	/* Extract the actual FEK from the decrypted raw S-expression. */
	fek_sexp = gcry_sexp_find_token(fek_sexp2, "value", 0);
	gcry_sexp_release(fek_sexp2);
	if (!fek_sexp) {
		fprintf(stderr, "Failed to find the decrypted file encryption "
				"key in the internal S-expression.\n");
		return 0;
	}
	/* Convert the decrypted FEK S-expression into MPI format. */
	fek_mpi = gcry_sexp_nth_mpi(fek_sexp, 1, GCRYMPI_FMT_USG);
	gcry_sexp_release(fek_sexp);
	if (!fek_mpi) {
		fprintf(stderr, "Failed to convert the decrypted file "
				"encryption key S-expression to internal MPI "
				"format.\n");
		return 0;
	}
	/* Convert the decrypted FEK from MPI format to binary data. */
	err = gcry_mpi_print(GCRYMPI_FMT_USG, fek, fek_size, &size, fek_mpi);
	gcry_mpi_release(fek_mpi);
	if (err != GPG_ERR_NO_ERROR || !size) {
		fprintf(stderr, "Failed to convert decrypted file encryption "
				"key from internal MPI format to binary data: "
				"%s\n", gcry_strerror(err));
		return 0;
	}
	/*
	 * Finally, remove the PKCS#1 padding and return the size of the
	 * decrypted FEK.
	 */
	padding = strnlen(fek, size) + 1;
	if (padding > size) {
		fprintf(stderr, "Failed to remove PKCS#1 padding from "
				"decrypted file encryption key.\n");
		return 0;
	}
	size -= padding;
	memmove(fek, fek + padding, size);
	return size;
}

/**
 * ntfs_desx_key_expand - expand a 128-bit desx key to the needed 192-bit key
 * @src:	source buffer containing 128-bit key
 *
 * Expands the on-disk 128-bit desx key to the needed des key, the in-, and the
 * out-whitening keys required to perform desx {de,en}cryption.
 */
static gcry_error_t ntfs_desx_key_expand(const u8 *src, u32 *des_key,
		u64 *out_whitening, u64 *in_whitening)
{
	static const u8 *salt1 = "Dan Simon  ";
	static const u8 *salt2 = "Scott Field";
	static const int salt_len = 12;
	gcry_md_hd_t hd1, hd2;
	u32 *md;
	gcry_error_t err;

	err = gcry_md_open(&hd1, GCRY_MD_MD5, 0);
	if (err != GPG_ERR_NO_ERROR) {
		fprintf(stderr, "Failed to open MD5 digest.\n");
		return err;
	}
	/* Hash the on-disk key. */
	gcry_md_write(hd1, src, 128 / 8);
	/* Copy the current hash for efficiency. */
	err = gcry_md_copy(&hd2, hd1);
	if (err != GPG_ERR_NO_ERROR) {
		fprintf(stderr, "Failed to copy MD5 digest object.\n");
		goto out;
	}
	/* Hash with the first salt and store the result. */
	gcry_md_write(hd1, salt1, salt_len);
	md = (u32*)gcry_md_read(hd1, 0);
	des_key[0] = md[0] ^ md[1];
	des_key[1] = md[2] ^ md[3];
	/* Hash with the second salt and store the result. */
	gcry_md_write(hd2, salt2, salt_len);
	md = (u32*)gcry_md_read(hd2, 0);
	*out_whitening = *(u64*)md;
	*in_whitening = *(u64*)(md + 2);
	gcry_md_close(hd2);
out:
	gcry_md_close(hd1);
	return err;
}

/**
 * ntfs_desx_setkey - libgcrypt set_key implementation for DES-X-MS128
 * @context:	pointer to a variable of type ntfs_desx_ctx
 * @key:	the 128 bit DES-X-MS128 key, concated with the DES handle
 * @keylen:	must always be 16
 * 
 * This is the libgcrypt set_key implementation for DES-X-MS128.
 */
static gcry_err_code_t ntfs_desx_setkey(void *context, const u8 *key,
		unsigned keylen)
{
	ntfs_desx_ctx *ctx = context;
	gcry_error_t err;
	u8 des_key[8];

	if (keylen != 16) {
		fprintf(stderr, "Key length for desx must be 16.\n");
		return GPG_ERR_INV_KEYLEN;
	}
	err = gcry_cipher_open(&ctx->gcry_cipher_hd, GCRY_CIPHER_DES,
			GCRY_CIPHER_MODE_ECB, 0);
	if (err != GPG_ERR_NO_ERROR) {
		fprintf(stderr, "Failed to open des cipher (error 0x%x).\n",
				err);
		return err;
	}
	err = ntfs_desx_key_expand(key, (u32*)des_key, &ctx->out_whitening,
			&ctx->in_whitening);
	if (err != GPG_ERR_NO_ERROR) {
		fprintf(stderr, "Failed to expand desx key (error 0x%x).\n",
				err);
		gcry_cipher_close(ctx->gcry_cipher_hd);
		return err;
	}
	err = gcry_cipher_setkey(ctx->gcry_cipher_hd, des_key, sizeof(des_key));
	if (err != GPG_ERR_NO_ERROR) {
		fprintf(stderr, "Failed to set des key (error 0x%x).\n", err);
		gcry_cipher_close(ctx->gcry_cipher_hd);
		return err;
	}
	/*
	 * Take a note of the ctx->gcry_cipher_hd since we need to close it at
	 * ntfs_decrypt_data_key_close() time.
	 */
	**(gcry_cipher_hd_t***)(key + ((keylen + 7) & ~7)) =
			&ctx->gcry_cipher_hd;
	return GPG_ERR_NO_ERROR;
}

static void ntfs_desx_decrypt(void *context, u8 *outbuf, const u8 *inbuf)
{
	ntfs_desx_ctx *ctx = context;
	gcry_error_t err;

	err = gcry_cipher_reset(ctx->gcry_cipher_hd);
	if (err != GPG_ERR_NO_ERROR)
		fprintf(stderr, "Failed to reset des cipher (error 0x%x).\n",
				err);
	*(u64*)outbuf = *(const u64*)inbuf ^ ctx->out_whitening;
	err = gcry_cipher_encrypt(ctx->gcry_cipher_hd, outbuf, 8, NULL, 0);
	if (err != GPG_ERR_NO_ERROR)
		fprintf(stderr, "Des decryption failed (error 0x%x).\n", err);
	*(u64*)outbuf ^= ctx->in_whitening;
}

static gcry_cipher_spec_t ntfs_desx_cipher = {
	.name = "DES-X-MS128",
	.blocksize = 8,
	.keylen = 128,
	.contextsize = sizeof(ntfs_desx_ctx),
	.setkey = ntfs_desx_setkey,
	.decrypt = ntfs_desx_decrypt,
};

//#define DO_CRYPTO_TESTS 1

#ifdef DO_CRYPTO_TESTS

/* Do not remove this test code from this file! AIA */
static BOOL ntfs_desx_key_expand_test(void)
{
	const u8 known_desx_on_disk_key[16] = {
		0xa1, 0xf9, 0xe0, 0xb2, 0x53, 0x23, 0x9e, 0x8f,
		0x0f, 0x91, 0x45, 0xd9, 0x8e, 0x20, 0xec, 0x30
	};
	const u8 known_des_key[8] = {
		0x27, 0xd1, 0x93, 0x09, 0xcb, 0x78, 0x93, 0x1f,
	};
	const u8 known_out_whitening[8] = {
		0xed, 0xda, 0x4c, 0x47, 0x60, 0x49, 0xdb, 0x8d,
	};
	const u8 known_in_whitening[8] = {
		0x75, 0xf6, 0xa0, 0x1a, 0xc0, 0xca, 0x28, 0x1e
	};
	u64 test_out_whitening, test_in_whitening;
	union {
		u64 u64;
		u32 u32[2];
	} test_des_key;
	gcry_error_t err;
	BOOL res;

	err = ntfs_desx_key_expand(known_desx_on_disk_key, test_des_key.u32,
			&test_out_whitening, &test_in_whitening);
	if (err != GPG_ERR_NO_ERROR)
		res = FALSE;
	else
		res = test_des_key.u64 == *(u64*)known_des_key &&
				test_out_whitening ==
				*(u64*)known_out_whitening &&
				test_in_whitening ==
				*(u64*)known_in_whitening;
	fprintf(stderr, "Testing whether ntfs_desx_key_expand() works: %s\n",
			res ? "SUCCESS" : "FAILED");
	return res;
}

static BOOL ntfs_des_test(void)
{
	const u8 known_des_key[8] = {
		0x27, 0xd1, 0x93, 0x09, 0xcb, 0x78, 0x93, 0x1f
	};
	const u8 known_des_encrypted_data[8] = {
		0xdc, 0xf7, 0x68, 0x2a, 0xaf, 0x48, 0x53, 0x0f
	};
	const u8 known_decrypted_data[8] = {
		0xd8, 0xd9, 0x15, 0x23, 0x5b, 0x88, 0x0e, 0x09
	};
	u8 test_decrypted_data[8];
	int res;
	gcry_error_t err;
	gcry_cipher_hd_t gcry_cipher_hd;

	err = gcry_cipher_open(&gcry_cipher_hd, GCRY_CIPHER_DES,
			GCRY_CIPHER_MODE_ECB, 0);
	if (err != GPG_ERR_NO_ERROR) {
		fprintf(stderr, "Failed to open des cipher (error 0x%x).\n",
				err);
		return FALSE;
	}
	err = gcry_cipher_setkey(gcry_cipher_hd, known_des_key,
			sizeof(known_des_key));
	if (err != GPG_ERR_NO_ERROR) {
		fprintf(stderr, "Failed to set des key (error 0x%x.\n", err);
		gcry_cipher_close(gcry_cipher_hd);
		return FALSE;
	}
	/*
	 * Apply DES decryption (ntfs actually uses encryption when decrypting).
	 */
	err = gcry_cipher_encrypt(gcry_cipher_hd, test_decrypted_data,
			sizeof(test_decrypted_data), known_des_encrypted_data,
			sizeof(known_des_encrypted_data));
	gcry_cipher_close(gcry_cipher_hd);
	if (err) {
		fprintf(stderr, "Failed to des decrypt test data (error "
				"0x%x).\n", err);
		return FALSE;
	}
	res = !memcmp(test_decrypted_data, known_decrypted_data,
			sizeof(known_decrypted_data));
	fprintf(stderr, "Testing whether des decryption works: %s\n",
			res ? "SUCCESS" : "FAILED");
	return res;
}

#else /* !defined(DO_CRYPTO_TESTS) */

static inline BOOL ntfs_desx_key_expand_test(void)
{
	return TRUE;
}

static inline BOOL ntfs_des_test(void)
{
	return TRUE;
}

#endif /* !defined(DO_CRYPTO_TESTS) */

static ntfs_fek *ntfs_fek_import_from_raw(u8 *fek_buf,
		unsigned fek_size __attribute__((unused)))
{
	ntfs_fek *fek;
	u32 key_size, wanted_key_size, gcry_algo;
	gcry_error_t err;

	// TODO: Sanity checking of sizes and offsets.
	key_size = *(u32*)fek_buf;
	fek = malloc(((((sizeof(*fek) + 7) & ~7) + key_size + 7) & ~7) +
			sizeof(gcry_cipher_hd_t));
	if (!fek) {
		errno = ENOMEM;
		return NULL;
	}
	fek->alg_id = *(u32*)(fek_buf + 8);
	fek->key_data = (u8*)fek + ((sizeof(*fek) + 7) & ~7);
	memcpy(fek->key_data, fek_buf + 16, key_size);
	fek->des_gcry_cipher_hd_ptr = NULL;
	*(gcry_cipher_hd_t***)(fek->key_data + ((key_size + 7) & ~7)) =
			&fek->des_gcry_cipher_hd_ptr;
	switch (fek->alg_id) {
	case CALG_DESX:
		if (!ntfs_desx_module) {
			if (!ntfs_desx_key_expand_test() || !ntfs_des_test()) {
				err = EINVAL;
				goto out;
			}
			err = gcry_cipher_register(&ntfs_desx_cipher,
					&ntfs_desx_algorithm_id,
					&ntfs_desx_module);
			if (err != GPG_ERR_NO_ERROR) {
				fprintf(stderr, "Failed to register desx "
						"cipher: %s\n",
						gcry_strerror(err));
				err = EINVAL;
				goto out;
			}
		}
		wanted_key_size = 16;
		gcry_algo = ntfs_desx_algorithm_id;
		break;
	case CALG_3DES:
		wanted_key_size = 24;
		gcry_algo = GCRY_CIPHER_3DES;
		break;
	case CALG_AES_256:
		wanted_key_size = 32;
		gcry_algo = GCRY_CIPHER_AES256;
		break;
	default:
		wanted_key_size = 8;
		gcry_algo = GCRY_CIPHER_DES;
		fprintf(stderr, "DES is not supported at present.  Please "
				"email linux-ntfs-dev@lists.sourceforge.net "
				"and say that you saw this message.  We will "
				"then implement support for DES.\n");
		err = EOPNOTSUPP;
		goto out;
	}
	if (key_size != wanted_key_size) {
		fprintf(stderr, "%s key of %u bytes but needed size is %u "
				"bytes, assuming corrupt key.  Aborting.\n",
				gcry_cipher_algo_name(gcry_algo),
				(unsigned)key_size, (unsigned)wanted_key_size);
		err = EIO;
		goto out;
	}
	err = gcry_cipher_open(&fek->gcry_cipher_hd, gcry_algo,
			GCRY_CIPHER_MODE_CBC, 0);
	if (err != GPG_ERR_NO_ERROR) {
		fprintf(stderr, "gcry_cipher_open() failed: %s\n",
				gcry_strerror(err));
		err = EINVAL;
		goto out;
	}
	err = gcry_cipher_setkey(fek->gcry_cipher_hd, fek->key_data, key_size);
	if (err != GPG_ERR_NO_ERROR) {
		fprintf(stderr, "gcry_cipher_setkey() failed: %s\n",
				gcry_strerror(err));
		gcry_cipher_close(fek->gcry_cipher_hd);
		err = EINVAL;
		goto out;
	}
	return fek;
out:
	free(fek);
	errno = err;
	return NULL;
}

static void ntfs_fek_release(ntfs_fek *fek)
{
	if (fek->des_gcry_cipher_hd_ptr)
		gcry_cipher_close(*fek->des_gcry_cipher_hd_ptr);
	gcry_cipher_close(fek->gcry_cipher_hd);
	free(fek);
}

static ntfs_fek *ntfs_df_array_fek_get(EFS_DF_ARRAY_HEADER *df_array,
		ntfs_rsa_private_key rsa_key)
{
	EFS_DF_HEADER *df_header;
	EFS_DF_CREDENTIAL_HEADER *df_cred;
	EFS_DF_CERT_THUMBPRINT_HEADER *df_cert;
	u8 *fek_buf;
	ntfs_fek *fek;
	u32 df_count, fek_size;
	unsigned i;

	df_header = (EFS_DF_HEADER*)(df_array + 1);
	df_count = le32_to_cpu(df_array->df_count);
	for (i = 0; i < df_count; i++) {
		df_cred = (EFS_DF_CREDENTIAL_HEADER*)((u8*)df_header +
				le32_to_cpu(df_header->cred_header_offset));
		df_cert = (EFS_DF_CERT_THUMBPRINT_HEADER*)((u8*)df_cred +
				le32_to_cpu(
				df_cred->cert_thumbprint_header_offset));
		fek_size = le32_to_cpu(df_header->fek_size);
		fek_buf = (u8*)df_header + le32_to_cpu(df_header->fek_offset);
		/* Decrypt the FEK.  Note: This is done in place. */
		fek_size = ntfs_raw_fek_decrypt(fek_buf, fek_size, rsa_key);
		if (fek_size) {
			/* Convert the FEK to our internal format. */
			fek = ntfs_fek_import_from_raw(fek_buf, fek_size);
			if (fek)
				return fek;
			fprintf(stderr, "Failed to convert the decrypted file "
					"encryption key to internal format.\n");
		} else
			fprintf(stderr, "Failed to decrypt the file "
					"encryption key.\n");
		df_header = (EFS_DF_HEADER*)((u8*)df_header +
				le32_to_cpu(df_header->df_length));
	}
	return NULL;
}

/**
 * ntfs_inode_fek_get -
 */
static ntfs_fek *ntfs_inode_fek_get(ntfs_inode *inode,
		ntfs_rsa_private_key rsa_key)
{
	ntfs_attr *na;
	EFS_ATTR_HEADER *efs;
	EFS_DF_ARRAY_HEADER *df_array;
	ntfs_fek *fek = NULL;

	/* Obtain the $EFS contents. */
	na = ntfs_attr_open(inode, AT_LOGGED_UTILITY_STREAM, EFS, 4);
	if (!na) {
		perror("Failed to open $EFS attribute");
		return NULL;
	}
	efs = malloc(na->data_size);
	if (!efs) {
		perror("Failed to allocate internal buffer");
		ntfs_attr_close(na);
		return NULL;
	}
	if (ntfs_attr_pread(na, 0, na->data_size, efs) != na->data_size) {
		perror("Failed to read $EFS attribute");
		free(efs);
		ntfs_attr_close(na);
		return NULL;
	}
	ntfs_attr_close(na);
	/* Iterate through the DDFs & DRFs until we obtain a key. */
	if (efs->offset_to_ddf_array) {
		df_array = (EFS_DF_ARRAY_HEADER*)((u8*)efs +
				le32_to_cpu(efs->offset_to_ddf_array));
		fek = ntfs_df_array_fek_get(df_array, rsa_key);
	}
	if (!fek && efs->offset_to_drf_array) {
		df_array = (EFS_DF_ARRAY_HEADER*)((u8*)efs +
				le32_to_cpu(efs->offset_to_drf_array));
		fek = ntfs_df_array_fek_get(df_array, rsa_key);
	}
	free(efs);
	return fek;
}

static int ntfs_fek_decrypt_sector(ntfs_fek *fek, u8 *data, const u64 offset)
{
	gcry_error_t err;

	err = gcry_cipher_reset(fek->gcry_cipher_hd);
	if (err != GPG_ERR_NO_ERROR) {
		fprintf(stderr, "Failed to reset cipher: %s\n",
				gcry_strerror(err));
		return -1;
	}
	/*
	 * Note: You may wonder why we are not calling gcry_cipher_setiv() here
	 * instead of doing it by hand after the decryption.  The answer is
	 * that gcry_cipher_setiv() wants an iv of length 8 bytes but we give
	 * it a length of 16 for AES256 so it does not like it.
	 */
	err = gcry_cipher_decrypt(fek->gcry_cipher_hd, data, 512, NULL, 0);
	if (err != GPG_ERR_NO_ERROR) {
		fprintf(stderr, "Decryption failed: %s\n", gcry_strerror(err));
		return -1;
	}
	/* Apply the IV. */
	if (fek->alg_id == CALG_AES_256) {
		((u64*)data)[0] ^= 0x5816657be9161312ULL + offset;
		((u64*)data)[1] ^= 0x1989adbe44918961ULL + offset;
	} else {
		/* All other algos (Des, 3Des, DesX) use the same IV. */
		((u64*)data)[0] ^= 0x169119629891ad13ULL + offset;
	}
	return 512;
}

/**
 * cat
 */
// TODO:
static int ntfs_cat_decrypt(ntfs_inode *inode, ntfs_fek *fek)
{
	int bufsize = 512;
	char *buffer;
	ntfs_attr *attr;
	s64 bytes_read, written, offset, total;
	s64 old_data_size, old_initialized_size;
	int i;

	buffer = malloc(bufsize);
	if (!buffer)
		return 1;
	attr = ntfs_attr_open(inode, AT_DATA, NULL, 0);
	if (!attr) {
		Eprintf("Cannot cat a directory.\n");
		free(buffer);
		return 1;
	}
	total = attr->data_size;

	// hack: make sure attr will not be commited to disk if you use this.
	// clear the encrypted bit, otherwise the library won't allow reading.
	NAttrClearEncrypted(attr);
	// extend the size, we may need to read past the end of the stream.
	old_data_size = attr->data_size;
	old_initialized_size = attr->initialized_size;
	attr->data_size = attr->initialized_size = attr->allocated_size;

	offset = 0;
	while (total > 0) {
		bytes_read = ntfs_attr_pread(attr, offset, 512, buffer);
		if (bytes_read == -1) {
			perror("ERROR: Couldn't read file");
			break;
		}
		if (!bytes_read)
			break;
		if ((i = ntfs_fek_decrypt_sector(fek, buffer, offset)) <
				bytes_read) {
			perror("ERROR: Couldn't decrypt all data!");
			Eprintf("%u/%lld/%lld/%lld\n", i, (long long)bytes_read,
					(long long)offset, (long long)total);
			break;
		}
		if (bytes_read > total)
			bytes_read = total;
		written = fwrite(buffer, 1, bytes_read, stdout);
		if (written != bytes_read) {
			perror("ERROR: Couldn't output all data!");
			break;
		}
		offset += bytes_read;
		total -= bytes_read;
	}
	attr->data_size = old_data_size;
	attr->initialized_size = old_initialized_size;
	NAttrSetEncrypted(attr);
	ntfs_attr_close(attr);
	free(buffer);
	return 0;
}

/**
 * main - Begin here
 *
 * Start from here.
 *
 * Return:  0  Success, the program worked
 *	    1  Error, something went wrong
 */
int main(int argc, char *argv[])
{
	u8 *pfx_buf, *password;
	ntfs_rsa_private_key rsa_key;
	ntfs_volume *vol;
	ntfs_inode *inode;
	ntfs_fek *fek;
	int pfx_size, res;

	if (!parse_options(argc, argv))
		return 1;
	utils_set_locale();

	/* Initialize crypto in ntfs. */
	if (ntfs_crypto_init()) {
		fprintf(stderr, "Failed to initialize crypto.  Aborting.\n");
		return 1;
	}
	/* Load the PKCS#12 (.pfx) file containing the user's private key. */
	if (ntfs_pkcs12_load_pfxfile(opts.keyfile, &pfx_buf, &pfx_size)) {
		fprintf(stderr, "Failed to load key file.  Aborting.\n");
		ntfs_crypto_deinit();
		return 1;
	}
	/* Ask the user for their password. */
	password = getpass("Enter the password with which the private key was "
			"encrypted: ");
	if (!password) {
		perror("Failed to obtain user password");
		free(pfx_buf);
		ntfs_crypto_deinit();
		return 1;
	}
	/* Obtain the user's private RSA key from the key file. */
	rsa_key = ntfs_pkcs12_extract_rsa_key(pfx_buf, pfx_size, password);
	/* Destroy the password. */
	memset(password, 0, strlen(password));
	/* No longer need the pfx file contents. */
	free(pfx_buf);
	if (!rsa_key) {
		fprintf(stderr, "Failed to extract the private RSA key.  Did "
				"you perhaps mistype the password?\n");
		ntfs_crypto_deinit();
		return 1;
	}
	/* Mount the ntfs volume. */
	vol = utils_mount_volume(opts.device, MS_RDONLY, opts.force);
	if (!vol) {
		fprintf(stderr, "Failed to mount ntfs volume.  Aborting.\n");
		ntfs_rsa_private_key_release(rsa_key);
		ntfs_crypto_deinit();
		return 1;
	}
	/* Open the encrypted ntfs file. */
	if (opts.inode != -1)
		inode = ntfs_inode_open(vol, opts.inode);
	else
		inode = ntfs_pathname_to_inode(vol, NULL, opts.file);
	if (!inode) {
		fprintf(stderr, "Failed to open encrypted file.  Aborting.\n");
		ntfs_umount(vol, FALSE);
		ntfs_rsa_private_key_release(rsa_key);
		ntfs_crypto_deinit();
		return 1;
	}
	/* Obtain the file encryption key of the encrypted file. */
	fek = ntfs_inode_fek_get(inode, rsa_key);
	ntfs_rsa_private_key_release(rsa_key);
	if (fek) {
		res = ntfs_cat_decrypt(inode, fek);
		ntfs_fek_release(fek);
	} else {
		fprintf(stderr, "Failed to obtain file encryption key.  "
				"Aborting.\n");
		res = 1;
	}
	ntfs_inode_close(inode);
	ntfs_umount(vol, FALSE);
	ntfs_crypto_deinit();
	return res;
}
