/* $Id: dsp_cmx.c,v 1.10 2005/10/26 14:12:13 keil Exp $
 *
 * Audio crossconnecting/conferrencing (hardware level).
 *
 * Copyright 2002 by Andreas Eversberg (jolly@jolly.de)
 *
 * This software may be used and distributed according to the terms
 * of the GNU General Public License, incorporated herein by reference.
 *
 */

/*
 * The process of adding and removing parties to/from a conference:
 *
 * There is a chain of conference_t which has one or more members in a chain
 * of conf_member_t.
 *
 * After a party is added, the conference is checked for hardware capability.
 * Also if a party is removed, the conference is checked again.
 *
 * There are 3 different solutions: -1 = software, 0 = hardware-crossconnect
 * 1-n = hardware-conference. The n will give the conference number.
 *
 * Depending on the change after removal or insertion of a party, hardware
 * commands are given.
 *
 * The current solution is stored within the conference_t entry.
 */

/* HOW THE CMX WORKS:
 *
 * There are 3 types of interaction: One member is alone, in this case only
 * data flow is done.
 * Two members will also exchange their data so they are crossconnected.
 * Three or more members will be added in a conference and will hear each
 * other but will not receive their own speech (echo) if not enabled.
 *
 * Features of CMX are:
 *  - Crossconnecting or even conference, if more than two members are together.
 *  - Force mixing of transmit data with other crossconnect/conference members.
 *  - Echo generation to benchmark the delay of audio processing.
 *  - Use hardware to minimize cpu load, disable FIFO load and minimize delay.
 *
 * There are 3 buffers:
 *
 * The conference buffer
 *
 *  R-3     R-2   R-1    W-2    W-1 W-3
 *   |       |     |      |      |   |
 * --+-------+-----+------+------+---+---------------
 *                        |          |
 *                      W-min      W-max
 *
 * The conference buffer is a ring buffer used to mix all data from all members.
 * To compensate the echo, data of individual member will later be substracted
 * before it is sent to that member. Each conference has a W-min and a W-max
 * pointer. Each individual member has a write pointer (W-x) and a read pointer
 * (R-x). W-min shows the lowest value of all W-x. The W-max shows the highest
 * value of all W-x. Whenever data is written, it is mixed by adding to the
 * existing sample value in the buffer. If W-max would increase, the additional
 * range is cleared so old data will be erased in the ring buffer.
 *
 *
 * RX-Buffer
 *                R-1           W-1
 *                 |             |
 * ----------------+-------------+-------------------
 * 
 * The rx-buffer is a ring buffer used to store the received data for each
 * individual member. To compensate echo, this data will later be substracted
 * from the conference's data before it is sent to that member. If only two
 * members are in one conference, this data is used to get the queued data from
 * the other member.
 *
 *
 * TX-Buffer
 *                  R        W
 *                  |        |
 * -----------------+--------+-----------------------
 *
 * The tx-buffer is a ring buffer to queue the transmit data from user space
 * until it will be mixed or sent. There are two pointers, R and W. If the write
 * pointer W would reach or overrun R, the buffer would overrun. In this case
 * (some) data is dropped so that it will not overrun.
 *
 *
 * If a member joins a conference:
 *
 * - If a member joins, its rx_buff is set to silence.
 * - If the conference reaches three members, the conf-buffer is cleared.
 * - When a member is joined, it will set its write and read pointer to W_max.
 *
 * The procedure of received data from card is explained in cmx_receive.
 * The procedure of received data from user space is explained in cmx_transmit.
 *
 *
 * LIMITS:
 *
 * The max_queue value is 2* the samples of largest packet ever received by any
 * conference member from her card. It also changes during life of conference.
 *
 *
 * AUDIO PROCESS:
 *
 * Writing data to conference's and member's buffer is done by adding the sample
 * value to the existing ring buffer. Writing user space data to the member's
 * buffer is done by substracting the sample value from the existing ring
 * buffer.
 *
 *
 * Interaction with other features:
 *
 * DTMF:
 * DTMF decoding is done before the data is crossconnected.
 *
 * Volume change:
 * Changing rx-volume is done before the data is crossconnected. The tx-volume
 * must be changed whenever data is transmitted to the card by the cmx.
 *
 * Tones:
 * If a tone is enabled, it will be processed whenever data is transmitted to
 * the card. It will replace the tx-data from the user space.
 * If tones are generated by hardware, this conference member is removed for
 * this time.
 *
 * Disable rx-data:
 * If cmx is realized in hardware, rx data will be disabled if requested by
 * the upper layer. If dtmf decoding is done by software and enabled, rx data
 * will not be diabled but blocked to the upper layer.
 *
 * HFC conference engine:
 * If it is possible to realize all features using hardware, hardware will be
 * used if not forbidden by control command. Disabling rx-data provides
 * absolutely traffic free audio processing. (except for the quick 1-frame
 * upload of a tone loop, only once for a new tone)
 *
 */

// delay.h is required for hw_lock.h
#include <linux/delay.h>
#include <linux/vmalloc.h>
#include "layer1.h"
#include "helper.h"
#include "debug.h"
#include "dsp.h"
#include "hw_lock.h"

//#define CMX_CONF_DEBUG /* debugging of multi party conference, by using conference even with two members */
//#define CMX_DEBUG /* massive read/write pointer output */

extern mISDN_HWlock_t dsp_lock;
LIST_HEAD(Conf_list);

/*
 * debug cmx memory structure
 */
void
dsp_cmx_debug(dsp_t *dsp)
{
	conference_t *conf;
	conf_member_t *member;
	dsp_t *odsp;

	printk(KERN_DEBUG "-----Current DSP\n");
	list_for_each_entry(odsp, &dsp_obj.ilist, list)
	{
		printk(KERN_DEBUG "* %s echo=%d txmix=%d", odsp->inst.name, odsp->echo, odsp->tx_mix);
		if (odsp->conf)
			printk(" (Conf %d)", odsp->conf->id);
		if (dsp == odsp)
			printk(" *this*");
		printk("\n");
	}
	
	printk(KERN_DEBUG "-----Current Conf:\n");
	list_for_each_entry(conf, &Conf_list, list)
	{
		printk(KERN_DEBUG "* Conf %d (0x%x)\n", conf->id, (u32)conf);
		list_for_each_entry(member, &conf->mlist, list)
		{
			printk(KERN_DEBUG "  - member = %s (slot_tx %d, bank_tx %d, slot_rx %d, bank_rx %d hfc_conf %d)%s\n", member->dsp->inst.name, member->dsp->pcm_slot_tx, member->dsp->pcm_bank_tx, member->dsp->pcm_slot_rx, member->dsp->pcm_bank_rx, member->dsp->hfc_conf, (member->dsp==dsp)?" *this*":"");
		}
	}
	printk(KERN_DEBUG "-----end\n");
}

/*
 * search conference
 */
static conference_t 
*dsp_cmx_search_conf(u32 id)
{
	conference_t *conf;

	if (!id) {
		printk(KERN_WARNING "%s: conference ID is 0.\n", 
			__FUNCTION__);
		return(NULL);
	}

	/* search conference */
	list_for_each_entry(conf, &Conf_list, list)
		if (conf->id == id)
			return(conf);

	return(NULL);
}


/*
 * add member to conference
 */
static int
dsp_cmx_add_conf_member(dsp_t *dsp, conference_t *conf)
{
	conf_member_t *member;

	if (!conf || !dsp) {
		printk(KERN_WARNING "%s: conf or dsp is 0.\n", __FUNCTION__);
		return(-EINVAL);
	}
	if (dsp->member) {
		printk(KERN_WARNING "%s: dsp is already member in a conf.\n", 
			__FUNCTION__);
		return(-EINVAL);
	}

	if (dsp->conf) {
		printk(KERN_WARNING "%s: dsp is already in a conf.\n", 
			__FUNCTION__);
		return(-EINVAL);
	}

	unlock_HW(&dsp_lock);
	if (!(member = vmalloc(sizeof(conf_member_t)))) {
		lock_HW(&dsp_lock, 0);
		printk(KERN_ERR "vmalloc conf_member_t failed\n");
		return(-ENOMEM);
	}
	lock_HW(&dsp_lock, 0);
	memset(member, 0, sizeof(conf_member_t));
	memset(dsp->rx_buff, dsp_silence, sizeof(dsp->rx_buff));
	member->dsp = dsp;
	/* set initial values */
	dsp->W_rx = conf->W_max;
	dsp->R_rx = conf->W_max;

	list_add_tail(&member->list, &conf->mlist);

	/* zero conf-buffer if we change from 2 to 3 members */
	if (3 == count_list_member(&conf->mlist))
		memset(conf->conf_buff, 0, sizeof(conf->conf_buff));

	dsp->conf = conf;
	dsp->member = member;

	return(0);
}


/*
 * del member from conference
 */
int
dsp_cmx_del_conf_member(dsp_t *dsp)
{
	conf_member_t *member;

	if (!dsp) {
		printk(KERN_WARNING "%s: dsp is 0.\n", 
			__FUNCTION__);
		return(-EINVAL);
	}

	if (!dsp->conf) {
		printk(KERN_WARNING "%s: dsp is not in a conf.\n", 
			__FUNCTION__);
		return(-EINVAL);
	}

	if (list_empty(&dsp->conf->mlist)) {
		printk(KERN_WARNING "%s: dsp has linked an empty conf.\n", 
			__FUNCTION__);
		return(-EINVAL);
	}

	/* find us in conf */
	list_for_each_entry(member, &dsp->conf->mlist, list) {
		if (member->dsp == dsp) {
			list_del(&member->list);
			dsp->conf = NULL;
			dsp->member = NULL;
			unlock_HW(&dsp_lock);
			vfree(member);
			lock_HW(&dsp_lock, 0);
			return(0);
		}
	}
	printk(KERN_WARNING "%s: dsp is not present in its own conf_meber list.\n", 
		__FUNCTION__);

	return(-EINVAL);
}


/*
 * new conference
 */
static conference_t
*dsp_cmx_new_conf(u32 id)
{
	conference_t *conf;

	if (!id) {
		printk(KERN_WARNING "%s: id is 0.\n", 
			__FUNCTION__);
		return(NULL);
	}

	unlock_HW(&dsp_lock);
	if (!(conf = vmalloc(sizeof(conference_t)))) {
		lock_HW(&dsp_lock, 0);
		printk(KERN_ERR "vmalloc conference_t failed\n");
		return(NULL);
	}
	lock_HW(&dsp_lock, 0);
	memset(conf, 0, sizeof(conference_t));
	INIT_LIST_HEAD(&conf->mlist);
	conf->id = id;

	list_add_tail(&conf->list, &Conf_list);

	return(conf);
}


/*
 * del conference
 */
int
dsp_cmx_del_conf(conference_t *conf)
{
	if (!conf) {
		printk(KERN_WARNING "%s: conf is null.\n", 
			__FUNCTION__);
		return(-EINVAL);
	}

	if (!list_empty(&conf->mlist)) {
		printk(KERN_WARNING "%s: conf not empty.\n", 
			__FUNCTION__);
		return(-EINVAL);
	}
	list_del(&conf->list);
	unlock_HW(&dsp_lock);
	vfree(conf);
	lock_HW(&dsp_lock, 0);

	return(0);
}


/*
 * send HW message to hfc card
 */
static void
dsp_cmx_hw_message(dsp_t *dsp, u32 message, u32 param1, u32 param2, u32 param3, u32 param4)
{
	struct sk_buff *nskb;
	u32 param[4];

	param[0] = param1;
	param[1] = param2;
	param[2] = param3;
	param[3] = param4;
	nskb = create_link_skb(PH_CONTROL | REQUEST, message, sizeof(param), param, 0);
	if (!nskb) {
		printk(KERN_ERR "%s: No mem for skb.\n", __FUNCTION__);
		return;
	}
	/* unlocking is not required, because we don't expect a response */
	if (dsp->inst.down.func(&dsp->inst.down, nskb))
		dev_kfree_skb(nskb);
}


/*
 * do hardware update and set the software/hardware flag
 *
 * either a conference or a dsp instance can be given
 * if only dsp instance is given, the instance is not associated with a conf
 * and therefore removed. if a conference is given, the dsp is expected to
 * be member of that conference.
 */
void 
dsp_cmx_hardware(conference_t *conf, dsp_t *dsp)
{
	conf_member_t *member, *nextm;
	dsp_t *finddsp;
	int memb = 0, i, ii, i1, i2;
	int freeunits[8];
	u_char freeslots[256];
	int same_hfc = -1, same_pcm = -1, current_conf = -1, all_conf = 1;

	/* dsp gets updated (no conf) */
//printk("-----1\n");
	if (!conf) {
//printk("-----2\n");
		if (!dsp)
			return;
//printk("-----3\n");
		if (dsp_debug & DEBUG_DSP_CMX)
			printk(KERN_DEBUG "%s checking dsp %s\n", __FUNCTION__, dsp->inst.name);
//printk("-----a\n");
		one_member:
		/* remove HFC conference if enabled */
		if (dsp->hfc_conf >= 0) {
			if (dsp_debug & DEBUG_DSP_CMX)
				printk(KERN_DEBUG "%s removing %s from HFC conf %d because dsp is split\n", __FUNCTION__, dsp->inst.name, dsp->hfc_conf);
			dsp_cmx_hw_message(dsp, HW_CONF_SPLIT, 0, 0, 0, 0);
			dsp->hfc_conf = -1;
		}
		if (!dsp->echo) {
			/* NO ECHO: remove PCM slot if assigned */
			if (dsp->pcm_slot_tx>=0 || dsp->pcm_slot_rx>=0) {
				if (dsp_debug & DEBUG_DSP_CMX)
					printk(KERN_DEBUG "%s removing %s from PCM slot %d (TX) %d (RX) because dsp is split (no echo)\n", __FUNCTION__, dsp->inst.name, dsp->pcm_slot_tx, dsp->pcm_slot_rx);
				dsp_cmx_hw_message(dsp, HW_PCM_DISC, 0, 0, 0, 0);
				dsp->pcm_slot_tx = -1;
				dsp->pcm_bank_tx = -1;
				dsp->pcm_slot_rx = -1;
				dsp->pcm_bank_rx = -1;
			}
			return;
		}
		/* ECHO: already echo */
		if (dsp->pcm_slot_tx>=0 && dsp->pcm_slot_rx<0
		 && dsp->pcm_bank_tx==2 && dsp->pcm_bank_rx==2)
			return;
		/* ECHO: if slot already assigned */
		if (dsp->pcm_slot_tx>=0) {
			dsp->pcm_slot_rx = dsp->pcm_slot_tx;
			dsp->pcm_bank_tx = 2; /* loop */
			dsp->pcm_bank_rx = 2;
			if (dsp_debug & DEBUG_DSP_CMX)
				printk(KERN_DEBUG "%s refresh %s for echo using slot %d\n", __FUNCTION__, dsp->inst.name, dsp->pcm_slot_tx);
			dsp_cmx_hw_message(dsp, HW_PCM_CONN, dsp->pcm_slot_tx, 2, dsp->pcm_slot_rx, 2);
			return;
		}
		/* ECHO: find slot */
		dsp->pcm_slot_tx = -1;
		dsp->pcm_slot_rx = -1;
		memset(freeslots, 1, sizeof(freeslots));
		list_for_each_entry(finddsp, &dsp_obj.ilist, list) {
			 if (finddsp->features.pcm_id==dsp->features.pcm_id) {
				if (finddsp->pcm_slot_rx>=0
				 && finddsp->pcm_slot_rx<sizeof(freeslots))
					freeslots[finddsp->pcm_slot_tx] = 0;
				if (finddsp->pcm_slot_tx>=0
				 && finddsp->pcm_slot_tx<sizeof(freeslots))
					freeslots[finddsp->pcm_slot_rx] = 0;
			}
		}
		i = 0;
		ii = dsp->features.pcm_slots;
		while(i < ii) {
			if (freeslots[i])
				break;
			i++;
		}
		if (i == ii) {
			if (dsp_debug & DEBUG_DSP_CMX)
				printk(KERN_DEBUG "%s no slot available for echo\n", __FUNCTION__);
			/* no more slots available */
			return;
		}
		/* assign free slot */
		dsp->pcm_slot_tx = i;
		dsp->pcm_slot_rx = i;
		dsp->pcm_bank_tx = 2; /* loop */
		dsp->pcm_bank_rx = 2;
		if (dsp_debug & DEBUG_DSP_CMX)
			printk(KERN_DEBUG "%s assign echo for %s using slot %d\n", __FUNCTION__, dsp->inst.name, dsp->pcm_slot_tx);
		dsp_cmx_hw_message(dsp, HW_PCM_CONN, dsp->pcm_slot_tx, 2, dsp->pcm_slot_rx, 2);
		return;
	}

//printk("-----4\n");
	/* conf gets updated (all members) */
	if (dsp_debug & DEBUG_DSP_CMX)
		printk(KERN_DEBUG "%s checking conference %d\n", __FUNCTION__, conf->id);
//printk("-----5\n");
	
	if (list_empty(&conf->mlist)) {
		printk(KERN_ERR "%s: conference whithout members\n", __FUNCTION__);
		return;
	}
	member = list_entry(conf->mlist.next, conf_member_t, list);
	same_hfc = member->dsp->features.hfc_id;
	same_pcm = member->dsp->features.pcm_id;
	/* check all members in our conference */
	list_for_each_entry(member, &conf->mlist, list) {
		/* check if member uses mixing */
		if (member->dsp->tx_mix) {
			if (dsp_debug & DEBUG_DSP_CMX)
				printk(KERN_DEBUG "%s dsp %s cannot form a conf, because tx_mix is turned on\n", __FUNCTION__, member->dsp->inst.name);
			conf_software:
			list_for_each_entry(member, &conf->mlist, list) {
				dsp = member->dsp;
				/* remove HFC conference if enabled */
				if (dsp->hfc_conf >= 0) {
					if (dsp_debug & DEBUG_DSP_CMX)
						printk(KERN_DEBUG "%s removing %s from HFC conf %d because not possible with hardware\n", __FUNCTION__, dsp->inst.name, dsp->hfc_conf);
					dsp_cmx_hw_message(dsp, HW_CONF_SPLIT, 0, 0, 0, 0);
					dsp->hfc_conf = -1;
				}
				/* remove PCM slot if assigned */
				if (dsp->pcm_slot_tx>=0 || dsp->pcm_slot_rx>=0) {
					if (dsp_debug & DEBUG_DSP_CMX)
						printk(KERN_DEBUG "%s removing %s from PCM slot %d (TX) slot %d (RX) because not possible with hardware\n", __FUNCTION__, dsp->inst.name, dsp->pcm_slot_tx, dsp->pcm_slot_rx);
					dsp_cmx_hw_message(dsp, HW_PCM_DISC, 0, 0, 0, 0);
					dsp->pcm_slot_tx = -1;
					dsp->pcm_bank_tx = -1;
					dsp->pcm_slot_rx = -1;
					dsp->pcm_bank_rx = -1;
				}
			}
			conf->hardware = 0;
			conf->software = 1;
			return;
		}
		/* check if member has echo turned on */
		if (member->dsp->echo) {
			if (dsp_debug & DEBUG_DSP_CMX)
				printk(KERN_DEBUG "%s dsp %s cannot form a conf, because echo is turned on\n", __FUNCTION__, member->dsp->inst.name);
			goto conf_software;
		}
		/* check if member has tx_mix turned on */
		if (member->dsp->tx_mix) {
			if (dsp_debug & DEBUG_DSP_CMX)
				printk(KERN_DEBUG "%s dsp %s cannot form a conf, because tx_mix is turned on\n", __FUNCTION__, member->dsp->inst.name);
			goto conf_software;
		}
		/* check if member changes volume at an not suppoted level */
		if (member->dsp->tx_volume) {
			if (dsp_debug & DEBUG_DSP_CMX)
				printk(KERN_DEBUG "%s dsp %s cannot form a conf, because tx_volume is changed\n", __FUNCTION__, member->dsp->inst.name);
			goto conf_software;
		}
		if (member->dsp->rx_volume) {
			if (dsp_debug & DEBUG_DSP_CMX)
				printk(KERN_DEBUG "%s dsp %s cannot form a conf, because rx_volume is changed\n", __FUNCTION__, member->dsp->inst.name);
			goto conf_software;
		}
		/* check if encryption is enabled */
		if (member->dsp->bf_enable) {
			if (dsp_debug & DEBUG_DSP_CMX)
				printk(KERN_DEBUG "%s dsp %s cannot form a conf, because encryption is enabled\n", __FUNCTION__, member->dsp->inst.name);
			goto conf_software;
		}
		/* check if member is on a card with PCM support */
		if (member->dsp->features.pcm_id < 0) {
			if (dsp_debug & DEBUG_DSP_CMX)
				printk(KERN_DEBUG "%s dsp %s cannot form a conf, because dsp has no PCM bus\n", __FUNCTION__, member->dsp->inst.name);
			goto conf_software;
		}
		/* check if relations are on the same PCM bus */
		if (member->dsp->features.pcm_id != same_pcm) {
			if (dsp_debug & DEBUG_DSP_CMX)
				printk(KERN_DEBUG "%s dsp %s cannot form a conf, because dsp is on a different PCM bus than the first dsp\n",
					__FUNCTION__, member->dsp->inst.name);
			goto conf_software;
		}
		/* determine if members are on the same hfc chip */
		if (same_hfc != member->dsp->features.hfc_id)
			same_hfc = -1;
		/* if there are members already in a conference */
		if (current_conf<0 && member->dsp->hfc_conf>=0)
			current_conf = member->dsp->hfc_conf;
		/* if any member is not in a conference */
		if (member->dsp->hfc_conf < 0)
			all_conf = 0;

		memb++;
	}

	/* if no member, this is an error */
	if (memb < 1)
		return;

	/* one member */
	if (memb == 1) {
		if (dsp_debug & DEBUG_DSP_CMX)
			printk(KERN_DEBUG "%s conf %d cannot form a HW conference, because dsp is alone\n", __FUNCTION__, conf->id);
		conf->hardware = 0;
		conf->software = 1;
		member = list_entry(conf->mlist.next, conf_member_t, list);
		dsp = member->dsp;
		goto one_member;
	}

	/* ok, now we are sure that all members are on the same pcm.
	 * now we will see if we have only two members, so we can do
	 * crossconnections, which don't have any limitations.
	 */

	/* if we have only two members */
	if (memb == 2) {
		member = list_entry(conf->mlist.next, conf_member_t, list);
		nextm = list_entry(member->list.next, conf_member_t, list);
		/* remove HFC conference if enabled */
		if (member->dsp->hfc_conf >= 0) {
			if (dsp_debug & DEBUG_DSP_CMX)
				printk(KERN_DEBUG "%s removing %s from HFC conf %d because two parties require only a PCM slot\n", __FUNCTION__, member->dsp->inst.name, member->dsp->hfc_conf);
			dsp_cmx_hw_message(member->dsp, HW_CONF_SPLIT, 0, 0, 0, 0);
			member->dsp->hfc_conf = -1;
		}
		if (nextm->dsp->hfc_conf >= 0) {
			if (dsp_debug & DEBUG_DSP_CMX)
				printk(KERN_DEBUG "%s removing %s from HFC conf %d because two parties require only a PCM slot\n", __FUNCTION__, nextm->dsp->inst.name, nextm->dsp->hfc_conf);
			dsp_cmx_hw_message(nextm->dsp, HW_CONF_SPLIT, 0, 0, 0, 0);
			nextm->dsp->hfc_conf = -1;
		}
		/* if members have two banks (and not on the same chip) */
		if (member->dsp->features.pcm_banks>1
		 && nextm->dsp->features.pcm_banks>1
		 && member->dsp->features.hfc_id!=nextm->dsp->features.hfc_id) {
			/* if both members have same slots with crossed banks */
			if (member->dsp->pcm_slot_tx>=0
			 && member->dsp->pcm_slot_rx>=0
			 && nextm->dsp->pcm_slot_tx>=0
			 && nextm->dsp->pcm_slot_rx>=0
			 && nextm->dsp->pcm_slot_tx==member->dsp->pcm_slot_rx
			 && nextm->dsp->pcm_slot_rx==member->dsp->pcm_slot_tx
			 && nextm->dsp->pcm_slot_tx==member->dsp->pcm_slot_tx
			 && member->dsp->pcm_bank_tx!=member->dsp->pcm_bank_rx
			 && nextm->dsp->pcm_bank_tx!=nextm->dsp->pcm_bank_rx) {
				/* all members have same slot */
				if (dsp_debug & DEBUG_DSP_CMX)
					printk(KERN_DEBUG "%s dsp %s & %s stay joined on PCM slot %d bank %d (TX) bank %d (RX) (on different chips)\n", __FUNCTION__,
						member->dsp->inst.name, nextm->dsp->inst.name,
						member->dsp->pcm_slot_tx, member->dsp->pcm_bank_tx, member->dsp->pcm_bank_rx);
				conf->hardware = 0;
				conf->software = 1;
				return;
			}
			/* find a new slot */
			memset(freeslots, 1, sizeof(freeslots));
			list_for_each_entry(dsp, &dsp_obj.ilist, list) {
				if (dsp!=member->dsp
				 && dsp!=nextm->dsp
				 && member->dsp->features.pcm_id==dsp->features.pcm_id) {
					if (dsp->pcm_slot_rx>=0
					 && dsp->pcm_slot_rx<sizeof(freeslots))
						freeslots[dsp->pcm_slot_tx] = 0;
					if (dsp->pcm_slot_tx>=0
					 && dsp->pcm_slot_tx<sizeof(freeslots))
						freeslots[dsp->pcm_slot_rx] = 0;
				}
			}
			i = 0;
			ii = member->dsp->features.pcm_slots;
			while(i < ii) {
				if (freeslots[i])
					break;
				i++;
			}
			if (i == ii) {
				if (dsp_debug & DEBUG_DSP_CMX)
					printk(KERN_DEBUG "%s no slot available for %s & %s\n", __FUNCTION__,
						member->dsp->inst.name, nextm->dsp->inst.name);
				/* no more slots available */
				goto conf_software;
			}
			/* assign free slot */
			member->dsp->pcm_slot_tx = i;
			member->dsp->pcm_slot_rx = i;
			nextm->dsp->pcm_slot_tx = i;
			nextm->dsp->pcm_slot_rx = i;
			member->dsp->pcm_bank_rx = 0;
			member->dsp->pcm_bank_tx = 1;
			nextm->dsp->pcm_bank_rx = 1;
			nextm->dsp->pcm_bank_tx = 0;
			if (dsp_debug & DEBUG_DSP_CMX)
				printk(KERN_DEBUG "%s adding %s & %s to new PCM slot %d (TX and RX on different chips) because both members have not same slots\n", __FUNCTION__,
					member->dsp->inst.name, nextm->dsp->inst.name, member->dsp->pcm_slot_tx);
			dsp_cmx_hw_message(member->dsp, HW_PCM_CONN, member->dsp->pcm_slot_tx, member->dsp->pcm_bank_tx,
				member->dsp->pcm_slot_rx, member->dsp->pcm_bank_rx);
			dsp_cmx_hw_message(nextm->dsp, HW_PCM_CONN, nextm->dsp->pcm_slot_tx, nextm->dsp->pcm_bank_tx,
				nextm->dsp->pcm_slot_rx, nextm->dsp->pcm_bank_rx);
			conf->hardware = 1;
			conf->software = 0;
			return;
		/* if members have one bank (or on the same chip) */
		} else {
			/* if both members have different crossed slots */
			if (member->dsp->pcm_slot_tx>=0
			 && member->dsp->pcm_slot_rx>=0
			 && nextm->dsp->pcm_slot_tx>=0
			 && nextm->dsp->pcm_slot_rx>=0
			 && nextm->dsp->pcm_slot_tx==member->dsp->pcm_slot_rx
			 && nextm->dsp->pcm_slot_rx==member->dsp->pcm_slot_tx
			 && member->dsp->pcm_slot_tx!=member->dsp->pcm_slot_rx
			 && member->dsp->pcm_bank_tx==0
			 && member->dsp->pcm_bank_rx==0
			 && nextm->dsp->pcm_bank_tx==0
			 && nextm->dsp->pcm_bank_rx==0) {
				/* all members have same slot */
				if (dsp_debug & DEBUG_DSP_CMX)
					printk(KERN_DEBUG "%s dsp %s & %s stay joined on PCM slot %d (TX) %d (RX) on same chip or one bank PCM)\n", __FUNCTION__,
						member->dsp->inst.name, nextm->dsp->inst.name, member->dsp->pcm_slot_tx, member->dsp->pcm_slot_rx);
				conf->hardware = 0;
				conf->software = 1;
				return;
			}
			/* find two new slot */
			memset(freeslots, 1, sizeof(freeslots));
			list_for_each_entry(dsp, &dsp_obj.ilist, list) {
				if (dsp!=member->dsp
				 && dsp!=nextm->dsp
				 && member->dsp->features.pcm_id==dsp->features.pcm_id) {
					if (dsp->pcm_slot_rx>=0
					 && dsp->pcm_slot_rx<sizeof(freeslots))
						freeslots[dsp->pcm_slot_tx] = 0;
					if (dsp->pcm_slot_tx>=0
					 && dsp->pcm_slot_tx<sizeof(freeslots))
						freeslots[dsp->pcm_slot_rx] = 0;
				}
			}
			i1 = 0;
			ii = member->dsp->features.pcm_slots;
			while(i1 < ii) {
				if (freeslots[i1])
					break;
				i1++;
			}
			if (i1 == ii) {
				if (dsp_debug & DEBUG_DSP_CMX)
					printk(KERN_DEBUG "%s no slot available for %s & %s\n", __FUNCTION__,
						member->dsp->inst.name, nextm->dsp->inst.name);
				/* no more slots available */
				goto conf_software;
			}
			i2 = i1+1;
			while(i2 < ii) {
				if (freeslots[i2])
					break;
				i2++;
			}
			if (i2 == ii) {
				if (dsp_debug & DEBUG_DSP_CMX)
					printk(KERN_DEBUG "%s no slot available for %s & %s\n", __FUNCTION__,
						member->dsp->inst.name, nextm->dsp->inst.name);
				/* no more slots available */
				goto conf_software;
			}
			/* assign free slots */
			member->dsp->pcm_slot_tx = i1;
			member->dsp->pcm_slot_rx = i2;
			nextm->dsp->pcm_slot_tx = i2;
			nextm->dsp->pcm_slot_rx = i1;
			member->dsp->pcm_bank_rx = 0;
			member->dsp->pcm_bank_tx = 0;
			nextm->dsp->pcm_bank_rx = 0;
			nextm->dsp->pcm_bank_tx = 0;
			if (dsp_debug & DEBUG_DSP_CMX)
				printk(KERN_DEBUG "%s adding %s & %s to new PCM slot %d (TX) %d (RX) on same chip or one bank PCM, because both members have not crossed slots\n", __FUNCTION__,
					member->dsp->inst.name, nextm->dsp->inst.name, member->dsp->pcm_slot_tx,
					member->dsp->pcm_slot_rx);
			dsp_cmx_hw_message(member->dsp, HW_PCM_CONN, member->dsp->pcm_slot_tx, member->dsp->pcm_bank_tx, member->dsp->pcm_slot_rx, member->dsp->pcm_bank_rx);
			dsp_cmx_hw_message(nextm->dsp, HW_PCM_CONN, nextm->dsp->pcm_slot_tx, nextm->dsp->pcm_bank_tx, nextm->dsp->pcm_slot_rx, nextm->dsp->pcm_bank_rx);
			conf->hardware = 1;
			conf->software = 0;
			return;
		}
	}

	/* if we have more than two, we may check if we have a conference
	 * unit available on the chip. also all members must be on the same
	 */

	/* if not the same HFC chip */
	if (same_hfc < 0) {
		if (dsp_debug & DEBUG_DSP_CMX)
			printk(KERN_DEBUG "%s conference %d cannot be formed, because members are on different chips or not on HFC chip\n",
				__FUNCTION__, conf->id);
		goto conf_software;
	}

	/* if all members already have the same conference */
	if (all_conf)
		return;

	/* if there is an existing conference, but not all members have joined
	 */
	if (current_conf >= 0) {
		join_members:
		list_for_each_entry(member, &conf->mlist, list) {
			/* join to current conference */
			if (member->dsp->hfc_conf == current_conf) {
				continue;
			}
			/* get a free timeslot first */
			memset(freeslots, 1, sizeof(freeslots));
			list_for_each_entry(dsp, &dsp_obj.ilist, list) {
				/* not checking current member, because
				 * slot will be overwritten.
				 */
				if (dsp!=member->dsp
				/* dsp must be on the same PCM */
				 && member->dsp->features.pcm_id==dsp->features.pcm_id) {
					/* dsp must be on a slot */
					if (dsp->pcm_slot_tx>=0
					 && dsp->pcm_slot_tx<sizeof(freeslots))
						freeslots[dsp->pcm_slot_tx] = 0;
					if (dsp->pcm_slot_rx>=0
					 && dsp->pcm_slot_rx<sizeof(freeslots))
						freeslots[dsp->pcm_slot_rx] = 0;
				}
			}
			i = 0;
			ii = member->dsp->features.pcm_slots;
			while(i < ii) {
				if (freeslots[i])
					break;
				i++;
			}
			if (i == ii) {
				/* no more slots available */
				if (dsp_debug & DEBUG_DSP_CMX)
					printk(KERN_DEBUG "%s conference %d cannot be formed, because no slot free\n", __FUNCTION__, conf->id);
				goto conf_software;
			}
			if (dsp_debug & DEBUG_DSP_CMX)
				printk(KERN_DEBUG "%s changing dsp %s to HW conference %d slot %d\n", __FUNCTION__, member->dsp->inst.name, current_conf, i);
			/* assign free slot & set PCM & join conf */
			member->dsp->pcm_slot_tx = i;
			member->dsp->pcm_slot_rx = i;
			member->dsp->pcm_bank_tx = 2; /* loop */
			member->dsp->pcm_bank_rx = 2;
			member->dsp->hfc_conf = current_conf;
			dsp_cmx_hw_message(member->dsp, HW_PCM_CONN, i, 2, i, 2);
			dsp_cmx_hw_message(member->dsp, HW_CONF_JOIN, current_conf, 0, 0, 0);
		}
		return;
	}

	/* no member is in a conference yet, so we find a free one
	 */
	memset(freeunits, 1, sizeof(freeunits));
	list_for_each_entry(dsp, &dsp_obj.ilist, list) {
		/* dsp must be on the same chip */
		if (dsp->features.hfc_id==same_hfc
		/* dsp must have joined a HW conference */
		 && dsp->hfc_conf>=0
		/* slot must be within range */
		 && dsp->hfc_conf<8)
			freeunits[dsp->hfc_conf] = 0;
	}
	i = 0;
	ii = 8;
	while(i < ii) {
		if (freeunits[i])
			break;
		i++;
	}
	if (i == ii) {
		/* no more conferences available */
		if (dsp_debug & DEBUG_DSP_CMX)
			printk(KERN_DEBUG "%s conference %d cannot be formed, because no conference number free\n", __FUNCTION__, conf->id);
		goto conf_software;
	}
	/* join all members */
	current_conf = i;
	goto join_members;
}


/*
 * conf_id != 0: join or change conference
 * conf_id == 0: split from conference if not already
 */
int
dsp_cmx_conf(dsp_t *dsp, u32 conf_id)
{
	int err;
	conference_t *conf;

	/* if conference doesn't change */
	if (dsp->conf_id == conf_id)
		return(0);

	/* first remove us from current conf */
	if (dsp->conf_id) {
		if (dsp_debug & DEBUG_DSP_CMX)
			printk(KERN_DEBUG "removing us from conference %d\n",
				dsp->conf->id);
		/* remove us from conf */
		conf = dsp->conf;
		err = dsp_cmx_del_conf_member(dsp);
		if (err)
			return(err);
		dsp->conf_id = 0;

		/* update hardware */
		dsp_cmx_hardware(NULL, dsp);

		/* conf now empty? */
		if (list_empty(&conf->mlist)) {
			if (dsp_debug & DEBUG_DSP_CMX)
				printk(KERN_DEBUG "conference is empty, so we remove it.\n");
			err = dsp_cmx_del_conf(conf);
			if (err)
				return(err);
		} else {
			/* update members left on conf */
			dsp_cmx_hardware(conf, NULL);
		}
	}

	/* if split */
	if (!conf_id)
		return(0);

	/* now add us to conf */
	if (dsp_debug & DEBUG_DSP_CMX)
		printk(KERN_DEBUG "searching conference %d\n",
			conf_id);
	conf = dsp_cmx_search_conf(conf_id);
	if (!conf) {
		if (dsp_debug & DEBUG_DSP_CMX)
			printk(KERN_DEBUG "conference doesn't exist yet, creating.\n");
		/* the conference doesn't exist, so we create */
		conf = dsp_cmx_new_conf(conf_id);
		if (!conf)
			return(-EINVAL);
	}
	/* add conference member */
	err = dsp_cmx_add_conf_member(dsp, conf);
	if (err)
		return(err);
	dsp->conf_id = conf_id;

	/* if we are alone, we do nothing! */
	if (list_empty(&conf->mlist)) {
		if (dsp_debug & DEBUG_DSP_CMX)
			printk(KERN_DEBUG "we are alone in this conference, so exit.\n");
		/* update hardware */
		dsp_cmx_hardware(NULL, dsp);
		return(0);
	}

	/* update members on conf */
	dsp_cmx_hardware(conf, NULL);

	return(0);
}


/*
 * audio data is received from card
 */

void 
dsp_cmx_receive(dsp_t *dsp, struct sk_buff *skb)
{
	conference_t *conf = dsp->conf;
	conf_member_t *member;
	s32 *c;
	u8 *d, *p;
	int len = skb->len;
	int w, ww, i, ii;
	int W_min, W_max;

	/* check if we have sompen */
	if (len < 1)
		return;

//#ifndef AUTOJITTER
	/* -> if length*2 is greater largest */
	if (dsp->largest < (len<<1))
		dsp->largest = (len<<1);
//#endif

	/* half of the buffer should be 4 time larger than maximum packet size */
	if (len >= (CMX_BUFF_HALF>>2)) {
		printk(KERN_ERR "%s line %d: packet from card is too large (%d bytes). please make card send smaller packets OR increase CMX_BUFF_SIZE\n", __FILE__, __LINE__, len);
		return;
	}

	/* STEP 1: WRITE DOWN WHAT WE GOT (into the buffer(s)) */

	/* -> new W-min & W-max is calculated:
	 * W_min will be the write pointer of this dsp (after writing 'len'
	 * of bytes).
	 * If there are other members in a conference, W_min will be the
	 * lowest of all member's writer pointers.
	 * W_max respectively
	 */
	W_max = W_min = (dsp->W_rx + len) & CMX_BUFF_MASK;
	if (conf) {
		/* -> who is larger? dsp or conf */
		if (conf->largest < dsp->largest)
			conf->largest = dsp->largest;
		else if (conf->largest > dsp->largest)
			dsp->largest = conf->largest;

		list_for_each_entry(member, &conf->mlist, list) {
			if (member != dsp->member) {
				/* if W_rx is lower */
				if (((member->dsp->W_rx - W_min) & CMX_BUFF_MASK) >= CMX_BUFF_HALF)
					W_min = member->dsp->W_rx;
				/* if W_rx is higher */
				if (((W_max - member->dsp->W_rx) & CMX_BUFF_MASK) >= CMX_BUFF_HALF)
					W_max = member->dsp->W_rx;
			}
		}
	}

#ifdef CMX_DEBUG
	printk(KERN_DEBUG "cmx_receive(dsp=%lx): W_rx(dsp)=%05x W_min=%05x W_max=%05x largest=%05x %s\n", dsp, dsp->W_rx, W_min, W_max, dsp->largest, dsp->inst.name);
#endif

	/* -> if data is not too fast (exceed maximum queue):
	 * data is written if 'new W_rx' is not too far behind W_min.
	 */
	if (((dsp->W_rx + len - W_min) & CMX_BUFF_MASK) <= dsp->largest) {
		/* -> received data is written to rx-buffer */
		p = skb->data;
		d = dsp->rx_buff;
		w = dsp->W_rx;
		i = 0;
		ii = len;
		while(i < ii) {
			d[w++ & CMX_BUFF_MASK] = *p++;
			i++;
		}
		/* -> if conference has three or more members */
		if (conf) {
#ifdef CMX_CONF_DEBUG
#warning CMX_CONF_DEBUG is enabled, it causes performance loss with normal 2-party crossconnects
			if (2 <= count_list_member(&conf->mlist)) {
#else
			if (3 <= count_list_member(&conf->mlist)) {
#endif
//printk(KERN_DEBUG "cmxing dsp:%s dsp->W_rx=%04x conf->W_max=%04x\n", dsp->inst.name, dsp->W_rx, conf->W_max);
				/* -> received data is added to conf-buffer
				 *    new space is overwritten */
				p = skb->data;
				c = conf->conf_buff;
				w = dsp->W_rx;
				ww = conf->W_max;
				i = 0;
				ii = len;
				/* loop until done or old W_max is reached */
				while(i<ii && w!=ww) {
					c[w] += dsp_audio_law_to_s32[*p++]; /* add to existing */
					w = (w+1) & CMX_BUFF_MASK; /* must be always masked, for loop condition */
					i++;
				}
				/* loop the rest */
				while(i < ii) {
					c[w++ & CMX_BUFF_MASK] = dsp_audio_law_to_s32[*p++]; /* write to new */
					i++;
				}
			}
			/* if W_max is lower new dsp->W_rx */
			if (((W_max - (dsp->W_rx+len)) & CMX_BUFF_MASK) >= CMX_BUFF_HALF)
				W_max = (dsp->W_rx + len) & CMX_BUFF_MASK;
			/* store for dsp_cmx_send */
			conf->W_min = W_min;
			/* -> write new W_max */
			conf->W_max = W_max;
		}
		/* -> write new W_rx */
		dsp->W_rx = (dsp->W_rx + len) & CMX_BUFF_MASK;
	} else {
		if (dsp_debug & DEBUG_DSP_CMX)
			printk(KERN_DEBUG "CMX: receiving too fast (rx_buff) dsp=%x\n", (u32)dsp);
#ifdef CMX_DEBUG
		printk(KERN_DEBUG "W_max=%x-W_min=%x = %d, largest = %d\n", W_max, W_min, (W_max - W_min) & CMX_BUFF_MASK, dsp->largest);
#endif
	}
}

/*
 * send mixed audio data to card
 */

struct sk_buff 
*dsp_cmx_send(dsp_t *dsp, int len, int dinfo)
{
	conference_t *conf = dsp->conf;
	dsp_t *member, *other;
	register s32 sample;
	s32 *c;
	u8 *d, *o, *p, *q;
	struct sk_buff *nskb;
	int r, rr, t, tt;

	/* PREPARE RESULT */
	nskb = alloc_skb(len, GFP_ATOMIC);
	if (!nskb) {
		printk(KERN_ERR "FATAL ERROR in mISDN_dsp.o: cannot alloc %d bytes\n", len);
		return(NULL);
	}
	mISDN_sethead(PH_DATA | REQUEST, dinfo, nskb);
	/* set pointers, indexes and stuff */
	member = dsp;
	p = dsp->tx_buff; /* transmit data */
	q = dsp->rx_buff; /* received data */
	d = skb_put(nskb, len); /* result */
	t = dsp->R_tx; /* tx-pointers */
	tt = dsp->W_tx;
	r = dsp->R_rx; /* rx-pointers */
	if (conf) {
		/* special hardware access */
		if (conf->hardware) {
			if (dsp->tone.tone && dsp->tone.software) {
				/* -> copy tone */
				dsp_tone_copy(dsp, d, len);
				dsp->R_tx = dsp->W_tx = 0; /* clear tx buffer */
				return(nskb);
			}
			if (t != tt) {
				while(len && t!=tt) {
					*d++ = p[t]; /* write tx_buff */
					t = (t+1) & CMX_BUFF_MASK;
					len--;
				}
			}
			if (len)
				memset(d, dsp_silence, len);
			dsp->R_tx = t;
			return(nskb);
		}
		/* W_min is also limit for read */
		rr = conf->W_min;
	} else
		rr = dsp->W_rx;

	/* increase r, if too far behind rr
	 * (this happens if interrupts get lost, so transmission is delayed) */
	if (((rr - r) & CMX_BUFF_MASK) > dsp->largest) {
		if (dsp_debug & DEBUG_DSP_CMX)
			printk(KERN_DEBUG "r=%04x is too far behind rr=%04x, correcting. (larger than %04x)\n", r, rr, dsp->largest);
		r = (rr - dsp->largest) & CMX_BUFF_MASK;
	}
	/* calculate actual r (if r+len would overrun rr) */
	if (((rr - r - len) & CMX_BUFF_MASK) >= CMX_BUFF_HALF) {
#ifdef CMX_DEBUG
		printk(KERN_DEBUG "r+len=%04x overruns rr=%04x\n", (r+len) & CMX_BUFF_MASK, rr);
#endif
		/* r is set "len" bytes before W_min */
		r = (rr - len) & CMX_BUFF_MASK;
		if (dsp_debug & DEBUG_DSP_CMX)
			printk(KERN_DEBUG "CMX: sending too fast (tx_buff) dsp=%x\n", (u32)dsp);
	} else
		/* rr is set "len" bytes after R_rx */
		rr = (r + len) & CMX_BUFF_MASK;
	dsp->R_rx = rr;
	/* now: rr is exactly "len" bytes after r now */
#ifdef CMX_DEBUG
	printk(KERN_DEBUG "CMX_SEND(dsp=%lx) %d bytes from tx:0x%05x-0x%05x rx:0x%05x-0x%05x echo=%d %s\n", dsp, len, t, tt, r, rr, dsp->echo, dsp->inst.name);
#endif

	/* STEP 2.0: PROCESS TONES/TX-DATA ONLY */
	if (dsp->tone.tone && dsp->tone.software) {
		/* -> copy tone */
		dsp_tone_copy(dsp, d, len);
		dsp->R_tx = dsp->W_tx = 0; /* clear tx buffer */
		return(nskb);
	}
	/* if we have tx-data but do not use mixing */
	if (!dsp->tx_mix && t!=tt) {
		/* -> send tx-data and continue when not enough */
		while(r!=rr && t!=tt) {
			*d++ = p[t]; /* write tx_buff */
			t = (t+1) & CMX_BUFF_MASK;
			r = (r+1) & CMX_BUFF_MASK;
		}
		if(r == rr) {
			dsp->R_tx = t;
			return(nskb);
		}
	}

	/* STEP 2.1: PROCESS DATA (one member / no conf) */
	if (!conf) {
		single:
		/* -> if echo is NOT enabled */
		if (!dsp->echo) {
			/* -> send tx-data if available or use 0-volume */
			while(r!=rr && t!=tt) {
				*d++ = p[t]; /* write tx_buff */
				t = (t+1) & CMX_BUFF_MASK;
				r = (r+1) & CMX_BUFF_MASK;
			}
			if(r != rr)
				memset(d, dsp_silence, (rr-r)&CMX_BUFF_MASK);
		/* -> if echo is enabled */
		} else {
			/* -> mix tx-data with echo if available, or use echo only */
			while(r!=rr && t!=tt) {
				*d++ = dsp_audio_mix_law[(p[t]<<8)|q[r]];
				t = (t+1) & CMX_BUFF_MASK;
				r = (r+1) & CMX_BUFF_MASK;
			}
			while(r != rr) {
				*d++ = q[r]; /* echo */
				r = (r+1) & CMX_BUFF_MASK;
			}
		}
		dsp->R_tx = t;
		return(nskb);
	}
	if (1 == count_list_member(&conf->mlist)) {
		goto single;
	}
	/* STEP 2.2: PROCESS DATA (two members) */
#ifdef CMX_CONF_DEBUG
	if (0) {
#else
	if (2 == count_list_member(&conf->mlist)) {
#endif
		/* "other" becomes other party */
		other = (list_entry(conf->mlist.next, conf_member_t, list))->dsp;
		if (other == member)
			other = (list_entry(conf->mlist.prev, conf_member_t, list))->dsp;
		o = other->rx_buff; /* received data */
		/* -> if echo is NOT enabled */
		if (!dsp->echo) {
			/* -> copy other member's rx-data, if tx-data is available, mix */
			while(r!=rr && t!=tt) {
				*d++ = dsp_audio_mix_law[(p[t]<<8)|o[r]];
				t = (t+1) & CMX_BUFF_MASK;
				r = (r+1) & CMX_BUFF_MASK;
			}
			while(r != rr) {
				*d++ = o[r];
				r = (r+1) & CMX_BUFF_MASK;
			}
		/* -> if echo is enabled */
		} else {
			/* -> mix other member's rx-data with echo, if tx-data is available, mix */
			while(r!=rr && t!=tt) {
				sample = dsp_audio_law_to_s32[p[t]] + dsp_audio_law_to_s32[o[r]] + dsp_audio_law_to_s32[q[r]];
				if (sample < -32768)
					sample = -32768;
				else if (sample > 32767)
					sample = 32767;
				*d++ = dsp_audio_s16_to_law[sample & 0xffff]; /* tx-data + rx_data + echo */
				t = (t+1) & CMX_BUFF_MASK;
				r = (r+1) & CMX_BUFF_MASK;
			}
			while(r != rr) {
				*d++ = dsp_audio_mix_law[(o[r]<<8)|q[r]];
				r = (r+1) & CMX_BUFF_MASK;
			}
		}
		dsp->R_tx = t;
		return(nskb);
	}
	/* STEP 2.3: PROCESS DATA (three or more members) */
	c = conf->conf_buff;
	/* -> if echo is NOT enabled */
	if (!dsp->echo) {
		/* -> substract rx-data from conf-data, if tx-data is available, mix */
		while(r!=rr && t!=tt) {
			sample = dsp_audio_law_to_s32[p[t]] + c[r] - dsp_audio_law_to_s32[q[r]];
			if (sample < -32768)
				sample = -32768;
			else if (sample > 32767)
				sample = 32767;
			*d++ = dsp_audio_s16_to_law[sample & 0xffff]; /* conf-rx+tx */
			r = (r+1) & CMX_BUFF_MASK;
			t = (t+1) & CMX_BUFF_MASK;
		}
		while(r != rr) {
			sample = c[r] - dsp_audio_law_to_s32[q[r]];
			if (sample < -32768)
				sample = -32768;
			else if (sample > 32767)
				sample = 32767;
			*d++ = dsp_audio_s16_to_law[sample & 0xffff]; /* conf-rx */
			r = (r+1) & CMX_BUFF_MASK;
		}
	/* -> if echo is enabled */
	} else {
		/* -> encode conf-data, if tx-data is available, mix */
		while(r!=rr && t!=tt) {
			sample = dsp_audio_law_to_s32[p[t]] + c[r];
			if (sample < -32768)
				sample = -32768;
			else if (sample > 32767)
				sample = 32767;
			*d++ = dsp_audio_s16_to_law[sample & 0xffff]; /* conf(echo)+tx */
			t = (t+1) & CMX_BUFF_MASK;
			r = (r+1) & CMX_BUFF_MASK;
		}
		while(r != rr) {
			sample = c[r];
			if (sample < -32768)
				sample = -32768;
			else if (sample > 32767)
				sample = 32767;
			*d++ = dsp_audio_s16_to_law[sample & 0xffff]; /* conf(echo) */
			r = (r+1) & CMX_BUFF_MASK;
		}
	}
	dsp->R_tx = t;
	return(nskb);
}

/*
 * audio data is transmitted from upper layer to the dsp
 */
void 
dsp_cmx_transmit(dsp_t *dsp, struct sk_buff *skb)
{
	u_int w, ww;
	u8 *d, *p;
	int space, l;
#ifdef AUTOJITTER
	int use;
#endif

	/* check if we have sompen */
	l = skb->len;
	w = dsp->W_tx;
	ww = dsp->R_tx;
	if (l < 1)
		return;

#ifdef AUTOJITTER
	/* check the delay */
	use = w-ww;
	if (use < 0)
		use += CMX_BUFF_SIZE;
	if (!dsp->tx_delay || dsp->tx_delay>use)
		dsp->tx_delay = use;
	dsp->tx_delay_count += l;
	if (dsp->tx_delay_count >= DELAY_CHECK) {
		/* now remove the delay */
		if (dsp_debug & DEBUG_DSP_DELAY)
			printk(KERN_DEBUG "%s(dsp=0x%x) removing delay of %d bytes\n", __FUNCTION__, (u32)dsp, dsp->tx_delay);
		dsp->tx_delay_count = 0;
		dsp->R_tx = ww = (ww + dsp->tx_delay) & CMX_BUFF_MASK;
		dsp->tx_delay = 0;
	}
#endif

	/* check if there is enough space, and then copy */
	p = dsp->tx_buff;
	d = skb->data;
	space = ww-w;
	if (space <= 0)
		space += CMX_BUFF_SIZE;
	/* write-pointer should not overrun nor reach read pointer */
	if (space-1 < skb->len)
		/* write to the space we have left */
		ww = (ww - 1) & CMX_BUFF_MASK;
	else
		/* write until all byte are copied */
		ww = (w + skb->len) & CMX_BUFF_MASK;
	dsp->W_tx = ww;

#ifdef CMX_DEBUG
	printk(KERN_DEBUG "cmx_transmit(dsp=%lx) %d bytes to 0x%x-0x%x. %s\n", dsp, (ww-w)&CMX_BUFF_MASK, w, ww, dsp->inst.name);
#endif

	/* copy transmit data to tx-buffer */
	while(w != ww) {
		p[w]= *d++;
		w = (w+1) & CMX_BUFF_MASK;
	}

	return;
}


