/*
 * drivers/staging/omapdrm/omap_drv.h
 *
 * Copyright (C) 2011 Texas Instruments
 * Author: Rob Clark <rob@ti.com>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 as published by
 * the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef __OMAP_DRV_H__
#define __OMAP_DRV_H__

#include <video/omapdss.h>
#include <linux/module.h>
#include <linux/types.h>
#include <drm/drmP.h>
#include <drm/drm_crtc_helper.h>
#include <plat/drm.h>
#include "omap_drm.h"

/* APIs we need from dispc.. TODO omapdss should export these */
void dispc_clear_irqs(u32 mask);
u32 dispc_read_irqs(void);
void dispc_set_irqs(u32 mask);
u32 dispc_error_irqs(void);
int dispc_runtime_get(void);
int dispc_runtime_put(void);

void dispc_mgr_enable(enum omap_channel channel, bool enable);
void dispc_mgr_setup(enum omap_channel channel,
		struct omap_overlay_manager_info *info);
void dispc_mgr_set_timings(enum omap_channel channel,
		struct omap_video_timings *timings);
bool dispc_mgr_go(enum omap_channel channel);
bool dispc_mgr_go_busy(enum omap_channel channel);
u32 dispc_mgr_get_vsync_irq(enum omap_channel channel);
u32 dispc_mgr_get_framedone_irq(enum omap_channel channel);

int dispc_ovl_enable(enum omap_plane plane, bool enable);
void dispc_ovl_set_channel_out(enum omap_plane plane,
		enum omap_channel channel);
int dispc_ovl_setup(enum omap_plane plane, struct omap_overlay_info *oi,
		bool ilace, bool replication,
		const struct omap_video_timings *mgr_timings);

int dss_feat_get_num_mgrs(void);
int dss_feat_get_num_ovls(void);
enum omap_display_type dss_feat_get_supported_displays(enum omap_channel channel);
enum omap_color_mode dss_feat_get_supported_color_modes(enum omap_plane plane);

void dispc_dump_regs(struct seq_file *s);
void dispc_dump_clocks(struct seq_file *s);
void dss_dump_clocks(struct seq_file *s);
void dss_dump_regs(struct seq_file *s);
void hdmi_dump_regs(struct seq_file *s);


#define DBG(fmt, ...) DRM_DEBUG(fmt"\n", ##__VA_ARGS__)
#define VERB(fmt, ...) if (0) DRM_DEBUG(fmt, ##__VA_ARGS__) /* verbose debug */

#define MODULE_NAME     "omapdrm"

/* max # of mapper-id's that can be assigned.. todo, come up with a better
 * (but still inexpensive) way to store/access per-buffer mapper private
 * data..
 */
#define MAX_MAPPERS 2

/* parameters which describe (unrotated) coordinates of scanout within a fb: */
struct omap_drm_window {
	uint32_t rotation;
	int32_t  crtc_x, crtc_y;		/* signed because can be offscreen */
	uint32_t crtc_w, crtc_h;
	uint32_t src_x, src_y;
	uint32_t src_w, src_h;
};

/* Once GO bit is set, we can't make further updates to shadowed registers
 * until the GO bit is cleared.  So various parts in the kms code that need
 * to update shadowed registers queue up a pair of callbacks, pre_apply
 * which is called before setting GO bit, and post_apply that is called
 * after GO bit is cleared.  The encoder manages the queuing, and everyone
 * else goes thru omap_encoder_apply() using these callbacks so that the
 * code which has to deal w/ GO bit state is centralized.
 */
struct omap_drm_apply {
	struct list_head pending_node, queued_node;
	bool queued;
	void (*pre_apply)(struct omap_drm_apply *apply);
	void (*post_apply)(struct omap_drm_apply *apply);
};

/* For transiently registering for different DSS irqs that various parts
 * of the KMS code need during setup/configuration.  We these are not
 * necessarily the same as what drm_vblank_get/put() are requesting, and
 * the hysteresis in drm_vblank_put() is not necessarily desirable for
 * internal housekeeping related irq usage.
 */
struct omap_drm_irq {
	struct list_head node;
	uint32_t irqmask;
	bool registered;
	void (*irq)(struct omap_drm_irq *irq, uint32_t irqstatus);
};

struct omap_drm_private {
	unsigned int num_crtcs;
	struct drm_crtc *crtcs[8];

	unsigned int num_planes;
	struct drm_plane *planes[8];

	unsigned int num_encoders;
	struct drm_encoder *encoders[8];

	unsigned int num_connectors;
	struct drm_connector *connectors[8];

	struct drm_fb_helper *fbdev;

	struct workqueue_struct *wq;

	/* list of GEM objects: */
	struct list_head obj_list;

	bool has_dmm;

	/* properties: */
	struct drm_property *rotation_prop;

	/* irq handling: */
	struct list_head irq_list;    /* list of omap_drm_irq */
	uint32_t vblank_mask;         /* irq bits set for userspace vblank */
	struct omap_drm_irq error_handler;
};

/* this should probably be in drm-core to standardize amongst drivers */
#define DRM_ROTATE_0	0
#define DRM_ROTATE_90	1
#define DRM_ROTATE_180	2
#define DRM_ROTATE_270	3
#define DRM_REFLECT_X	4
#define DRM_REFLECT_Y	5

#ifdef CONFIG_DEBUG_FS
int omap_debugfs_init(struct drm_minor *minor);
void omap_debugfs_cleanup(struct drm_minor *minor);
void omap_framebuffer_describe(struct drm_framebuffer *fb, struct seq_file *m);
void omap_gem_describe(struct drm_gem_object *obj, struct seq_file *m);
void omap_gem_describe_objects(struct list_head *list, struct seq_file *m);
#endif

int omap_irq_enable_vblank(struct drm_device *dev, int crtc);
void omap_irq_disable_vblank(struct drm_device *dev, int crtc);
irqreturn_t omap_irq_handler(DRM_IRQ_ARGS);
void omap_irq_preinstall(struct drm_device *dev);
int omap_irq_postinstall(struct drm_device *dev);
void omap_irq_uninstall(struct drm_device *dev);
void omap_irq_register(struct drm_device *dev, struct omap_drm_irq *irq);
void omap_irq_unregister(struct drm_device *dev, struct omap_drm_irq *irq);

struct drm_fb_helper *omap_fbdev_init(struct drm_device *dev);
void omap_fbdev_free(struct drm_device *dev);

const struct omap_video_timings *omap_crtc_timings(struct drm_crtc *crtc);
enum omap_channel omap_crtc_channel(struct drm_crtc *crtc);
int omap_crtc_apply(struct drm_crtc *crtc,
		struct omap_drm_apply *apply);
struct drm_crtc *omap_crtc_init(struct drm_device *dev,
		struct drm_plane *plane, enum omap_channel channel, int id);

struct drm_plane *omap_plane_init(struct drm_device *dev,
		int plane_id, bool private_plane);
int omap_plane_dpms(struct drm_plane *plane, int mode);
int omap_plane_mode_set(struct drm_plane *plane,
		struct drm_crtc *crtc, struct drm_framebuffer *fb,
		int crtc_x, int crtc_y,
		unsigned int crtc_w, unsigned int crtc_h,
		uint32_t src_x, uint32_t src_y,
		uint32_t src_w, uint32_t src_h,
		void (*fxn)(void *), void *arg);
void omap_plane_install_properties(struct drm_plane *plane,
		struct drm_mode_object *obj);
int omap_plane_set_property(struct drm_plane *plane,
		struct drm_property *property, uint64_t val);

struct drm_encoder *omap_encoder_init(struct drm_device *dev);
struct drm_encoder *omap_connector_attached_encoder(
		struct drm_connector *connector);

struct drm_connector *omap_connector_init(struct drm_device *dev,
		int connector_type, struct omap_dss_device *dssdev,
		struct drm_encoder *encoder);
int omap_connector_mode_set(struct drm_connector *connector,
		struct drm_display_mode *mode,
		struct omap_video_timings *timings);
void omap_connector_flush(struct drm_connector *connector,
		int x, int y, int w, int h);

uint32_t omap_framebuffer_get_formats(uint32_t *pixel_formats,
		uint32_t max_formats, enum omap_color_mode supported_modes);
struct drm_framebuffer *omap_framebuffer_create(struct drm_device *dev,
		struct drm_file *file, struct drm_mode_fb_cmd2 *mode_cmd);
struct drm_framebuffer *omap_framebuffer_init(struct drm_device *dev,
		struct drm_mode_fb_cmd2 *mode_cmd, struct drm_gem_object **bos);
struct drm_gem_object *omap_framebuffer_bo(struct drm_framebuffer *fb, int p);
int omap_framebuffer_replace(struct drm_framebuffer *a,
		struct drm_framebuffer *b, void *arg,
		void (*unpin)(void *arg, struct drm_gem_object *bo));
void omap_framebuffer_update_scanout(struct drm_framebuffer *fb,
		struct omap_drm_window *win, struct omap_overlay_info *info);
struct drm_connector *omap_framebuffer_get_next_connector(
		struct drm_framebuffer *fb, struct drm_connector *from);
void omap_framebuffer_flush(struct drm_framebuffer *fb,
		int x, int y, int w, int h);

void omap_gem_init(struct drm_device *dev);
void omap_gem_deinit(struct drm_device *dev);

struct drm_gem_object *omap_gem_new(struct drm_device *dev,
		union omap_gem_size gsize, uint32_t flags);
int omap_gem_new_handle(struct drm_device *dev, struct drm_file *file,
		union omap_gem_size gsize, uint32_t flags, uint32_t *handle);
void omap_gem_free_object(struct drm_gem_object *obj);
int omap_gem_init_object(struct drm_gem_object *obj);
void *omap_gem_vaddr(struct drm_gem_object *obj);
int omap_gem_dumb_map_offset(struct drm_file *file, struct drm_device *dev,
		uint32_t handle, uint64_t *offset);
int omap_gem_dumb_destroy(struct drm_file *file, struct drm_device *dev,
		uint32_t handle);
int omap_gem_dumb_create(struct drm_file *file, struct drm_device *dev,
		struct drm_mode_create_dumb *args);
int omap_gem_mmap(struct file *filp, struct vm_area_struct *vma);
int omap_gem_mmap_obj(struct drm_gem_object *obj,
		struct vm_area_struct *vma);
int omap_gem_fault(struct vm_area_struct *vma, struct vm_fault *vmf);
int omap_gem_op_start(struct drm_gem_object *obj, enum omap_gem_op op);
int omap_gem_op_finish(struct drm_gem_object *obj, enum omap_gem_op op);
int omap_gem_op_sync(struct drm_gem_object *obj, enum omap_gem_op op);
int omap_gem_op_async(struct drm_gem_object *obj, enum omap_gem_op op,
		void (*fxn)(void *arg), void *arg);
int omap_gem_roll(struct drm_gem_object *obj, uint32_t roll);
void omap_gem_cpu_sync(struct drm_gem_object *obj, int pgoff);
void omap_gem_dma_sync(struct drm_gem_object *obj,
		enum dma_data_direction dir);
int omap_gem_get_paddr(struct drm_gem_object *obj,
		dma_addr_t *paddr, bool remap);
int omap_gem_put_paddr(struct drm_gem_object *obj);
int omap_gem_get_pages(struct drm_gem_object *obj, struct page ***pages,
		bool remap);
int omap_gem_put_pages(struct drm_gem_object *obj);
uint32_t omap_gem_flags(struct drm_gem_object *obj);
int omap_gem_rotated_paddr(struct drm_gem_object *obj, uint32_t orient,
		int x, int y, dma_addr_t *paddr);
uint64_t omap_gem_mmap_offset(struct drm_gem_object *obj);
size_t omap_gem_mmap_size(struct drm_gem_object *obj);
int omap_gem_tiled_size(struct drm_gem_object *obj, uint16_t *w, uint16_t *h);
int omap_gem_tiled_stride(struct drm_gem_object *obj, uint32_t orient);

struct dma_buf * omap_gem_prime_export(struct drm_device *dev,
		struct drm_gem_object *obj, int flags);
struct drm_gem_object * omap_gem_prime_import(struct drm_device *dev,
		struct dma_buf *buffer);

static inline int align_pitch(int pitch, int width, int bpp)
{
	int bytespp = (bpp + 7) / 8;
	/* in case someone tries to feed us a completely bogus stride: */
	pitch = max(pitch, width * bytespp);
	/* PVR needs alignment to 8 pixels.. right now that is the most
	 * restrictive stride requirement..
	 */
	return ALIGN(pitch, 8 * bytespp);
}

static inline enum omap_channel pipe2chan(int pipe)
{
	int num_mgrs = dss_feat_get_num_mgrs();

	/*
	 * We usually don't want to create a CRTC for each manager,
	 * at least not until we have a way to expose private planes
	 * to userspace.  Otherwise there would not be enough video
	 * pipes left for drm planes.  The higher #'d managers tend
	 * to have more features so start in reverse order.
	 */
	return num_mgrs - pipe - 1;
}

/* map crtc to vblank mask */
static inline uint32_t pipe2vbl(int crtc)
{
	enum omap_channel channel = pipe2chan(crtc);
	return dispc_mgr_get_vsync_irq(channel);
}

/* should these be made into common util helpers?
 */

static inline int objects_lookup(struct drm_device *dev,
		struct drm_file *filp, uint32_t pixel_format,
		struct drm_gem_object **bos, uint32_t *handles)
{
	int i, n = drm_format_num_planes(pixel_format);

	for (i = 0; i < n; i++) {
		bos[i] = drm_gem_object_lookup(dev, filp, handles[i]);
		if (!bos[i]) {
			goto fail;
		}
	}

	return 0;

fail:
	while (--i > 0) {
		drm_gem_object_unreference_unlocked(bos[i]);
	}
	return -ENOENT;
}


/****** PLUGIN API specific ******/

/* interface that plug-in drivers (for now just PVR) can implement */
struct omap_drm_plugin {
	const char *name;

	/* drm functions */
	int (*load)(struct drm_device *dev, unsigned long flags);
	int (*unload)(struct drm_device *dev);
	int (*open)(struct drm_device *dev, struct drm_file *file);
	int (*release)(struct drm_device *dev, struct drm_file *file);

	struct drm_ioctl_desc *ioctls;
	int num_ioctls;
	int ioctl_base;

	struct list_head list;  /* note, this means struct can't be const.. */
};

int omap_drm_register_plugin(struct omap_drm_plugin *plugin);
int omap_drm_unregister_plugin(struct omap_drm_plugin *plugin);

int omap_drm_register_mapper(void);
void omap_drm_unregister_mapper(int id);

void * omap_drm_file_priv(struct drm_file *file, int mapper_id);
void omap_drm_file_set_priv(struct drm_file *file, int mapper_id, void *priv);

void * omap_gem_priv(struct drm_gem_object *obj, int mapper_id);
void omap_gem_set_priv(struct drm_gem_object *obj, int mapper_id, void *priv);
void omap_gem_vm_open(struct vm_area_struct *vma);
void omap_gem_vm_close(struct vm_area_struct *vma);

/* for external plugin buffers wrapped as GEM object (via. omap_gem_new_ext())
 * a vm_ops struct can be provided to get callback notification of various
 * events..
 */
struct omap_gem_vm_ops {
	void (*open)(struct vm_area_struct * area);
	void (*close)(struct vm_area_struct * area);
	//maybe: int (*fault)(struct vm_area_struct *vma, struct vm_fault *vmf);

	/* note: mmap is not expected to do anything.. it is just to allow buffer
	 * allocate to update it's own internal state
	 */
	void (*mmap)(struct file *, struct vm_area_struct *);
};

struct drm_gem_object * omap_gem_new_ext(struct drm_device *dev,
		union omap_gem_size gsize, uint32_t flags,
		dma_addr_t paddr, struct page **pages,
		struct omap_gem_vm_ops *ops);

void omap_gem_op_update(void);
int omap_gem_set_sync_object(struct drm_gem_object *obj, void *syncobj);
/*********************************/


#endif /* __OMAP_DRV_H__ */
