#!/usr/bin/env texlua  

VERSION = "0.2"

--[[
     ptex2pdf.lua: convert pTeX (and friends) based files to pdf

     Originally based on musixtex.lua from Bob Tennent.

     (c) Copyright 2012 Bob Tennent rdt@cs.queensu.ca
     (c) Copyright 2013 Norbert Preining norbert@preining.info

     This program is free software; you can redistribute it and/or modify it
     under the terms of the GNU General Public License as published by the
     Free Software Foundation; either version 2 of the License, or (at your
     option) any later version.

     This program is distributed in the hope that it will be useful,
     but WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
     Public License for more details.

     You should have received a copy of the GNU General Public License along
     with this program; if not, write to the Free Software Foundation, Inc.,
     51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.

--]]

--[[

  ChangeLog:

     version 0.1  2013-03-08 NP
       Initial release on blog
     version 0.2  2013-03-10 NP
       import into git repository
       support passing options on to tex and dvipdfm
       add README with TeXworks config options

--]]

function usage()
  print("Usage:  [texlua] ptex2pdf[.lua] { option | basename[.tex] } ... ")
  print("options: -v  version")
  print("         -h  help")
  print("         -e  use eptex class of programs")
  print("         -u  use uptex class of programs")
  print("         -l  use latex based formats")
  print("         -s  stop at dvi")
  print("         -i  retain intermediate files")
  print("         -ot '<opts>' extra options for TeX")
  print("         -od '<opts>' extra options for dvipdfmx")
end

function whoami ()
  print("This is ptex2pdf.lua version ".. VERSION .. ".")
end

whoami()
if #arg == 0 then
  usage()
  os.exit(0)
end

-- defaults:
tex = "ptex"
texopts = ""
dvipdf = "dvipdfmx"
dvipdfopts = ""
intermediate = 1

use_eptex = 0
use_uptex = 0
use_latex = 0
filename = ""
exit_code = 0
narg = 1
repeat
  this_arg = arg[narg]
  if this_arg == "-v" then
    os.exit(0)
  elseif this_arg == "-h" then
    usage()
    os.exit(0)
  elseif this_arg == "-e" then
    use_eptex = 1
  elseif this_arg == "-u" then
    use_uptex = 1
  elseif this_arg == "-l" then
    use_latex = 1
  elseif this_arg == "-s" then
    dvipdf = ""
  elseif this_arg == "-i" then
    intermediate = 0
  elseif this_arg == "-ot" then
    narg = narg+1
    texopts = arg[narg]
  elseif this_arg == "-od" then
    narg = narg+1
    dvipdfopts = arg[narg]
  else
    filename = this_arg 
  end --if this_arg == ...
  narg = narg+1
until narg > #arg 

if use_eptex == 1 then
  if use_uptex == 1 then
    if use_latex == 1 then
      tex = "uplatex"	-- uptex already as etex extension
    else
      tex = "uptex"
    end
  else
    if use_latex == 1 then
      tex = "platex"    -- latex needs etex anyway
    else
      tex = "eptex"
    end
  end
else
  if use_uptex == 1 then
    if use_latex == 1 then
      tex = "uplatex"
    else
      tex = "uptex"
    end
  else
    if use_latex == 1 then
      tex = "platex"
    else
      tex = "ptex"
    end
  end
end

if filename ~= "" and string.sub(filename, -4, -1) == ".tex" then
  filename = string.sub(filename, 1, -5)
end
if not io.open(filename .. ".tex", "r") then
  print("Non-existent file: ", filename .. ".tex")
  exit_code = 1
else
  print("Processing ".. filename .. ".tex.")
  if (os.execute(tex .. " " .. texopts .. " " .. filename) == 0) and
     (dvipdf == "" or  (os.execute(dvipdf .. " " .. dvipdfopts .. " " .. filename) == 0))
    then 
      if dvipdf ~= "" then 
        print(filename .. ".pdf generated by " .. dvipdf .. ".")
      end
      if intermediate == 1 then -- clean-up:
        if dvipdf ~= "" then
          os.remove( filename .. ".dvi" )
        end
      end
    else
      print("ptex2pdf processing of " .. filename .. ".tex fails.\n")
      exit_code = 2
      --[[ uncomment for debugging
      print("tex = ", tex)
      print("dvipdf = ", dvipdf)
      --]]
    end
end --if not io.open ...

os.exit( exit_code )

-- Local Variables:
-- lua-indent-level: 2
-- tab-width: 2
-- indent-tabs-mode: nil
-- End:
-- vim:set tabstop=2 expandtab: #
