/** -*- C++ -*-
 * @file cache/component/state.h
 * @author Peter Rockai <me@mornfall.net>
 */
#include <apt-front/workarounds.h>
#include <apt-pkg/pkgcache.h>
#include <apt-pkg/depcache.h>
#include <apt-pkg/policy.h>
#include <apt-pkg/algorithms.h>
#include <apt-front/cache/cache.h>
#include <apt-front/cache/entity/package.h>

#ifndef CACHE_COMPONENT_STATE_H
#define CACHE_COMPONENT_STATE_H

class pkgPolicy;

namespace aptFront {
namespace cache {
namespace component {
class Packages;

struct PackagePolicy : public pkgPolicy
{
    PackagePolicy( const PackagePolicy &p )
        : pkgPolicy( *static_cast<const pkgPolicy *>( &p ) )
    {
        unsigned c = Cache->Head().PackageCount;
        Pins = new Pin[c];
        std::copy( p.Pins, p.Pins+c, Pins );
        PFPriority = new signed short[c];
        std::copy( p.PFPriority, p.PFPriority+c, PFPriority );
        // i really hate APT API.......
    }
    PackagePolicy( aptFront::cache::Cache *owner )
        : pkgPolicy( &(owner->packages().baseReference()) )
    {}
};

struct State : public pkgDepCache, public Implementation<State>
{
    State( cache::Cache *cache );
    State( const State & ); // copy ctor - deep copying needed!
    virtual ~State();
    entity::Package::State packageState( const entity::Package &i );
    bool isInstalled( const entity::Package &i );

    // @todo add a Version variant of the above
    void markInstall( entity::Package p ) {
        notifyPreChange();
        pkgProblemResolver fix( &baseReference() );
        fix.Clear( p );
        fix.Protect( p );
        MarkInstall( p, true );
        fix.InstallProtect();
        fix.Resolve( true );
        notifyPostChange();
    }

    void markRemove( entity::Package p ) {
        notifyPreChange();
        pkgProblemResolver fix( &baseReference() );
        fix.Clear( p );
        fix.Protect( p );
        fix.Remove( p );
        MarkDelete( p );
        fix.Resolve( true );
        notifyPostChange();
    }

    void markKeep( entity::Package p ) {
        notifyPreChange();
        pkgProblemResolver fix( &baseReference() );
        fix.Clear( p );
        fix.Protect( p );
        MarkKeep( p, true );
        fix.Resolve( true );
        notifyPostChange();
    }

    PackagePolicy &policy() { return m_policy; }

    void upgrade() {
        notifyPreChange();
        pkgAllUpgrade( *this );
        notifyPostChange();
    }
    void distUpgrade() {
        notifyPreChange();
        pkgDistUpgrade( *this );
        notifyPostChange();
    }
    void notifyPostChange();
    pkgDepCache &baseReference() { return *this; }

    void setOwnerCache( cache::Cache *c ) {
        Base::setOwnerCache( c );
        updateCounts();
    }

    int brokenCount() { return BrokenCount(); }
    int removeCount() { return m_removeCount; }
    int newInstallCount() { return m_newInstallCount; } 
    int upgradeCount() { return m_upgradeCount; } 

    int installedCount() { return m_installedCount; }
    int upgradableCount() { return m_upgradableCount; }
    int availableCount() { return m_availableCount; }

    bool changed();

    void updateCounts();
protected:
    PackagePolicy m_policy;
    int m_removeCount;
    int m_newInstallCount;
    int m_upgradeCount;
    int m_installedCount;
    int m_upgradableCount;
    int m_availableCount;
};


}
}
}
#endif
