/** -*- C++ -*-
 * @file cache/component/packagetags.h
 * @author Enrico Zini (enrico) <enrico@enricozini.org>
 */

/*
 * System tag database
 *
 * Copyright (C) 2003,2004,2005  Enrico Zini <enrico@debian.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <apt-front/cache/component/debtags/serializer.h>
#include <apt-front/cache/component/base.h>
#include <apt-front/cache/entity/package.h>
#include <tagcoll/TDBReadonlyDiskIndex.h>

#ifndef CACHE_COMPONENT_DEBTAGS_H
#define CACHE_COMPONENT_DEBTAGS_H

namespace aptFront {
namespace cache {
namespace component {

/**
 * Access the Debtags tag database.
 */
class PackageTags : public Implementation<PackageTags>
{
protected:
	// Debtags database
	Tagcoll::TDBReadonlyDiskIndex<entity::Package, entity::Tag> _tags;

	Tagcoll::Converter<entity::Package, std::string> fromitem;
	Tagcoll::Converter<entity::Tag, std::string> fromtag;
	Tagcoll::Converter<std::string, entity::Package> toitem;
	Tagcoll::Converter<std::string, entity::Tag> totag;

	std::string rcdir;

public:
	/** Create a new accessor for the Debtags database
	 * 
	 * \param editable
	 * Specifies if recording of modifications should be enabled.  If editable
	 * is true, then the local state directory will be created when the object
	 * is instantiated.
	 */
    PackageTags (bool editable = false);
    virtual ~PackageTags () {}

	virtual void setOwnerCache( Cache *c )
	{
		toitem.setOwnerCache(c);
		totag.setOwnerCache(c);
	}

	Tagcoll::TDBReadonlyDiskIndex<entity::Package, entity::Tag>& tagdb() { return _tags; }
	const Tagcoll::TDBReadonlyDiskIndex<entity::Package, entity::Tag>& tagdb() const { return _tags; }

	/**
	 * Check if the tag database has been created (i.e. if something
	 * equivalend to debtags update has been run)
	 */
	static bool hasTagDatabase();


	/**
	 * Save in the state storage directory a patch that can be used to turn
	 * the system database into the collection given
	 */
	void savePatch();

	/**
	 * Save in the state storage directory a patch to turn the system database
	 * into the collection given
	 */
	void savePatch(const Tagcoll::PatchList<entity::Package, entity::Tag>& patch);

	/**
	 * Send to the central archive a patch that can be used to turn
	 * the system database into the collection given
	 */
	void sendPatch();

	/**
	 * Send the given patch to the central archive
	 */
	void sendPatch(const Tagcoll::PatchList<entity::Package, entity::Tag>& patch);
	

	/** Output the current Debian tags database to a TagcollConsumer
	 * \note The collection is sent to 'cons' without merging repeated items
	 */
	void outputSystem(Tagcoll::Consumer<std::string, std::string>& cons);

	/** Output the current Debian tags database to a TagcollConsumer
	 * \note The collection is sent to 'cons' without merging repeated items
	 */
	void outputSystem(Tagcoll::Consumer<entity::Package, entity::Tag>& cons);

	/**
	 * Output the current Debian tags database, patched with local patch, to a TagcollConsumer
	 * \note The collection is sent to 'cons' without merging repeated items
	 */
	void outputPatched(Tagcoll::Consumer<std::string, std::string>& cons);

	/**
	 * Output the current Debian tags database, patched with local patch, to a TagcollConsumer
	 * \note The collection is sent to 'cons' without merging repeated items
	 */
	void outputPatched(Tagcoll::Consumer<entity::Package, entity::Tag>& cons);
};


}
}
}

// vim:set ts=4 sw=4:
#endif
