/* -*- Mode: C++; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/*
 * This file is part of the libe-book project.
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 */

#include <algorithm>
#include <cstring>
#include <iterator>
#include <vector>

#include <librevenge/librevenge.h>

#include "FB2Content.h"
#include "FB2ContentCollector.h"
#include "FB2ContentMap.h"
#include "FB2Style.h"

namespace libebook
{

FB2ContentCollector::FB2ContentCollector(
  librevenge::RVNGTextInterface *document, const librevenge::RVNGPropertyList &metadata,
  const FB2ContentMap &notes, const FB2ContentMap &bitmaps)
  : m_document(document)
  , m_metadata(metadata)
  , m_notes(notes)
  , m_bitmaps(bitmaps)
  , m_currentFootnote(1)
{
}

void FB2ContentCollector::defineMetadataEntry(const char *, const char *)
{
}

void FB2ContentCollector::openMetadataEntry(const char *)
{
}

void FB2ContentCollector::closeMetadataEntry()
{
}

void FB2ContentCollector::defineID(const char *)
{
}

void FB2ContentCollector::openPageSpan()
{
  m_document->openPageSpan(librevenge::RVNGPropertyList());
}

void FB2ContentCollector::closePageSpan()
{
  m_document->closePageSpan();
}

void FB2ContentCollector::openBlock()
{
}

void FB2ContentCollector::closeBlock()
{
}

void FB2ContentCollector::openParagraph(const FB2BlockFormat &format)
{
  m_document->openParagraph(makePropertyList(format));
}

void FB2ContentCollector::closeParagraph()
{
  m_document->closeParagraph();
}

void FB2ContentCollector::openSpan(const FB2Style &style)
{
  m_document->openSpan(makePropertyList(style));
}

void FB2ContentCollector::closeSpan()
{
  m_document->closeSpan();
}

void FB2ContentCollector::insertText(const char *text)
{
  m_document->insertText(librevenge::RVNGString(text));
}

void FB2ContentCollector::openTable(const FB2BlockFormat &format)
{
  m_document->openTable(makePropertyList(format));
}

void FB2ContentCollector::closeTable()
{
  m_document->closeTable();
}

void FB2ContentCollector::openTableRow(const FB2BlockFormat &format)
{
  librevenge::RVNGPropertyList props;
  if (format.headerRow)
    props.insert("fo:is-header-row", true);

  m_document->openTableRow(props);
}

void FB2ContentCollector::closeTableRow()
{
  m_document->closeTableRow();
}

void FB2ContentCollector::openTableCell(int rowspan, int colspan)
{
  librevenge::RVNGPropertyList props;
  if (colspan > 0)
    props.insert("table:number-columns-spanned", colspan);
  if (rowspan > 0)
    props.insert("table:number-rows-spanned", rowspan);

  m_document->openTableCell(props);
}

void FB2ContentCollector::closeTableCell()
{
  m_document->closeTableCell();
}

void FB2ContentCollector::insertCoveredTableCell()
{
  m_document->insertCoveredTableCell(librevenge::RVNGPropertyList());
}

void FB2ContentCollector::insertFootnote(const char *id)
{
  const FB2Content *const note = m_notes.get(id);
  if (note)
  {
    librevenge::RVNGPropertyList props;
    props.insert("librevenge:number", m_currentFootnote);
    ++m_currentFootnote;

    m_document->openFootnote(props);
    note->unfold(*this);
    m_document->closeFootnote();
  }
}

void FB2ContentCollector::insertBitmap(const char *id)
{
  const FB2Content *const bitmap = m_bitmaps.get(id);
  if (bitmap)
    bitmap->unfold(*this);
}

void FB2ContentCollector::insertBitmapData(const char *contentType, const char *base64Data)
{
  librevenge::RVNGPropertyList props;
  props.insert("librevenge:mimetype", librevenge::RVNGString(contentType));

  const librevenge::RVNGBinaryData data(base64Data);
  props.insert("office:binary-data", data);

  m_document->insertBinaryObject(props);
}

}

/* vim:set shiftwidth=2 softtabstop=2 expandtab: */
