// -*- mode: c++; tab-width: 4; indent-tabs-mode: t -*-
/* @file cache/component/tagmap.h
 * @author Enrico Zini (enrico) <enrico@enricozini.org>
 */

/*
 * System tag database
 *
 * Copyright (C) 2003-2006  Enrico Zini <enrico@debian.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <ept/forward.h>
#include <ept/cache/debtags/pkgidx.h>
#include <tagcoll/coll/base.h>
#include <tagcoll/coll/intdiskindex.h>
#include <tagcoll/coll/patched.h>

#ifndef EPT_CACHE_DEBTAGS_TAGMAP_H
#define EPT_CACHE_DEBTAGS_TAGMAP_H

namespace ept {
namespace t {
namespace cache {
namespace debtags {
template< typename _ > class TagMap;
}
}
}
}

namespace tagcoll {
template< typename _, typename _1 > class PatchList;

namespace coll {

template<typename C>
struct coll_traits< ept::t::cache::debtags::TagMap<C> >
{
	typedef typename C::Package item_type;
	typedef typename C::Tag tag_type;
	typedef typename std::set< typename C::Tag > tagset_type;
	typedef typename std::set< typename C::Package > itemset_type;
};
}
}

namespace ept {
namespace t {
namespace cache {

template<typename C> class Package;
template<typename C> class Tag;

namespace debtags {

/**
 * Access the Debtags tag database.
 */
template<typename C>
class TagMap : public tagcoll::coll::Collection< TagMap<C> >
{
public:
	typedef typename tagcoll::coll::coll_traits< TagMap<C> >::item_type Package;
	typedef typename tagcoll::coll::coll_traits< TagMap<C> >::tag_type Tag;
	typedef typename tagcoll::coll::coll_traits< TagMap<C> >::itemset_type PackageSet;
	typedef typename tagcoll::coll::coll_traits< TagMap<C> >::tagset_type TagSet;

	typedef typename C::Aggregator Aggregator;
	typedef typename C::Vocabulary Vocabulary;

protected:
	// Master mmap index container
	tagcoll::diskindex::MasterMMap mastermmap;

	// Debtags database
	tagcoll::coll::IntDiskIndex m_rocoll;
	tagcoll::coll::Patched< tagcoll::coll::IntDiskIndex > m_coll;

#if 0
	Tagcoll::TDBReadonlyDiskIndex<entity::Package, entity::Tag> _tags;

	Tagcoll::Converter<entity::Package, std::string> fromitem;
	Tagcoll::Converter<entity::Tag, std::string> fromtag;
	Tagcoll::Converter<std::string, entity::Package> toitem;
	Tagcoll::Converter<std::string, entity::Tag> totag;
#endif

	std::string rcdir;

	time_t m_timestamp;

	PkgIdx<C> m_pkgidx;

	Aggregator& m_pkgs;

	Package packageByID(int id) const
	{
		return m_pkgs.index().packageByName(m_pkgidx.name(id));
	}

	template<typename IDS>
	std::set<Package> packagesById(const IDS& ids) const
	{
		std::set<Package> pkgs;
		for (typename IDS::const_iterator i = ids.begin();
				i != ids.end(); ++i)
			pkgs.insert(packageByID(*i));
		return pkgs;
	}

public:
	typedef tagcoll::coll::Patched< tagcoll::coll::IntDiskIndex > coll_type;
	typedef std::pair< Package, std::set<Tag> > value_type;

	class const_iterator
	{
		const TagMap<C>& coll;
		typename TagMap<C>::coll_type::const_iterator ci;
		mutable const typename TagMap<C>::value_type* cached_val;

	protected:
		const_iterator(const TagMap<C>& coll,
						const typename TagMap<C>::coll_type::const_iterator& ci)
			: coll(coll), ci(ci), cached_val(0) {}

	public:
		~const_iterator()
		{
			if (cached_val)
				delete cached_val;
		}
		const typename TagMap<C>::value_type operator*() const
		{
			if (cached_val)
				return *cached_val;

			return make_pair(coll.packageByID(ci->first), coll.vocabulary().tagsByID(ci->second));
		}
		const typename TagMap<C>::value_type* operator->() const
		{
			if (cached_val)
				return cached_val;
			return cached_val = new typename TagMap<C>::value_type(*(*this));
		}
		const_iterator& operator++()
		{
			++ci;
			if (cached_val)
			{
				delete cached_val;
				cached_val = 0;
			}
			return *this;
		}
		bool operator==(const const_iterator& iter) const
		{
			return ci == iter.ci;
		}
		bool operator!=(const const_iterator& iter) const
		{
			return ci != iter.ci;
		}

		friend class TagMap<C>;
	};
	const_iterator begin() const { return const_iterator(*this, m_coll.begin()); }
	const_iterator end() const { return const_iterator(*this, m_coll.end()); }

	/** Create a new accessor for the Debtags database
	 *
	 * \param editable
	 * Specifies if recording of modifications should be enabled.  If editable
	 * is true, then the local state directory will be created when the object
	 * is instantiated.
	 */
    TagMap(Aggregator& pkgs, bool editable = false);
    ~TagMap() {}

	time_t timestamp() const { return m_timestamp; }

	//void init(Cache& cache, bool editable = false);

	coll_type& tagdb() { return m_coll; }
	const coll_type& tagdb() const { return m_coll; }
	tagcoll::PatchList<Package, Tag> changes() const;

#if 0
	template<typename ITEMS, typename TAGS>
	void insert(const ITEMS& items, const TAGS& tags)
	{
		for (typename ITEMS::const_iterator i = items.begin();
				i != items.end(); ++i)
			m_changes.addPatch(Patch(*i, tags, TagSet()));
	}

	template<typename ITEMS>
	void insert(const ITEMS& items, const wibble::Empty<Tag>& tags)
	{
		// Nothing to do in this case
	}

	/**
	 * Get the changes that have been applied to this collection
	 */
	const Patches& changes() const { return m_changes; }

	/**
	 * Throw away all changes previously applied to this collection
	 */
	void resetChanges() { m_changes.clear(); }

	/**
	 * Set the changes list to a specific patch list
	 */
	void setChanges(const Patches& changes);

	/**
	 * Add a specific patch list to the changes list
	 */
	void addChanges(const Patches& changes);
#endif

    bool hasTag(const Tag& tag) const { return m_coll.hasTag(tag.id()); }

	TagSet getTagsOfItem(const Package& item) const
	{
		return vocabulary().tagsByID(m_coll.getTagsOfItem(item.ondiskId()));
	}

	template<typename ITEMS>
	TagSet getTagsOfItems(const ITEMS& items) const
	{
		std::set<int> pkgs;
		for (typename ITEMS::const_iterator i = items.begin();
				i != items.end(); ++i)
			pkgs.insert(i->ondiskId());
		return vocabulary().tagsByID(m_coll.getTagsOfItems(pkgs));
	}

	PackageSet getItemsHavingTag(const Tag& tag) const
	{
		return packagesById(m_coll.getItemsHavingTag(tag.id()));
	}
	template<typename TAGS>
	PackageSet getItemsHavingTags(const TAGS& tags) const
	{
		std::set<int> itags;
		for (typename TAGS::const_iterator i = tags.begin();
				i != tags.end(); ++i)
			itags.insert(i->id());
		return packagesById(m_coll.getItemsHavingTags(itags));
	}

#if 0
	ItemSet getTaggedItems() const;
#endif
	TagSet getAllTags() const
	{
		return vocabulary().tagsByID(m_coll.getAllTags());
	}

    Vocabulary &vocabulary() { return m_pkgs.vocabulary(); }
    const Vocabulary &vocabulary() const { return m_pkgs.vocabulary(); }

	int getCardinality(const Tag& tag) const
	{
		return m_coll.getCardinality(tag.id());
	}

	void applyChange(const tagcoll::PatchList<Package, Tag>& change)
	{
		using namespace tagcoll;
		PatchList<int, int> intp;
		for (typename PatchList<Package, Tag>::const_iterator i = change.begin();
				i != change.end(); ++i)
		{
			Patch<int, int> p(i->first.ondiskId());
			for (typename std::set<Tag>::const_iterator j = i->second.added.begin();
					j != i->second.added.end(); ++j)
				p.add(j->id());
			for (typename std::set<Tag>::const_iterator j = i->second.removed.begin();
					j != i->second.removed.end(); ++j)
				p.remove(j->id());
			intp.addPatch(p);
		}
		m_coll.applyChange(intp);
	}

#if 0
	template<typename OUT>
	void output(OUT out) const
	{
		for (const_iterator i = begin(); i != end(); ++i)
		{
			*out = *i;
			++out;
		}
	}
#endif













	/**
	 * Check if the tag database has been created (i.e. if something
	 * equivalend to debtags update has been run)
	 */
	static bool hasTagDatabase();


	/**
	 * Save in the state storage directory a patch that can be used to turn
	 * the system database into the collection given
	 */
	void savePatch();

	/**
	 * Save in the state storage directory a patch to turn the system database
	 * into the collection given
	 */
	void savePatch(const tagcoll::PatchList<std::string, std::string>& patch);

	/**
	 * Save in the state storage directory a patch to turn the system database
	 * into the collection given
	 */
	void savePatch(const tagcoll::PatchList<Package, Tag>& patch);

	/**
	 * Send to the central archive a patch that can be used to turn
	 * the system database into the collection given
	 */
	void sendPatch();

	/**
	 * Send the given patch to the central archive
	 */
	void sendPatch(const tagcoll::PatchList<std::string, std::string>& patch);

	/**
	 * Send the given patch to the central archive
	 */
	void sendPatch(const tagcoll::PatchList<Package, Tag>& patch);


	/** Output the current Debian tags database to a TagcollConsumer
	 * \note The collection is sent to 'cons' without merging repeated items
	 */
	template<typename OUT>
	void outputSystem(const OUT& cons);

	/**
	 * Output the current Debian tags database, patched with local patch, to a TagcollConsumer
	 * \note The collection is sent to 'cons' without merging repeated items
	 */
	template<typename OUT>
	void outputPatched(const OUT& cons);

	static std::string componentName();
};


}
}
}
}

// vim:set ts=4 sw=4:
#endif
