/*
 * Gksu -- a library providing access to su functionality
 * Copyright (C) 2004 Gustavo Noronha Silva
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <pty.h>
#include <pwd.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <sys/select.h>
#include <errno.h>

#include <glib.h>
#include <locale.h>

#include "defines.h"
#include "../config.h"

#include "gksu-context.h"

/* local function declarations */
static void
gksu_context_init (GTypeInstance *instance, gpointer g_class);

static void
gksu_context_dispose (GObject *object);

static void
gksu_context_finalize (GObject *object);

static void
gksu_context_class_init (GksuContextClass *klass);

/**
 * gksu_context_new:
 *
 * Creates a new context to use gksu with. You must
 * manually free the context with gksu_context_free
 * after use.
 *
 * Returns: a new #GksuContext.
 */
GksuContext*
gksu_context_new ()
{
  g_type_init ();
  return (GksuContext*) g_type_create_instance (GKSU_TYPE_CONTEXT);
}

/**
 * gksu_context_free:
 * @context: the #GksuContext to be freed.
 *
 * Frees the provided #GksuContext.
 */
void
gksu_context_free (GksuContext *context)
{
  g_object_unref (G_OBJECT(context));
}

/**
 * gksu_context_set_user:
 * @context: the #GksuContext you want to modify
 * @username: the target username
 *
 * Sets up what user the command will be run as. The default
 * is root, but you can run the command as any user.
 *
 */
void
gksu_context_set_user (GksuContext *context, gchar *username)
{
  g_assert (username != NULL);

  if (context->user)
    g_free (context->user);
  context->user = g_strdup (username);
}

/**
 * gksu_context_get_user:
 * @context: the #GksuContext from which to grab the information
 *
 * Gets the user the command will be run as, as set 
 * by gksu_context_set_user.
 *
 * Returns: a pointer to the string containing the username.
 */
const gchar*
gksu_context_get_user (GksuContext *context)
{
  return context->user;
}

/**
 * gksu_context_set_command:
 * @context: the #GksuContext you want to modify
 * @command: the command that shall be ran
 *
 * Sets up what command will run with the target user.
 *
 */
void
gksu_context_set_command (GksuContext *context, gchar *command)
{
  g_assert (command != NULL);

  if (context->command)
    g_free (context->command);
  context->command = g_strdup (command);
}

/**
 * gksu_context_get_command:
 * @context: the #GksuContext from which to grab the information
 *
 * Gets the command that will be run, as set by 
 * gksu_context_set_command.
 *
 * Returns: a pointer to the string containing the command.
 */
const gchar*
gksu_context_get_command (GksuContext *context)
{
  return context->command;
}

/**
 * gksu_context_set_password:
 * @context: the #GksuContext you want to modify
 * @password: the password
 *
 * Sets up what is the password to be passed to su/sudo. *
 */
void
gksu_context_set_password (GksuContext *context, gchar *password)
{
  g_assert (password != NULL);

  if (context->password)
    g_free (context->password);

  if (password[strlen(password) - 1] == '\n')
    context->password = g_strdup (password);
  else
    context->password = g_strdup_printf ("%s\n", password);
}

/**
 * gksu_context_get_password:
 * @context: the #GksuContext from which to grab the information
 *
 * Gets the password that is set for calling the su/sudo
 * programs.
 *
 * Returns: a pointer to the string containing the password.
 */
const gchar*
gksu_context_get_password (GksuContext *context)
{
  return context->password;
}

/**
 * gksu_context_set_login_shell:
 * @context: the #GksuContext you want to modify
 * @value: TRUE or FALSE
 *
 * Should the shell in which the command will be run be
 * a login shell?
 */
void
gksu_context_set_login_shell (GksuContext *context, gboolean value)
{
  context->login_shell = value;
}

/**
 * gksu_context_get_login_shell:
 * @context: the #GksuContext from which to grab the information
 *
 * Finds out if the shell created by the underlying su process
 * will be a login shell.
 *
 * Returns: TRUE if the shell will be a login shell, FALSE otherwise.
 */
gboolean
gksu_context_get_login_shell (GksuContext *context)
{
  return context->login_shell;
}

/**
 * gksu_context_set_keep_env:
 * @context: the #GksuContext you want to modify
 * @value: TRUE or FALSE
 *
 * Should the environment be kept as it is? Defaults to
 * TRUE. Notice that setting this to FALSE may cause the
 * X authorization stuff to fail.
 */
void
gksu_context_set_keep_env (GksuContext *context, gboolean value)
{
  context->keep_env = value;
}

/**
 * gksu_context_get_keep_env:
 * @context: the #GksuContext from which to grab the information
 *
 * Finds out if the environment in which the program will be
 * run will be reset.
 *
 * Returns: TRUE if the environment is going to be kept, 
 * FALSE otherwise.
 */
gboolean
gksu_context_get_keep_env (GksuContext *context)
{
  return context->keep_env;
}

/**
 * gksu_context_set_debug:
 * @context: the #GksuContext you want to modify
 * @value: TRUE or FALSE
 *
 * Set up if debuging information should be printed.
 */
void
gksu_context_set_debug (GksuContext *context, gboolean value)
{
  context->debug = value;
}

/**
 * gksu_context_get_debug:
 * @context: the #GksuContext from which to grab the information
 *
 * Finds out if the library is configured to print debuging
 * information.
 *
 * Returns: TRUE if it is, FALSE otherwise.
 */
gboolean
gksu_context_get_debug (GksuContext *context)
{
  return context->debug;
}

/**
 * gksu_context_set_ssh_fwd:
 * @context: the #GksuContext you want to modify
 * @value: TRUE or FALSE
 *
 * Set up if we are running inside a sshd forwarding
 * session. That's because the X authorization semantics
 * differ a bit.
 * -> THIS SETTING IS CURRENTLY USED FOR NOTHING
 */
void
gksu_context_set_ssh_fwd (GksuContext *context, gboolean value)
{
  context->ssh_fwd = value;
}

/**
 * gksu_context_get_ssh_fwd:
 * @context: the #GksuContext from which to grab the information
 *
 * Finds out if the library is configured as being run inside a
 * ssh X11 forwarding tunnel.
 * -> THIS SETTING IS CURRENTLY USED FOR NOTHING
 *
 * Returns: TRUE if it is, FALSE otherwise.
 */
gboolean
gksu_context_get_ssh_fwd (GksuContext *context)
{
  return context->ssh_fwd;
}

GType
gksu_context_get_type (void)
{
  static GType type = 0;

  if (type == 0)
    {
      static const GTypeInfo info =
	{
	  sizeof (GksuContextClass), /* size of class */
	  NULL, /* base_init */
	  NULL, /* base_finalize */
	  (GClassInitFunc) gksu_context_class_init,
	  NULL, /* class_finalize */
	  NULL, /* class_data */
	  sizeof (GksuContext), /* size of object */
	  0, /* n_preallocs */
	  (GInstanceInitFunc) gksu_context_init /* instance_init */
	};
      type = g_type_register_static (G_TYPE_OBJECT,
				     "GksuContextType",
				     &info, 0);
    }

  return type;
}

static void
gksu_context_init (GTypeInstance *instance, gpointer g_class)
{
  GksuContext *self = (GksuContext*)instance;

  self->xauth = NULL;
  self->dir = NULL;
  self->display = NULL;
  self->user = g_strdup ("root");
  self->password = NULL;
  self->command = NULL;

  self->login_shell = FALSE;
  self->keep_env = FALSE;

  self->debug = FALSE;
  self->ssh_fwd = FALSE;
}

static void
gksu_context_dispose (GObject *object)
{
  GksuContext *self = (GksuContext*)object;

  g_free (self->xauth);
  g_free (self->dir);
  g_free (self->display);
  g_free (self->user);
  g_free (self->password);
  g_free (self->command);

  g_free (self->user);
}

static void
gksu_context_finalize (GObject *object)
{
}

static void
gksu_context_class_init (GksuContextClass *klass)
{
  GObjectClass *gobject_class = G_OBJECT_CLASS (klass);

  gobject_class->dispose = gksu_context_dispose;
  gobject_class->finalize = gksu_context_finalize;

}

GType
gksu_context_error_get_type (void)
{
  static GType etype = 0;
  if (etype == 0) {
    static const GEnumValue values[] = {
      { GKSU_CONTEXT_ERROR_HELPER, "GKSU_CONTEXT_ERROR_HELPER", "helper" },
      { GKSU_CONTEXT_ERROR_NOCOMMAND, "GKSU_CONTEXT_ERROR_NOCOMMAND", "nocommand" },
      { GKSU_CONTEXT_ERROR_NOPASSWORD, "GKSU_CONTEXT_ERROR_NOPASSWORD", "nopassword" },
      { GKSU_CONTEXT_ERROR_FORK, "GKSU_CONTEXT_ERROR_FORK", "fork" },
      { GKSU_CONTEXT_ERROR_EXEC, "GKSU_CONTEXT_ERROR_EXEC", "exec" },
      { GKSU_CONTEXT_ERROR_PIPE, "GKSU_CONTEXT_ERROR_PIPE", "pipe" },
      { GKSU_CONTEXT_ERROR_PIPEREAD, "GKSU_CONTEXT_ERROR_PIPEREAD", "piperead" },
      { GKSU_CONTEXT_ERROR_WRONGPASS, "GKSU_CONTEXT_ERROR_WRONGPASS", "wrongpass" },
      { GKSU_CONTEXT_ERROR_CHILDFAILED, "GKSU_CONTEXT_ERROR_CHILDFAILED", "childfailed" },
      { 0, NULL, NULL }
    };
    etype = g_enum_register_static ("GksuContextError", values);
  }
  return etype;
}

/**
 * prepare_xauth:
 *
 * Sets up the variables with values for the $DISPLAY 
 * environment variable and xauth-related stuff. Also
 * creates a temporary directory to hold a .Xauthority
 *
 * Returns: 0 if it suceeds.
 */
static int
prepare_xauth (GksuContext *context)
{
  FILE *xauth_output;
  gchar xauth[256] = {0};

  gchar *tmp = NULL;
  gchar **tmpv = NULL;
  gchar *display = NULL;

  /* avoid problems with "network" DISPLAY's */
  display = g_strdup (getenv ("DISPLAY"));
  tmpv = g_strsplit (display, ":", 3);
  g_free (display);
  
  context->display = g_strdup_printf (":%s", tmpv[1]);
  
  g_strfreev (tmpv);

  /* get the authorization token */
  tmp = g_strdup_printf ("/usr/X11R6/bin/xauth list %s | "
			 "grep 'MIT-MAGIC-COOKIE-1' | "
			 "cut -d ' ' -f 5",
			 context->display);
  if ((xauth_output = popen (tmp, "r")) == NULL)
    {
      fprintf (stderr,
	       _("Failed to obtain xauth key: %s"),
	       strerror(errno));
      return 1;
    }
  fread (xauth, sizeof(char), 256, xauth_output);
  xauth[strlen(xauth) - 1] = '\0';
  pclose (xauth_output);
  g_free (tmp);

  context->xauth = g_strdup (xauth);

  if (context->debug)
    {
      fprintf(stderr, 
	      "xauth: -%s-\n"
	      "display: -%s-\n",
	      context->xauth, context->display);
    }

  return 0;
}

/* Write all of buf, even if write(2) is interrupted. */
static ssize_t
full_write (int d, const char *buf, size_t nbytes)
{
  ssize_t r, w = 0;

  /* Loop until nbytes of buf have been written. */
  while (w < nbytes) {
    /* Keep trying until write succeeds without interruption. */
    do {
      r = write(d, buf + w, nbytes - w);
    } while (r < 0 && errno == EINTR);

    if (r < 0) {
      return -1;
    }

    w += r;
  }

  return w;
}

static gboolean
copy (const char *fn, const char *dir)
{
  int in, out;
  int r;
  char *newfn;
  char buf[BUFSIZ];

  newfn = g_strdup_printf("%s/.Xauthority", dir);

  out = open(newfn, O_WRONLY | O_CREAT | O_EXCL);
  if (out == -1) 
    {
      if (errno == EEXIST)
	fprintf (stderr,
		   _("The X authority file i am trying to create for"
		     " the target user already exists! This is highly"
		     " suspicous!"));
      else
	fprintf (stderr,
		   _("Error copying '%s' to '%s': %s"),
		   fn, dir, strerror(errno));

      return TRUE;
    }
  
  in = open(fn, O_RDONLY);
  if (in == -1)
    {
      fprintf (stderr,
	       _("Error copying '%s' to '%s': %s"),
	       fn, dir, strerror(errno));
      return TRUE;
    }
  
  while ((r = read(in, buf, BUFSIZ)) > 0) 
    {
      if (full_write(out, buf, r) == -1)
	{
	  fprintf (stderr,
		   _("Error copying '%s' to '%s': %s"),
		   fn, dir, strerror(errno));
	  return TRUE;
	}
    }

  if (r == -1)
    {
      fprintf (stderr,
	       _("Error copying '%s' to '%s': %s"),
	       fn, dir, strerror(errno));
      return TRUE;
    }

  return FALSE;
}

static gboolean
sudo_prepare_xauth (GksuContext *context)
{
  gchar template[] = "/tmp/" PACKAGE "-XXXXXX";
  gboolean error_copying = FALSE;
  gchar *xauth;

  context->dir = g_strdup (mkdtemp(template));
  if (!context->dir)
    {
      fprintf (stderr, strerror(errno));
      return 1;
    }
  
  xauth = g_strdup(g_getenv ("XAUTHORITY"));
  if (xauth == NULL)
    xauth = g_strdup_printf ("%s/.Xauthority", g_get_home_dir());

  error_copying = copy (xauth, context->dir);
  g_free (xauth);

  if (error_copying)
    return 1;

  return 0;
}

static void
sudo_reset_xauth (GksuContext *context, gchar *xauth, 
		  gchar *xauth_env)
{
  /* reset the env var as it was before or clean it  */
  if (xauth_env)
    setenv ("XAUTHORITY", xauth_env, TRUE);
  else
    unsetenv ("XAUTHORITY");
  
  if (context->debug)
    fprintf (stderr, "xauth: %s\nxauth_env: %s\ndir: %s\n",
	     xauth, xauth_env, context->dir);
  
  unlink (xauth);
  rmdir (context->dir);

  g_free (xauth);
}

/**
 * gksu_context_run:
 * @context: a #GksuContext
 * @error: a #GError object to be filled with the error code or NULL
 *
 * This could be considered one of the main functions in GKSu.
 * it is responsible for doing the 'user changing' magic.
 *
 * Returns: the child's error status, 0 if all went fine, -1 if failed
 */
gboolean
gksu_context_run (GksuContext *context, GError **error)
{
  int i = 0;
  gboolean auth_failed = FALSE;
  GQuark gksu_quark;

  gchar auxcommand[] = PREFIX "/lib/" PACKAGE "/gksu-run-helper";

  int fdpty;
  pid_t pid;

  gksu_quark = g_quark_from_string (PACKAGE);

  if (!context->command)
    {
      g_set_error (error, gksu_quark, GKSU_CONTEXT_ERROR_NOCOMMAND,
		   _("gksu_run needs a command to be run, "
		     "none was provided."));
      return -1;
    }

  if (!g_file_test (auxcommand, G_FILE_TEST_IS_EXECUTABLE))
    {
      g_set_error (error, gksu_quark, GKSU_CONTEXT_ERROR_HELPER,
		   _("The gksu-run-helper command was not found or "
		     "is not executable."));
      return -1;
    }

  /* 
     FIXME: need to check if we are in X and also need to
     provide GError stuff
  */
  prepare_xauth (context);

  pid = forkpty (&fdpty, NULL, NULL, NULL);
  if (pid == 0)
    {
      gchar **cmd = g_malloc (sizeof(gchar*)*7);

      cmd[i] = g_strdup ("/bin/su"); i++;
      if (context->login_shell)
	{
	  cmd[i] = g_strdup ("-"); i++;
	}
      cmd[i] = g_strdup (context->user); i++;
      if (context->keep_env)
	{
	  cmd[i] = g_strdup ("-p"); i++;
	}
      cmd[i] = g_strdup ("-c"); i++;

      /* needs to get X authorization prior to running the program */
      cmd[i] = g_strdup_printf ("%s \"%s\"", auxcommand, 
				context->command); i++;

      cmd[i] = NULL;

      /* executes the command */
      if (execv (cmd[0], cmd) == -1)
	{
	  fprintf (stderr, 
		   _("Unable to run /bin/su: %s"),
		   strerror(errno));
	}

      for (i = 0 ; cmd[i] != NULL ; i++)
	g_free (cmd[i]);
      g_free(cmd);
    }
  else if (pid == -1)
    {
      g_set_error (error, gksu_quark, GKSU_CONTEXT_ERROR_FORK,
		   _("Failed to fork new process: %s"),
		   strerror(errno));
      return -1;
    }
  else
    {
      fd_set rfds;

      struct timeval tv;

      struct passwd *pwd = NULL;
      gint target_uid = -1;
      gint my_uid = 0;

      gchar buf[256];
      gint status;

      /* 
	 for controling the conversation with the
	 helper
      */
      gboolean first_line_read = FALSE;
      
      read (fdpty, buf, 256);
      if (context->debug)
	fprintf (stderr, "gksu_context_run: buf: -%s-\n", buf);
      
      my_uid = getuid();
      pwd = getpwnam (context->user);
      if (pwd)
	target_uid = pwd->pw_uid;

      /* no need to ask for password if we're already root */
      if (my_uid != target_uid && my_uid)
	{
	  if (context->password == NULL)
	    {
	      /* FIXME: need to check if we can run the program
		 without using a password (pam_wheel?) and if
		 not, return a comprehensive error message/code */
	      write (fdpty, "\n", 1);
	      return -1;
	    }

	  sleep (1);
	  write (fdpty, context->password, 
		 strlen(context->password));
	} 
      
      FD_ZERO(&rfds);
      FD_SET(fdpty, &rfds);

      tv.tv_sec = 0;
      tv.tv_usec = 100;

      while (!waitpid (pid, &status, WNOHANG))
	{
	  if (select (fdpty+1, &rfds, NULL, NULL, &tv) < 0)
	    {
	      g_set_error (error, gksu_quark, GKSU_CONTEXT_ERROR_PIPEREAD,
			   _("Could not read from the pipe "
			     "with the child: %s"),
			   strerror(errno));
	      return -1;
	    }

	  if(FD_ISSET(fdpty, &rfds))
	    {
	      bzero(buf, 256);
	      read (fdpty, buf, 255);

	      if (!first_line_read)
		{
		  read (fdpty, buf, 255);

		  if (!strncmp (buf, "su: Authentication failure", 26))
		    auth_failed = TRUE;
		  else if (!strncmp (buf, "gksu: waiting", 13))
		    {
		      gchar *line;
		      
		      line = g_strdup_printf ("gksu-run: %s\n", context->display);
		      write (fdpty, line, strlen(line));
		      g_free (line);
		      
		      line = g_strdup_printf ("gksu-run: %s\n", context->xauth);
		      write (fdpty, line, strlen(line));
		      g_free (line);

		      bzero (buf, 255);
		    }

		  first_line_read = TRUE;
		}
	      else if (!strncmp (buf, "gksu-run:", 8))
		{
		  gchar *tmp = NULL;

		  tmp = g_strrstr (buf, "\n");
		  while (!tmp)
		    {
		      bzero(buf, 256);
		      read (fdpty, buf, 255);
		      tmp = g_strrstr (tmp, "\n");
		    }
		  if (tmp)
		    fprintf (stderr, "%s", tmp+1);
		  bzero (buf, 255);
		}
	      else
		fprintf (stderr, "%s", buf);
	    }
	  else
	    FD_SET(fdpty, &rfds);

	  usleep (200);
	}

      while (1)
	{
	  bzero(buf, 256);
	  if(read (fdpty, buf, 255) <= 0)
	    break;
	  fprintf (stderr, "%s", buf);
	}

      if (WIFEXITED(status))
	{
	  if (WEXITSTATUS(status))
	    {
	      if (auth_failed)
		g_set_error (error, gksu_quark, GKSU_CONTEXT_ERROR_WRONGPASS,
			     _("Wrong password."));
	      else
		g_set_error (error, gksu_quark, GKSU_CONTEXT_ERROR_CHILDFAILED,
			     _("Child terminated with %d status"),
			     WEXITSTATUS(status));
	      return WEXITSTATUS(status);
	    }
	}
    }

  return 0;
}

/**
 * gksu_context_sudo_run:
 * @context: a #GksuContext
 * @error: a #GError object to be filled with the error code or NULL
 *
 * This could be considered one of the main functions in GKSu.
 * it is responsible for doing the 'user changing' magic by
 * calling gksu_ask_password() if it needs the user's password
 * it behaves like sudo.
 *
 * Returns: the child's error status, 0 if all went fine, -1 if failed
 */
gboolean
gksu_context_sudo_run (GksuContext *context, GError **error)
{
  char **cmd;
  char buffer[256];
  int argcount = 8;
  int i, j;

  GQuark gksu_quark;

  gchar *xauth = NULL, 
    *xauth_env = NULL;

  pid_t pid;
  int status;
  size_t r;
  FILE *infile, *outfile;
  int parent_pipe[2];	/* For talking to the parent */
  int child_pipe[2];	/* For talking to the child */
  
  gksu_quark = g_quark_from_string (PACKAGE);

  if (!context->command)
    {
      g_set_error (error, gksu_quark, GKSU_CONTEXT_ERROR_NOCOMMAND,
		   _("gksu_sudo_run needs a command to be run, "
		     "none was provided."));
      return -1;
    }

  /* 
     FIXME: need to check if we are in X
  */
  if (sudo_prepare_xauth (context) == 1)
    {
      g_set_error (error, gksu_quark, GKSU_CONTEXT_ERROR_XAUTH,
		   _("Unable to copy the user's Xauthorization file."));
      return -1;
    }

  /* sets XAUTHORITY */
  xauth = g_strdup_printf ("%s/.Xauthority", context->dir);
  xauth_env = getenv ("XAUTHORITY");
  setenv ("XAUTHORITY", xauth, TRUE);

  if (context->debug)
    fprintf (stderr, "xauth: %s\n", xauth);

  cmd = g_new (gchar *, argcount + 1);
  
  argcount = 0;

  /* sudo binary */
  cmd[argcount] = g_strdup("/usr/bin/sudo");
  argcount++;

  if (!context->keep_env)
    {
      /* Make sudo set $HOME */
      cmd[argcount] = g_strdup("-H");
      argcount++;
    }

  /* Make sudo read from stdin */
  cmd[argcount] = g_strdup("-S");
  argcount++;

  /* Make sudo use next arg as prompt */
  cmd[argcount] = g_strdup("-p");
  argcount++;

  /* prompt */
  cmd[argcount] = g_strdup("GNOME_SUDO_PASS");
  argcount++;

  /* Make sudo use the selected user */
  cmd[argcount] = g_strdup("-u");
  argcount++;

  /* user */
  cmd[argcount] = g_strdup(context->user);
  argcount++;

  /* sudo does not understand this if we do not use -H
     weird.
  */
  if (!context->keep_env)
    {
      /* Make sudo stop processing options */
      cmd[argcount] = g_strdup("--");
      argcount++;
    }

  for (i = j = 0; ; i++)
    {
      if (context->command[i] == ' ' || context->command[i] == '\0')
	{
	  buffer[j] = '\0';
	  cmd = g_realloc (cmd, sizeof(gchar*) * (argcount + 1));
	  cmd[argcount] = g_strdup (buffer);
	  bzero (buffer, 256);
	  argcount = argcount + 1;
	  j = 0;

	  if (context->command[i] == '\0')
	    break;
	}
      else
	{
	  if (context->command[i] == '\\')
	    i = i + 1;
	  buffer[j] = context->command[i];
	  j = j + 1;
	}
    }
  cmd = g_realloc (cmd, sizeof(gchar*) * (argcount + 1));
  cmd[argcount] = NULL;

  if (context->debug)
    {
      for (i = 0; cmd[i] != NULL; i++)
	fprintf (stderr, "cmd[%d]: %s\n", i, cmd[i]);
    }

  if ((pipe(parent_pipe)) == -1)
    {
      g_set_error (error, gksu_quark, GKSU_CONTEXT_ERROR_PIPE,
		   _("Error creating pipe: %s"),
		   strerror(errno));
      sudo_reset_xauth (context, xauth, xauth_env);
      return -1;
    }

  if ((pipe(child_pipe)) == -1)
    {
      g_set_error (error, gksu_quark, GKSU_CONTEXT_ERROR_PIPE,
		   _("Error creating pipe: %s"),
		   strerror(errno));
      sudo_reset_xauth (context, xauth, xauth_env);
      return -1;
    }

  pid = fork();
  if (pid == -1)
    {
      g_set_error (error, gksu_quark, GKSU_CONTEXT_ERROR_FORK,
		   _("Failed to fork new process: %s"),
		   strerror(errno));
      sudo_reset_xauth (context, xauth, xauth_env);
      return -1;
    }
  else if (pid == 0) 
    {
      // Child
      close(child_pipe[1]);
      dup2(child_pipe[0], STDIN_FILENO);
      dup2(parent_pipe[1], STDERR_FILENO);
      
      execv(cmd[0], cmd);
      
      g_set_error (error, gksu_quark, GKSU_CONTEXT_ERROR_EXEC,
		   _("Failed to exec new process: %s"),
		   strerror(errno));
      sudo_reset_xauth (context, xauth, xauth_env);
      return -1;
    } 
  else 
    {
      gboolean auth_failed = FALSE;

      // Parent
      close(parent_pipe[1]);
      
      infile = fdopen(parent_pipe[0], "r");
      if (!infile)
	{
	  g_set_error (error, gksu_quark, GKSU_CONTEXT_ERROR_PIPE,
		       _("Error opening pipe: %s"),
		       strerror(errno));
	  sudo_reset_xauth (context, xauth, xauth_env);
	  return -1;
	}

      outfile = fdopen(child_pipe[1], "w");
      if (!outfile)
	{
	  g_set_error (error, gksu_quark, GKSU_CONTEXT_ERROR_PIPE,
		       _("Error opening pipe: %s"),
		       strerror(errno));
	  sudo_reset_xauth (context, xauth, xauth_env);
	  return -1;
	}

      /*
	we are expecting to receive a GNOME_SUDO_PASS
	if we don't there are two possibilities: an error
	or a password is not needed
      */
      r = read(parent_pipe[0], buffer, 16);

      if (context->debug)
	fprintf (stderr, "buffer: -%s-\n", buffer);

      if (strncmp(buffer, "GNOME_SUDO_PASS", 16) == 0) 
	{
	  if (context->debug)
	    fprintf (stderr, "Yeah, we're in...\n");

	  if (context->password == NULL)
	    {
	      g_set_error (error, gksu_quark, GKSU_CONTEXT_ERROR_NOPASSWORD,
			   _("No password was supplied and sudo needs it."));
	      fprintf (outfile, "\n");
	      sudo_reset_xauth (context, xauth, xauth_env);
	      return -1;
	    }

	  fprintf (outfile, "%s", context->password);
	  fclose (outfile);
	} 
      else 
	{
	  if (context->debug)
	    fprintf (stderr, "Oops... what's up?\n");

	  buffer[16] = 0;
	  fprintf (stderr, "%s", buffer);
	}

      /* ignore the newline that comes right after sudo receives
	 the password
      */
      fgets (buffer, 255, infile);
      fgets (buffer, 255, infile);
      if (!strcmp (buffer, "Sorry, try again.\n"))
	auth_failed = TRUE;

      while (!waitpid (pid, &status, WNOHANG))
	{
	  bzero(buffer, 256);
	  if(!fgets (buffer, 255, infile))
	    break;
	  fprintf (stderr, "%s", buffer);
	}

      /* make sure we did read everything */
      while (1)
	{
	  bzero(buffer, 256);
	  if(!fread (buffer, sizeof(gchar), 255, infile))
	    break;
	  fprintf (stderr, "%s", buffer);
	  fflush (stderr);
	}
      
      if (WIFEXITED(status))
	{
	  if (WEXITSTATUS(status))
	    {
	      if (auth_failed)
		g_set_error (error, gksu_quark, GKSU_CONTEXT_ERROR_WRONGPASS,
			     _("Wrong password."));
	      else
		g_set_error (error, gksu_quark, GKSU_CONTEXT_ERROR_CHILDFAILED,
			     _("Child terminated with %d status"),
			     WEXITSTATUS(status));

	      sudo_reset_xauth (context, xauth, xauth_env);
	      return TRUE;
	    }
	}
      
      fclose(infile);
    }
  
  sudo_reset_xauth (context, xauth, xauth_env);
  return FALSE;
}

