/* LanguageTool, a natural language style checker 
 * Copyright (C) 2007 Daniel Naber (http://www.danielnaber.de)
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301
 * USA
 */
package de.danielnaber.languagetool.language;

import java.util.*;

import de.danielnaber.languagetool.Language;
import de.danielnaber.languagetool.rules.*;
import de.danielnaber.languagetool.rules.patterns.Unifier;
import de.danielnaber.languagetool.synthesis.Synthesizer;
import de.danielnaber.languagetool.synthesis.es.SpanishSynthesizer;
import de.danielnaber.languagetool.tagging.Tagger;
import de.danielnaber.languagetool.tagging.disambiguation.Disambiguator;
import de.danielnaber.languagetool.tagging.disambiguation.rules.es.SpanishRuleDisambiguator;
import de.danielnaber.languagetool.tagging.es.SpanishTagger;
import de.danielnaber.languagetool.tokenizers.SRXSentenceTokenizer;
import de.danielnaber.languagetool.tokenizers.SentenceTokenizer;
import de.danielnaber.languagetool.tokenizers.Tokenizer;
import de.danielnaber.languagetool.tokenizers.es.SpanishWordTokenizer;

public class Spanish extends Language {

  private static final Unifier SPANISH_UNIFIER = new Unifier();

  private final SentenceTokenizer sentenceTokenizer = new SRXSentenceTokenizer(getShortName());
  private final Tokenizer wordTokenizer = new SpanishWordTokenizer();
  private final Synthesizer synthesizer = new SpanishSynthesizer();
  private final Tagger tagger = new SpanishTagger();
  private final Disambiguator disambiguator = new SpanishRuleDisambiguator();

  @Override
  public Locale getLocale() {
    return new Locale(getShortName());
  }

  @Override
  public String getName() {
    return "Spanish";
  }

  @Override
  public String getShortName() {
    return "es";
  }

  @Override
  public String[] getCountryVariants() {
    return new String[]{
            "ES", "", "MX", "GT", "CR", "PA", "DO",
            "VE", "PE", "AR", "EC", "CL", "UY", "PY",
            "BO", "SV", "HN", "NI", "PR", "US", "CU"
    };
  }
  
  @Override
  public String[] getUnpairedRuleStartSymbols() {
    return new String[]{ "[", "(", "{", "“", "«", "¿", "¡" };
  }

  @Override
  public String[] getUnpairedRuleEndSymbols() {
    return new String[]{ "]", ")", "}", "”", "»", "?", "!" };
  }
  
  @Override
  public Tagger getTagger() {
    return tagger;
  }
  
  @Override
  public Disambiguator getDisambiguator() {
	    return disambiguator;
  }
  
  @Override
  public Unifier getUnifier() {
	    return SPANISH_UNIFIER;
  }
  
  @Override
  public final Tokenizer getWordTokenizer() {
	    return wordTokenizer;
	  }
  
  @Override
  public final Synthesizer getSynthesizer() {
	    return synthesizer;
  }

  @Override
  public final SentenceTokenizer getSentenceTokenizer() {
	    return sentenceTokenizer;
  }
  
  @Override
  public Contributor[] getMaintainers() {
	  final Contributor contributor = new Contributor("Juan Martorell");
    contributor.setUrl("http://languagetool-es.blogspot.com/");
    return new Contributor[] { contributor };
  }

  @Override
  public List<Class<? extends Rule>> getRelevantRules() {
    return Arrays.asList(
            CommaWhitespaceRule.class,
            DoublePunctuationRule.class,
            GenericUnpairedBracketsRule.class,
            UppercaseSentenceStartRule.class,
            WordRepeatRule.class,
            WhitespaceRule.class
    );
  }

}
