/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
* This file is part of the libgltf project.
*
* This Source Code Form is subject to the terms of the Mozilla Public
* License, v. 2.0. If a copy of the MPL was not distributed with this
* file, You can obtain one at http://mozilla.org/MPL/2.0/.
*/

#include "libgltf.h"
#include "RenderScene.h"
#include <cassert>

namespace libgltf
{

glTFHandle* gltf_renderer_init(const std::string& jsonfile, std::vector<glTFFile>& o_glTFFiles)
{
    RenderScene* renderScene = new RenderScene();
    glTFHandle* gltf_result = renderScene->initScene(jsonfile, o_glTFFiles);
    if( gltf_result )
        gltf_result->renderer = renderScene;
    return gltf_result;
}

int gltf_renderer_set_content(glTFHandle* handle, const std::vector<glTFFile>& inputFiles)
{
    assert(handle);
    if( !handle )
        return LIBGLTF_UNKNOWN_ERROR;

    RenderScene* renderScene = reinterpret_cast<RenderScene*>(handle->renderer);
    return renderScene->initRender(inputFiles);
}

void gltf_render_FPS_enable(glTFHandle* handle)
{
    assert(handle);
    if( handle )
    {
        RenderScene* renderScene = reinterpret_cast<RenderScene*>(handle->renderer);
        renderScene->initFPS();
    }
}

int gltf_prepare_renderer(glTFHandle* handle)
{
    assert(handle);
    if( handle )
    {
        RenderScene* renderScene = reinterpret_cast<RenderScene*>(handle->renderer);
        return renderScene->prepareRender(&handle->viewport);
    }
    else
    {
        return LIBGLTF_UNKNOWN_ERROR;
    }
}

void gltf_renderer(glTFHandle* handle)
{
    assert(handle);
    if( handle )
    {
        RenderScene* renderScene = reinterpret_cast<RenderScene*>(handle->renderer);
        renderScene->render();
    }
}

void gltf_complete_renderer(glTFHandle* handle)
{
    assert(handle);
    if( handle )
    {
        RenderScene* renderScene = reinterpret_cast<RenderScene*>(handle->renderer);
        renderScene->completeRender();
    }
}

void gltf_renderer_release(glTFHandle* handle)
{
    assert(handle);
    if( handle )
    {
        RenderScene* renderScene = reinterpret_cast<RenderScene*>(handle->renderer);
        renderScene->releaseRender();
        delete renderScene;
    }
}

glm::vec3* gltf_get_model_center_pos(glTFHandle* handle)
{
    assert(handle);
    if( handle )
    {
        RenderScene* renderScene = reinterpret_cast<RenderScene*>(handle->renderer);
        return renderScene->getModelCenterPos();
    }
    else
    {
        return 0;
    }
}

void gltf_get_camera_pos(glTFHandle* handle, glm::vec3* eye,
                         glm::vec3* view, glm::vec3* up)
{
    assert(handle);
    if( handle )
    {
        RenderScene* renderScene = reinterpret_cast<RenderScene*>(handle->renderer);
        assert(eye);
        assert(view);
        assert(up);

        if( eye && view && up )
        {
            renderScene->getCameraPos(eye, view, up);
        }
    }
    else
    {
        *eye = glm::vec3(0.0f);
        *view = glm::vec3(0.0f);
        *up = glm::vec3(0.0f);
    }
}

double gltf_get_model_size(glTFHandle* handle)
{
    assert(handle);
    if( handle )
    {
        RenderScene* renderScene = reinterpret_cast<RenderScene*>(handle->renderer);
        return renderScene->getModelSize();
    }
    else
    {
        return 0.0;
    }
}

void gltf_renderer_move_camera(glTFHandle* handle, double x, double y,
                               double z, double time)
{
    assert(handle);
    if( handle )
    {
        RenderScene* renderScene = reinterpret_cast<RenderScene*>(handle->renderer);
        renderScene->getCamera().moveCamera(x, y, z, time);
    }
}

void gltf_renderer_rotate_model(glTFHandle* handle, double horizontal,
                                double vertical, double planar)
{
    assert(handle);
    if( handle )
    {
        RenderScene* renderScene = reinterpret_cast<RenderScene*>(handle->renderer);
        if( renderScene->isRotationEnabled() )
            renderScene->getCamera().rotateObjectMouse(horizontal, vertical, planar);
    }
}

void gltf_renderer_rotate_camera(glTFHandle* handle, double horizontal,
                                 double vertical, double planar)
{
    assert(handle);
    if( handle )
    {
        RenderScene* renderScene = reinterpret_cast<RenderScene*>(handle->renderer);
        if( renderScene->isRotationEnabled() )
        {
            renderScene->getCamera().rotateCamera(horizontal, vertical, planar);
        }
    }
}

void gltf_enable_rotation(glTFHandle* handle)
{
    assert(handle);
    if( handle )
    {
        RenderScene* renderScene = reinterpret_cast<RenderScene*>(handle->renderer);
        renderScene->enableRotation();
    }
}

void gltf_disable_rotation(glTFHandle* handle)
{
    assert(handle);
    if( handle )
    {
        RenderScene* renderScene = reinterpret_cast<RenderScene*>(handle->renderer);
        renderScene->disableRotation();
    }
}

void gltf_enable_transparency(glTFHandle* handle)
{
    assert(handle);
    if( handle )
    {
        RenderScene* renderScene = reinterpret_cast<RenderScene*>(handle->renderer);
        renderScene->enableTransparency();
    }
}

void gltf_disable_transparency(glTFHandle* handle)
{
    assert(handle);
    if( handle )
    {
        RenderScene* renderScene = reinterpret_cast<RenderScene*>(handle->renderer);
        renderScene->disableTransparency();
    }
}

int gltf_renderer_get_bitmap(glTFHandle** handle, int size,
                             char* buffer, GLenum format, double time)
{
    if (size <= 0)
        return LIBGLTF_INVALID_SIZE;
    assert(handle[0]);
    if( !handle || !handle[0] )
        return LIBGLTF_UNKNOWN_ERROR;
    RenderScene* renderScene = reinterpret_cast<RenderScene*>(handle[0]->renderer);
    int status = renderScene->prepareRenderBitmap(&(handle[0]->viewport));
    if (LIBGLTF_SUCCESS != status)
    {
        return status;
    }
    for (int i = 0; i < size; i++)
    {
        assert(handle[i]);
        if( !handle[i] )
            return LIBGLTF_UNKNOWN_ERROR;
        RenderScene* tRenderScene = reinterpret_cast<RenderScene*>(handle[i]->renderer);
        tRenderScene->renderBitmap(time);
    }
    renderScene->completeRenderBitmap(&(handle[0]->viewport),
                                      (unsigned char*)buffer, format);
    return LIBGLTF_SUCCESS;
}

void gltf_animation_start(glTFHandle* handle)
{
    assert(handle);
    if( handle )
    {
        RenderScene* renderScene = reinterpret_cast<RenderScene*>(handle->renderer);
        renderScene->startAnimation();
    }
}

void gltf_animation_stop(glTFHandle* handle)
{
    assert(handle);
    if( handle )
    {
        RenderScene* renderScene = reinterpret_cast<RenderScene*>(handle->renderer);
        renderScene->stopAnimation();
    }
}

void gltf_orbit_mode_start(glTFHandle* handle)
{
    assert(handle);
    if( handle )
    {
        RenderScene* renderScene = reinterpret_cast<RenderScene*>(handle->renderer);
        renderScene->startAerialView();
    }
}

void gltf_orbit_mode_stop(glTFHandle* handle)
{
    assert(handle);
    if( handle )
    {
        RenderScene* renderScene = reinterpret_cast<RenderScene*>(handle->renderer);
        renderScene->stopAerialView();
    }
}

bool gltf_animation_is_playing(glTFHandle* handle)
{
    assert(handle);
    if( handle )
    {
        RenderScene* renderScene = reinterpret_cast<RenderScene*>(handle->renderer);
        return renderScene->isAnimPlay();
    }
    else
    {
        return false;
    }
}

void gltf_animation_set_looping(glTFHandle* handle, bool loop)
{
    assert(handle);
    if( handle )
    {
        RenderScene* renderScene = reinterpret_cast<RenderScene*>(handle->renderer);
        renderScene->setAnimLoop(loop);
    }
}

bool gltf_animation_get_looping(glTFHandle* handle)
{
    assert(handle);
    if( handle )
    {
        RenderScene* renderScene = reinterpret_cast<RenderScene*>(handle->renderer);
        return renderScene->getAnimLoop();
    }
    else
    {
        return false;
    }
}

double gltf_animation_get_duration(glTFHandle* handle)
{
    assert(handle);
    if( handle )
    {
        RenderScene* renderScene = reinterpret_cast<RenderScene*>(handle->renderer);
        return renderScene->getAnimDuration();
    }
    else
    {
        return 0.0;
    }
}

void gltf_animation_set_time(glTFHandle* handle, double time)
{
    assert(handle);
    if( handle )
    {
        RenderScene* renderScene = reinterpret_cast<RenderScene*>(handle->renderer);
        if (time <= 0)
        {
            renderScene->setAnimTime(0.0);
        }
        else
        {
            renderScene->setAnimTime(time);
        }
    }
}

double gltf_animation_get_time(glTFHandle* handle)
{
    assert(handle);
    if( handle )
    {
        RenderScene* renderScene = reinterpret_cast<RenderScene*>(handle->renderer);
        return renderScene->getAnimTime();
    }
    else
    {
        return 0.0;
    }
}

void gltf_animation_resume(glTFHandle* handle)
{
    assert(handle);
    if( handle )
    {
        RenderScene* renderScene = reinterpret_cast<RenderScene*>(handle->renderer);
        renderScene->resumeAnim();
    }
}

} // namespace libgltf

/* vim:set shiftwidth=4 softtabstop=4 expandtab: */
