/*
 * Copyright (C) 2011 Canonical, Ltd.
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License
 * version 3.0 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3.0 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library. If not, see
 * <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */

using GLib;
using Dee;

namespace Unity {

/* Encapsulates information about a search from Unity.
 * A bit sparse at the moment with information, but in the future will encapsulate
 * hint information too
 */
public class LensSearch : InitiallyUnowned
{
  public string search_string { get; construct; }
  public HashTable<string, Variant> hints { get; construct; }
  public Dee.SerializableModel results_model { get; construct; }

  [Signal (action = true)]
  public signal void finished ();
  /* We want nice method also for C and python consumers */
  [CCode (cname = "unity_lens_search_finished")]
  public void emit_finished ()
  {
    finished ();
  }

  public LensSearch (string search_string, HashTable<string, Variant> hints, 
                     Dee.SerializableModel results_model)
  {
    Object (search_string:search_string, hints:hints,
            results_model:results_model);
  }

  private bool got_finished = false;

  construct
  {
    this.finished.connect (() => { got_finished = true; });
  }

  internal bool was_finished ()
  {
    return got_finished;
  }

  public bool equals (LensSearch? other)
  {
    if (other == null ||
        search_string != other.search_string)
      return false;

    return true;
  }

  private HashTable<string, Variant>? _reply_hints = null;

  public void set_reply_hint (string key, Variant variant)
  {
    if (_reply_hints == null)
    {
      _reply_hints = new HashTable<string, Variant> (str_hash, str_equal);
    }

    _reply_hints.insert (key, variant);
  }

  internal HashTable<string, Variant>? get_reply_hints ()
  {
    return _reply_hints;
  }
}

} /* namespace */
