/*
 * Copyright (C) 2011 Canonical, Ltd.
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License
 * version 3.0 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3.0 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library. If not, see
 * <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */

using GLib;
using Gee;
using Config;


namespace Unity {

/*
 * Finds Scopes for a the Lens with the given ID and creates ScopeProxy objects
 */
private class ScopeFactory : GLib.Object
{
  public string lens_id { get; construct; }
  public string lenses_directory { get; construct; }

  public Gee.ArrayList<ScopeProxy> scopes;

  public signal void scope_added (ScopeProxy scope);

  public ScopeFactory (string lens_id_, string lenses_directory_="")
  {
    Object (lens_id:lens_id_, lenses_directory:lenses_directory_);
  }
  
  construct
  {
    if (lenses_directory == "")
      lenses_directory = "%s/unity/lenses/%s".printf (Config.DATADIR, lens_id);

    scopes = new Gee.ArrayList<ScopeProxy> ();

    list_lens_directory.begin ();
  }

  private async void list_lens_directory ()
  {
    Trace.log_object (this, "Searching for Scopes in %s", lenses_directory);

    var dir = File.new_for_path (lenses_directory);
    try {
      var e = yield dir.enumerate_children_async (FileAttribute.STANDARD_NAME,
                                                  0,
                                                  Priority.DEFAULT);
      while (true)
      {
        var files = yield e.next_files_async (10, Priority.DEFAULT);
        if (files == null)
          break;

        foreach (var info in files)
        {
          var name = info.get_name ();
          if (name.has_suffix (".scope"))
            load_scope_file (name);
        }
      }
    } catch (Error error) {
      warning ("Unable to search for scopes: %s", error.message);
    }
  }

  private void load_scope_file (string name)
  {
    const string GROUP = "Scope";
    const string DBUS_NAME = "DBusName";
    const string DBUS_PATH = "DBusPath";

    var path = Path.build_filename (lenses_directory, name);
    var key_file = new KeyFile ();

    try {
      key_file.load_from_file (path, KeyFileFlags.NONE);
      if (key_file.has_group ("Scope"))
      {
        var dbus_name = key_file.get_string (GROUP, DBUS_NAME);
        var dbus_path = key_file.get_string (GROUP, DBUS_PATH);
        
        var scope = new ScopeProxyRemote (dbus_name, dbus_path);
        scopes.add (scope);
        scope_added (scope);

        Trace.log_object (this, "Successfully loaded %s: %s@%s",
                          path, dbus_path, dbus_name);
      }
      else
      {
        warning ("Unable to load %s: Does not contain 'Scope' group", path);
      }
    } catch (Error e) {
      warning ("Unable to load %s: %s", path, e.message);
    }
  }

  public void add_local_scope (Scope scope)
  {
    var proxy = new ScopeProxyLocal (scope);
    scope.set_local (true);
    scopes.add (proxy);
    scope_added (proxy);
  }
}

} /* namespace */
