/*
 * Copyright (C) 2014 Etnaviv Project
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the next
 * paragraph) shall be included in all copies or substantial portions of the
 * Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 * Authors:
 *    Christian Gmeiner <christian.gmeiner@gmail.com>
 */

#include "util/hash_table.h"
#include "util/os_file.h"

#include "etnaviv_priv.h"
#include "etnaviv_drmif.h"

static pthread_mutex_t etna_drm_table_lock = PTHREAD_MUTEX_INITIALIZER;

struct etna_device *etna_device_new(int fd)
{
	struct etna_device *dev = calloc(sizeof(*dev), 1);
	struct drm_etnaviv_param req = {
		.param = ETNAVIV_PARAM_SOFTPIN_START_ADDR,
	};
	int ret;

	if (!dev)
		return NULL;

	p_atomic_set(&dev->refcnt, 1);
	dev->fd = fd;
	dev->handle_table = _mesa_hash_table_create(NULL, _mesa_hash_u32, _mesa_key_u32_equal);
	dev->name_table = _mesa_hash_table_create(NULL, _mesa_hash_u32, _mesa_key_u32_equal);
	etna_bo_cache_init(&dev->bo_cache);

	ret = drmCommandWriteRead(dev->fd, DRM_ETNAVIV_GET_PARAM, &req, sizeof(req));
	if (!ret && req.value != ~0ULL) {
		const uint64_t _4GB = 1ull << 32;

		util_vma_heap_init(&dev->address_space, req.value, _4GB - req.value);
		dev->use_softpin = 1;
	}

	return dev;
}

/* like etna_device_new() but creates it's own private dup() of the fd
 * which is close()d when the device is finalized. */
struct etna_device *etna_device_new_dup(int fd)
{
	int dup_fd = os_dupfd_cloexec(fd);
	struct etna_device *dev = etna_device_new(dup_fd);

	if (dev)
		dev->closefd = 1;
	else
		close(dup_fd);

	return dev;
}

struct etna_device *etna_device_ref(struct etna_device *dev)
{
	p_atomic_inc(&dev->refcnt);

	return dev;
}

static void etna_device_del_impl(struct etna_device *dev)
{
	etna_bo_cache_cleanup(&dev->bo_cache, 0);

	if (dev->use_softpin)
		util_vma_heap_finish(&dev->address_space);

	_mesa_hash_table_destroy(dev->handle_table, NULL);
	_mesa_hash_table_destroy(dev->name_table, NULL);

	if (dev->closefd)
		close(dev->fd);

	free(dev);
}

void etna_device_del_locked(struct etna_device *dev)
{
	if (!p_atomic_dec_zero(&dev->refcnt))
		return;

	etna_device_del_impl(dev);
}

void etna_device_del(struct etna_device *dev)
{
	if (!p_atomic_dec_zero(&dev->refcnt))
		return;

	pthread_mutex_lock(&etna_drm_table_lock);
	etna_device_del_impl(dev);
	pthread_mutex_unlock(&etna_drm_table_lock);
}

int etna_device_fd(struct etna_device *dev)
{
   return dev->fd;
}

bool etnaviv_device_softpin_capable(struct etna_device *dev)
{
	return !!dev->use_softpin;
}
