//
// UseCorrectCasingRule class
//
// Authors:
//	Daniel Abramov <ex@vingrad.ru>
//	Sebastien Pouliot <sebastien@ximian.com>
//
// (C) 2007 Daniel Abramov
// Copyright (C) 2008 Novell, Inc (http://www.novell.com)
//
// Permission is hereby granted, free of charge, to any person obtaining
// a copy of this software and associated documentation files (the
// "Software"), to deal in the Software without restriction, including
// without limitation the rights to use, copy, modify, merge, publish,
// distribute, sublicense, and/or sell copies of the Software, and to
// permit persons to whom the Software is furnished to do so, subject to
// the following conditions:
//
// The above copyright notice and this permission notice shall be
// included in all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
// EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
// MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
// NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
// LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
// OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
// WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
//

using System;
using System.Collections.Generic;

using Mono.Cecil;

using Gendarme.Framework;
using Gendarme.Framework.Rocks;

namespace Gendarme.Rules.Naming {

	public class UseCorrectCasingRule : ITypeRule, IMethodRule {

		// check if name is PascalCased
		private static bool IsPascalCase (string name)
		{
			if (String.IsNullOrEmpty (name))
				return true;

			return Char.IsUpper (name [0]);
		}

		// convert name to PascalCase
		private static string PascalCase (string name)
		{
			if (String.IsNullOrEmpty (name))
				return String.Empty;

			if (name.Length == 1)
				return name.ToUpperInvariant ();

			int index = IndexOfFirstCorrectChar (name);
			return Char.ToUpperInvariant (name [index]) + name.Substring (index + 1);
		}

		// check if name is camelCased
		private static bool IsCamelCase (string name)
		{
			if (String.IsNullOrEmpty (name))
				return true;

			return Char.IsLower (name [0]);
		}

		// convert name to camelCase
		private static string CamelCase (string name)
		{
			if (String.IsNullOrEmpty (name))
				return String.Empty;

			if (name.Length == 1)
				return name.ToLowerInvariant ();

			int index = IndexOfFirstCorrectChar (name);
			return Char.ToLowerInvariant (name [index]) + name.Substring (index + 1);
		}

		private static int IndexOfFirstCorrectChar (string s)
		{
			int index = 0;
			while ((index < s.Length) && (s [index] == '_'))
				index++;
			// it's possible that we won't find one, e.g. something called "_"
			return (index == s.Length) ? 0 : index;
		}

		public MessageCollection CheckType (TypeDefinition typeDefinition, Runner runner)
		{
			// rule does not apply to generated code (outside developer's control)
			if (typeDefinition.IsGeneratedCode ())
				return runner.RuleSuccess;

			// <Module> isn't tagged as generated by the compiler but still must be ignored
			if (typeDefinition.Name == Constants.ModuleType)
				return runner.RuleSuccess;

			// ok, rule applies

			// types should all be PascalCased
			if (IsPascalCase (typeDefinition.Name))
				return runner.RuleSuccess;

			Location location = new Location (typeDefinition);
			string errorMessage = string.Format ("By existing naming conventions, the type names should all be pascal-cased (e.g. MyClass). Rename '{0}' type to '{1}'.",
							     typeDefinition.Name, PascalCase (typeDefinition.Name));
			return new MessageCollection (new Message (errorMessage, location, MessageType.Error));
		}

		public MessageCollection CheckMethod (MethodDefinition methodDefinition, Runner runner)
		{
			if (methodDefinition.IsConstructor)
				return runner.RuleSuccess;

			string name = methodDefinition.Name;
			MethodSemanticsAttributes attrs = methodDefinition.SemanticsAttributes;
			MethodSemanticsAttributes mask = MethodSemanticsAttributes.Getter | MethodSemanticsAttributes.Setter
				| MethodSemanticsAttributes.AddOn | MethodSemanticsAttributes.RemoveOn;
			if ((attrs & mask) != 0) {
				// it's something special
				int underscore = methodDefinition.Name.IndexOf ('_');
				if (underscore != -1)
					name = name.Substring (underscore + 1);
			} else if (methodDefinition.IsSpecialName || methodDefinition.IsGeneratedCode ()) {
				return runner.RuleSuccess;
			}

			MessageCollection messages = null;
			Location location = new Location (methodDefinition);

			// like types, methods/props should all be PascalCased, too
			if (!IsPascalCase (name)) {
				string errorMessage = string.Format ("By existing naming conventions, all the method and property names should all be pascal-cased (e.g. MyOperation). Rename '{0}' to '{1}'.",
								     name, PascalCase (name));
				messages = new MessageCollection (new Message (errorMessage, location, MessageType.Error));
			}
			// check parameters
			List<string> parameterNames = new List<string> ();
			foreach (ParameterDefinition paramDefinition in methodDefinition.Parameters) {
				if (!parameterNames.Contains (paramDefinition.Name)) // somewhy they duplicate sometimes
					parameterNames.Add (paramDefinition.Name);
			}
			foreach (string param in parameterNames) {
				// params should all be camelCased
				if (!IsCamelCase (param)) {
					if (messages == null)
						messages = new MessageCollection ();

					string errorMessage = string.Format ("By existing naming conventions, the parameter names should all be camel-cased (e.g. myParameter). Rename '{0}' parameter to '{1}'.",
										param, CamelCase (param));
					messages.Add (new Message (errorMessage, location, MessageType.Error));
				}
			}

			return messages;
		}
	}
}
