// !!! DO NOT EDIT - This file is generated automatically - DO NOT EDIT !!!
//
// X509CertificateTest.cs - NUnit2 Test Cases for X509Certificate
//
// Author:
//	Sebastien Pouliot (spouliot@motus.com)
//
// (C) 2002, 2003 Motus Technologies Inc. (http://www.motus.com)
//

using NUnit.Framework;
using System;
using System.Globalization;
using System.IO;
using System.Security.Cryptography;
using System.Security.Cryptography.X509Certificates;
using System.Text;
using System.Threading;

namespace MonoTests.System.Security.Cryptography.X509Certificates {

[TestFixture]
public class X509CertificateTest {

private CultureInfo oldcult;

[SetUp]
protected void SetUp () {
	// the current culture determines the result of formatting
	oldcult = Thread.CurrentThread.CurrentCulture;
	Thread.CurrentThread.CurrentCulture = new CultureInfo ("");
}

[TearDown]
protected void TearDown () {
	Thread.CurrentThread.CurrentCulture = oldcult;
	File.Delete("temp.cer");
	File.Delete("temp.b64");
}

public void AssertEquals (string msg, byte[] array1, byte[] array2)
{
	Assert.AreEqual (array1, array2, msg);
}

public string ToString (byte[] data)
{
	StringBuilder sb = new StringBuilder ();
	for (int i = 0; i < data.Length; i++)
		sb.Append (data[i].ToString ("X2"));
	return sb.ToString ();
}

//--8<-- NON GENERATED CODE STARTS HERE --8<----8<----8<----8<----8<----8<--


[Test]
public void Equality ()
{
	// copy from Certificate1()
	byte[] cert = { 0x30,0x82,0x01,0xFF,0x30,0x82,0x01,0x6C,0x02,0x05,0x02,0x72,0x00,0x06,0xE8,0x30,0x0D,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x01,0x02,0x05,0x00,0x30,0x5F,0x31,0x0B,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x55,0x53,0x31,0x20,0x30,0x1E,0x06,0x03,0x55,0x04,0x0A,0x13,0x17,0x52,0x53,0x41,0x20,0x44,0x61,0x74,0x61,0x20,0x53,0x65,0x63,0x75,0x72,0x69,0x74,0x79,0x2C,0x20,0x49,0x6E,0x63,0x2E,0x31,0x2E,0x30,0x2C,0x06,0x03,0x55,0x04,0x0B,0x13,0x25,0x53,0x65,0x63,0x75,0x72,0x65,0x20,0x53,0x65,0x72,0x76,
		0x65,0x72,0x20,0x43,0x65,0x72,0x74,0x69,0x66,0x69,0x63,0x61,0x74,0x69,0x6F,0x6E,0x20,0x41,0x75,0x74,0x68,0x6F,0x72,0x69,0x74,0x79,0x30,0x1E,0x17,0x0D,0x39,0x36,0x30,0x33,0x31,0x32,0x31,0x38,0x33,0x38,0x34,0x37,0x5A,0x17,0x0D,0x39,0x37,0x30,0x33,0x31,0x32,0x31,0x38,0x33,0x38,0x34,0x36,0x5A,0x30,0x61,0x31,0x0B,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x55,0x53,0x31,0x13,0x30,0x11,0x06,0x03,0x55,0x04,0x08,0x13,0x0A,0x43,0x61,0x6C,0x69,0x66,0x6F,0x72,0x6E,0x69,0x61,0x31,0x14,0x30,0x12,0x06,0x03,
		0x55,0x04,0x0A,0x13,0x0B,0x43,0x6F,0x6D,0x6D,0x65,0x72,0x63,0x65,0x4E,0x65,0x74,0x31,0x27,0x30,0x25,0x06,0x03,0x55,0x04,0x0B,0x13,0x1E,0x53,0x65,0x72,0x76,0x65,0x72,0x20,0x43,0x65,0x72,0x74,0x69,0x66,0x69,0x63,0x61,0x74,0x69,0x6F,0x6E,0x20,0x41,0x75,0x74,0x68,0x6F,0x72,0x69,0x74,0x79,0x30,0x70,0x30,0x0D,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x01,0x01,0x05,0x00,0x03,0x5F,0x00,0x30,0x5C,0x02,0x55,0x2D,0x58,0xE9,0xBF,0xF0,0x31,0xCD,0x79,0x06,0x50,0x5A,0xD5,0x9E,0x0E,0x2C,0xE6,0xC2,0xF7,0xF9,
		0xD2,0xCE,0x55,0x64,0x85,0xB1,0x90,0x9A,0x92,0xB3,0x36,0xC1,0xBC,0xEA,0xC8,0x23,0xB7,0xAB,0x3A,0xA7,0x64,0x63,0x77,0x5F,0x84,0x22,0x8E,0xE5,0xB6,0x45,0xDD,0x46,0xAE,0x0A,0xDD,0x00,0xC2,0x1F,0xBA,0xD9,0xAD,0xC0,0x75,0x62,0xF8,0x95,0x82,0xA2,0x80,0xB1,0x82,0x69,0xFA,0xE1,0xAF,0x7F,0xBC,0x7D,0xE2,0x7C,0x76,0xD5,0xBC,0x2A,0x80,0xFB,0x02,0x03,0x01,0x00,0x01,0x30,0x0D,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x01,0x02,0x05,0x00,0x03,0x7E,0x00,0x54,0x20,0x67,0x12,0xBB,0x66,0x14,0xC3,0x26,0x6B,0x7F,
		0xDA,0x4A,0x25,0x4D,0x8B,0xE0,0xFD,0x1E,0x53,0x6D,0xAC,0xA2,0xD0,0x89,0xB8,0x2E,0x90,0xA0,0x27,0x43,0xA4,0xEE,0x4A,0x26,0x86,0x40,0xFF,0xB8,0x72,0x8D,0x1E,0xE7,0xB7,0x77,0xDC,0x7D,0xD8,0x3F,0x3A,0x6E,0x55,0x10,0xA6,0x1D,0xB5,0x58,0xF2,0xF9,0x0F,0x2E,0xB4,0x10,0x55,0x48,0xDC,0x13,0x5F,0x0D,0x08,0x26,0x88,0xC9,0xAF,0x66,0xF2,0x2C,0x9C,0x6F,0x3D,0xC3,0x2B,0x69,0x28,0x89,0x40,0x6F,0x8F,0x35,0x3B,0x9E,0xF6,0x8E,0xF1,0x11,0x17,0xFB,0x0C,0x98,0x95,0xA1,0xC2,0xBA,0x89,0x48,0xEB,0xB4,0x06,0x6A,0x22,0x54,
		0xD7,0xBA,0x18,0x3A,0x48,0xA6,0xCB,0xC2,0xFD,0x20,0x57,0xBC,0x63,0x1C };

	X509Certificate x509 = new X509Certificate (cert);
	Assert.IsFalse (x509.Equals (null), "#1");

#if !NET_2_0
	X509Certificate nullobj = new X509Certificate (IntPtr.Zero);
	Assert.IsFalse (x509.Equals (nullobj), "#2");
	Assert.IsFalse (nullobj.Equals (null), "#3");
	Assert.IsFalse (nullobj.Equals (x509), "#4");
#endif
}

[Test]
#if NET_2_0
[ExpectedException (typeof (ArgumentNullException))]
#endif
public void ConstructorX509CertificateNull ()
{
	X509Certificate nullcopy = new X509Certificate ((X509Certificate) null);
}

//-->8-- NON GENERATED CODE ENDS HERE   -->8---->8---->8---->8---->8---->8--

// Certificate: basic\COMMERCE.cer
// - X.509v1 (i.e. without any extensions)
// - md2RSA (MD2 isn't part of the Cryptography namespace)
// - Uncommon key size (RSA 680 bits)
[Test]
public void Certificate1 ()
{
	// certificate can be loaded by constructor
	byte[] cert = { 0x30,0x82,0x01,0xFF,0x30,0x82,0x01,0x6C,0x02,0x05,0x02,0x72,0x00,0x06,0xE8,0x30,0x0D,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x01,0x02,0x05,0x00,0x30,0x5F,0x31,0x0B,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x55,0x53,0x31,0x20,0x30,0x1E,0x06,0x03,0x55,0x04,0x0A,0x13,0x17,0x52,0x53,0x41,0x20,0x44,0x61,0x74,0x61,0x20,0x53,0x65,0x63,0x75,0x72,0x69,0x74,0x79,0x2C,0x20,0x49,0x6E,0x63,0x2E,0x31,0x2E,0x30,0x2C,0x06,0x03,0x55,0x04,0x0B,0x13,0x25,0x53,0x65,0x63,0x75,0x72,0x65,0x20,0x53,0x65,0x72,0x76,
		0x65,0x72,0x20,0x43,0x65,0x72,0x74,0x69,0x66,0x69,0x63,0x61,0x74,0x69,0x6F,0x6E,0x20,0x41,0x75,0x74,0x68,0x6F,0x72,0x69,0x74,0x79,0x30,0x1E,0x17,0x0D,0x39,0x36,0x30,0x33,0x31,0x32,0x31,0x38,0x33,0x38,0x34,0x37,0x5A,0x17,0x0D,0x39,0x37,0x30,0x33,0x31,0x32,0x31,0x38,0x33,0x38,0x34,0x36,0x5A,0x30,0x61,0x31,0x0B,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x55,0x53,0x31,0x13,0x30,0x11,0x06,0x03,0x55,0x04,0x08,0x13,0x0A,0x43,0x61,0x6C,0x69,0x66,0x6F,0x72,0x6E,0x69,0x61,0x31,0x14,0x30,0x12,0x06,0x03,
		0x55,0x04,0x0A,0x13,0x0B,0x43,0x6F,0x6D,0x6D,0x65,0x72,0x63,0x65,0x4E,0x65,0x74,0x31,0x27,0x30,0x25,0x06,0x03,0x55,0x04,0x0B,0x13,0x1E,0x53,0x65,0x72,0x76,0x65,0x72,0x20,0x43,0x65,0x72,0x74,0x69,0x66,0x69,0x63,0x61,0x74,0x69,0x6F,0x6E,0x20,0x41,0x75,0x74,0x68,0x6F,0x72,0x69,0x74,0x79,0x30,0x70,0x30,0x0D,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x01,0x01,0x05,0x00,0x03,0x5F,0x00,0x30,0x5C,0x02,0x55,0x2D,0x58,0xE9,0xBF,0xF0,0x31,0xCD,0x79,0x06,0x50,0x5A,0xD5,0x9E,0x0E,0x2C,0xE6,0xC2,0xF7,0xF9,
		0xD2,0xCE,0x55,0x64,0x85,0xB1,0x90,0x9A,0x92,0xB3,0x36,0xC1,0xBC,0xEA,0xC8,0x23,0xB7,0xAB,0x3A,0xA7,0x64,0x63,0x77,0x5F,0x84,0x22,0x8E,0xE5,0xB6,0x45,0xDD,0x46,0xAE,0x0A,0xDD,0x00,0xC2,0x1F,0xBA,0xD9,0xAD,0xC0,0x75,0x62,0xF8,0x95,0x82,0xA2,0x80,0xB1,0x82,0x69,0xFA,0xE1,0xAF,0x7F,0xBC,0x7D,0xE2,0x7C,0x76,0xD5,0xBC,0x2A,0x80,0xFB,0x02,0x03,0x01,0x00,0x01,0x30,0x0D,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x01,0x02,0x05,0x00,0x03,0x7E,0x00,0x54,0x20,0x67,0x12,0xBB,0x66,0x14,0xC3,0x26,0x6B,0x7F,
		0xDA,0x4A,0x25,0x4D,0x8B,0xE0,0xFD,0x1E,0x53,0x6D,0xAC,0xA2,0xD0,0x89,0xB8,0x2E,0x90,0xA0,0x27,0x43,0xA4,0xEE,0x4A,0x26,0x86,0x40,0xFF,0xB8,0x72,0x8D,0x1E,0xE7,0xB7,0x77,0xDC,0x7D,0xD8,0x3F,0x3A,0x6E,0x55,0x10,0xA6,0x1D,0xB5,0x58,0xF2,0xF9,0x0F,0x2E,0xB4,0x10,0x55,0x48,0xDC,0x13,0x5F,0x0D,0x08,0x26,0x88,0xC9,0xAF,0x66,0xF2,0x2C,0x9C,0x6F,0x3D,0xC3,0x2B,0x69,0x28,0x89,0x40,0x6F,0x8F,0x35,0x3B,0x9E,0xF6,0x8E,0xF1,0x11,0x17,0xFB,0x0C,0x98,0x95,0xA1,0xC2,0xBA,0x89,0x48,0xEB,0xB4,0x06,0x6A,0x22,0x54,
		0xD7,0xBA,0x18,0x3A,0x48,0xA6,0xCB,0xC2,0xFD,0x20,0x57,0xBC,0x63,0x1C };

	X509Certificate x509 = new X509Certificate (cert);
	X509Certificate clone = new X509Certificate (x509);

	Assertion.Assert ("Equals", x509.Equals (clone));
	Assertion.Assert ("Equals", clone.Equals (x509));

	FileStream fs = File.OpenWrite ("temp.cer");
	fs.Write (cert, 0, cert.Length);
	fs.Close ();

	X509Certificate disk = X509Certificate.CreateFromCertFile ("temp.cer");
	Assertion.Assert ("Equals", disk.Equals (clone));
	Assertion.Assert ("Equals", disk.Equals (x509));
	byte[] hash = { 0xD6,0x2F,0x48,0xD0,0x13,0xEE,0x7F,0xB5,0x8B,0x79,0x07,0x45,0x12,0x67,0x0D,0x9C,0x5B,0x3A,0x5D,0xA9 };
	AssertEquals ("GetCertHash", hash, x509.GetCertHash ());
	Assertion.AssertEquals ("GetCertHashString", "D62F48D013EE7FB58B79074512670D9C5B3A5DA9", x509.GetCertHashString ());
#if NET_2_0
	DateTime from = DateTime.ParseExact (x509.GetEffectiveDateString (), "MM/dd/yyyy HH:mm:ss", null).ToUniversalTime ();
	Assert.AreEqual ("03/12/1996 18:38:47", from.ToString (), "GetEffectiveDateString");
	DateTime until = DateTime.ParseExact (x509.GetExpirationDateString (), "MM/dd/yyyy HH:mm:ss", null).ToUniversalTime ();
	Assert.AreEqual ("03/12/1997 18:38:46", until.ToString (), "GetExpirationDateString");
	Assert.AreEqual ("02720006E8", x509.GetSerialNumberString (), "GetSerialNumberString");
#else
	// fx 1.x has a bug where the returned dates were always in the Seattle time zone
	Assertion.AssertEquals ("GetEffectiveDateString", "03/12/1996 10:38:47", x509.GetEffectiveDateString ());
	Assertion.AssertEquals ("GetExpirationDateString", "03/12/1997 10:38:46", x509.GetExpirationDateString ());
	// which was making it easier to test the dates ;-)
	Assert.AreEqual ("E806007202", x509.GetSerialNumberString (), "GetSerialNumberString");
#endif
	Assertion.AssertEquals ("GetFormat", "X509", x509.GetFormat ());
	Assertion.AssertEquals ("GetHashCode", -701544240, x509.GetHashCode ());
	Assertion.AssertEquals ("GetIssuerName", "C=US, O=\"RSA Data Security, Inc.\", OU=Secure Server Certification Authority", x509.GetIssuerName ());
	Assertion.AssertEquals ("GetKeyAlgorithm", "1.2.840.113549.1.1.1", x509.GetKeyAlgorithm ());
	byte[] keyparams = { 0x05,0x00 };
	AssertEquals ("GetKeyAlgorithmParameters", keyparams, x509.GetKeyAlgorithmParameters ());
	Assertion.AssertEquals ("GetKeyAlgorithmParametersString", "0500", x509.GetKeyAlgorithmParametersString ());
	Assertion.AssertEquals ("GetName", "C=US, S=California, O=CommerceNet, OU=Server Certification Authority", x509.GetName ());
	byte[] pubkey = { 0x30,0x5C,0x02,0x55,0x2D,0x58,0xE9,0xBF,0xF0,0x31,0xCD,0x79,0x06,0x50,0x5A,0xD5,0x9E,0x0E,0x2C,0xE6,0xC2,0xF7,0xF9,0xD2,0xCE,0x55,0x64,0x85,0xB1,0x90,0x9A,0x92,0xB3,0x36,0xC1,0xBC,0xEA,0xC8,0x23,0xB7,0xAB,0x3A,0xA7,0x64,0x63,0x77,0x5F,0x84,0x22,0x8E,0xE5,0xB6,0x45,0xDD,0x46,0xAE,0x0A,0xDD,0x00,0xC2,0x1F,0xBA,0xD9,0xAD,0xC0,0x75,0x62,0xF8,0x95,0x82,0xA2,0x80,0xB1,0x82,0x69,0xFA,0xE1,0xAF,0x7F,0xBC,0x7D,0xE2,0x7C,0x76,0xD5,0xBC,0x2A,0x80,0xFB,0x02,0x03,0x01,0x00,0x01 };
	AssertEquals ("GetPublicKey", pubkey, x509.GetPublicKey ());
	Assertion.AssertEquals ("GetPublicKeyString", "305C02552D58E9BFF031CD7906505AD59E0E2CE6C2F7F9D2CE556485B1909A92B336C1BCEAC823B7AB3AA76463775F84228EE5B645DD46AE0ADD00C21FBAD9ADC07562F89582A280B18269FAE1AF7FBC7DE27C76D5BC2A80FB0203010001", x509.GetPublicKeyString ());
	AssertEquals ("GetRawCertData", cert, x509.GetRawCertData ());
	Assertion.AssertEquals ("GetRawCertDataString", ToString (cert), x509.GetRawCertDataString ());
	byte[] serial = { 0xE8,0x06,0x00,0x72,0x02 };
	AssertEquals ("GetSerialNumber", serial, x509.GetSerialNumber ());
	byte[] ToStringArray = { 0x43,0x45,0x52,0x54,0x49,0x46,0x49,0x43,0x41,0x54,0x45,0x3A,0x0D,0x0A,0x09,0x46,0x6F,0x72,0x6D,0x61,0x74,0x3A,0x20,0x20,0x58,0x35,0x30,0x39,0x0D,0x0A,0x09,0x4E,0x61,0x6D,0x65,0x3A,0x20,0x20,0x43,0x3D,0x55,0x53,0x2C,0x20,0x53,0x3D,0x43,0x61,0x6C,0x69,0x66,0x6F,0x72,0x6E,0x69,0x61,0x2C,0x20,0x4F,0x3D,0x43,0x6F,0x6D,0x6D,0x65,0x72,0x63,0x65,0x4E,0x65,0x74,0x2C,0x20,0x4F,0x55,0x3D,0x53,0x65,0x72,0x76,0x65,0x72,0x20,0x43,0x65,0x72,0x74,0x69,0x66,0x69,0x63,0x61,0x74,0x69,0x6F,0x6E,0x20,0x41,0x75,0x74,0x68,
		0x6F,0x72,0x69,0x74,0x79,0x0D,0x0A,0x09,0x49,0x73,0x73,0x75,0x69,0x6E,0x67,0x20,0x43,0x41,0x3A,0x20,0x20,0x43,0x3D,0x55,0x53,0x2C,0x20,0x4F,0x3D,0x22,0x52,0x53,0x41,0x20,0x44,0x61,0x74,0x61,0x20,0x53,0x65,0x63,0x75,0x72,0x69,0x74,0x79,0x2C,0x20,0x49,0x6E,0x63,0x2E,0x22,0x2C,0x20,0x4F,0x55,0x3D,0x53,0x65,0x63,0x75,0x72,0x65,0x20,0x53,0x65,0x72,0x76,0x65,0x72,0x20,0x43,0x65,0x72,0x74,0x69,0x66,0x69,0x63,0x61,0x74,0x69,0x6F,0x6E,0x20,0x41,0x75,0x74,0x68,0x6F,0x72,0x69,0x74,0x79,0x0D,0x0A,0x09,0x4B,
		0x65,0x79,0x20,0x41,0x6C,0x67,0x6F,0x72,0x69,0x74,0x68,0x6D,0x3A,0x20,0x20,0x31,0x2E,0x32,0x2E,0x38,0x34,0x30,0x2E,0x31,0x31,0x33,0x35,0x34,0x39,0x2E,0x31,0x2E,0x31,0x2E,0x31,0x0D,0x0A,0x09,0x53,0x65,0x72,0x69,0x61,0x6C,0x20,0x4E,0x75,0x6D,0x62,0x65,0x72,0x3A,0x20,0x20,0x45,0x38,0x30,0x36,0x30,0x30,0x37,0x32,0x30,0x32,0x0D,0x0A,0x09,0x4B,0x65,0x79,0x20,0x41,0x6C,0x6F,0x67,0x72,0x69,0x74,0x68,0x6D,0x20,0x50,0x61,0x72,0x61,0x6D,0x65,0x74,0x65,0x72,0x73,0x3A,0x20,0x20,0x30,0x35,0x30,0x30,0x0D,0x0A,
		0x09,0x50,0x75,0x62,0x6C,0x69,0x63,0x20,0x4B,0x65,0x79,0x3A,0x20,0x20,0x33,0x30,0x35,0x43,0x30,0x32,0x35,0x35,0x32,0x44,0x35,0x38,0x45,0x39,0x42,0x46,0x46,0x30,0x33,0x31,0x43,0x44,0x37,0x39,0x30,0x36,0x35,0x30,0x35,0x41,0x44,0x35,0x39,0x45,0x30,0x45,0x32,0x43,0x45,0x36,0x43,0x32,0x46,0x37,0x46,0x39,0x44,0x32,0x43,0x45,0x35,0x35,0x36,0x34,0x38,0x35,0x42,0x31,0x39,0x30,0x39,0x41,0x39,0x32,0x42,0x33,0x33,0x36,0x43,0x31,0x42,0x43,0x45,0x41,0x43,0x38,0x32,0x33,0x42,0x37,0x41,0x42,0x33,0x41,0x41,0x37,
		0x36,0x34,0x36,0x33,0x37,0x37,0x35,0x46,0x38,0x34,0x32,0x32,0x38,0x45,0x45,0x35,0x42,0x36,0x34,0x35,0x44,0x44,0x34,0x36,0x41,0x45,0x30,0x41,0x44,0x44,0x30,0x30,0x43,0x32,0x31,0x46,0x42,0x41,0x44,0x39,0x41,0x44,0x43,0x30,0x37,0x35,0x36,0x32,0x46,0x38,0x39,0x35,0x38,0x32,0x41,0x32,0x38,0x30,0x42,0x31,0x38,0x32,0x36,0x39,0x46,0x41,0x45,0x31,0x41,0x46,0x37,0x46,0x42,0x43,0x37,0x44,0x45,0x32,0x37,0x43,0x37,0x36,0x44,0x35,0x42,0x43,0x32,0x41,0x38,0x30,0x46,0x42,0x30,0x32,0x30,0x33,0x30,0x31,0x30,0x30,
		0x30,0x31,0x0D,0x0A,0x0D,0x0A };
	string tsa = Encoding.Default.GetString (ToStringArray).Replace ("\r\n", Environment.NewLine);
	//Assertion.AssertEquals ("ToString(true)", tsa, x509.ToString (true));
}


// Certificate: basic\ATT.cer
// - X.509v1 (i.e. without any extensions)
// - md5RSA, RSA 1024 bits, self-signed
// - has a ampersand (&) in its DN
[Test]
public void Certificate2 ()
{
	// certificate can be loaded by constructor
	byte[] cert = { 0x30,0x82,0x01,0xDF,0x30,0x82,0x01,0x48,0x02,0x01,0x00,0x30,0x0D,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x01,0x04,0x05,0x00,0x30,0x39,0x31,0x0B,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x55,0x53,0x31,0x0D,0x30,0x0B,0x06,0x03,0x55,0x04,0x0A,0x14,0x04,0x41,0x54,0x26,0x54,0x31,0x1B,0x30,0x19,0x06,0x03,0x55,0x04,0x0B,0x14,0x12,0x44,0x69,0x72,0x65,0x63,0x74,0x6F,0x72,0x79,0x20,0x53,0x65,0x72,0x76,0x69,0x63,0x65,0x73,0x30,0x1E,0x17,0x0D,0x39,0x36,0x30,0x31,0x31,0x38,0x32,0x31,0x30,0x33,0x35,0x32,
		0x5A,0x17,0x0D,0x30,0x31,0x30,0x31,0x31,0x36,0x32,0x31,0x30,0x33,0x35,0x32,0x5A,0x30,0x39,0x31,0x0B,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x55,0x53,0x31,0x0D,0x30,0x0B,0x06,0x03,0x55,0x04,0x0A,0x14,0x04,0x41,0x54,0x26,0x54,0x31,0x1B,0x30,0x19,0x06,0x03,0x55,0x04,0x0B,0x14,0x12,0x44,0x69,0x72,0x65,0x63,0x74,0x6F,0x72,0x79,0x20,0x53,0x65,0x72,0x76,0x69,0x63,0x65,0x73,0x30,0x81,0x9D,0x30,0x0D,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x01,0x01,0x05,0x00,0x03,0x81,0x8B,0x00,0x30,0x81,0x87,
		0x02,0x81,0x81,0x00,0x87,0x64,0x72,0x89,0x0B,0x20,0x8F,0x87,0x27,0xAC,0xC6,0x22,0xFE,0x00,0x40,0x69,0x48,0xAF,0xC6,0x86,0xCD,0x23,0x33,0xE3,0x11,0xC5,0x31,0x1A,0x1F,0x7E,0x9E,0x92,0x13,0xB6,0xA2,0xAC,0xE3,0xB0,0x1F,0x2A,0x07,0x6C,0xB6,0xD4,0xDE,0x4B,0xFA,0xF1,0xA2,0xA0,0x7D,0xCE,0x4B,0xBE,0xBE,0x26,0x48,0x09,0x8C,0x85,0x11,0xDE,0xCB,0x22,0xE7,0xC2,0xEE,0x44,0x51,0xFE,0x67,0xD5,0x5B,0x5A,0xE0,0x16,0x37,0x54,0x04,0xB8,0x3B,0x32,0x12,0x94,0x83,0x9E,0xB1,0x4D,0x80,0x6C,0xA4,0xA9,0x76,0xAC,0xB8,0xA4,
		0x97,0xF7,0xAB,0x0B,0x6C,0xA5,0x43,0xBA,0x6E,0x4F,0xC5,0x4E,0x00,0x30,0x16,0x3C,0x3F,0x99,0x14,0xDA,0xA2,0x20,0x08,0x8B,0xBA,0xED,0x76,0xAC,0x97,0x00,0xD5,0x6D,0x02,0x01,0x0F,0x30,0x0D,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x01,0x04,0x05,0x00,0x03,0x81,0x81,0x00,0x38,0x50,0x1D,0x0A,0xD3,0x1B,0xBB,0xA2,0x9F,0x6C,0x8D,0x10,0xAA,0x42,0x1B,0x05,0x8D,0xE4,0x25,0xAB,0xFB,0x55,0xAE,0x6D,0xBA,0x53,0x67,0x15,0x07,0x9A,0xEC,0x55,0x9F,0x72,0x89,0x5F,0x24,0xB0,0xDB,0xCA,0x64,0xBD,0x64,0xAA,0xC2,0x8C,
		0xD9,0x3D,0xA2,0x45,0xB7,0xC6,0x92,0x71,0x51,0xEF,0xED,0xE1,0x51,0x54,0x97,0x56,0x35,0xA1,0xCE,0xE4,0x44,0xC4,0x47,0x66,0xFF,0x91,0xDA,0x88,0x9C,0x23,0xC2,0xB3,0xD4,0x62,0x4A,0xBC,0x94,0x55,0x9C,0x80,0x8E,0xB3,0xDD,0x4F,0x1A,0xED,0x12,0x5A,0xB5,0x2E,0xBC,0xF8,0x4B,0xCE,0xC6,0xD4,0x70,0xB3,0xB3,0x22,0xF8,0x5E,0x5C,0x36,0x7A,0xA6,0xB8,0x39,0x73,0x46,0x43,0x5C,0x9B,0x9A,0xBD,0x1E,0x7E,0xA7,0x04,0xCF,0x25,0x36 };

	X509Certificate x509 = new X509Certificate (cert);
	X509Certificate clone = new X509Certificate (x509);

	Assertion.Assert ("Equals", x509.Equals (clone));
	Assertion.Assert ("Equals", clone.Equals (x509));

	FileStream fs = File.OpenWrite ("temp.cer");
	fs.Write (cert, 0, cert.Length);
	fs.Close ();

	X509Certificate disk = X509Certificate.CreateFromCertFile ("temp.cer");
	Assertion.Assert ("Equals", disk.Equals (clone));
	Assertion.Assert ("Equals", disk.Equals (x509));
	byte[] hash = { 0x9E,0x87,0x80,0x3E,0xC5,0x68,0x9A,0xEF,0xE7,0x7F,0x92,0xF9,0x1A,0xBF,0xA7,0x46,0x7C,0x76,0xED,0x02 };
	AssertEquals ("GetCertHash", hash, x509.GetCertHash ());
	Assertion.AssertEquals ("GetCertHashString", "9E87803EC5689AEFE77F92F91ABFA7467C76ED02", x509.GetCertHashString ());
#if NET_2_0
	DateTime from = DateTime.ParseExact (x509.GetEffectiveDateString (), "MM/dd/yyyy HH:mm:ss", null).ToUniversalTime ();
	Assert.AreEqual ("01/18/1996 21:03:52", from.ToString (), "GetEffectiveDateString");
	DateTime until = DateTime.ParseExact (x509.GetExpirationDateString (), "MM/dd/yyyy HH:mm:ss", null).ToUniversalTime ();
	Assert.AreEqual ("01/16/2001 21:03:52", until.ToString (), "GetExpirationDateString");
#else
	// fx 1.x has a bug where the returned dates were always in the Seattle time zone
	Assertion.AssertEquals ("GetEffectiveDateString", "01/18/1996 13:03:52", x509.GetEffectiveDateString ());
	Assertion.AssertEquals ("GetExpirationDateString", "01/16/2001 13:03:52", x509.GetExpirationDateString ());
	// which was making it easier to test the dates ;-)
#endif
	Assertion.AssertEquals ("GetFormat", "X509", x509.GetFormat ());
	Assertion.AssertEquals ("GetHashCode", -1635286978, x509.GetHashCode ());
	Assertion.AssertEquals ("GetIssuerName", "C=US, O=AT&T, OU=Directory Services", x509.GetIssuerName ());
	Assertion.AssertEquals ("GetKeyAlgorithm", "1.2.840.113549.1.1.1", x509.GetKeyAlgorithm ());
	byte[] keyparams = { 0x05,0x00 };
	AssertEquals ("GetKeyAlgorithmParameters", keyparams, x509.GetKeyAlgorithmParameters ());
	Assertion.AssertEquals ("GetKeyAlgorithmParametersString", "0500", x509.GetKeyAlgorithmParametersString ());
	Assertion.AssertEquals ("GetName", "C=US, O=AT&T, OU=Directory Services", x509.GetName ());
	byte[] pubkey = { 0x30,0x81,0x87,0x02,0x81,0x81,0x00,0x87,0x64,0x72,0x89,0x0B,0x20,0x8F,0x87,0x27,0xAC,0xC6,0x22,0xFE,0x00,0x40,0x69,0x48,0xAF,0xC6,0x86,0xCD,0x23,0x33,0xE3,0x11,0xC5,0x31,0x1A,0x1F,0x7E,0x9E,0x92,0x13,0xB6,0xA2,0xAC,0xE3,0xB0,0x1F,0x2A,0x07,0x6C,0xB6,0xD4,0xDE,0x4B,0xFA,0xF1,0xA2,0xA0,0x7D,0xCE,0x4B,0xBE,0xBE,0x26,0x48,0x09,0x8C,0x85,0x11,0xDE,0xCB,0x22,0xE7,0xC2,0xEE,0x44,0x51,0xFE,0x67,0xD5,0x5B,0x5A,0xE0,0x16,0x37,0x54,0x04,0xB8,0x3B,0x32,0x12,0x94,0x83,0x9E,0xB1,0x4D,0x80,0x6C,0xA4,0xA9,0x76,0xAC,
		0xB8,0xA4,0x97,0xF7,0xAB,0x0B,0x6C,0xA5,0x43,0xBA,0x6E,0x4F,0xC5,0x4E,0x00,0x30,0x16,0x3C,0x3F,0x99,0x14,0xDA,0xA2,0x20,0x08,0x8B,0xBA,0xED,0x76,0xAC,0x97,0x00,0xD5,0x6D,0x02,0x01,0x0F };
	AssertEquals ("GetPublicKey", pubkey, x509.GetPublicKey ());
	Assertion.AssertEquals ("GetPublicKeyString", "30818702818100876472890B208F8727ACC622FE00406948AFC686CD2333E311C5311A1F7E9E9213B6A2ACE3B01F2A076CB6D4DE4BFAF1A2A07DCE4BBEBE2648098C8511DECB22E7C2EE4451FE67D55B5AE016375404B83B321294839EB14D806CA4A976ACB8A497F7AB0B6CA543BA6E4FC54E0030163C3F9914DAA220088BBAED76AC9700D56D02010F", x509.GetPublicKeyString ());
	AssertEquals ("GetRawCertData", cert, x509.GetRawCertData ());
	Assertion.AssertEquals ("GetRawCertDataString", ToString (cert), x509.GetRawCertDataString ());
	byte[] serial = { 0x00 };
	AssertEquals ("GetSerialNumber", serial, x509.GetSerialNumber ());
	Assertion.AssertEquals ("GetSerialNumberString", "00", x509.GetSerialNumberString ());
	byte[] ToStringArray = { 0x43,0x45,0x52,0x54,0x49,0x46,0x49,0x43,0x41,0x54,0x45,0x3A,0x0D,0x0A,0x09,0x46,0x6F,0x72,0x6D,0x61,0x74,0x3A,0x20,0x20,0x58,0x35,0x30,0x39,0x0D,0x0A,0x09,0x4E,0x61,0x6D,0x65,0x3A,0x20,0x20,0x43,0x3D,0x55,0x53,0x2C,0x20,0x4F,0x3D,0x41,0x54,0x26,0x54,0x2C,0x20,0x4F,0x55,0x3D,0x44,0x69,0x72,0x65,0x63,0x74,0x6F,0x72,0x79,0x20,0x53,0x65,0x72,0x76,0x69,0x63,0x65,0x73,0x0D,0x0A,0x09,0x49,0x73,0x73,0x75,0x69,0x6E,0x67,0x20,0x43,0x41,0x3A,0x20,0x20,0x43,0x3D,0x55,0x53,0x2C,0x20,0x4F,0x3D,0x41,0x54,0x26,0x54,
		0x2C,0x20,0x4F,0x55,0x3D,0x44,0x69,0x72,0x65,0x63,0x74,0x6F,0x72,0x79,0x20,0x53,0x65,0x72,0x76,0x69,0x63,0x65,0x73,0x0D,0x0A,0x09,0x4B,0x65,0x79,0x20,0x41,0x6C,0x67,0x6F,0x72,0x69,0x74,0x68,0x6D,0x3A,0x20,0x20,0x31,0x2E,0x32,0x2E,0x38,0x34,0x30,0x2E,0x31,0x31,0x33,0x35,0x34,0x39,0x2E,0x31,0x2E,0x31,0x2E,0x31,0x0D,0x0A,0x09,0x53,0x65,0x72,0x69,0x61,0x6C,0x20,0x4E,0x75,0x6D,0x62,0x65,0x72,0x3A,0x20,0x20,0x30,0x30,0x0D,0x0A,0x09,0x4B,0x65,0x79,0x20,0x41,0x6C,0x6F,0x67,0x72,0x69,0x74,0x68,0x6D,0x20,
		0x50,0x61,0x72,0x61,0x6D,0x65,0x74,0x65,0x72,0x73,0x3A,0x20,0x20,0x30,0x35,0x30,0x30,0x0D,0x0A,0x09,0x50,0x75,0x62,0x6C,0x69,0x63,0x20,0x4B,0x65,0x79,0x3A,0x20,0x20,0x33,0x30,0x38,0x31,0x38,0x37,0x30,0x32,0x38,0x31,0x38,0x31,0x30,0x30,0x38,0x37,0x36,0x34,0x37,0x32,0x38,0x39,0x30,0x42,0x32,0x30,0x38,0x46,0x38,0x37,0x32,0x37,0x41,0x43,0x43,0x36,0x32,0x32,0x46,0x45,0x30,0x30,0x34,0x30,0x36,0x39,0x34,0x38,0x41,0x46,0x43,0x36,0x38,0x36,0x43,0x44,0x32,0x33,0x33,0x33,0x45,0x33,0x31,0x31,0x43,0x35,0x33,
		0x31,0x31,0x41,0x31,0x46,0x37,0x45,0x39,0x45,0x39,0x32,0x31,0x33,0x42,0x36,0x41,0x32,0x41,0x43,0x45,0x33,0x42,0x30,0x31,0x46,0x32,0x41,0x30,0x37,0x36,0x43,0x42,0x36,0x44,0x34,0x44,0x45,0x34,0x42,0x46,0x41,0x46,0x31,0x41,0x32,0x41,0x30,0x37,0x44,0x43,0x45,0x34,0x42,0x42,0x45,0x42,0x45,0x32,0x36,0x34,0x38,0x30,0x39,0x38,0x43,0x38,0x35,0x31,0x31,0x44,0x45,0x43,0x42,0x32,0x32,0x45,0x37,0x43,0x32,0x45,0x45,0x34,0x34,0x35,0x31,0x46,0x45,0x36,0x37,0x44,0x35,0x35,0x42,0x35,0x41,0x45,0x30,0x31,0x36,0x33,
		0x37,0x35,0x34,0x30,0x34,0x42,0x38,0x33,0x42,0x33,0x32,0x31,0x32,0x39,0x34,0x38,0x33,0x39,0x45,0x42,0x31,0x34,0x44,0x38,0x30,0x36,0x43,0x41,0x34,0x41,0x39,0x37,0x36,0x41,0x43,0x42,0x38,0x41,0x34,0x39,0x37,0x46,0x37,0x41,0x42,0x30,0x42,0x36,0x43,0x41,0x35,0x34,0x33,0x42,0x41,0x36,0x45,0x34,0x46,0x43,0x35,0x34,0x45,0x30,0x30,0x33,0x30,0x31,0x36,0x33,0x43,0x33,0x46,0x39,0x39,0x31,0x34,0x44,0x41,0x41,0x32,0x32,0x30,0x30,0x38,0x38,0x42,0x42,0x41,0x45,0x44,0x37,0x36,0x41,0x43,0x39,0x37,0x30,0x30,0x44,
		0x35,0x36,0x44,0x30,0x32,0x30,0x31,0x30,0x46,0x0D,0x0A,0x0D,0x0A };
	string tsa = Encoding.Default.GetString (ToStringArray).Replace ("\r\n", Environment.NewLine);
	//Assertion.AssertEquals ("ToString(true)", tsa, x509.ToString (true));
}


// Certificate: basic\MYSET.cer
// - X.509v3, sha1RSA, RSA 512 bits
// - unreadable subject name (CN)
[Test]
public void Certificate3 ()
{
	// certificate can be loaded by constructor
	byte[] cert = { 0x30,0x82,0x03,0x4E,0x30,0x82,0x02,0xB7,0xA0,0x03,0x02,0x01,0x02,0x02,0x20,0x03,0x53,0xD7,0x8B,0xDB,0x3E,0x16,0x15,0x80,0x55,0xC4,0x05,0x40,0x02,0x73,0x4D,0x0C,0x20,0xF8,0x0D,0x88,0x00,0x5F,0x65,0x7A,0xAC,0xBA,0x86,0xBD,0x1C,0xD7,0xE4,0x30,0x0D,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x01,0x05,0x05,0x00,0x30,0x49,0x31,0x0B,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x55,0x53,0x31,0x18,0x30,0x16,0x06,0x03,0x55,0x04,0x0A,0x13,0x0F,0x43,0x43,0x41,0x20,0x2D,0x20,0x55,0x6E,0x69,0x71,0x75,0x65,0x20,
		0x49,0x44,0x31,0x20,0x30,0x1E,0x06,0x03,0x55,0x04,0x03,0x13,0x17,0x42,0x72,0x61,0x6E,0x64,0x20,0x4E,0x61,0x6D,0x65,0x3A,0x50,0x72,0x6F,0x64,0x75,0x63,0x74,0x20,0x54,0x79,0x70,0x65,0x30,0x1E,0x17,0x0D,0x39,0x36,0x30,0x38,0x30,0x37,0x30,0x30,0x30,0x30,0x30,0x30,0x5A,0x17,0x0D,0x39,0x36,0x30,0x38,0x33,0x31,0x32,0x33,0x35,0x39,0x35,0x39,0x5A,0x30,0x6E,0x31,0x0B,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x55,0x53,0x31,0x10,0x30,0x0E,0x06,0x03,0x55,0x04,0x0A,0x13,0x07,0x42,0x72,0x61,0x6E,0x64,0x49,
		0x44,0x31,0x26,0x30,0x24,0x06,0x03,0x55,0x04,0x0B,0x13,0x1D,0x49,0x73,0x73,0x75,0x69,0x6E,0x67,0x20,0x46,0x69,0x6E,0x61,0x6E,0x63,0x69,0x61,0x6C,0x20,0x49,0x6E,0x73,0x74,0x69,0x74,0x75,0x74,0x69,0x6F,0x6E,0x31,0x25,0x30,0x23,0x06,0x03,0x55,0x04,0x03,0x13,0x1C,0x30,0x2B,0x57,0x4B,0x4A,0x78,0x2B,0x77,0x59,0x45,0x5A,0x61,0x62,0x53,0x53,0x50,0x56,0x58,0x39,0x6B,0x4C,0x73,0x6E,0x78,0x39,0x32,0x73,0x3D,0x30,0x5C,0x30,0x0D,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x01,0x01,0x05,0x00,0x03,0x4B,0x00,
		0x30,0x48,0x02,0x41,0x00,0xAC,0xC4,0x0E,0x05,0x25,0xBC,0xEA,0xEF,0x0C,0x22,0x7F,0xC4,0x0C,0x4A,0x69,0x31,0x00,0xF9,0x3F,0xE9,0xE1,0x6C,0x54,0x97,0x77,0x4E,0x18,0xC6,0x4A,0x95,0xE0,0xD4,0x58,0x29,0x5C,0x17,0x5D,0x1D,0x1E,0x56,0xBC,0x49,0x3D,0xE0,0xF9,0x9F,0xBB,0x01,0xF9,0x86,0xB6,0xA6,0x95,0xDD,0xE1,0x04,0x32,0x01,0x52,0x4E,0x8F,0x86,0x30,0xF7,0x02,0x03,0x01,0x00,0x01,0xA3,0x82,0x01,0x44,0x30,0x82,0x01,0x40,0x30,0x5C,0x06,0x03,0x55,0x1D,0x23,0x04,0x55,0x30,0x53,0x81,0x4B,0x84,0x49,0x31,0x0B,0x30,
		0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x55,0x53,0x31,0x18,0x30,0x16,0x06,0x03,0x55,0x04,0x0A,0x13,0x0F,0x43,0x43,0x41,0x20,0x2D,0x20,0x55,0x6E,0x69,0x71,0x75,0x65,0x20,0x49,0x44,0x31,0x20,0x30,0x1E,0x06,0x03,0x55,0x04,0x03,0x13,0x17,0x42,0x72,0x61,0x6E,0x64,0x20,0x4E,0x61,0x6D,0x65,0x3A,0x50,0x72,0x6F,0x64,0x75,0x63,0x74,0x20,0x54,0x79,0x70,0x65,0x82,0x04,0x32,0x06,0xAC,0x10,0x30,0x0E,0x06,0x03,0x55,0x1D,0x0F,0x01,0x01,0xFF,0x04,0x04,0x03,0x02,0x07,0x80,0x30,0x2E,0x06,0x03,0x55,0x1D,0x10,0x01,
		0x01,0xFF,0x04,0x24,0x30,0x22,0x80,0x0F,0x31,0x39,0x39,0x36,0x30,0x38,0x30,0x37,0x30,0x37,0x34,0x39,0x30,0x30,0x5A,0x81,0x0F,0x31,0x39,0x39,0x36,0x30,0x39,0x30,0x37,0x30,0x37,0x34,0x39,0x30,0x30,0x5A,0x30,0x18,0x06,0x03,0x55,0x1D,0x20,0x04,0x11,0x30,0x0F,0x30,0x0D,0x06,0x0B,0x60,0x86,0x48,0x01,0x86,0xF8,0x45,0x01,0x07,0x01,0x01,0x30,0x0C,0x06,0x03,0x55,0x1D,0x13,0x01,0x01,0xFF,0x04,0x02,0x30,0x00,0x30,0x14,0x06,0x09,0x60,0x86,0x48,0x01,0x86,0xF8,0x45,0x02,0x03,0x01,0x01,0xFF,0x04,0x04,0x03,0x02,
		0x07,0x80,0x30,0x62,0x06,0x09,0x60,0x86,0x48,0x01,0x86,0xF8,0x45,0x02,0x07,0x01,0x01,0xFF,0x04,0x52,0x30,0x50,0x04,0x14,0x33,0x39,0x38,0x32,0x33,0x39,0x38,0x37,0x32,0x33,0x37,0x38,0x39,0x31,0x33,0x34,0x39,0x37,0x38,0x32,0x30,0x09,0x06,0x05,0x2B,0x0D,0x03,0x02,0x1A,0x05,0x00,0x16,0x0F,0x74,0x65,0x72,0x73,0x65,0x20,0x73,0x74,0x61,0x74,0x65,0x6D,0x65,0x6E,0x74,0x1D,0x00,0x16,0x1A,0x67,0x65,0x74,0x73,0x65,0x74,0x2D,0x63,0x65,0x6E,0x74,0x65,0x72,0x40,0x76,0x65,0x72,0x69,0x73,0x69,0x67,0x6E,0x2E,0x63,
		0x6F,0x6D,0x30,0x0D,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x01,0x05,0x05,0x00,0x03,0x81,0x81,0x00,0x1D,0x6C,0x90,0xF2,0x02,0x10,0xD5,0xA0,0x7B,0xEB,0x07,0x14,0x1D,0xAE,0x2C,0xD5,0xB1,0x4C,0x23,0x30,0xB2,0x31,0x7F,0x96,0x3C,0xD1,0x41,0x11,0xEA,0x08,0x0D,0x80,0x5F,0x46,0x66,0x9D,0x0A,0xF2,0x91,0xE8,0x7C,0xCE,0xC0,0xAD,0xE6,0x96,0x19,0x9C,0x26,0xC3,0x4E,0xEB,0x6F,0xF4,0x4A,0x69,0x4C,0xFE,0x4C,0x76,0xB3,0x51,0xCA,0x44,0x03,0xFC,0xD4,0xF4,0xF9,0x32,0x2A,0xAE,0x03,0x1B,0x5F,0xA6,0xBF,0x16,0x61,
		0xA0,0x07,0x86,0x85,0xA7,0xD6,0x0D,0xEF,0x88,0x9B,0x2A,0xBA,0xB8,0xD4,0x5C,0x94,0x7C,0x63,0xE1,0xE0,0x69,0xEB,0x3D,0x2F,0xC1,0x71,0x56,0x47,0x55,0x4B,0xB8,0xFD,0xCD,0xD6,0xC7,0x6F,0xA7,0x47,0xE9,0x43,0x91,0x2E,0xCF,0x93,0xA4,0x05,0x18,0xC5,0x98 };

	X509Certificate x509 = new X509Certificate (cert);
	X509Certificate clone = new X509Certificate (x509);

	Assertion.Assert ("Equals", x509.Equals (clone));
	Assertion.Assert ("Equals", clone.Equals (x509));

	FileStream fs = File.OpenWrite ("temp.cer");
	fs.Write (cert, 0, cert.Length);
	fs.Close ();

	X509Certificate disk = X509Certificate.CreateFromCertFile ("temp.cer");
	Assertion.Assert ("Equals", disk.Equals (clone));
	Assertion.Assert ("Equals", disk.Equals (x509));
	byte[] hash = { 0x39,0x5F,0xBB,0xFC,0x14,0x89,0x33,0x27,0x76,0x89,0xB7,0x59,0x3A,0x7D,0x1C,0xFF,0xF1,0x56,0xF1,0x62 };
	AssertEquals ("GetCertHash", hash, x509.GetCertHash ());
	Assertion.AssertEquals ("GetCertHashString", "395FBBFC148933277689B7593A7D1CFFF156F162", x509.GetCertHashString ());
#if NET_2_0
	DateTime from = DateTime.ParseExact (x509.GetEffectiveDateString (), "MM/dd/yyyy HH:mm:ss", null).ToUniversalTime ();
	Assert.AreEqual ("08/07/1996 00:00:00", from.ToString (), "GetEffectiveDateString");
	DateTime until = DateTime.ParseExact (x509.GetExpirationDateString (), "MM/dd/yyyy HH:mm:ss", null).ToUniversalTime ();
	Assert.AreEqual ("08/31/1996 23:59:59", until.ToString (), "GetExpirationDateString");
	Assert.AreEqual ("0353D78BDB3E16158055C4054002734D0C20F80D88005F657AACBA86BD1CD7E4", x509.GetSerialNumberString (), "GetSerialNumberString");
#else
	// fx 1.x has a bug where the returned dates were always in the Seattle time zone
	Assertion.AssertEquals ("GetEffectiveDateString", "08/06/1996 16:00:00", x509.GetEffectiveDateString ());
	Assertion.AssertEquals ("GetExpirationDateString", "08/31/1996 15:59:59", x509.GetExpirationDateString ());
	// which was making it easier to test the dates ;-)
	Assert.AreEqual ("E4D71CBD86BAAC7A655F00880DF8200C4D73024005C4558015163EDB8BD75303", x509.GetSerialNumberString (), "GetSerialNumberString");
#endif
	Assertion.AssertEquals ("GetFormat", "X509", x509.GetFormat ());
	Assertion.AssertEquals ("GetHashCode", 962575356, x509.GetHashCode ());
	Assertion.AssertEquals ("GetIssuerName", "C=US, O=CCA - Unique ID, CN=Brand Name:Product Type", x509.GetIssuerName ());
	Assertion.AssertEquals ("GetKeyAlgorithm", "1.2.840.113549.1.1.1", x509.GetKeyAlgorithm ());
	byte[] keyparams = { 0x05,0x00 };
	AssertEquals ("GetKeyAlgorithmParameters", keyparams, x509.GetKeyAlgorithmParameters ());
	Assertion.AssertEquals ("GetKeyAlgorithmParametersString", "0500", x509.GetKeyAlgorithmParametersString ());
	Assertion.AssertEquals ("GetName", "C=US, O=BrandID, OU=Issuing Financial Institution, CN=\"0+WKJx+wYEZabSSPVX9kLsnx92s=\"", x509.GetName ());
	byte[] pubkey = { 0x30,0x48,0x02,0x41,0x00,0xAC,0xC4,0x0E,0x05,0x25,0xBC,0xEA,0xEF,0x0C,0x22,0x7F,0xC4,0x0C,0x4A,0x69,0x31,0x00,0xF9,0x3F,0xE9,0xE1,0x6C,0x54,0x97,0x77,0x4E,0x18,0xC6,0x4A,0x95,0xE0,0xD4,0x58,0x29,0x5C,0x17,0x5D,0x1D,0x1E,0x56,0xBC,0x49,0x3D,0xE0,0xF9,0x9F,0xBB,0x01,0xF9,0x86,0xB6,0xA6,0x95,0xDD,0xE1,0x04,0x32,0x01,0x52,0x4E,0x8F,0x86,0x30,0xF7,0x02,0x03,0x01,0x00,0x01 };
	AssertEquals ("GetPublicKey", pubkey, x509.GetPublicKey ());
	Assertion.AssertEquals ("GetPublicKeyString", "3048024100ACC40E0525BCEAEF0C227FC40C4A693100F93FE9E16C5497774E18C64A95E0D458295C175D1D1E56BC493DE0F99FBB01F986B6A695DDE1043201524E8F8630F70203010001", x509.GetPublicKeyString ());
	AssertEquals ("GetRawCertData", cert, x509.GetRawCertData ());
	Assertion.AssertEquals ("GetRawCertDataString", ToString (cert), x509.GetRawCertDataString ());
	byte[] serial = { 0xE4,0xD7,0x1C,0xBD,0x86,0xBA,0xAC,0x7A,0x65,0x5F,0x00,0x88,0x0D,0xF8,0x20,0x0C,0x4D,0x73,0x02,0x40,0x05,0xC4,0x55,0x80,0x15,0x16,0x3E,0xDB,0x8B,0xD7,0x53,0x03 };
	AssertEquals ("GetSerialNumber", serial, x509.GetSerialNumber ());
	byte[] ToStringArray = { 0x43,0x45,0x52,0x54,0x49,0x46,0x49,0x43,0x41,0x54,0x45,0x3A,0x0D,0x0A,0x09,0x46,0x6F,0x72,0x6D,0x61,0x74,0x3A,0x20,0x20,0x58,0x35,0x30,0x39,0x0D,0x0A,0x09,0x4E,0x61,0x6D,0x65,0x3A,0x20,0x20,0x43,0x3D,0x55,0x53,0x2C,0x20,0x4F,0x3D,0x42,0x72,0x61,0x6E,0x64,0x49,0x44,0x2C,0x20,0x4F,0x55,0x3D,0x49,0x73,0x73,0x75,0x69,0x6E,0x67,0x20,0x46,0x69,0x6E,0x61,0x6E,0x63,0x69,0x61,0x6C,0x20,0x49,0x6E,0x73,0x74,0x69,0x74,0x75,0x74,0x69,0x6F,0x6E,0x2C,0x20,0x43,0x4E,0x3D,0x22,0x30,0x2B,0x57,0x4B,0x4A,0x78,0x2B,0x77,
		0x59,0x45,0x5A,0x61,0x62,0x53,0x53,0x50,0x56,0x58,0x39,0x6B,0x4C,0x73,0x6E,0x78,0x39,0x32,0x73,0x3D,0x22,0x0D,0x0A,0x09,0x49,0x73,0x73,0x75,0x69,0x6E,0x67,0x20,0x43,0x41,0x3A,0x20,0x20,0x43,0x3D,0x55,0x53,0x2C,0x20,0x4F,0x3D,0x43,0x43,0x41,0x20,0x2D,0x20,0x55,0x6E,0x69,0x71,0x75,0x65,0x20,0x49,0x44,0x2C,0x20,0x43,0x4E,0x3D,0x42,0x72,0x61,0x6E,0x64,0x20,0x4E,0x61,0x6D,0x65,0x3A,0x50,0x72,0x6F,0x64,0x75,0x63,0x74,0x20,0x54,0x79,0x70,0x65,0x0D,0x0A,0x09,0x4B,0x65,0x79,0x20,0x41,0x6C,0x67,0x6F,0x72,
		0x69,0x74,0x68,0x6D,0x3A,0x20,0x20,0x31,0x2E,0x32,0x2E,0x38,0x34,0x30,0x2E,0x31,0x31,0x33,0x35,0x34,0x39,0x2E,0x31,0x2E,0x31,0x2E,0x31,0x0D,0x0A,0x09,0x53,0x65,0x72,0x69,0x61,0x6C,0x20,0x4E,0x75,0x6D,0x62,0x65,0x72,0x3A,0x20,0x20,0x45,0x34,0x44,0x37,0x31,0x43,0x42,0x44,0x38,0x36,0x42,0x41,0x41,0x43,0x37,0x41,0x36,0x35,0x35,0x46,0x30,0x30,0x38,0x38,0x30,0x44,0x46,0x38,0x32,0x30,0x30,0x43,0x34,0x44,0x37,0x33,0x30,0x32,0x34,0x30,0x30,0x35,0x43,0x34,0x35,0x35,0x38,0x30,0x31,0x35,0x31,0x36,0x33,0x45,
		0x44,0x42,0x38,0x42,0x44,0x37,0x35,0x33,0x30,0x33,0x0D,0x0A,0x09,0x4B,0x65,0x79,0x20,0x41,0x6C,0x6F,0x67,0x72,0x69,0x74,0x68,0x6D,0x20,0x50,0x61,0x72,0x61,0x6D,0x65,0x74,0x65,0x72,0x73,0x3A,0x20,0x20,0x30,0x35,0x30,0x30,0x0D,0x0A,0x09,0x50,0x75,0x62,0x6C,0x69,0x63,0x20,0x4B,0x65,0x79,0x3A,0x20,0x20,0x33,0x30,0x34,0x38,0x30,0x32,0x34,0x31,0x30,0x30,0x41,0x43,0x43,0x34,0x30,0x45,0x30,0x35,0x32,0x35,0x42,0x43,0x45,0x41,0x45,0x46,0x30,0x43,0x32,0x32,0x37,0x46,0x43,0x34,0x30,0x43,0x34,0x41,0x36,0x39,
		0x33,0x31,0x30,0x30,0x46,0x39,0x33,0x46,0x45,0x39,0x45,0x31,0x36,0x43,0x35,0x34,0x39,0x37,0x37,0x37,0x34,0x45,0x31,0x38,0x43,0x36,0x34,0x41,0x39,0x35,0x45,0x30,0x44,0x34,0x35,0x38,0x32,0x39,0x35,0x43,0x31,0x37,0x35,0x44,0x31,0x44,0x31,0x45,0x35,0x36,0x42,0x43,0x34,0x39,0x33,0x44,0x45,0x30,0x46,0x39,0x39,0x46,0x42,0x42,0x30,0x31,0x46,0x39,0x38,0x36,0x42,0x36,0x41,0x36,0x39,0x35,0x44,0x44,0x45,0x31,0x30,0x34,0x33,0x32,0x30,0x31,0x35,0x32,0x34,0x45,0x38,0x46,0x38,0x36,0x33,0x30,0x46,0x37,0x30,0x32,
		0x30,0x33,0x30,0x31,0x30,0x30,0x30,0x31,0x0D,0x0A,0x0D,0x0A };
	string tsa = Encoding.Default.GetString (ToStringArray).Replace ("\r\n", Environment.NewLine);
	//Assertion.AssertEquals ("ToString(true)", tsa, x509.ToString (true));
}


// Certificate: basic\RSA.cer
// - X.509v1 (i.e. without any extensions) self-signed certificate
// - md2RSA (MD2 isn't part of the Cryptography namespace)
// - Uncommon key size (RSA 1000 bits)
[Test]
public void Certificate4 ()
{
	// certificate can be loaded by constructor
	byte[] cert = { 0x30,0x82,0x02,0x29,0x30,0x82,0x01,0x96,0x02,0x05,0x02,0x41,0x00,0x00,0x01,0x30,0x0D,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x01,0x02,0x05,0x00,0x30,0x5F,0x31,0x0B,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x55,0x53,0x31,0x20,0x30,0x1E,0x06,0x03,0x55,0x04,0x0A,0x13,0x17,0x52,0x53,0x41,0x20,0x44,0x61,0x74,0x61,0x20,0x53,0x65,0x63,0x75,0x72,0x69,0x74,0x79,0x2C,0x20,0x49,0x6E,0x63,0x2E,0x31,0x2E,0x30,0x2C,0x06,0x03,0x55,0x04,0x0B,0x13,0x25,0x53,0x65,0x63,0x75,0x72,0x65,0x20,0x53,0x65,0x72,0x76,
		0x65,0x72,0x20,0x43,0x65,0x72,0x74,0x69,0x66,0x69,0x63,0x61,0x74,0x69,0x6F,0x6E,0x20,0x41,0x75,0x74,0x68,0x6F,0x72,0x69,0x74,0x79,0x30,0x1E,0x17,0x0D,0x39,0x34,0x31,0x31,0x30,0x39,0x32,0x33,0x35,0x34,0x31,0x37,0x5A,0x17,0x0D,0x39,0x39,0x31,0x32,0x33,0x31,0x32,0x33,0x35,0x34,0x31,0x37,0x5A,0x30,0x5F,0x31,0x0B,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x55,0x53,0x31,0x20,0x30,0x1E,0x06,0x03,0x55,0x04,0x0A,0x13,0x17,0x52,0x53,0x41,0x20,0x44,0x61,0x74,0x61,0x20,0x53,0x65,0x63,0x75,0x72,0x69,0x74,
		0x79,0x2C,0x20,0x49,0x6E,0x63,0x2E,0x31,0x2E,0x30,0x2C,0x06,0x03,0x55,0x04,0x0B,0x13,0x25,0x53,0x65,0x63,0x75,0x72,0x65,0x20,0x53,0x65,0x72,0x76,0x65,0x72,0x20,0x43,0x65,0x72,0x74,0x69,0x66,0x69,0x63,0x61,0x74,0x69,0x6F,0x6E,0x20,0x41,0x75,0x74,0x68,0x6F,0x72,0x69,0x74,0x79,0x30,0x81,0x9B,0x30,0x0D,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x01,0x01,0x05,0x00,0x03,0x81,0x89,0x00,0x30,0x81,0x85,0x02,0x7E,0x00,0x92,0xCE,0x7A,0xC1,0xAE,0x83,0x3E,0x5A,0xAA,0x89,0x83,0x57,0xAC,0x25,0x01,0x76,0x0C,
		0xAD,0xAE,0x8E,0x2C,0x37,0xCE,0xEB,0x35,0x78,0x64,0x54,0x03,0xE5,0x84,0x40,0x51,0xC9,0xBF,0x8F,0x08,0xE2,0x8A,0x82,0x08,0xD2,0x16,0x86,0x37,0x55,0xE9,0xB1,0x21,0x02,0xAD,0x76,0x68,0x81,0x9A,0x05,0xA2,0x4B,0xC9,0x4B,0x25,0x66,0x22,0x56,0x6C,0x88,0x07,0x8F,0xF7,0x81,0x59,0x6D,0x84,0x07,0x65,0x70,0x13,0x71,0x76,0x3E,0x9B,0x77,0x4C,0xE3,0x50,0x89,0x56,0x98,0x48,0xB9,0x1D,0xA7,0x29,0x1A,0x13,0x2E,0x4A,0x11,0x59,0x9C,0x1E,0x15,0xD5,0x49,0x54,0x2C,0x73,0x3A,0x69,0x82,0xB1,0x97,0x39,0x9C,0x6D,0x70,0x67,
		0x48,0xE5,0xDD,0x2D,0xD6,0xC8,0x1E,0x7B,0x02,0x03,0x01,0x00,0x01,0x30,0x0D,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x01,0x02,0x05,0x00,0x03,0x7E,0x00,0x88,0xD1,0xD1,0x79,0x21,0xCE,0xE2,0x8B,0xE8,0xF8,0xC1,0x7D,0x34,0x53,0x3F,0x61,0x83,0xD9,0xB6,0x0B,0x38,0x17,0xB6,0xE8,0xBE,0x21,0x8D,0x8F,0x00,0xB8,0x8B,0x53,0x7E,0x44,0x67,0x1E,0x22,0xBD,0x97,0x27,0xE0,0x9C,0x85,0xCC,0x4A,0xF6,0x85,0x3B,0xB2,0xE2,0xBE,0x92,0xD3,0xE5,0x0D,0xE9,0xAF,0x5C,0x0E,0x0C,0x46,0x95,0xFF,0xA1,0x1C,0x5E,0x3E,0xE8,0x36,
		0x58,0x7A,0x73,0xA6,0x0A,0xF8,0x22,0x11,0x6B,0xC3,0x09,0x38,0x7E,0x26,0xBB,0x73,0xEF,0x00,0xBD,0x02,0xA4,0xF3,0x14,0x0D,0x30,0x3F,0x61,0x70,0x7B,0x20,0xFE,0x32,0xA3,0x9F,0xB3,0xF4,0x67,0x52,0xDC,0xB4,0xEE,0x84,0x8C,0x96,0x36,0x20,0xDE,0x81,0x08,0x83,0x71,0x21,0x8A,0x0F,0x9E,0xA9 };

	X509Certificate x509 = new X509Certificate (cert);
	X509Certificate clone = new X509Certificate (x509);

	Assertion.Assert ("Equals", x509.Equals (clone));
	Assertion.Assert ("Equals", clone.Equals (x509));

	FileStream fs = File.OpenWrite ("temp.cer");
	fs.Write (cert, 0, cert.Length);
	fs.Close ();

	X509Certificate disk = X509Certificate.CreateFromCertFile ("temp.cer");
	Assertion.Assert ("Equals", disk.Equals (clone));
	Assertion.Assert ("Equals", disk.Equals (x509));
	byte[] hash = { 0x0D,0x97,0x44,0x61,0x70,0x37,0x13,0xCB,0x74,0x93,0x2D,0x2A,0x75,0xAC,0xBC,0x71,0x4B,0x28,0x12,0x66 };
	AssertEquals ("GetCertHash", hash, x509.GetCertHash ());
	Assertion.AssertEquals ("GetCertHashString", "0D974461703713CB74932D2A75ACBC714B281266", x509.GetCertHashString ());
#if NET_2_0
	DateTime from = DateTime.ParseExact (x509.GetEffectiveDateString (), "MM/dd/yyyy HH:mm:ss", null).ToUniversalTime ();
	Assert.AreEqual ("11/09/1994 23:54:17", from.ToString (), "GetEffectiveDateString");
	DateTime until = DateTime.ParseExact (x509.GetExpirationDateString (), "MM/dd/yyyy HH:mm:ss", null).ToUniversalTime ();
	Assert.AreEqual ("12/31/1999 23:54:17", until.ToString (), "GetExpirationDateString");
	Assert.AreEqual ("0241000001", x509.GetSerialNumberString (), "GetSerialNumberString");
#else
	// fx 1.x has a bug where the returned dates were always in the Seattle time zone
	Assertion.AssertEquals ("GetEffectiveDateString", "11/09/1994 15:54:17", x509.GetEffectiveDateString ());
	Assertion.AssertEquals ("GetExpirationDateString", "12/31/1999 15:54:17", x509.GetExpirationDateString ());
	// which was making it easier to test the dates ;-)
	Assert.AreEqual ("0100004102", x509.GetSerialNumberString (), "GetSerialNumberString");
#endif
	Assertion.AssertEquals ("GetFormat", "X509", x509.GetFormat ());
	Assertion.AssertEquals ("GetHashCode", 228017249, x509.GetHashCode ());
	Assertion.AssertEquals ("GetIssuerName", "C=US, O=\"RSA Data Security, Inc.\", OU=Secure Server Certification Authority", x509.GetIssuerName ());
	Assertion.AssertEquals ("GetKeyAlgorithm", "1.2.840.113549.1.1.1", x509.GetKeyAlgorithm ());
	byte[] keyparams = { 0x05,0x00 };
	AssertEquals ("GetKeyAlgorithmParameters", keyparams, x509.GetKeyAlgorithmParameters ());
	Assertion.AssertEquals ("GetKeyAlgorithmParametersString", "0500", x509.GetKeyAlgorithmParametersString ());
	Assertion.AssertEquals ("GetName", "C=US, O=\"RSA Data Security, Inc.\", OU=Secure Server Certification Authority", x509.GetName ());
	byte[] pubkey = { 0x30,0x81,0x85,0x02,0x7E,0x00,0x92,0xCE,0x7A,0xC1,0xAE,0x83,0x3E,0x5A,0xAA,0x89,0x83,0x57,0xAC,0x25,0x01,0x76,0x0C,0xAD,0xAE,0x8E,0x2C,0x37,0xCE,0xEB,0x35,0x78,0x64,0x54,0x03,0xE5,0x84,0x40,0x51,0xC9,0xBF,0x8F,0x08,0xE2,0x8A,0x82,0x08,0xD2,0x16,0x86,0x37,0x55,0xE9,0xB1,0x21,0x02,0xAD,0x76,0x68,0x81,0x9A,0x05,0xA2,0x4B,0xC9,0x4B,0x25,0x66,0x22,0x56,0x6C,0x88,0x07,0x8F,0xF7,0x81,0x59,0x6D,0x84,0x07,0x65,0x70,0x13,0x71,0x76,0x3E,0x9B,0x77,0x4C,0xE3,0x50,0x89,0x56,0x98,0x48,0xB9,0x1D,0xA7,0x29,0x1A,0x13,
		0x2E,0x4A,0x11,0x59,0x9C,0x1E,0x15,0xD5,0x49,0x54,0x2C,0x73,0x3A,0x69,0x82,0xB1,0x97,0x39,0x9C,0x6D,0x70,0x67,0x48,0xE5,0xDD,0x2D,0xD6,0xC8,0x1E,0x7B,0x02,0x03,0x01,0x00,0x01 };
	AssertEquals ("GetPublicKey", pubkey, x509.GetPublicKey ());
	Assertion.AssertEquals ("GetPublicKeyString", "308185027E0092CE7AC1AE833E5AAA898357AC2501760CADAE8E2C37CEEB3578645403E5844051C9BF8F08E28A8208D216863755E9B12102AD7668819A05A24BC94B256622566C88078FF781596D840765701371763E9B774CE35089569848B91DA7291A132E4A11599C1E15D549542C733A6982B197399C6D706748E5DD2DD6C81E7B0203010001", x509.GetPublicKeyString ());
	AssertEquals ("GetRawCertData", cert, x509.GetRawCertData ());
	Assertion.AssertEquals ("GetRawCertDataString", ToString (cert), x509.GetRawCertDataString ());
	byte[] serial = { 0x01,0x00,0x00,0x41,0x02 };
	AssertEquals ("GetSerialNumber", serial, x509.GetSerialNumber ());
	byte[] ToStringArray = { 0x43,0x45,0x52,0x54,0x49,0x46,0x49,0x43,0x41,0x54,0x45,0x3A,0x0D,0x0A,0x09,0x46,0x6F,0x72,0x6D,0x61,0x74,0x3A,0x20,0x20,0x58,0x35,0x30,0x39,0x0D,0x0A,0x09,0x4E,0x61,0x6D,0x65,0x3A,0x20,0x20,0x43,0x3D,0x55,0x53,0x2C,0x20,0x4F,0x3D,0x22,0x52,0x53,0x41,0x20,0x44,0x61,0x74,0x61,0x20,0x53,0x65,0x63,0x75,0x72,0x69,0x74,0x79,0x2C,0x20,0x49,0x6E,0x63,0x2E,0x22,0x2C,0x20,0x4F,0x55,0x3D,0x53,0x65,0x63,0x75,0x72,0x65,0x20,0x53,0x65,0x72,0x76,0x65,0x72,0x20,0x43,0x65,0x72,0x74,0x69,0x66,0x69,0x63,0x61,0x74,0x69,
		0x6F,0x6E,0x20,0x41,0x75,0x74,0x68,0x6F,0x72,0x69,0x74,0x79,0x0D,0x0A,0x09,0x49,0x73,0x73,0x75,0x69,0x6E,0x67,0x20,0x43,0x41,0x3A,0x20,0x20,0x43,0x3D,0x55,0x53,0x2C,0x20,0x4F,0x3D,0x22,0x52,0x53,0x41,0x20,0x44,0x61,0x74,0x61,0x20,0x53,0x65,0x63,0x75,0x72,0x69,0x74,0x79,0x2C,0x20,0x49,0x6E,0x63,0x2E,0x22,0x2C,0x20,0x4F,0x55,0x3D,0x53,0x65,0x63,0x75,0x72,0x65,0x20,0x53,0x65,0x72,0x76,0x65,0x72,0x20,0x43,0x65,0x72,0x74,0x69,0x66,0x69,0x63,0x61,0x74,0x69,0x6F,0x6E,0x20,0x41,0x75,0x74,0x68,0x6F,0x72,
		0x69,0x74,0x79,0x0D,0x0A,0x09,0x4B,0x65,0x79,0x20,0x41,0x6C,0x67,0x6F,0x72,0x69,0x74,0x68,0x6D,0x3A,0x20,0x20,0x31,0x2E,0x32,0x2E,0x38,0x34,0x30,0x2E,0x31,0x31,0x33,0x35,0x34,0x39,0x2E,0x31,0x2E,0x31,0x2E,0x31,0x0D,0x0A,0x09,0x53,0x65,0x72,0x69,0x61,0x6C,0x20,0x4E,0x75,0x6D,0x62,0x65,0x72,0x3A,0x20,0x20,0x30,0x31,0x30,0x30,0x30,0x30,0x34,0x31,0x30,0x32,0x0D,0x0A,0x09,0x4B,0x65,0x79,0x20,0x41,0x6C,0x6F,0x67,0x72,0x69,0x74,0x68,0x6D,0x20,0x50,0x61,0x72,0x61,0x6D,0x65,0x74,0x65,0x72,0x73,0x3A,0x20,
		0x20,0x30,0x35,0x30,0x30,0x0D,0x0A,0x09,0x50,0x75,0x62,0x6C,0x69,0x63,0x20,0x4B,0x65,0x79,0x3A,0x20,0x20,0x33,0x30,0x38,0x31,0x38,0x35,0x30,0x32,0x37,0x45,0x30,0x30,0x39,0x32,0x43,0x45,0x37,0x41,0x43,0x31,0x41,0x45,0x38,0x33,0x33,0x45,0x35,0x41,0x41,0x41,0x38,0x39,0x38,0x33,0x35,0x37,0x41,0x43,0x32,0x35,0x30,0x31,0x37,0x36,0x30,0x43,0x41,0x44,0x41,0x45,0x38,0x45,0x32,0x43,0x33,0x37,0x43,0x45,0x45,0x42,0x33,0x35,0x37,0x38,0x36,0x34,0x35,0x34,0x30,0x33,0x45,0x35,0x38,0x34,0x34,0x30,0x35,0x31,0x43,
		0x39,0x42,0x46,0x38,0x46,0x30,0x38,0x45,0x32,0x38,0x41,0x38,0x32,0x30,0x38,0x44,0x32,0x31,0x36,0x38,0x36,0x33,0x37,0x35,0x35,0x45,0x39,0x42,0x31,0x32,0x31,0x30,0x32,0x41,0x44,0x37,0x36,0x36,0x38,0x38,0x31,0x39,0x41,0x30,0x35,0x41,0x32,0x34,0x42,0x43,0x39,0x34,0x42,0x32,0x35,0x36,0x36,0x32,0x32,0x35,0x36,0x36,0x43,0x38,0x38,0x30,0x37,0x38,0x46,0x46,0x37,0x38,0x31,0x35,0x39,0x36,0x44,0x38,0x34,0x30,0x37,0x36,0x35,0x37,0x30,0x31,0x33,0x37,0x31,0x37,0x36,0x33,0x45,0x39,0x42,0x37,0x37,0x34,0x43,0x45,
		0x33,0x35,0x30,0x38,0x39,0x35,0x36,0x39,0x38,0x34,0x38,0x42,0x39,0x31,0x44,0x41,0x37,0x32,0x39,0x31,0x41,0x31,0x33,0x32,0x45,0x34,0x41,0x31,0x31,0x35,0x39,0x39,0x43,0x31,0x45,0x31,0x35,0x44,0x35,0x34,0x39,0x35,0x34,0x32,0x43,0x37,0x33,0x33,0x41,0x36,0x39,0x38,0x32,0x42,0x31,0x39,0x37,0x33,0x39,0x39,0x43,0x36,0x44,0x37,0x30,0x36,0x37,0x34,0x38,0x45,0x35,0x44,0x44,0x32,0x44,0x44,0x36,0x43,0x38,0x31,0x45,0x37,0x42,0x30,0x32,0x30,0x33,0x30,0x31,0x30,0x30,0x30,0x31,0x0D,0x0A,0x0D,0x0A };
	string tsa = Encoding.Default.GetString (ToStringArray).Replace ("\r\n", Environment.NewLine);
	//Assertion.AssertEquals ("ToString(true)", tsa, x509.ToString (true));
}


// Certificate: basic\TL-CERT.cer
// - X.509v3 self-signed certificate
// - sha1RSA, RSA 2048 bits
[Test]
public void Certificate5 ()
{
	// certificate can be loaded by constructor
	byte[] cert = { 0x30,0x82,0x06,0x2F,0x30,0x82,0x05,0x17,0xA0,0x03,0x02,0x01,0x02,0x02,0x01,0x6A,0x30,0x0D,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x01,0x05,0x05,0x00,0x30,0x20,0x31,0x0B,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x55,0x53,0x31,0x11,0x30,0x0F,0x06,0x03,0x55,0x04,0x0A,0x13,0x08,0x53,0x45,0x54,0x20,0x52,0x6F,0x6F,0x74,0x30,0x1E,0x17,0x0D,0x39,0x37,0x30,0x37,0x31,0x35,0x30,0x30,0x30,0x30,0x30,0x30,0x5A,0x17,0x0D,0x30,0x34,0x30,0x37,0x31,0x35,0x30,0x30,0x30,0x30,0x30,0x30,0x5A,0x30,0x20,0x31,0x0B,
		0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x55,0x53,0x31,0x11,0x30,0x0F,0x06,0x03,0x55,0x04,0x0A,0x13,0x08,0x53,0x45,0x54,0x20,0x52,0x6F,0x6F,0x74,0x30,0x82,0x01,0x22,0x30,0x0D,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x01,0x01,0x05,0x00,0x03,0x82,0x01,0x0F,0x00,0x30,0x82,0x01,0x0A,0x02,0x82,0x01,0x01,0x00,0xD4,0xDC,0x3E,0xBA,0xE1,0x94,0xF7,0xBE,0xCD,0xED,0x21,0x77,0xCF,0xDA,0x88,0x58,0x51,0x0E,0x8F,0xF4,0xDA,0x00,0x14,0x1E,0x0D,0xA7,0xAD,0xB6,0x96,0x5A,0xC6,0xD3,0xEC,0x25,0xED,0xD8,0x43,
		0x0B,0x6E,0x7F,0x3F,0x9E,0x1E,0x74,0xA0,0x1E,0x97,0x76,0x30,0xCA,0x6F,0x0A,0x63,0xC0,0xA3,0x31,0x40,0x25,0x80,0xB8,0xBD,0x28,0xEB,0x7D,0x75,0x0B,0x4E,0x41,0x34,0xC4,0x20,0x00,0xC2,0xCB,0xF4,0x9A,0x20,0x00,0x58,0xD9,0xF4,0x40,0x13,0x18,0x77,0x0C,0xB5,0x04,0xDE,0xB7,0xB6,0x43,0x8B,0xA4,0xCC,0x36,0x76,0x79,0xC5,0x0B,0x17,0xCB,0x7E,0x88,0xA1,0x33,0xB0,0xD3,0x34,0x27,0xBF,0x3B,0x61,0xDA,0xC5,0x20,0xEB,0xF9,0x94,0x9A,0x8B,0x79,0xB2,0xA8,0x8E,0xCB,0xC1,0xD9,0x94,0x4A,0x99,0x66,0x50,0x55,0xB2,0x83,0x28,
		0x7D,0x22,0x3D,0xEC,0xDC,0xA3,0xE8,0x39,0xDB,0x83,0x54,0xC9,0x89,0xA9,0xDF,0x59,0x52,0x9F,0x7A,0xEF,0x7C,0x11,0x62,0x52,0xEC,0xE6,0x67,0xBA,0x3D,0xEA,0xAB,0x47,0xDB,0xE4,0xF4,0x1F,0x73,0xC3,0x3D,0xEC,0x7E,0x84,0x7D,0x2F,0x29,0xFE,0x6C,0x17,0x3F,0x75,0x6D,0x56,0x6E,0xC0,0x4E,0xB5,0xBF,0x2A,0x20,0x8A,0xE4,0x57,0xAE,0xC0,0x2E,0x68,0xC9,0x09,0xCF,0x85,0x77,0x0A,0xEF,0x3A,0x37,0xCB,0x60,0x4C,0x45,0x73,0x7F,0x90,0x3E,0x86,0x1D,0xFA,0xC3,0xFC,0x50,0x8A,0xB2,0xC5,0x8A,0x34,0xF0,0xF2,0x43,0xEE,0x3C,0x56,
		0xBA,0x24,0xE9,0xE0,0xA5,0x87,0x1E,0x7C,0x30,0x33,0x77,0xFD,0x5D,0xE0,0x57,0x0D,0x6C,0x19,0x39,0x02,0x03,0x01,0x00,0x01,0xA3,0x82,0x03,0x72,0x30,0x82,0x03,0x6E,0x30,0x12,0x06,0x03,0x55,0x1D,0x13,0x01,0x01,0xFF,0x04,0x08,0x30,0x06,0x01,0x01,0xFF,0x02,0x01,0x03,0x30,0x82,0x02,0xC9,0x06,0x03,0x55,0x1D,0x20,0x01,0x01,0xFF,0x04,0x82,0x02,0xBD,0x30,0x82,0x02,0xB9,0x30,0x82,0x02,0xB5,0x06,0x04,0x67,0x2A,0x05,0x00,0x30,0x82,0x02,0xAB,0x30,0x82,0x02,0xA7,0x06,0x04,0x67,0x2A,0x07,0x06,0x30,0x82,0x02,0x9D,
		0x30,0x82,0x02,0x99,0x1A,0x82,0x02,0x95,0x54,0x68,0x69,0x73,0x20,0x53,0x45,0x54,0x20,0x52,0x6F,0x6F,0x74,0x20,0x43,0x65,0x72,0x74,0x69,0x66,0x69,0x63,0x61,0x74,0x65,0x20,0x61,0x6E,0x64,0x20,0x61,0x6E,0x79,0x20,0x63,0x65,0x72,0x74,0x69,0x66,0x69,0x63,0x61,0x74,0x65,0x20,0x61,0x75,0x74,0x68,0x65,0x6E,0x74,0x69,0x63,0x61,0x74,0x65,0x64,0x20,0x64,0x69,0x72,0x65,0x63,0x74,0x6C,0x79,0x20,0x6F,0x72,0x20,0x69,0x6E,0x64,0x69,0x72,0x65,0x63,0x74,0x6C,0x79,0x20,0x62,0x79,0x20,0x74,0x68,0x69,0x73,0x20,0x63,
		0x65,0x72,0x74,0x69,0x66,0x69,0x63,0x61,0x74,0x65,0x2C,0x20,0x6D,0x61,0x79,0x20,0x6F,0x6E,0x6C,0x79,0x20,0x62,0x65,0x20,0x75,0x73,0x65,0x64,0x20,0x74,0x6F,0x20,0x65,0x6E,0x61,0x62,0x6C,0x65,0x20,0x22,0x53,0x65,0x63,0x75,0x72,0x65,0x20,0x46,0x69,0x6E,0x61,0x6E,0x63,0x69,0x61,0x6C,0x20,0x54,0x72,0x61,0x6E,0x73,0x61,0x63,0x74,0x69,0x6F,0x6E,0x73,0x22,0x20,0x61,0x73,0x20,0x64,0x65,0x66,0x69,0x6E,0x65,0x64,0x20,0x69,0x6E,0x20,0x74,0x68,0x65,0x20,0x53,0x45,0x54,0x20,0x52,0x6F,0x6F,0x74,0x20,0x43,0x65,
		0x72,0x74,0x69,0x66,0x69,0x63,0x61,0x74,0x65,0x20,0x50,0x72,0x61,0x63,0x74,0x69,0x63,0x65,0x20,0x53,0x74,0x61,0x74,0x65,0x6D,0x65,0x6E,0x74,0x20,0x61,0x6E,0x64,0x2C,0x20,0x77,0x68,0x65,0x6E,0x20,0x61,0x70,0x70,0x72,0x6F,0x70,0x72,0x69,0x61,0x74,0x65,0x2C,0x20,0x69,0x6E,0x20,0x61,0x20,0x53,0x45,0x54,0x20,0x42,0x72,0x61,0x6E,0x64,0x20,0x43,0x65,0x72,0x74,0x69,0x66,0x69,0x63,0x61,0x74,0x65,0x20,0x50,0x72,0x61,0x63,0x74,0x69,0x63,0x65,0x20,0x53,0x74,0x61,0x74,0x65,0x6D,0x65,0x6E,0x74,0x2E,0x20,0x20,
		0x4E,0x6F,0x20,0x50,0x61,0x72,0x74,0x79,0x20,0x6D,0x61,0x79,0x20,0x72,0x65,0x6C,0x79,0x20,0x75,0x70,0x6F,0x6E,0x20,0x74,0x68,0x65,0x20,0x53,0x45,0x54,0x20,0x52,0x6F,0x6F,0x74,0x20,0x43,0x65,0x72,0x74,0x69,0x66,0x69,0x63,0x61,0x74,0x65,0x20,0x66,0x6F,0x72,0x20,0x61,0x6E,0x79,0x20,0x6F,0x74,0x68,0x65,0x72,0x20,0x70,0x75,0x72,0x70,0x6F,0x73,0x65,0x2E,0x20,0x20,0x41,0x20,0x53,0x45,0x54,0x20,0x42,0x72,0x61,0x6E,0x64,0x20,0x73,0x68,0x61,0x6C,0x6C,0x20,0x62,0x65,0x20,0x61,0x6E,0x79,0x20,0x70,0x61,0x79,
		0x6D,0x65,0x6E,0x74,0x20,0x62,0x72,0x61,0x6E,0x64,0x20,0x77,0x68,0x6F,0x73,0x65,0x20,0x53,0x45,0x54,0x20,0x63,0x65,0x72,0x74,0x69,0x66,0x69,0x63,0x61,0x74,0x65,0x20,0x69,0x73,0x20,0x73,0x69,0x67,0x6E,0x65,0x64,0x20,0x62,0x79,0x20,0x74,0x68,0x65,0x20,0x70,0x72,0x69,0x76,0x61,0x74,0x65,0x20,0x6B,0x65,0x79,0x20,0x63,0x6F,0x72,0x72,0x65,0x73,0x70,0x6F,0x6E,0x64,0x69,0x6E,0x67,0x20,0x74,0x6F,0x20,0x74,0x68,0x65,0x20,0x70,0x75,0x62,0x6C,0x69,0x63,0x20,0x6B,0x65,0x79,0x20,0x63,0x6F,0x6E,0x74,0x61,0x69,
		0x6E,0x65,0x64,0x20,0x69,0x6E,0x20,0x74,0x68,0x69,0x73,0x20,0x63,0x65,0x72,0x74,0x69,0x66,0x69,0x63,0x61,0x74,0x65,0x2E,0x20,0x20,0x41,0x6C,0x6C,0x20,0x6D,0x61,0x74,0x74,0x65,0x72,0x73,0x20,0x72,0x65,0x6C,0x61,0x74,0x69,0x6E,0x67,0x20,0x74,0x6F,0x20,0x75,0x73,0x61,0x67,0x65,0x2C,0x20,0x6C,0x69,0x61,0x62,0x69,0x6C,0x69,0x74,0x79,0x20,0x61,0x6E,0x64,0x20,0x70,0x72,0x6F,0x63,0x65,0x64,0x75,0x72,0x65,0x73,0x20,0x77,0x69,0x74,0x68,0x20,0x53,0x45,0x54,0x20,0x63,0x65,0x72,0x74,0x69,0x66,0x69,0x63,0x61,
		0x74,0x65,0x73,0x20,0x69,0x73,0x73,0x75,0x65,0x64,0x20,0x62,0x65,0x6E,0x65,0x61,0x74,0x68,0x20,0x61,0x20,0x53,0x45,0x54,0x20,0x42,0x72,0x61,0x6E,0x64,0x20,0x73,0x68,0x61,0x6C,0x6C,0x20,0x62,0x65,0x20,0x64,0x65,0x74,0x65,0x72,0x6D,0x69,0x6E,0x65,0x64,0x20,0x62,0x79,0x20,0x74,0x68,0x61,0x74,0x20,0x53,0x45,0x54,0x20,0x42,0x72,0x61,0x6E,0x64,0x2E,0x30,0x0E,0x06,0x03,0x55,0x1D,0x0F,0x01,0x01,0xFF,0x04,0x04,0x03,0x02,0x01,0x06,0x30,0x2B,0x06,0x03,0x55,0x1D,0x10,0x04,0x24,0x30,0x22,0x80,0x0F,0x31,0x39,
		0x39,0x37,0x30,0x37,0x31,0x35,0x30,0x30,0x30,0x30,0x30,0x30,0x5A,0x81,0x0F,0x31,0x39,0x39,0x38,0x30,0x37,0x31,0x35,0x30,0x30,0x30,0x30,0x30,0x30,0x5A,0x30,0x10,0x06,0x04,0x67,0x2A,0x07,0x01,0x01,0x01,0xFF,0x04,0x05,0x03,0x03,0x07,0x00,0x80,0x30,0x3C,0x06,0x04,0x67,0x2A,0x07,0x00,0x01,0x01,0xFF,0x04,0x31,0x30,0x2F,0x30,0x2D,0x02,0x01,0x00,0x30,0x09,0x06,0x05,0x2B,0x0E,0x03,0x02,0x1A,0x05,0x00,0x30,0x07,0x06,0x05,0x67,0x2A,0x03,0x00,0x00,0x04,0x14,0xC8,0x57,0x44,0x4F,0xD7,0x91,0x56,0x3E,0xC6,0xF3,
		0xE0,0xE6,0x08,0x2E,0x9A,0xAF,0x61,0x11,0x43,0x5D,0x30,0x0D,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x01,0x05,0x05,0x00,0x03,0x82,0x01,0x01,0x00,0x91,0x6D,0x0D,0x97,0xB7,0x8D,0x44,0x23,0xB9,0x49,0xAD,0x23,0xA9,0x8B,0xED,0x93,0x33,0x97,0x4C,0xE1,0x6E,0xB1,0x34,0x96,0x18,0xF3,0x58,0xB3,0x9C,0xBF,0x63,0x0F,0x61,0x46,0xC7,0xD1,0x01,0x41,0x0C,0xC8,0x42,0x55,0x6B,0x54,0x71,0x06,0x3B,0xF7,0xD1,0x77,0x65,0xDF,0x16,0xE7,0x63,0x03,0x7B,0x23,0x26,0x28,0xEC,0x94,0xF8,0x9F,0x94,0x04,0x0F,0xE5,0x45,0x99,
		0x4E,0xB5,0x1B,0xBC,0xB9,0xC4,0xB0,0xE2,0x8A,0x3E,0x05,0xA6,0xE3,0x56,0x7D,0x01,0x77,0xAB,0xC2,0xA6,0x72,0x90,0x23,0xD3,0x15,0x8F,0x0F,0xEA,0x7B,0x31,0xDE,0x89,0x31,0xF0,0x1B,0x81,0x6B,0x5F,0xA8,0x13,0xC6,0x62,0x7D,0xFE,0x74,0x14,0x40,0x2A,0x14,0xC2,0xA1,0x1B,0x9C,0xB2,0xD6,0xEF,0x2A,0x6D,0xA5,0xF7,0xA6,0x38,0x8F,0xD4,0x94,0x74,0x30,0x10,0x9E,0xBA,0xA9,0xAB,0x6B,0x61,0x6B,0xFC,0xB2,0x3F,0x87,0x6B,0x19,0x82,0x83,0x70,0xE7,0xD8,0xEA,0x28,0x7B,0xB4,0x29,0x47,0xF4,0x59,0xB3,0x3E,0x4B,0x6A,0x9D,0x54,
		0x0D,0x4E,0x1C,0xD0,0x29,0xB4,0xD1,0xE1,0x19,0x79,0x41,0x73,0xF6,0x57,0x72,0xBE,0x75,0x03,0x94,0xD7,0x58,0xA8,0xC4,0x08,0x71,0xA2,0xE3,0x16,0x31,0xCD,0xC0,0xEE,0x1C,0x21,0x26,0x52,0x55,0x7B,0x00,0x54,0x6D,0xA6,0x44,0xC2,0x4F,0xEA,0x8F,0x04,0x1C,0x3A,0xA2,0xE3,0x5B,0xD7,0x9D,0xE2,0x57,0x30,0x2C,0xF5,0xAE,0x62,0x3B,0xB5,0x49,0x89,0xCB,0x01,0xD1,0x5A,0x38,0xDE,0x97,0x57,0x85,0x91,0x68,0x6B,0xFD,0xEC,0xD3,0x80,0xF0,0x82,0xBF,0x9A };

	X509Certificate x509 = new X509Certificate (cert);
	X509Certificate clone = new X509Certificate (x509);

	Assertion.Assert ("Equals", x509.Equals (clone));
	Assertion.Assert ("Equals", clone.Equals (x509));

	FileStream fs = File.OpenWrite ("temp.cer");
	fs.Write (cert, 0, cert.Length);
	fs.Close ();

	X509Certificate disk = X509Certificate.CreateFromCertFile ("temp.cer");
	Assertion.Assert ("Equals", disk.Equals (clone));
	Assertion.Assert ("Equals", disk.Equals (x509));
	byte[] hash = { 0x63,0xDB,0x7A,0x42,0x67,0x39,0x75,0x39,0x2A,0xC6,0xA4,0xFA,0xE3,0xD7,0x84,0x46,0xB4,0x67,0xC3,0x26 };
	AssertEquals ("GetCertHash", hash, x509.GetCertHash ());
	Assertion.AssertEquals ("GetCertHashString", "63DB7A42673975392AC6A4FAE3D78446B467C326", x509.GetCertHashString ());
#if NET_2_0
	DateTime from = DateTime.ParseExact (x509.GetEffectiveDateString (), "MM/dd/yyyy HH:mm:ss", null).ToUniversalTime ();
	Assert.AreEqual ("07/15/1997 00:00:00", from.ToString (), "GetEffectiveDateString");
	DateTime until = DateTime.ParseExact (x509.GetExpirationDateString (), "MM/dd/yyyy HH:mm:ss", null).ToUniversalTime ();
	Assert.AreEqual ("07/15/2004 00:00:00", until.ToString (), "GetExpirationDateString");
#else
	// fx 1.x has a bug where the returned dates were always in the Seattle time zone
	Assertion.AssertEquals ("GetEffectiveDateString", "07/14/1997 16:00:00", x509.GetEffectiveDateString ());
	Assertion.AssertEquals ("GetExpirationDateString", "07/14/2004 16:00:00", x509.GetExpirationDateString ());
	// which was making it easier to test the dates ;-)
#endif
	Assertion.AssertEquals ("GetFormat", "X509", x509.GetFormat ());
	Assertion.AssertEquals ("GetHashCode", 1675328066, x509.GetHashCode ());
	Assertion.AssertEquals ("GetIssuerName", "C=US, O=SET Root", x509.GetIssuerName ());
	Assertion.AssertEquals ("GetKeyAlgorithm", "1.2.840.113549.1.1.1", x509.GetKeyAlgorithm ());
	byte[] keyparams = { 0x05,0x00 };
	AssertEquals ("GetKeyAlgorithmParameters", keyparams, x509.GetKeyAlgorithmParameters ());
	Assertion.AssertEquals ("GetKeyAlgorithmParametersString", "0500", x509.GetKeyAlgorithmParametersString ());
	Assertion.AssertEquals ("GetName", "C=US, O=SET Root", x509.GetName ());
	byte[] pubkey = { 0x30,0x82,0x01,0x0A,0x02,0x82,0x01,0x01,0x00,0xD4,0xDC,0x3E,0xBA,0xE1,0x94,0xF7,0xBE,0xCD,0xED,0x21,0x77,0xCF,0xDA,0x88,0x58,0x51,0x0E,0x8F,0xF4,0xDA,0x00,0x14,0x1E,0x0D,0xA7,0xAD,0xB6,0x96,0x5A,0xC6,0xD3,0xEC,0x25,0xED,0xD8,0x43,0x0B,0x6E,0x7F,0x3F,0x9E,0x1E,0x74,0xA0,0x1E,0x97,0x76,0x30,0xCA,0x6F,0x0A,0x63,0xC0,0xA3,0x31,0x40,0x25,0x80,0xB8,0xBD,0x28,0xEB,0x7D,0x75,0x0B,0x4E,0x41,0x34,0xC4,0x20,0x00,0xC2,0xCB,0xF4,0x9A,0x20,0x00,0x58,0xD9,0xF4,0x40,0x13,0x18,0x77,0x0C,0xB5,0x04,0xDE,0xB7,0xB6,0x43,
		0x8B,0xA4,0xCC,0x36,0x76,0x79,0xC5,0x0B,0x17,0xCB,0x7E,0x88,0xA1,0x33,0xB0,0xD3,0x34,0x27,0xBF,0x3B,0x61,0xDA,0xC5,0x20,0xEB,0xF9,0x94,0x9A,0x8B,0x79,0xB2,0xA8,0x8E,0xCB,0xC1,0xD9,0x94,0x4A,0x99,0x66,0x50,0x55,0xB2,0x83,0x28,0x7D,0x22,0x3D,0xEC,0xDC,0xA3,0xE8,0x39,0xDB,0x83,0x54,0xC9,0x89,0xA9,0xDF,0x59,0x52,0x9F,0x7A,0xEF,0x7C,0x11,0x62,0x52,0xEC,0xE6,0x67,0xBA,0x3D,0xEA,0xAB,0x47,0xDB,0xE4,0xF4,0x1F,0x73,0xC3,0x3D,0xEC,0x7E,0x84,0x7D,0x2F,0x29,0xFE,0x6C,0x17,0x3F,0x75,0x6D,0x56,0x6E,0xC0,0x4E,
		0xB5,0xBF,0x2A,0x20,0x8A,0xE4,0x57,0xAE,0xC0,0x2E,0x68,0xC9,0x09,0xCF,0x85,0x77,0x0A,0xEF,0x3A,0x37,0xCB,0x60,0x4C,0x45,0x73,0x7F,0x90,0x3E,0x86,0x1D,0xFA,0xC3,0xFC,0x50,0x8A,0xB2,0xC5,0x8A,0x34,0xF0,0xF2,0x43,0xEE,0x3C,0x56,0xBA,0x24,0xE9,0xE0,0xA5,0x87,0x1E,0x7C,0x30,0x33,0x77,0xFD,0x5D,0xE0,0x57,0x0D,0x6C,0x19,0x39,0x02,0x03,0x01,0x00,0x01 };
	AssertEquals ("GetPublicKey", pubkey, x509.GetPublicKey ());
	Assertion.AssertEquals ("GetPublicKeyString", "3082010A0282010100D4DC3EBAE194F7BECDED2177CFDA8858510E8FF4DA00141E0DA7ADB6965AC6D3EC25EDD8430B6E7F3F9E1E74A01E977630CA6F0A63C0A331402580B8BD28EB7D750B4E4134C42000C2CBF49A200058D9F4401318770CB504DEB7B6438BA4CC367679C50B17CB7E88A133B0D33427BF3B61DAC520EBF9949A8B79B2A88ECBC1D9944A99665055B283287D223DECDCA3E839DB8354C989A9DF59529F7AEF7C116252ECE667BA3DEAAB47DBE4F41F73C33DEC7E847D2F29FE6C173F756D566EC04EB5BF2A208AE457AEC02E68C909CF85770AEF3A37CB604C45737F903E861DFAC3FC508AB2C58A34F0F243EE3C56BA24E9E0A5871E7C303377FD5DE0570D6C19390203010001", x509.GetPublicKeyString ());
	AssertEquals ("GetRawCertData", cert, x509.GetRawCertData ());
	Assertion.AssertEquals ("GetRawCertDataString", ToString (cert), x509.GetRawCertDataString ());
	byte[] serial = { 0x6A };
	AssertEquals ("GetSerialNumber", serial, x509.GetSerialNumber ());
	Assertion.AssertEquals ("GetSerialNumberString", "6A", x509.GetSerialNumberString ());
	byte[] ToStringArray = { 0x43,0x45,0x52,0x54,0x49,0x46,0x49,0x43,0x41,0x54,0x45,0x3A,0x0D,0x0A,0x09,0x46,0x6F,0x72,0x6D,0x61,0x74,0x3A,0x20,0x20,0x58,0x35,0x30,0x39,0x0D,0x0A,0x09,0x4E,0x61,0x6D,0x65,0x3A,0x20,0x20,0x43,0x3D,0x55,0x53,0x2C,0x20,0x4F,0x3D,0x53,0x45,0x54,0x20,0x52,0x6F,0x6F,0x74,0x0D,0x0A,0x09,0x49,0x73,0x73,0x75,0x69,0x6E,0x67,0x20,0x43,0x41,0x3A,0x20,0x20,0x43,0x3D,0x55,0x53,0x2C,0x20,0x4F,0x3D,0x53,0x45,0x54,0x20,0x52,0x6F,0x6F,0x74,0x0D,0x0A,0x09,0x4B,0x65,0x79,0x20,0x41,0x6C,0x67,0x6F,0x72,0x69,0x74,0x68,
		0x6D,0x3A,0x20,0x20,0x31,0x2E,0x32,0x2E,0x38,0x34,0x30,0x2E,0x31,0x31,0x33,0x35,0x34,0x39,0x2E,0x31,0x2E,0x31,0x2E,0x31,0x0D,0x0A,0x09,0x53,0x65,0x72,0x69,0x61,0x6C,0x20,0x4E,0x75,0x6D,0x62,0x65,0x72,0x3A,0x20,0x20,0x36,0x41,0x0D,0x0A,0x09,0x4B,0x65,0x79,0x20,0x41,0x6C,0x6F,0x67,0x72,0x69,0x74,0x68,0x6D,0x20,0x50,0x61,0x72,0x61,0x6D,0x65,0x74,0x65,0x72,0x73,0x3A,0x20,0x20,0x30,0x35,0x30,0x30,0x0D,0x0A,0x09,0x50,0x75,0x62,0x6C,0x69,0x63,0x20,0x4B,0x65,0x79,0x3A,0x20,0x20,0x33,0x30,0x38,0x32,0x30,
		0x31,0x30,0x41,0x30,0x32,0x38,0x32,0x30,0x31,0x30,0x31,0x30,0x30,0x44,0x34,0x44,0x43,0x33,0x45,0x42,0x41,0x45,0x31,0x39,0x34,0x46,0x37,0x42,0x45,0x43,0x44,0x45,0x44,0x32,0x31,0x37,0x37,0x43,0x46,0x44,0x41,0x38,0x38,0x35,0x38,0x35,0x31,0x30,0x45,0x38,0x46,0x46,0x34,0x44,0x41,0x30,0x30,0x31,0x34,0x31,0x45,0x30,0x44,0x41,0x37,0x41,0x44,0x42,0x36,0x39,0x36,0x35,0x41,0x43,0x36,0x44,0x33,0x45,0x43,0x32,0x35,0x45,0x44,0x44,0x38,0x34,0x33,0x30,0x42,0x36,0x45,0x37,0x46,0x33,0x46,0x39,0x45,0x31,0x45,0x37,
		0x34,0x41,0x30,0x31,0x45,0x39,0x37,0x37,0x36,0x33,0x30,0x43,0x41,0x36,0x46,0x30,0x41,0x36,0x33,0x43,0x30,0x41,0x33,0x33,0x31,0x34,0x30,0x32,0x35,0x38,0x30,0x42,0x38,0x42,0x44,0x32,0x38,0x45,0x42,0x37,0x44,0x37,0x35,0x30,0x42,0x34,0x45,0x34,0x31,0x33,0x34,0x43,0x34,0x32,0x30,0x30,0x30,0x43,0x32,0x43,0x42,0x46,0x34,0x39,0x41,0x32,0x30,0x30,0x30,0x35,0x38,0x44,0x39,0x46,0x34,0x34,0x30,0x31,0x33,0x31,0x38,0x37,0x37,0x30,0x43,0x42,0x35,0x30,0x34,0x44,0x45,0x42,0x37,0x42,0x36,0x34,0x33,0x38,0x42,0x41,
		0x34,0x43,0x43,0x33,0x36,0x37,0x36,0x37,0x39,0x43,0x35,0x30,0x42,0x31,0x37,0x43,0x42,0x37,0x45,0x38,0x38,0x41,0x31,0x33,0x33,0x42,0x30,0x44,0x33,0x33,0x34,0x32,0x37,0x42,0x46,0x33,0x42,0x36,0x31,0x44,0x41,0x43,0x35,0x32,0x30,0x45,0x42,0x46,0x39,0x39,0x34,0x39,0x41,0x38,0x42,0x37,0x39,0x42,0x32,0x41,0x38,0x38,0x45,0x43,0x42,0x43,0x31,0x44,0x39,0x39,0x34,0x34,0x41,0x39,0x39,0x36,0x36,0x35,0x30,0x35,0x35,0x42,0x32,0x38,0x33,0x32,0x38,0x37,0x44,0x32,0x32,0x33,0x44,0x45,0x43,0x44,0x43,0x41,0x33,0x45,
		0x38,0x33,0x39,0x44,0x42,0x38,0x33,0x35,0x34,0x43,0x39,0x38,0x39,0x41,0x39,0x44,0x46,0x35,0x39,0x35,0x32,0x39,0x46,0x37,0x41,0x45,0x46,0x37,0x43,0x31,0x31,0x36,0x32,0x35,0x32,0x45,0x43,0x45,0x36,0x36,0x37,0x42,0x41,0x33,0x44,0x45,0x41,0x41,0x42,0x34,0x37,0x44,0x42,0x45,0x34,0x46,0x34,0x31,0x46,0x37,0x33,0x43,0x33,0x33,0x44,0x45,0x43,0x37,0x45,0x38,0x34,0x37,0x44,0x32,0x46,0x32,0x39,0x46,0x45,0x36,0x43,0x31,0x37,0x33,0x46,0x37,0x35,0x36,0x44,0x35,0x36,0x36,0x45,0x43,0x30,0x34,0x45,0x42,0x35,0x42,
		0x46,0x32,0x41,0x32,0x30,0x38,0x41,0x45,0x34,0x35,0x37,0x41,0x45,0x43,0x30,0x32,0x45,0x36,0x38,0x43,0x39,0x30,0x39,0x43,0x46,0x38,0x35,0x37,0x37,0x30,0x41,0x45,0x46,0x33,0x41,0x33,0x37,0x43,0x42,0x36,0x30,0x34,0x43,0x34,0x35,0x37,0x33,0x37,0x46,0x39,0x30,0x33,0x45,0x38,0x36,0x31,0x44,0x46,0x41,0x43,0x33,0x46,0x43,0x35,0x30,0x38,0x41,0x42,0x32,0x43,0x35,0x38,0x41,0x33,0x34,0x46,0x30,0x46,0x32,0x34,0x33,0x45,0x45,0x33,0x43,0x35,0x36,0x42,0x41,0x32,0x34,0x45,0x39,0x45,0x30,0x41,0x35,0x38,0x37,0x31,
		0x45,0x37,0x43,0x33,0x30,0x33,0x33,0x37,0x37,0x46,0x44,0x35,0x44,0x45,0x30,0x35,0x37,0x30,0x44,0x36,0x43,0x31,0x39,0x33,0x39,0x30,0x32,0x30,0x33,0x30,0x31,0x30,0x30,0x30,0x31,0x0D,0x0A,0x0D,0x0A };
	string tsa = Encoding.Default.GetString (ToStringArray).Replace ("\r\n", Environment.NewLine);
	//Assertion.AssertEquals ("ToString(true)", tsa, x509.ToString (true));
}


// Certificate: basic\ATT-wrongsig.cer
// - same as ATT.cer
// - last byte has been modified (invalidating it's signature)
[Test]
public void Certificate6 ()
{
	// certificate can be loaded by constructor
	byte[] cert = { 0x30,0x82,0x01,0xDF,0x30,0x82,0x01,0x48,0x02,0x01,0x00,0x30,0x0D,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x01,0x04,0x05,0x00,0x30,0x39,0x31,0x0B,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x55,0x53,0x31,0x0D,0x30,0x0B,0x06,0x03,0x55,0x04,0x0A,0x14,0x04,0x41,0x54,0x26,0x54,0x31,0x1B,0x30,0x19,0x06,0x03,0x55,0x04,0x0B,0x14,0x12,0x44,0x69,0x72,0x65,0x63,0x74,0x6F,0x72,0x79,0x20,0x53,0x65,0x72,0x76,0x69,0x63,0x65,0x73,0x30,0x1E,0x17,0x0D,0x39,0x36,0x30,0x31,0x31,0x38,0x32,0x31,0x30,0x33,0x35,0x32,
		0x5A,0x17,0x0D,0x30,0x31,0x30,0x31,0x31,0x36,0x32,0x31,0x30,0x33,0x35,0x32,0x5A,0x30,0x39,0x31,0x0B,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x55,0x53,0x31,0x0D,0x30,0x0B,0x06,0x03,0x55,0x04,0x0A,0x14,0x04,0x41,0x54,0x26,0x54,0x31,0x1B,0x30,0x19,0x06,0x03,0x55,0x04,0x0B,0x14,0x12,0x44,0x69,0x72,0x65,0x63,0x74,0x6F,0x72,0x79,0x20,0x53,0x65,0x72,0x76,0x69,0x63,0x65,0x73,0x30,0x81,0x9D,0x30,0x0D,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x01,0x01,0x05,0x00,0x03,0x81,0x8B,0x00,0x30,0x81,0x87,
		0x02,0x81,0x81,0x00,0x87,0x64,0x72,0x89,0x0B,0x20,0x8F,0x87,0x27,0xAC,0xC6,0x22,0xFE,0x00,0x40,0x69,0x48,0xAF,0xC6,0x86,0xCD,0x23,0x33,0xE3,0x11,0xC5,0x31,0x1A,0x1F,0x7E,0x9E,0x92,0x13,0xB6,0xA2,0xAC,0xE3,0xB0,0x1F,0x2A,0x07,0x6C,0xB6,0xD4,0xDE,0x4B,0xFA,0xF1,0xA2,0xA0,0x7D,0xCE,0x4B,0xBE,0xBE,0x26,0x48,0x09,0x8C,0x85,0x11,0xDE,0xCB,0x22,0xE7,0xC2,0xEE,0x44,0x51,0xFE,0x67,0xD5,0x5B,0x5A,0xE0,0x16,0x37,0x54,0x04,0xB8,0x3B,0x32,0x12,0x94,0x83,0x9E,0xB1,0x4D,0x80,0x6C,0xA4,0xA9,0x76,0xAC,0xB8,0xA4,
		0x97,0xF7,0xAB,0x0B,0x6C,0xA5,0x43,0xBA,0x6E,0x4F,0xC5,0x4E,0x00,0x30,0x16,0x3C,0x3F,0x99,0x14,0xDA,0xA2,0x20,0x08,0x8B,0xBA,0xED,0x76,0xAC,0x97,0x00,0xD5,0x6D,0x02,0x01,0x0F,0x30,0x0D,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x01,0x04,0x05,0x00,0x03,0x81,0x81,0x00,0x38,0x50,0x1D,0x0A,0xD3,0x1B,0xBB,0xA2,0x9F,0x6C,0x8D,0x10,0xAA,0x42,0x1B,0x05,0x8D,0xE4,0x25,0xAB,0xFB,0x55,0xAE,0x6D,0xBA,0x53,0x67,0x15,0x07,0x9A,0xEC,0x55,0x9F,0x72,0x89,0x5F,0x24,0xB0,0xDB,0xCA,0x64,0xBD,0x64,0xAA,0xC2,0x8C,
		0xD9,0x3D,0xA2,0x45,0xB7,0xC6,0x92,0x71,0x51,0xEF,0xED,0xE1,0x51,0x54,0x97,0x56,0x35,0xA1,0xCE,0xE4,0x44,0xC4,0x47,0x66,0xFF,0x91,0xDA,0x88,0x9C,0x23,0xC2,0xB3,0xD4,0x62,0x4A,0xBC,0x94,0x55,0x9C,0x80,0x8E,0xB3,0xDD,0x4F,0x1A,0xED,0x12,0x5A,0xB5,0x2E,0xBC,0xF8,0x4B,0xCE,0xC6,0xD4,0x70,0xB3,0xB3,0x22,0xF8,0x5E,0x5C,0x36,0x7A,0xA6,0xB8,0x39,0x73,0x46,0x43,0x5C,0x9B,0x9A,0xBD,0x1E,0x7E,0xA7,0x04,0xCF,0x25,0x35 };

	X509Certificate x509 = new X509Certificate (cert);
	X509Certificate clone = new X509Certificate (x509);

	Assertion.Assert ("Equals", x509.Equals (clone));
	Assertion.Assert ("Equals", clone.Equals (x509));

	FileStream fs = File.OpenWrite ("temp.cer");
	fs.Write (cert, 0, cert.Length);
	fs.Close ();

	X509Certificate disk = X509Certificate.CreateFromCertFile ("temp.cer");
	Assertion.Assert ("Equals", disk.Equals (clone));
	Assertion.Assert ("Equals", disk.Equals (x509));
	byte[] hash = { 0x95,0xD4,0xE2,0xEA,0x34,0x0B,0xBF,0x33,0x27,0x1C,0x1F,0xBA,0x8B,0x52,0x9F,0x17,0x72,0x1F,0x8A,0x99 };
	AssertEquals ("GetCertHash", hash, x509.GetCertHash ());
	Assertion.AssertEquals ("GetCertHashString", "95D4E2EA340BBF33271C1FBA8B529F17721F8A99", x509.GetCertHashString ());
#if NET_2_0
	DateTime from = DateTime.ParseExact (x509.GetEffectiveDateString (), "MM/dd/yyyy HH:mm:ss", null).ToUniversalTime ();
	Assert.AreEqual ("01/18/1996 21:03:52", from.ToString (), "GetEffectiveDateString");
	DateTime until = DateTime.ParseExact (x509.GetExpirationDateString (), "MM/dd/yyyy HH:mm:ss", null).ToUniversalTime ();
	Assert.AreEqual ("01/16/2001 21:03:52", until.ToString (), "GetExpirationDateString");
#else
	// fx 1.x has a bug where the returned dates were always in the Seattle time zone
	Assertion.AssertEquals ("GetEffectiveDateString", "01/18/1996 13:03:52", x509.GetEffectiveDateString ());
	Assertion.AssertEquals ("GetExpirationDateString", "01/16/2001 13:03:52", x509.GetExpirationDateString ());
	// which was making it easier to test the dates ;-)
#endif
	Assertion.AssertEquals ("GetFormat", "X509", x509.GetFormat ());
	Assertion.AssertEquals ("GetHashCode", -1781210390, x509.GetHashCode ());
	Assertion.AssertEquals ("GetIssuerName", "C=US, O=AT&T, OU=Directory Services", x509.GetIssuerName ());
	Assertion.AssertEquals ("GetKeyAlgorithm", "1.2.840.113549.1.1.1", x509.GetKeyAlgorithm ());
	byte[] keyparams = { 0x05,0x00 };
	AssertEquals ("GetKeyAlgorithmParameters", keyparams, x509.GetKeyAlgorithmParameters ());
	Assertion.AssertEquals ("GetKeyAlgorithmParametersString", "0500", x509.GetKeyAlgorithmParametersString ());
	Assertion.AssertEquals ("GetName", "C=US, O=AT&T, OU=Directory Services", x509.GetName ());
	byte[] pubkey = { 0x30,0x81,0x87,0x02,0x81,0x81,0x00,0x87,0x64,0x72,0x89,0x0B,0x20,0x8F,0x87,0x27,0xAC,0xC6,0x22,0xFE,0x00,0x40,0x69,0x48,0xAF,0xC6,0x86,0xCD,0x23,0x33,0xE3,0x11,0xC5,0x31,0x1A,0x1F,0x7E,0x9E,0x92,0x13,0xB6,0xA2,0xAC,0xE3,0xB0,0x1F,0x2A,0x07,0x6C,0xB6,0xD4,0xDE,0x4B,0xFA,0xF1,0xA2,0xA0,0x7D,0xCE,0x4B,0xBE,0xBE,0x26,0x48,0x09,0x8C,0x85,0x11,0xDE,0xCB,0x22,0xE7,0xC2,0xEE,0x44,0x51,0xFE,0x67,0xD5,0x5B,0x5A,0xE0,0x16,0x37,0x54,0x04,0xB8,0x3B,0x32,0x12,0x94,0x83,0x9E,0xB1,0x4D,0x80,0x6C,0xA4,0xA9,0x76,0xAC,
		0xB8,0xA4,0x97,0xF7,0xAB,0x0B,0x6C,0xA5,0x43,0xBA,0x6E,0x4F,0xC5,0x4E,0x00,0x30,0x16,0x3C,0x3F,0x99,0x14,0xDA,0xA2,0x20,0x08,0x8B,0xBA,0xED,0x76,0xAC,0x97,0x00,0xD5,0x6D,0x02,0x01,0x0F };
	AssertEquals ("GetPublicKey", pubkey, x509.GetPublicKey ());
	Assertion.AssertEquals ("GetPublicKeyString", "30818702818100876472890B208F8727ACC622FE00406948AFC686CD2333E311C5311A1F7E9E9213B6A2ACE3B01F2A076CB6D4DE4BFAF1A2A07DCE4BBEBE2648098C8511DECB22E7C2EE4451FE67D55B5AE016375404B83B321294839EB14D806CA4A976ACB8A497F7AB0B6CA543BA6E4FC54E0030163C3F9914DAA220088BBAED76AC9700D56D02010F", x509.GetPublicKeyString ());
	AssertEquals ("GetRawCertData", cert, x509.GetRawCertData ());
	Assertion.AssertEquals ("GetRawCertDataString", ToString (cert), x509.GetRawCertDataString ());
	byte[] serial = { 0x00 };
	AssertEquals ("GetSerialNumber", serial, x509.GetSerialNumber ());
	Assertion.AssertEquals ("GetSerialNumberString", "00", x509.GetSerialNumberString ());
	byte[] ToStringArray = { 0x43,0x45,0x52,0x54,0x49,0x46,0x49,0x43,0x41,0x54,0x45,0x3A,0x0D,0x0A,0x09,0x46,0x6F,0x72,0x6D,0x61,0x74,0x3A,0x20,0x20,0x58,0x35,0x30,0x39,0x0D,0x0A,0x09,0x4E,0x61,0x6D,0x65,0x3A,0x20,0x20,0x43,0x3D,0x55,0x53,0x2C,0x20,0x4F,0x3D,0x41,0x54,0x26,0x54,0x2C,0x20,0x4F,0x55,0x3D,0x44,0x69,0x72,0x65,0x63,0x74,0x6F,0x72,0x79,0x20,0x53,0x65,0x72,0x76,0x69,0x63,0x65,0x73,0x0D,0x0A,0x09,0x49,0x73,0x73,0x75,0x69,0x6E,0x67,0x20,0x43,0x41,0x3A,0x20,0x20,0x43,0x3D,0x55,0x53,0x2C,0x20,0x4F,0x3D,0x41,0x54,0x26,0x54,
		0x2C,0x20,0x4F,0x55,0x3D,0x44,0x69,0x72,0x65,0x63,0x74,0x6F,0x72,0x79,0x20,0x53,0x65,0x72,0x76,0x69,0x63,0x65,0x73,0x0D,0x0A,0x09,0x4B,0x65,0x79,0x20,0x41,0x6C,0x67,0x6F,0x72,0x69,0x74,0x68,0x6D,0x3A,0x20,0x20,0x31,0x2E,0x32,0x2E,0x38,0x34,0x30,0x2E,0x31,0x31,0x33,0x35,0x34,0x39,0x2E,0x31,0x2E,0x31,0x2E,0x31,0x0D,0x0A,0x09,0x53,0x65,0x72,0x69,0x61,0x6C,0x20,0x4E,0x75,0x6D,0x62,0x65,0x72,0x3A,0x20,0x20,0x30,0x30,0x0D,0x0A,0x09,0x4B,0x65,0x79,0x20,0x41,0x6C,0x6F,0x67,0x72,0x69,0x74,0x68,0x6D,0x20,
		0x50,0x61,0x72,0x61,0x6D,0x65,0x74,0x65,0x72,0x73,0x3A,0x20,0x20,0x30,0x35,0x30,0x30,0x0D,0x0A,0x09,0x50,0x75,0x62,0x6C,0x69,0x63,0x20,0x4B,0x65,0x79,0x3A,0x20,0x20,0x33,0x30,0x38,0x31,0x38,0x37,0x30,0x32,0x38,0x31,0x38,0x31,0x30,0x30,0x38,0x37,0x36,0x34,0x37,0x32,0x38,0x39,0x30,0x42,0x32,0x30,0x38,0x46,0x38,0x37,0x32,0x37,0x41,0x43,0x43,0x36,0x32,0x32,0x46,0x45,0x30,0x30,0x34,0x30,0x36,0x39,0x34,0x38,0x41,0x46,0x43,0x36,0x38,0x36,0x43,0x44,0x32,0x33,0x33,0x33,0x45,0x33,0x31,0x31,0x43,0x35,0x33,
		0x31,0x31,0x41,0x31,0x46,0x37,0x45,0x39,0x45,0x39,0x32,0x31,0x33,0x42,0x36,0x41,0x32,0x41,0x43,0x45,0x33,0x42,0x30,0x31,0x46,0x32,0x41,0x30,0x37,0x36,0x43,0x42,0x36,0x44,0x34,0x44,0x45,0x34,0x42,0x46,0x41,0x46,0x31,0x41,0x32,0x41,0x30,0x37,0x44,0x43,0x45,0x34,0x42,0x42,0x45,0x42,0x45,0x32,0x36,0x34,0x38,0x30,0x39,0x38,0x43,0x38,0x35,0x31,0x31,0x44,0x45,0x43,0x42,0x32,0x32,0x45,0x37,0x43,0x32,0x45,0x45,0x34,0x34,0x35,0x31,0x46,0x45,0x36,0x37,0x44,0x35,0x35,0x42,0x35,0x41,0x45,0x30,0x31,0x36,0x33,
		0x37,0x35,0x34,0x30,0x34,0x42,0x38,0x33,0x42,0x33,0x32,0x31,0x32,0x39,0x34,0x38,0x33,0x39,0x45,0x42,0x31,0x34,0x44,0x38,0x30,0x36,0x43,0x41,0x34,0x41,0x39,0x37,0x36,0x41,0x43,0x42,0x38,0x41,0x34,0x39,0x37,0x46,0x37,0x41,0x42,0x30,0x42,0x36,0x43,0x41,0x35,0x34,0x33,0x42,0x41,0x36,0x45,0x34,0x46,0x43,0x35,0x34,0x45,0x30,0x30,0x33,0x30,0x31,0x36,0x33,0x43,0x33,0x46,0x39,0x39,0x31,0x34,0x44,0x41,0x41,0x32,0x32,0x30,0x30,0x38,0x38,0x42,0x42,0x41,0x45,0x44,0x37,0x36,0x41,0x43,0x39,0x37,0x30,0x30,0x44,
		0x35,0x36,0x44,0x30,0x32,0x30,0x31,0x30,0x46,0x0D,0x0A,0x0D,0x0A };
	string tsa = Encoding.Default.GetString (ToStringArray).Replace ("\r\n", Environment.NewLine);
	//Assertion.AssertEquals ("ToString(true)", tsa, x509.ToString (true));
}


// Certificate: basic\BADCERT.cer
// - Bad certificate (will throw an exception)
[Test]
public void Certificate7 ()
{
	// cannot be loaded - will throw an exception
	/*System.Security.Cryptography.CryptographicException: Input data cannot be coded as a valid certificate.
   at System.Security.Cryptography.X509Certificates.X509Certificate.SetX509Certificate(Byte[] data)
   at System.Security.Cryptography.X509Certificates.X509Certificate..ctor(Byte[] data)
   at System.Security.Cryptography.X509Certificates.X509Certificate.CreateFromCertFile(String filename)
   at vx509.Class1.CreateEntry(String filename) in d:\documents and settings\administrator\my documents\visual studio projects\solution1\vx509\x509nunitgenerator.cs:line 73*/
	try {
	byte[] cert = { 0x30,0x80,0x30,0x82,0x02,0x08,0xA0,0x03,0x02,0x01,0x02,0x02,0x01,0x02,0x30,0x0D,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x01,0x04,0x05,0x00,0x30,0x58,0x31,0x0B,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x45,0x53,0x31,0x12,0x30,0x10,0x06,0x03,0x55,0x04,0x09,0x13,0x09,0x43,0x61,0x74,0x61,0x6C,0x75,0x6E,0x79,0x61,0x31,0x12,0x30,0x10,0x06,0x03,0x55,0x04,0x07,0x13,0x09,0x42,0x61,0x72,0x63,0x65,0x6C,0x6F,0x6E,0x61,0x31,0x0C,0x30,0x0A,0x06,0x03,0x55,0x04,0x0A,0x13,0x03,0x55,0x50,0x43,0x31,0x13,0x30,
		0x11,0x06,0x03,0x55,0x04,0x0B,0x13,0x0A,0x65,0x73,0x43,0x45,0x52,0x54,0x2D,0x55,0x50,0x43,0x30,0x1E,0x17,0x0D,0x39,0x36,0x30,0x35,0x32,0x31,0x30,0x39,0x35,0x38,0x32,0x36,0x5A,0x17,0x0D,0x39,0x37,0x30,0x35,0x32,0x31,0x30,0x39,0x35,0x38,0x32,0x36,0x5A,0x30,0x70,0x31,0x0B,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x45,0x53,0x31,0x12,0x30,0x10,0x06,0x03,0x55,0x04,0x09,0x13,0x09,0x43,0x61,0x74,0x61,0x6C,0x75,0x6E,0x79,0x61,0x31,0x12,0x30,0x10,0x06,0x03,0x55,0x04,0x07,0x13,0x09,0x42,0x61,0x72,0x63,
		0x65,0x6C,0x6F,0x6E,0x61,0x31,0x0C,0x30,0x0A,0x06,0x03,0x55,0x04,0x0A,0x13,0x03,0x55,0x50,0x43,0x31,0x13,0x30,0x11,0x06,0x03,0x55,0x04,0x0B,0x13,0x0A,0x65,0x73,0x43,0x45,0x52,0x54,0x2D,0x55,0x50,0x43,0x31,0x16,0x30,0x14,0x06,0x03,0x55,0x04,0x03,0x13,0x0D,0x65,0x73,0x63,0x65,0x72,0x74,0x2E,0x75,0x70,0x63,0x2E,0x65,0x73,0x30,0x7C,0x30,0x0D,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x01,0x01,0x05,0x00,0x03,0x6B,0x00,0x30,0x68,0x02,0x61,0x00,0xBE,0xAA,0x8B,0x77,0x54,0xA3,0xAF,0xCA,0x77,0x9F,0x2F,
		0xB0,0xCF,0x43,0x88,0xFF,0xA6,0x6D,0x79,0x55,0x5B,0x61,0x8C,0x68,0xEC,0x48,0x1E,0x8A,0x86,0x38,0xA4,0xFE,0x19,0xB8,0x62,0x17,0x1D,0x9D,0x0F,0x47,0x2C,0xFF,0x63,0x8F,0x29,0x91,0x04,0xD1,0x52,0xBC,0x7F,0x67,0xB6,0xB2,0x8F,0x74,0x55,0xC1,0x33,0x21,0x6C,0x8F,0xAB,0x01,0x95,0x24,0xC8,0xB2,0x73,0x93,0x9D,0x22,0x61,0x50,0xA9,0x35,0xFB,0x9D,0x57,0x50,0x32,0xEF,0x56,0x52,0x50,0x93,0xAB,0xB1,0x88,0x94,0x78,0x56,0x15,0xC6,0x1C,0x8B,0x02,0x03,0x01,0x00,0x01,0xA3,0x81,0x84,0x30,0x81,0x81,0x30,0x0B,0x06,0x03,
		0x55,0x1D,0x0F,0x04,0x04,0x03,0x02,0x07,0x80,0x30,0x09,0x06,0x03,0x55,0x1D,0x13,0x04,0x02,0x30,0x00,0x30,0x3D,0x06,0x03,0x55,0x1D,0x11,0x04,0x36,0x30,0x34,0xA1,0x1A,0x16,0x18,0x65,0x73,0x63,0x65,0x72,0x74,0x2D,0x75,0x70,0x63,0x40,0x65,0x73,0x63,0x65,0x72,0x74,0x2E,0x75,0x70,0x63,0x2E,0x65,0x73,0xA6,0x16,0x16,0x14,0x68,0x74,0x74,0x70,0x3A,0x2F,0x2F,0x65,0x73,0x63,0x65,0x72,0x74,0x2E,0x75,0x70,0x63,0x2E,0x65,0x73,0x30,0x28,0x06,0x03,0x55,0x1D,0x20,0x04,0x21,0x30,0x1F,0x30,0x1D,0x06,0x04,0x2A,0x84,
		0x80,0x00,0x30,0x15,0x30,0x07,0x06,0x05,0x2A,0x84,0x80,0x00,0x01,0x30,0x0A,0x06,0x05,0x2A,0x84,0x80,0x00,0x02,0x02,0x01,0x0A,0x30,0x80,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x01,0x04,0x05,0x00,0x00,0x00,0x03,0x81,0x81,0x00,0x5B,0xFD,0xC2,0xA7,0x04,0xD4,0x83,0x4E,0x17,0x6D,0xA6,0xFA,0x27,0xE7,0xC6,0xF8,0xAB,0xB9,0x5D,0x9F,0xD0,0xA1,0xDF,0xD7,0x97,0x9F,0xE0,0x20,0xA6,0xC5,0x7A,0x64,0xCD,0x52,0x2F,0xE9,0xAE,0xDA,0xBE,0x9C,0xE4,0xD5,0x97,0xED,0xF1,0x84,0xC0,0xD0,0xFE,0x9B,0xEF,0x54,0xB1,0x80,
		0xE5,0xBF,0x3C,0xC9,0xED,0x93,0x20,0x2D,0x52,0x21,0xE9,0xBC,0xB9,0xE3,0x4F,0xAC,0x11,0x65,0x0E,0x8F,0xA1,0x68,0x99,0x63,0x47,0xE5,0x3D,0xE4,0x42,0x73,0x13,0xFA,0xC5,0xC8,0x34,0x8C,0xC0,0x41,0x18,0x89,0xD5,0xE6,0xA0,0x18,0x5B,0x5D,0x86,0x1C,0x1E,0xC6,0x70,0xD8,0x0E,0x89,0x64,0x94,0x83,0x8E,0x3B,0x40,0x7C,0x59,0xCF,0x2B,0x2F,0xB7,0xCE,0x97,0x98,0x12,0x15,0xEF,0x13,0xD4,0x00,0x00 };

		X509Certificate bad = new X509Certificate (cert);
		Assertion.Fail ("Expected CryptographicException but got none");
	}
	catch (CryptographicException) {
		// this is expected 
	}
	catch (Exception e) {
		Assertion.Fail ("Expected CryptographicException but got :" + e.ToString ());
	}
}


// Certificate: basic\ArmedForcesDSARoot.cer
// - X.509v3 self-signed certificate
// - DSA 1024 bits
[Test]
public void Certificate8 ()
{
	// certificate can be loaded by constructor
	byte[] cert = { 0x30,0x82,0x03,0x04,0x30,0x82,0x02,0xC4,0xA0,0x03,0x02,0x01,0x02,0x02,0x01,0x03,0x30,0x09,0x06,0x07,0x2A,0x86,0x48,0xCE,0x38,0x04,0x03,0x30,0x51,0x31,0x0B,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x55,0x53,0x31,0x18,0x30,0x16,0x06,0x03,0x55,0x04,0x0A,0x13,0x0F,0x55,0x2E,0x53,0x2E,0x20,0x47,0x6F,0x76,0x65,0x72,0x6E,0x6D,0x65,0x6E,0x74,0x31,0x0C,0x30,0x0A,0x06,0x03,0x55,0x04,0x0B,0x13,0x03,0x44,0x6F,0x44,0x31,0x1A,0x30,0x18,0x06,0x03,0x55,0x04,0x03,0x13,0x11,0x41,0x72,0x6D,0x65,0x64,0x20,0x46,0x6F,
		0x72,0x63,0x65,0x73,0x20,0x52,0x6F,0x6F,0x74,0x30,0x1E,0x17,0x0D,0x30,0x30,0x31,0x30,0x32,0x35,0x30,0x30,0x30,0x30,0x30,0x30,0x5A,0x17,0x0D,0x30,0x33,0x30,0x31,0x30,0x31,0x30,0x30,0x30,0x30,0x30,0x30,0x5A,0x30,0x51,0x31,0x0B,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x55,0x53,0x31,0x18,0x30,0x16,0x06,0x03,0x55,0x04,0x0A,0x13,0x0F,0x55,0x2E,0x53,0x2E,0x20,0x47,0x6F,0x76,0x65,0x72,0x6E,0x6D,0x65,0x6E,0x74,0x31,0x0C,0x30,0x0A,0x06,0x03,0x55,0x04,0x0B,0x13,0x03,0x44,0x6F,0x44,0x31,0x1A,0x30,0x18,
		0x06,0x03,0x55,0x04,0x03,0x13,0x11,0x41,0x72,0x6D,0x65,0x64,0x20,0x46,0x6F,0x72,0x63,0x65,0x73,0x20,0x52,0x6F,0x6F,0x74,0x30,0x82,0x01,0xB6,0x30,0x82,0x01,0x2B,0x06,0x07,0x2A,0x86,0x48,0xCE,0x38,0x04,0x01,0x30,0x82,0x01,0x1E,0x02,0x81,0x81,0x00,0x90,0x89,0x3E,0x18,0x1B,0xFE,0xA3,0x1D,0x16,0x89,0x00,0xB4,0xD5,0x40,0x82,0x4C,0x2E,0xEC,0x3D,0x66,0x0D,0x0D,0xB9,0x17,0x40,0x6E,0x3A,0x5C,0x03,0x7B,0x1B,0x93,0x28,0x0C,0xEF,0xB9,0x97,0xE3,0xA1,0xEB,0xE2,0xA3,0x7C,0x61,0xDD,0x6F,0xD5,0xAD,0x15,0x69,0x00,
		0x16,0xB2,0xC3,0x08,0x3D,0xC4,0x59,0xC6,0xF2,0x70,0xA5,0xB0,0xF5,0x1F,0x1D,0xF4,0xB0,0x15,0xDA,0x7E,0x28,0x39,0x24,0x99,0x36,0x5B,0xEC,0x39,0x25,0xFA,0x92,0x49,0x65,0xD2,0x43,0x05,0x6A,0x9E,0xA3,0x7B,0xF0,0xDE,0xA3,0x2F,0xD3,0x6F,0x3A,0xF9,0x35,0xC3,0x29,0xD4,0x45,0x6C,0x56,0x9A,0xDE,0x36,0x6E,0xFE,0x12,0x68,0x96,0x7B,0x45,0x1D,0x2C,0xFF,0xB9,0x2D,0xF5,0x52,0x8C,0xDF,0x3E,0x2F,0x63,0x02,0x15,0x00,0x81,0xA9,0xB5,0xD0,0x04,0xF2,0x9B,0xA7,0xD8,0x55,0x4C,0x3B,0x32,0xA1,0x45,0x32,0x4F,0xF5,0x51,0xDD,
		0x02,0x81,0x80,0x64,0x7A,0x88,0x0B,0xF2,0x3E,0x91,0x81,0x59,0x9C,0xF4,0xEA,0xC6,0x7B,0x0E,0xBE,0xEA,0x05,0xE8,0x77,0xFD,0x20,0x34,0x87,0xA1,0xC4,0x69,0xF6,0xC8,0x8B,0x19,0xDA,0xCD,0xFA,0x21,0x8A,0x57,0xA9,0x7A,0x26,0x0A,0x56,0xD4,0xED,0x4B,0x1B,0x7C,0x70,0xED,0xB4,0xE6,0x7A,0x6A,0xDE,0xD3,0x29,0xE2,0xE9,0x9A,0x33,0xED,0x09,0x8D,0x9E,0xDF,0xDA,0x2E,0x4A,0xC1,0x50,0x92,0xEE,0x2F,0xE5,0x5A,0xF3,0x85,0x62,0x6A,0x48,0xDC,0x1B,0x02,0x98,0xA6,0xB0,0xD1,0x09,0x4B,0x10,0xD1,0xF0,0xFA,0xE0,0xB1,0x1D,0x13,
		0x54,0x4B,0xC0,0xA8,0x40,0xEF,0x71,0xE8,0x56,0x6B,0xA2,0x29,0xCB,0x1E,0x09,0x7D,0x27,0x39,0x91,0x3B,0x20,0x4F,0x98,0x39,0xE8,0x39,0xCA,0x98,0xC5,0xAF,0x54,0x03,0x81,0x84,0x00,0x02,0x81,0x80,0x54,0xA8,0x88,0xB5,0x8F,0x01,0x56,0xCE,0x18,0x8F,0xA6,0xD6,0x7C,0x29,0x29,0x75,0x45,0xE8,0x31,0xA4,0x07,0x17,0xED,0x1E,0x5D,0xB2,0x7B,0xBB,0xCE,0x3C,0x97,0x67,0x1E,0x88,0x0A,0xFE,0x7D,0x00,0x22,0x27,0x1D,0x66,0xEE,0xF6,0x1B,0xB6,0x95,0x7F,0x5A,0xFF,0x06,0x34,0x02,0x43,0xC3,0x83,0xC4,0x66,0x2C,0xA1,0x05,0x0E,
		0x68,0xB3,0xCA,0xDC,0xD3,0xF9,0x0C,0xC0,0x66,0xDF,0x85,0x84,0x4B,0x20,0x5D,0x41,0xAC,0xC0,0xEC,0x37,0x92,0x0E,0x97,0x19,0xBF,0x53,0x35,0x63,0x27,0x18,0x33,0x35,0x42,0x4D,0xF0,0x2D,0x6D,0xA7,0xA4,0x98,0xAA,0x57,0xF3,0xD2,0xB8,0x6E,0x4E,0x8F,0xFF,0xBE,0x6F,0x4E,0x0F,0x0B,0x44,0x24,0xEE,0xDF,0x4C,0x22,0x5B,0x44,0x98,0x94,0xCB,0xB8,0xA3,0x2F,0x30,0x2D,0x30,0x1D,0x06,0x03,0x55,0x1D,0x0E,0x04,0x16,0x04,0x14,0x9D,0x2D,0x73,0xC3,0xB8,0xE3,0x4D,0x29,0x28,0xC3,0x65,0xBE,0xA9,0x98,0xCB,0xD6,0x8A,0x06,0x68,
		0x9C,0x30,0x0C,0x06,0x03,0x55,0x1D,0x13,0x04,0x05,0x30,0x03,0x01,0x01,0xFF,0x30,0x09,0x06,0x07,0x2A,0x86,0x48,0xCE,0x38,0x04,0x03,0x03,0x2F,0x00,0x30,0x2C,0x02,0x14,0x5A,0x1B,0x2D,0x08,0x0E,0xE6,0x99,0x38,0x8F,0xB5,0x09,0xC9,0x89,0x79,0x7E,0x01,0x30,0xBD,0xCE,0xF0,0x02,0x14,0x71,0x7B,0x08,0x51,0x97,0xCE,0x4D,0x1F,0x6A,0x84,0x47,0x3A,0xC0,0xBD,0x13,0x89,0x81,0xB9,0x01,0x97 };

	X509Certificate x509 = new X509Certificate (cert);
	X509Certificate clone = new X509Certificate (x509);

	Assertion.Assert ("Equals", x509.Equals (clone));
	Assertion.Assert ("Equals", clone.Equals (x509));

	FileStream fs = File.OpenWrite ("temp.cer");
	fs.Write (cert, 0, cert.Length);
	fs.Close ();

	X509Certificate disk = X509Certificate.CreateFromCertFile ("temp.cer");
	Assertion.Assert ("Equals", disk.Equals (clone));
	Assertion.Assert ("Equals", disk.Equals (x509));
	byte[] hash = { 0xA3,0x0A,0x06,0xB1,0xB0,0xD4,0xF8,0x71,0x67,0x8A,0x60,0xC5,0x4E,0xE4,0xD8,0x66,0x36,0x06,0xCC,0xE6 };
	AssertEquals ("GetCertHash", hash, x509.GetCertHash ());
	Assertion.AssertEquals ("GetCertHashString", "A30A06B1B0D4F871678A60C54EE4D8663606CCE6", x509.GetCertHashString ());
#if NET_2_0
	DateTime from = DateTime.ParseExact (x509.GetEffectiveDateString (), "MM/dd/yyyy HH:mm:ss", null).ToUniversalTime ();
	Assert.AreEqual ("10/25/2000 00:00:00", from.ToString (), "GetEffectiveDateString");
	DateTime until = DateTime.ParseExact (x509.GetExpirationDateString (), "MM/dd/yyyy HH:mm:ss", null).ToUniversalTime ();
	Assert.AreEqual ("01/01/2003 00:00:00", until.ToString (), "GetExpirationDateString");
#else
	// fx 1.x has a bug where the returned dates were always in the Seattle time zone
	Assertion.AssertEquals ("GetEffectiveDateString", "10/24/2000 16:00:00", x509.GetEffectiveDateString ());
	Assertion.AssertEquals ("GetExpirationDateString", "12/31/2002 16:00:00", x509.GetExpirationDateString ());
	// which was making it easier to test the dates ;-)
#endif
	Assertion.AssertEquals ("GetFormat", "X509", x509.GetFormat ());
	Assertion.AssertEquals ("GetHashCode", -1559624015, x509.GetHashCode ());
	Assertion.AssertEquals ("GetIssuerName", "C=US, O=U.S. Government, OU=DoD, CN=Armed Forces Root", x509.GetIssuerName ());
	Assertion.AssertEquals ("GetKeyAlgorithm", "1.2.840.10040.4.1", x509.GetKeyAlgorithm ());
	byte[] keyparams = { 0x30,0x82,0x01,0x1E,0x02,0x81,0x81,0x00,0x90,0x89,0x3E,0x18,0x1B,0xFE,0xA3,0x1D,0x16,0x89,0x00,0xB4,0xD5,0x40,0x82,0x4C,0x2E,0xEC,0x3D,0x66,0x0D,0x0D,0xB9,0x17,0x40,0x6E,0x3A,0x5C,0x03,0x7B,0x1B,0x93,0x28,0x0C,0xEF,0xB9,0x97,0xE3,0xA1,0xEB,0xE2,0xA3,0x7C,0x61,0xDD,0x6F,0xD5,0xAD,0x15,0x69,0x00,0x16,0xB2,0xC3,0x08,0x3D,0xC4,0x59,0xC6,0xF2,0x70,0xA5,0xB0,0xF5,0x1F,0x1D,0xF4,0xB0,0x15,0xDA,0x7E,0x28,0x39,0x24,0x99,0x36,0x5B,0xEC,0x39,0x25,0xFA,0x92,0x49,0x65,0xD2,0x43,0x05,0x6A,0x9E,0xA3,0x7B,0xF0,0xDE,
		0xA3,0x2F,0xD3,0x6F,0x3A,0xF9,0x35,0xC3,0x29,0xD4,0x45,0x6C,0x56,0x9A,0xDE,0x36,0x6E,0xFE,0x12,0x68,0x96,0x7B,0x45,0x1D,0x2C,0xFF,0xB9,0x2D,0xF5,0x52,0x8C,0xDF,0x3E,0x2F,0x63,0x02,0x15,0x00,0x81,0xA9,0xB5,0xD0,0x04,0xF2,0x9B,0xA7,0xD8,0x55,0x4C,0x3B,0x32,0xA1,0x45,0x32,0x4F,0xF5,0x51,0xDD,0x02,0x81,0x80,0x64,0x7A,0x88,0x0B,0xF2,0x3E,0x91,0x81,0x59,0x9C,0xF4,0xEA,0xC6,0x7B,0x0E,0xBE,0xEA,0x05,0xE8,0x77,0xFD,0x20,0x34,0x87,0xA1,0xC4,0x69,0xF6,0xC8,0x8B,0x19,0xDA,0xCD,0xFA,0x21,0x8A,0x57,0xA9,0x7A,
		0x26,0x0A,0x56,0xD4,0xED,0x4B,0x1B,0x7C,0x70,0xED,0xB4,0xE6,0x7A,0x6A,0xDE,0xD3,0x29,0xE2,0xE9,0x9A,0x33,0xED,0x09,0x8D,0x9E,0xDF,0xDA,0x2E,0x4A,0xC1,0x50,0x92,0xEE,0x2F,0xE5,0x5A,0xF3,0x85,0x62,0x6A,0x48,0xDC,0x1B,0x02,0x98,0xA6,0xB0,0xD1,0x09,0x4B,0x10,0xD1,0xF0,0xFA,0xE0,0xB1,0x1D,0x13,0x54,0x4B,0xC0,0xA8,0x40,0xEF,0x71,0xE8,0x56,0x6B,0xA2,0x29,0xCB,0x1E,0x09,0x7D,0x27,0x39,0x91,0x3B,0x20,0x4F,0x98,0x39,0xE8,0x39,0xCA,0x98,0xC5,0xAF,0x54 };
	AssertEquals ("GetKeyAlgorithmParameters", keyparams, x509.GetKeyAlgorithmParameters ());
	Assertion.AssertEquals ("GetKeyAlgorithmParametersString", "3082011E0281810090893E181BFEA31D168900B4D540824C2EEC3D660D0DB917406E3A5C037B1B93280CEFB997E3A1EBE2A37C61DD6FD5AD15690016B2C3083DC459C6F270A5B0F51F1DF4B015DA7E28392499365BEC3925FA924965D243056A9EA37BF0DEA32FD36F3AF935C329D4456C569ADE366EFE1268967B451D2CFFB92DF5528CDF3E2F6302150081A9B5D004F29BA7D8554C3B32A145324FF551DD028180647A880BF23E9181599CF4EAC67B0EBEEA05E877FD203487A1C469F6C88B19DACDFA218A57A97A260A56D4ED4B1B7C70EDB4E67A6ADED329E2E99A33ED098D9EDFDA2E4AC15092EE2FE55AF385626A48DC1B0298A6B0D1094B10D1F0FAE0B11D13544BC0A840EF71E8566BA229CB1E097D2739913B204F9839E839CA98C5AF54", x509.GetKeyAlgorithmParametersString ());
	Assertion.AssertEquals ("GetName", "C=US, O=U.S. Government, OU=DoD, CN=Armed Forces Root", x509.GetName ());
	byte[] pubkey = { 0x02,0x81,0x80,0x54,0xA8,0x88,0xB5,0x8F,0x01,0x56,0xCE,0x18,0x8F,0xA6,0xD6,0x7C,0x29,0x29,0x75,0x45,0xE8,0x31,0xA4,0x07,0x17,0xED,0x1E,0x5D,0xB2,0x7B,0xBB,0xCE,0x3C,0x97,0x67,0x1E,0x88,0x0A,0xFE,0x7D,0x00,0x22,0x27,0x1D,0x66,0xEE,0xF6,0x1B,0xB6,0x95,0x7F,0x5A,0xFF,0x06,0x34,0x02,0x43,0xC3,0x83,0xC4,0x66,0x2C,0xA1,0x05,0x0E,0x68,0xB3,0xCA,0xDC,0xD3,0xF9,0x0C,0xC0,0x66,0xDF,0x85,0x84,0x4B,0x20,0x5D,0x41,0xAC,0xC0,0xEC,0x37,0x92,0x0E,0x97,0x19,0xBF,0x53,0x35,0x63,0x27,0x18,0x33,0x35,0x42,0x4D,0xF0,0x2D,
		0x6D,0xA7,0xA4,0x98,0xAA,0x57,0xF3,0xD2,0xB8,0x6E,0x4E,0x8F,0xFF,0xBE,0x6F,0x4E,0x0F,0x0B,0x44,0x24,0xEE,0xDF,0x4C,0x22,0x5B,0x44,0x98,0x94,0xCB,0xB8 };
	AssertEquals ("GetPublicKey", pubkey, x509.GetPublicKey ());
	Assertion.AssertEquals ("GetPublicKeyString", "02818054A888B58F0156CE188FA6D67C29297545E831A40717ED1E5DB27BBBCE3C97671E880AFE7D0022271D66EEF61BB6957F5AFF06340243C383C4662CA1050E68B3CADCD3F90CC066DF85844B205D41ACC0EC37920E9719BF53356327183335424DF02D6DA7A498AA57F3D2B86E4E8FFFBE6F4E0F0B4424EEDF4C225B449894CBB8", x509.GetPublicKeyString ());
	AssertEquals ("GetRawCertData", cert, x509.GetRawCertData ());
	Assertion.AssertEquals ("GetRawCertDataString", ToString (cert), x509.GetRawCertDataString ());
	byte[] serial = { 0x03 };
	AssertEquals ("GetSerialNumber", serial, x509.GetSerialNumber ());
	Assertion.AssertEquals ("GetSerialNumberString", "03", x509.GetSerialNumberString ());
	byte[] ToStringArray = { 0x43,0x45,0x52,0x54,0x49,0x46,0x49,0x43,0x41,0x54,0x45,0x3A,0x0D,0x0A,0x09,0x46,0x6F,0x72,0x6D,0x61,0x74,0x3A,0x20,0x20,0x58,0x35,0x30,0x39,0x0D,0x0A,0x09,0x4E,0x61,0x6D,0x65,0x3A,0x20,0x20,0x43,0x3D,0x55,0x53,0x2C,0x20,0x4F,0x3D,0x55,0x2E,0x53,0x2E,0x20,0x47,0x6F,0x76,0x65,0x72,0x6E,0x6D,0x65,0x6E,0x74,0x2C,0x20,0x4F,0x55,0x3D,0x44,0x6F,0x44,0x2C,0x20,0x43,0x4E,0x3D,0x41,0x72,0x6D,0x65,0x64,0x20,0x46,0x6F,0x72,0x63,0x65,0x73,0x20,0x52,0x6F,0x6F,0x74,0x0D,0x0A,0x09,0x49,0x73,0x73,0x75,0x69,0x6E,0x67,
		0x20,0x43,0x41,0x3A,0x20,0x20,0x43,0x3D,0x55,0x53,0x2C,0x20,0x4F,0x3D,0x55,0x2E,0x53,0x2E,0x20,0x47,0x6F,0x76,0x65,0x72,0x6E,0x6D,0x65,0x6E,0x74,0x2C,0x20,0x4F,0x55,0x3D,0x44,0x6F,0x44,0x2C,0x20,0x43,0x4E,0x3D,0x41,0x72,0x6D,0x65,0x64,0x20,0x46,0x6F,0x72,0x63,0x65,0x73,0x20,0x52,0x6F,0x6F,0x74,0x0D,0x0A,0x09,0x4B,0x65,0x79,0x20,0x41,0x6C,0x67,0x6F,0x72,0x69,0x74,0x68,0x6D,0x3A,0x20,0x20,0x31,0x2E,0x32,0x2E,0x38,0x34,0x30,0x2E,0x31,0x30,0x30,0x34,0x30,0x2E,0x34,0x2E,0x31,0x0D,0x0A,0x09,0x53,0x65,
		0x72,0x69,0x61,0x6C,0x20,0x4E,0x75,0x6D,0x62,0x65,0x72,0x3A,0x20,0x20,0x30,0x33,0x0D,0x0A,0x09,0x4B,0x65,0x79,0x20,0x41,0x6C,0x6F,0x67,0x72,0x69,0x74,0x68,0x6D,0x20,0x50,0x61,0x72,0x61,0x6D,0x65,0x74,0x65,0x72,0x73,0x3A,0x20,0x20,0x33,0x30,0x38,0x32,0x30,0x31,0x31,0x45,0x30,0x32,0x38,0x31,0x38,0x31,0x30,0x30,0x39,0x30,0x38,0x39,0x33,0x45,0x31,0x38,0x31,0x42,0x46,0x45,0x41,0x33,0x31,0x44,0x31,0x36,0x38,0x39,0x30,0x30,0x42,0x34,0x44,0x35,0x34,0x30,0x38,0x32,0x34,0x43,0x32,0x45,0x45,0x43,0x33,0x44,
		0x36,0x36,0x30,0x44,0x30,0x44,0x42,0x39,0x31,0x37,0x34,0x30,0x36,0x45,0x33,0x41,0x35,0x43,0x30,0x33,0x37,0x42,0x31,0x42,0x39,0x33,0x32,0x38,0x30,0x43,0x45,0x46,0x42,0x39,0x39,0x37,0x45,0x33,0x41,0x31,0x45,0x42,0x45,0x32,0x41,0x33,0x37,0x43,0x36,0x31,0x44,0x44,0x36,0x46,0x44,0x35,0x41,0x44,0x31,0x35,0x36,0x39,0x30,0x30,0x31,0x36,0x42,0x32,0x43,0x33,0x30,0x38,0x33,0x44,0x43,0x34,0x35,0x39,0x43,0x36,0x46,0x32,0x37,0x30,0x41,0x35,0x42,0x30,0x46,0x35,0x31,0x46,0x31,0x44,0x46,0x34,0x42,0x30,0x31,0x35,
		0x44,0x41,0x37,0x45,0x32,0x38,0x33,0x39,0x32,0x34,0x39,0x39,0x33,0x36,0x35,0x42,0x45,0x43,0x33,0x39,0x32,0x35,0x46,0x41,0x39,0x32,0x34,0x39,0x36,0x35,0x44,0x32,0x34,0x33,0x30,0x35,0x36,0x41,0x39,0x45,0x41,0x33,0x37,0x42,0x46,0x30,0x44,0x45,0x41,0x33,0x32,0x46,0x44,0x33,0x36,0x46,0x33,0x41,0x46,0x39,0x33,0x35,0x43,0x33,0x32,0x39,0x44,0x34,0x34,0x35,0x36,0x43,0x35,0x36,0x39,0x41,0x44,0x45,0x33,0x36,0x36,0x45,0x46,0x45,0x31,0x32,0x36,0x38,0x39,0x36,0x37,0x42,0x34,0x35,0x31,0x44,0x32,0x43,0x46,0x46,
		0x42,0x39,0x32,0x44,0x46,0x35,0x35,0x32,0x38,0x43,0x44,0x46,0x33,0x45,0x32,0x46,0x36,0x33,0x30,0x32,0x31,0x35,0x30,0x30,0x38,0x31,0x41,0x39,0x42,0x35,0x44,0x30,0x30,0x34,0x46,0x32,0x39,0x42,0x41,0x37,0x44,0x38,0x35,0x35,0x34,0x43,0x33,0x42,0x33,0x32,0x41,0x31,0x34,0x35,0x33,0x32,0x34,0x46,0x46,0x35,0x35,0x31,0x44,0x44,0x30,0x32,0x38,0x31,0x38,0x30,0x36,0x34,0x37,0x41,0x38,0x38,0x30,0x42,0x46,0x32,0x33,0x45,0x39,0x31,0x38,0x31,0x35,0x39,0x39,0x43,0x46,0x34,0x45,0x41,0x43,0x36,0x37,0x42,0x30,0x45,
		0x42,0x45,0x45,0x41,0x30,0x35,0x45,0x38,0x37,0x37,0x46,0x44,0x32,0x30,0x33,0x34,0x38,0x37,0x41,0x31,0x43,0x34,0x36,0x39,0x46,0x36,0x43,0x38,0x38,0x42,0x31,0x39,0x44,0x41,0x43,0x44,0x46,0x41,0x32,0x31,0x38,0x41,0x35,0x37,0x41,0x39,0x37,0x41,0x32,0x36,0x30,0x41,0x35,0x36,0x44,0x34,0x45,0x44,0x34,0x42,0x31,0x42,0x37,0x43,0x37,0x30,0x45,0x44,0x42,0x34,0x45,0x36,0x37,0x41,0x36,0x41,0x44,0x45,0x44,0x33,0x32,0x39,0x45,0x32,0x45,0x39,0x39,0x41,0x33,0x33,0x45,0x44,0x30,0x39,0x38,0x44,0x39,0x45,0x44,0x46,
		0x44,0x41,0x32,0x45,0x34,0x41,0x43,0x31,0x35,0x30,0x39,0x32,0x45,0x45,0x32,0x46,0x45,0x35,0x35,0x41,0x46,0x33,0x38,0x35,0x36,0x32,0x36,0x41,0x34,0x38,0x44,0x43,0x31,0x42,0x30,0x32,0x39,0x38,0x41,0x36,0x42,0x30,0x44,0x31,0x30,0x39,0x34,0x42,0x31,0x30,0x44,0x31,0x46,0x30,0x46,0x41,0x45,0x30,0x42,0x31,0x31,0x44,0x31,0x33,0x35,0x34,0x34,0x42,0x43,0x30,0x41,0x38,0x34,0x30,0x45,0x46,0x37,0x31,0x45,0x38,0x35,0x36,0x36,0x42,0x41,0x32,0x32,0x39,0x43,0x42,0x31,0x45,0x30,0x39,0x37,0x44,0x32,0x37,0x33,0x39,
		0x39,0x31,0x33,0x42,0x32,0x30,0x34,0x46,0x39,0x38,0x33,0x39,0x45,0x38,0x33,0x39,0x43,0x41,0x39,0x38,0x43,0x35,0x41,0x46,0x35,0x34,0x0D,0x0A,0x09,0x50,0x75,0x62,0x6C,0x69,0x63,0x20,0x4B,0x65,0x79,0x3A,0x20,0x20,0x30,0x32,0x38,0x31,0x38,0x30,0x35,0x34,0x41,0x38,0x38,0x38,0x42,0x35,0x38,0x46,0x30,0x31,0x35,0x36,0x43,0x45,0x31,0x38,0x38,0x46,0x41,0x36,0x44,0x36,0x37,0x43,0x32,0x39,0x32,0x39,0x37,0x35,0x34,0x35,0x45,0x38,0x33,0x31,0x41,0x34,0x30,0x37,0x31,0x37,0x45,0x44,0x31,0x45,0x35,0x44,0x42,0x32,
		0x37,0x42,0x42,0x42,0x43,0x45,0x33,0x43,0x39,0x37,0x36,0x37,0x31,0x45,0x38,0x38,0x30,0x41,0x46,0x45,0x37,0x44,0x30,0x30,0x32,0x32,0x32,0x37,0x31,0x44,0x36,0x36,0x45,0x45,0x46,0x36,0x31,0x42,0x42,0x36,0x39,0x35,0x37,0x46,0x35,0x41,0x46,0x46,0x30,0x36,0x33,0x34,0x30,0x32,0x34,0x33,0x43,0x33,0x38,0x33,0x43,0x34,0x36,0x36,0x32,0x43,0x41,0x31,0x30,0x35,0x30,0x45,0x36,0x38,0x42,0x33,0x43,0x41,0x44,0x43,0x44,0x33,0x46,0x39,0x30,0x43,0x43,0x30,0x36,0x36,0x44,0x46,0x38,0x35,0x38,0x34,0x34,0x42,0x32,0x30,
		0x35,0x44,0x34,0x31,0x41,0x43,0x43,0x30,0x45,0x43,0x33,0x37,0x39,0x32,0x30,0x45,0x39,0x37,0x31,0x39,0x42,0x46,0x35,0x33,0x33,0x35,0x36,0x33,0x32,0x37,0x31,0x38,0x33,0x33,0x33,0x35,0x34,0x32,0x34,0x44,0x46,0x30,0x32,0x44,0x36,0x44,0x41,0x37,0x41,0x34,0x39,0x38,0x41,0x41,0x35,0x37,0x46,0x33,0x44,0x32,0x42,0x38,0x36,0x45,0x34,0x45,0x38,0x46,0x46,0x46,0x42,0x45,0x36,0x46,0x34,0x45,0x30,0x46,0x30,0x42,0x34,0x34,0x32,0x34,0x45,0x45,0x44,0x46,0x34,0x43,0x32,0x32,0x35,0x42,0x34,0x34,0x39,0x38,0x39,0x34,
		0x43,0x42,0x42,0x38,0x0D,0x0A,0x0D,0x0A };
	string tsa = Encoding.Default.GetString (ToStringArray).Replace ("\r\n", Environment.NewLine);
	//Assertion.AssertEquals ("ToString(true)", tsa, x509.ToString (true));
}


// Certificate: basic\VERI.cer
// - X.509v3 end-entity certificate
// - md5RSA, RSA 512 bits
// - has a E= (email) in it's DN
[Test]
public void Certificate9 ()
{
	// certificate can be loaded by constructor
	byte[] cert = { 0x30,0x82,0x09,0xB9,0x30,0x82,0x09,0x22,0xA0,0x03,0x02,0x01,0x02,0x02,0x10,0x20,0x0B,0x35,0x5E,0xCE,0xC4,0xB0,0x63,0xB7,0xDE,0xC6,0x34,0xB9,0x70,0x34,0x44,0x30,0x0D,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x01,0x04,0x05,0x00,0x30,0x62,0x31,0x11,0x30,0x0F,0x06,0x03,0x55,0x04,0x07,0x13,0x08,0x49,0x6E,0x74,0x65,0x72,0x6E,0x65,0x74,0x31,0x17,0x30,0x15,0x06,0x03,0x55,0x04,0x0A,0x13,0x0E,0x56,0x65,0x72,0x69,0x53,0x69,0x67,0x6E,0x2C,0x20,0x49,0x6E,0x63,0x2E,0x31,0x34,0x30,0x32,0x06,0x03,0x55,0x04,0x0B,
		0x13,0x2B,0x56,0x65,0x72,0x69,0x53,0x69,0x67,0x6E,0x20,0x43,0x6C,0x61,0x73,0x73,0x20,0x31,0x20,0x43,0x41,0x20,0x2D,0x20,0x49,0x6E,0x64,0x69,0x76,0x69,0x64,0x75,0x61,0x6C,0x20,0x53,0x75,0x62,0x73,0x63,0x72,0x69,0x62,0x65,0x72,0x30,0x1E,0x17,0x0D,0x39,0x36,0x30,0x38,0x32,0x31,0x30,0x30,0x30,0x30,0x30,0x30,0x5A,0x17,0x0D,0x39,0x37,0x30,0x38,0x32,0x30,0x32,0x33,0x35,0x39,0x35,0x39,0x5A,0x30,0x82,0x01,0x0A,0x31,0x11,0x30,0x0F,0x06,0x03,0x55,0x04,0x07,0x13,0x08,0x49,0x6E,0x74,0x65,0x72,0x6E,0x65,0x74,
		0x31,0x17,0x30,0x15,0x06,0x03,0x55,0x04,0x0A,0x13,0x0E,0x56,0x65,0x72,0x69,0x53,0x69,0x67,0x6E,0x2C,0x20,0x49,0x6E,0x63,0x2E,0x31,0x34,0x30,0x32,0x06,0x03,0x55,0x04,0x0B,0x13,0x2B,0x56,0x65,0x72,0x69,0x53,0x69,0x67,0x6E,0x20,0x43,0x6C,0x61,0x73,0x73,0x20,0x31,0x20,0x43,0x41,0x20,0x2D,0x20,0x49,0x6E,0x64,0x69,0x76,0x69,0x64,0x75,0x61,0x6C,0x20,0x53,0x75,0x62,0x73,0x63,0x72,0x69,0x62,0x65,0x72,0x31,0x46,0x30,0x44,0x06,0x03,0x55,0x04,0x0B,0x13,0x3D,0x77,0x77,0x77,0x2E,0x76,0x65,0x72,0x69,0x73,0x69,
		0x67,0x6E,0x2E,0x63,0x6F,0x6D,0x2F,0x72,0x65,0x70,0x6F,0x73,0x69,0x74,0x6F,0x72,0x79,0x2F,0x43,0x50,0x53,0x20,0x49,0x6E,0x63,0x6F,0x72,0x70,0x2E,0x20,0x62,0x79,0x20,0x52,0x65,0x66,0x2E,0x2C,0x4C,0x49,0x41,0x42,0x2E,0x4C,0x54,0x44,0x28,0x63,0x29,0x39,0x36,0x31,0x26,0x30,0x24,0x06,0x03,0x55,0x04,0x0B,0x13,0x1D,0x44,0x69,0x67,0x69,0x74,0x61,0x6C,0x20,0x49,0x44,0x20,0x43,0x6C,0x61,0x73,0x73,0x20,0x31,0x20,0x2D,0x20,0x4E,0x65,0x74,0x73,0x63,0x61,0x70,0x65,0x31,0x16,0x30,0x14,0x06,0x03,0x55,0x04,0x03,
		0x13,0x0D,0x44,0x61,0x76,0x69,0x64,0x20,0x54,0x2E,0x20,0x47,0x72,0x61,0x79,0x31,0x1E,0x30,0x1C,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x09,0x01,0x16,0x0F,0x64,0x61,0x76,0x69,0x64,0x40,0x66,0x6F,0x72,0x6D,0x61,0x6C,0x2E,0x69,0x65,0x30,0x5C,0x30,0x0D,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x01,0x01,0x05,0x00,0x03,0x4B,0x00,0x30,0x48,0x02,0x41,0x00,0xC5,0x81,0x07,0xA2,0xEB,0x0F,0xB8,0xFF,0xF8,0xF8,0x1C,0xEE,0x32,0xFF,0xBF,0x12,0x35,0x6A,0xF9,0x6B,0xC8,0xBE,0x2F,0xFB,0x3E,0xAF,0x04,0x51,
		0x4A,0xAC,0xDD,0x10,0x29,0xA8,0xCD,0x40,0x5B,0x66,0x1E,0x98,0xEF,0xF2,0x4C,0x77,0xFA,0x8F,0x86,0xD1,0x21,0x67,0x92,0x44,0x4A,0xC4,0x89,0xC9,0x83,0xCF,0x88,0x9F,0x6F,0xE2,0x32,0x35,0x02,0x03,0x01,0x00,0x01,0xA3,0x82,0x07,0x08,0x30,0x82,0x07,0x04,0x30,0x09,0x06,0x03,0x55,0x1D,0x13,0x04,0x02,0x30,0x00,0x30,0x82,0x02,0x1F,0x06,0x03,0x55,0x1D,0x03,0x04,0x82,0x02,0x16,0x30,0x82,0x02,0x12,0x30,0x82,0x02,0x0E,0x30,0x82,0x02,0x0A,0x06,0x0B,0x60,0x86,0x48,0x01,0x86,0xF8,0x45,0x01,0x07,0x01,0x01,0x30,0x82,
		0x01,0xF9,0x16,0x82,0x01,0xA7,0x54,0x68,0x69,0x73,0x20,0x63,0x65,0x72,0x74,0x69,0x66,0x69,0x63,0x61,0x74,0x65,0x20,0x69,0x6E,0x63,0x6F,0x72,0x70,0x6F,0x72,0x61,0x74,0x65,0x73,0x20,0x62,0x79,0x20,0x72,0x65,0x66,0x65,0x72,0x65,0x6E,0x63,0x65,0x2C,0x20,0x61,0x6E,0x64,0x20,0x69,0x74,0x73,0x20,0x75,0x73,0x65,0x20,0x69,0x73,0x20,0x73,0x74,0x72,0x69,0x63,0x74,0x6C,0x79,0x20,0x73,0x75,0x62,0x6A,0x65,0x63,0x74,0x20,0x74,0x6F,0x2C,0x20,0x74,0x68,0x65,0x20,0x56,0x65,0x72,0x69,0x53,0x69,0x67,0x6E,0x20,0x43,
		0x65,0x72,0x74,0x69,0x66,0x69,0x63,0x61,0x74,0x69,0x6F,0x6E,0x20,0x50,0x72,0x61,0x63,0x74,0x69,0x63,0x65,0x20,0x53,0x74,0x61,0x74,0x65,0x6D,0x65,0x6E,0x74,0x20,0x28,0x43,0x50,0x53,0x29,0x2C,0x20,0x61,0x76,0x61,0x69,0x6C,0x61,0x62,0x6C,0x65,0x20,0x61,0x74,0x3A,0x20,0x68,0x74,0x74,0x70,0x73,0x3A,0x2F,0x2F,0x77,0x77,0x77,0x2E,0x76,0x65,0x72,0x69,0x73,0x69,0x67,0x6E,0x2E,0x63,0x6F,0x6D,0x2F,0x43,0x50,0x53,0x3B,0x20,0x62,0x79,0x20,0x45,0x2D,0x6D,0x61,0x69,0x6C,0x20,0x61,0x74,0x20,0x43,0x50,0x53,0x2D,
		0x72,0x65,0x71,0x75,0x65,0x73,0x74,0x73,0x40,0x76,0x65,0x72,0x69,0x73,0x69,0x67,0x6E,0x2E,0x63,0x6F,0x6D,0x3B,0x20,0x6F,0x72,0x20,0x62,0x79,0x20,0x6D,0x61,0x69,0x6C,0x20,0x61,0x74,0x20,0x56,0x65,0x72,0x69,0x53,0x69,0x67,0x6E,0x2C,0x20,0x49,0x6E,0x63,0x2E,0x2C,0x20,0x32,0x35,0x39,0x33,0x20,0x43,0x6F,0x61,0x73,0x74,0x20,0x41,0x76,0x65,0x2E,0x2C,0x20,0x4D,0x6F,0x75,0x6E,0x74,0x61,0x69,0x6E,0x20,0x56,0x69,0x65,0x77,0x2C,0x20,0x43,0x41,0x20,0x39,0x34,0x30,0x34,0x33,0x20,0x55,0x53,0x41,0x20,0x54,0x65,
		0x6C,0x2E,0x20,0x2B,0x31,0x20,0x28,0x34,0x31,0x35,0x29,0x20,0x39,0x36,0x31,0x2D,0x38,0x38,0x33,0x30,0x20,0x43,0x6F,0x70,0x79,0x72,0x69,0x67,0x68,0x74,0x20,0x28,0x63,0x29,0x20,0x31,0x39,0x39,0x36,0x20,0x56,0x65,0x72,0x69,0x53,0x69,0x67,0x6E,0x2C,0x20,0x49,0x6E,0x63,0x2E,0x20,0x20,0x41,0x6C,0x6C,0x20,0x52,0x69,0x67,0x68,0x74,0x73,0x20,0x52,0x65,0x73,0x65,0x72,0x76,0x65,0x64,0x2E,0x20,0x43,0x45,0x52,0x54,0x41,0x49,0x4E,0x20,0x57,0x41,0x52,0x52,0x41,0x4E,0x54,0x49,0x45,0x53,0x20,0x44,0x49,0x53,0x43,
		0x4C,0x41,0x49,0x4D,0x45,0x44,0x20,0x61,0x6E,0x64,0x20,0x4C,0x49,0x41,0x42,0x49,0x4C,0x49,0x54,0x59,0x20,0x4C,0x49,0x4D,0x49,0x54,0x45,0x44,0x2E,0xA0,0x0E,0x06,0x0C,0x60,0x86,0x48,0x01,0x86,0xF8,0x45,0x01,0x07,0x01,0x01,0x01,0xA1,0x0E,0x06,0x0C,0x60,0x86,0x48,0x01,0x86,0xF8,0x45,0x01,0x07,0x01,0x01,0x02,0x30,0x2C,0x30,0x2A,0x16,0x28,0x68,0x74,0x74,0x70,0x73,0x3A,0x2F,0x2F,0x77,0x77,0x77,0x2E,0x76,0x65,0x72,0x69,0x73,0x69,0x67,0x6E,0x2E,0x63,0x6F,0x6D,0x2F,0x72,0x65,0x70,0x6F,0x73,0x69,0x74,0x6F,
		0x72,0x79,0x2F,0x43,0x50,0x53,0x20,0x30,0x11,0x06,0x09,0x60,0x86,0x48,0x01,0x86,0xF8,0x42,0x01,0x01,0x04,0x04,0x03,0x02,0x07,0x80,0x30,0x36,0x06,0x09,0x60,0x86,0x48,0x01,0x86,0xF8,0x42,0x01,0x08,0x04,0x29,0x16,0x27,0x68,0x74,0x74,0x70,0x73,0x3A,0x2F,0x2F,0x77,0x77,0x77,0x2E,0x76,0x65,0x72,0x69,0x73,0x69,0x67,0x6E,0x2E,0x63,0x6F,0x6D,0x2F,0x72,0x65,0x70,0x6F,0x73,0x69,0x74,0x6F,0x72,0x79,0x2F,0x43,0x50,0x53,0x30,0x82,0x04,0x87,0x06,0x09,0x60,0x86,0x48,0x01,0x86,0xF8,0x42,0x01,0x0D,0x04,0x82,0x04,
		0x78,0x16,0x82,0x04,0x74,0x43,0x41,0x55,0x54,0x49,0x4F,0x4E,0x3A,0x20,0x54,0x68,0x65,0x20,0x43,0x6F,0x6D,0x6D,0x6F,0x6E,0x20,0x4E,0x61,0x6D,0x65,0x20,0x69,0x6E,0x20,0x74,0x68,0x69,0x73,0x20,0x43,0x6C,0x61,0x73,0x73,0x20,0x31,0x20,0x44,0x69,0x67,0x69,0x74,0x61,0x6C,0x20,0x0A,0x49,0x44,0x20,0x69,0x73,0x20,0x6E,0x6F,0x74,0x20,0x61,0x75,0x74,0x68,0x65,0x6E,0x74,0x69,0x63,0x61,0x74,0x65,0x64,0x20,0x62,0x79,0x20,0x56,0x65,0x72,0x69,0x53,0x69,0x67,0x6E,0x2E,0x20,0x49,0x74,0x20,0x6D,0x61,0x79,0x20,0x62,
		0x65,0x20,0x74,0x68,0x65,0x0A,0x68,0x6F,0x6C,0x64,0x65,0x72,0x27,0x73,0x20,0x72,0x65,0x61,0x6C,0x20,0x6E,0x61,0x6D,0x65,0x20,0x6F,0x72,0x20,0x61,0x6E,0x20,0x61,0x6C,0x69,0x61,0x73,0x2E,0x20,0x56,0x65,0x72,0x69,0x53,0x69,0x67,0x6E,0x20,0x64,0x6F,0x65,0x73,0x20,0x61,0x75,0x74,0x68,0x2D,0x0A,0x65,0x6E,0x74,0x69,0x63,0x61,0x74,0x65,0x20,0x74,0x68,0x65,0x20,0x65,0x2D,0x6D,0x61,0x69,0x6C,0x20,0x61,0x64,0x64,0x72,0x65,0x73,0x73,0x20,0x6F,0x66,0x20,0x74,0x68,0x65,0x20,0x68,0x6F,0x6C,0x64,0x65,0x72,0x2E,
		0x0A,0x0A,0x54,0x68,0x69,0x73,0x20,0x63,0x65,0x72,0x74,0x69,0x66,0x69,0x63,0x61,0x74,0x65,0x20,0x69,0x6E,0x63,0x6F,0x72,0x70,0x6F,0x72,0x61,0x74,0x65,0x73,0x20,0x62,0x79,0x20,0x72,0x65,0x66,0x65,0x72,0x65,0x6E,0x63,0x65,0x2C,0x20,0x61,0x6E,0x64,0x20,0x0A,0x69,0x74,0x73,0x20,0x75,0x73,0x65,0x20,0x69,0x73,0x20,0x73,0x74,0x72,0x69,0x63,0x74,0x6C,0x79,0x20,0x73,0x75,0x62,0x6A,0x65,0x63,0x74,0x20,0x74,0x6F,0x2C,0x20,0x74,0x68,0x65,0x20,0x56,0x65,0x72,0x69,0x53,0x69,0x67,0x6E,0x20,0x0A,0x43,0x65,0x72,
		0x74,0x69,0x66,0x69,0x63,0x61,0x74,0x69,0x6F,0x6E,0x20,0x50,0x72,0x61,0x63,0x74,0x69,0x63,0x65,0x20,0x53,0x74,0x61,0x74,0x65,0x6D,0x65,0x6E,0x74,0x20,0x28,0x43,0x50,0x53,0x29,0x2C,0x20,0x61,0x76,0x61,0x69,0x6C,0x61,0x62,0x6C,0x65,0x0A,0x69,0x6E,0x20,0x74,0x68,0x65,0x20,0x56,0x65,0x72,0x69,0x53,0x69,0x67,0x6E,0x20,0x72,0x65,0x70,0x6F,0x73,0x69,0x74,0x6F,0x72,0x79,0x20,0x61,0x74,0x3A,0x20,0x0A,0x68,0x74,0x74,0x70,0x73,0x3A,0x2F,0x2F,0x77,0x77,0x77,0x2E,0x76,0x65,0x72,0x69,0x73,0x69,0x67,0x6E,0x2E,
		0x63,0x6F,0x6D,0x3B,0x20,0x62,0x79,0x20,0x45,0x2D,0x6D,0x61,0x69,0x6C,0x20,0x61,0x74,0x0A,0x43,0x50,0x53,0x2D,0x72,0x65,0x71,0x75,0x65,0x73,0x74,0x73,0x40,0x76,0x65,0x72,0x69,0x73,0x69,0x67,0x6E,0x2E,0x63,0x6F,0x6D,0x3B,0x20,0x6F,0x72,0x20,0x62,0x79,0x20,0x6D,0x61,0x69,0x6C,0x20,0x61,0x74,0x20,0x56,0x65,0x72,0x69,0x53,0x69,0x67,0x6E,0x2C,0x0A,0x49,0x6E,0x63,0x2E,0x2C,0x20,0x32,0x35,0x39,0x33,0x20,0x43,0x6F,0x61,0x73,0x74,0x20,0x41,0x76,0x65,0x2E,0x2C,0x20,0x4D,0x6F,0x75,0x6E,0x74,0x61,0x69,0x6E,
		0x20,0x56,0x69,0x65,0x77,0x2C,0x20,0x43,0x41,0x20,0x39,0x34,0x30,0x34,0x33,0x20,0x55,0x53,0x41,0x0A,0x0A,0x43,0x6F,0x70,0x79,0x72,0x69,0x67,0x68,0x74,0x20,0x28,0x63,0x29,0x31,0x39,0x39,0x36,0x20,0x56,0x65,0x72,0x69,0x53,0x69,0x67,0x6E,0x2C,0x20,0x49,0x6E,0x63,0x2E,0x20,0x20,0x41,0x6C,0x6C,0x20,0x52,0x69,0x67,0x68,0x74,0x73,0x20,0x0A,0x52,0x65,0x73,0x65,0x72,0x76,0x65,0x64,0x2E,0x20,0x43,0x45,0x52,0x54,0x41,0x49,0x4E,0x20,0x57,0x41,0x52,0x52,0x41,0x4E,0x54,0x49,0x45,0x53,0x20,0x44,0x49,0x53,0x43,
		0x4C,0x41,0x49,0x4D,0x45,0x44,0x20,0x41,0x4E,0x44,0x20,0x0A,0x4C,0x49,0x41,0x42,0x49,0x4C,0x49,0x54,0x59,0x20,0x4C,0x49,0x4D,0x49,0x54,0x45,0x44,0x2E,0x0A,0x0A,0x57,0x41,0x52,0x4E,0x49,0x4E,0x47,0x3A,0x20,0x54,0x48,0x45,0x20,0x55,0x53,0x45,0x20,0x4F,0x46,0x20,0x54,0x48,0x49,0x53,0x20,0x43,0x45,0x52,0x54,0x49,0x46,0x49,0x43,0x41,0x54,0x45,0x20,0x49,0x53,0x20,0x53,0x54,0x52,0x49,0x43,0x54,0x4C,0x59,0x0A,0x53,0x55,0x42,0x4A,0x45,0x43,0x54,0x20,0x54,0x4F,0x20,0x54,0x48,0x45,0x20,0x56,0x45,0x52,0x49,
		0x53,0x49,0x47,0x4E,0x20,0x43,0x45,0x52,0x54,0x49,0x46,0x49,0x43,0x41,0x54,0x49,0x4F,0x4E,0x20,0x50,0x52,0x41,0x43,0x54,0x49,0x43,0x45,0x0A,0x53,0x54,0x41,0x54,0x45,0x4D,0x45,0x4E,0x54,0x2E,0x20,0x20,0x54,0x48,0x45,0x20,0x49,0x53,0x53,0x55,0x49,0x4E,0x47,0x20,0x41,0x55,0x54,0x48,0x4F,0x52,0x49,0x54,0x59,0x20,0x44,0x49,0x53,0x43,0x4C,0x41,0x49,0x4D,0x53,0x20,0x43,0x45,0x52,0x54,0x41,0x49,0x4E,0x0A,0x49,0x4D,0x50,0x4C,0x49,0x45,0x44,0x20,0x41,0x4E,0x44,0x20,0x45,0x58,0x50,0x52,0x45,0x53,0x53,0x20,
		0x57,0x41,0x52,0x52,0x41,0x4E,0x54,0x49,0x45,0x53,0x2C,0x20,0x49,0x4E,0x43,0x4C,0x55,0x44,0x49,0x4E,0x47,0x20,0x57,0x41,0x52,0x52,0x41,0x4E,0x54,0x49,0x45,0x53,0x0A,0x4F,0x46,0x20,0x4D,0x45,0x52,0x43,0x48,0x41,0x4E,0x54,0x41,0x42,0x49,0x4C,0x49,0x54,0x59,0x20,0x4F,0x52,0x20,0x46,0x49,0x54,0x4E,0x45,0x53,0x53,0x20,0x46,0x4F,0x52,0x20,0x41,0x20,0x50,0x41,0x52,0x54,0x49,0x43,0x55,0x4C,0x41,0x52,0x0A,0x50,0x55,0x52,0x50,0x4F,0x53,0x45,0x2C,0x20,0x41,0x4E,0x44,0x20,0x57,0x49,0x4C,0x4C,0x20,0x4E,0x4F,
		0x54,0x20,0x42,0x45,0x20,0x4C,0x49,0x41,0x42,0x4C,0x45,0x20,0x46,0x4F,0x52,0x20,0x43,0x4F,0x4E,0x53,0x45,0x51,0x55,0x45,0x4E,0x54,0x49,0x41,0x4C,0x2C,0x0A,0x50,0x55,0x4E,0x49,0x54,0x49,0x56,0x45,0x2C,0x20,0x41,0x4E,0x44,0x20,0x43,0x45,0x52,0x54,0x41,0x49,0x4E,0x20,0x4F,0x54,0x48,0x45,0x52,0x20,0x44,0x41,0x4D,0x41,0x47,0x45,0x53,0x2E,0x20,0x53,0x45,0x45,0x20,0x54,0x48,0x45,0x20,0x43,0x50,0x53,0x0A,0x46,0x4F,0x52,0x20,0x44,0x45,0x54,0x41,0x49,0x4C,0x53,0x2E,0x0A,0x0A,0x43,0x6F,0x6E,0x74,0x65,0x6E,
		0x74,0x73,0x20,0x6F,0x66,0x20,0x74,0x68,0x65,0x20,0x56,0x65,0x72,0x69,0x53,0x69,0x67,0x6E,0x20,0x72,0x65,0x67,0x69,0x73,0x74,0x65,0x72,0x65,0x64,0x0A,0x6E,0x6F,0x6E,0x76,0x65,0x72,0x69,0x66,0x69,0x65,0x64,0x53,0x75,0x62,0x6A,0x65,0x63,0x74,0x41,0x74,0x74,0x72,0x69,0x62,0x75,0x74,0x65,0x73,0x20,0x65,0x78,0x74,0x65,0x6E,0x73,0x69,0x6F,0x6E,0x20,0x76,0x61,0x6C,0x75,0x65,0x20,0x73,0x68,0x61,0x6C,0x6C,0x20,0x0A,0x6E,0x6F,0x74,0x20,0x62,0x65,0x20,0x63,0x6F,0x6E,0x73,0x69,0x64,0x65,0x72,0x65,0x64,0x20,
		0x61,0x73,0x20,0x61,0x63,0x63,0x75,0x72,0x61,0x74,0x65,0x20,0x69,0x6E,0x66,0x6F,0x72,0x6D,0x61,0x74,0x69,0x6F,0x6E,0x20,0x76,0x61,0x6C,0x69,0x64,0x61,0x74,0x65,0x64,0x20,0x0A,0x62,0x79,0x20,0x74,0x68,0x65,0x20,0x49,0x41,0x2E,0x30,0x0D,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x01,0x04,0x05,0x00,0x03,0x81,0x81,0x00,0x2B,0x3D,0x44,0xC7,0x32,0x59,0xAE,0xF1,0x5F,0x8F,0x3F,0x87,0xE3,0x3E,0xEB,0x81,0x30,0xF8,0xA9,0x96,0xDB,0x01,0x42,0x0B,0x04,0xEF,0x37,0x02,0x3F,0xD4,0x20,0x61,0x58,0xC4,0x4A,0x3A,
		0x39,0xB3,0xFB,0xD9,0xF8,0xA5,0xC4,0x5E,0x33,0x5A,0x0E,0xFA,0x93,0x56,0x2F,0x6F,0xD6,0x61,0xA2,0xAF,0xA5,0x0C,0x1D,0xE2,0x41,0x65,0xF3,0x40,0x75,0x66,0x83,0xD2,0x5A,0xB4,0xB7,0x56,0x0B,0x8E,0x0D,0xA1,0x33,0x13,0x7D,0x49,0xC3,0xB1,0x00,0x68,0x83,0x7F,0xB5,0x66,0xD4,0x32,0x32,0xFE,0x8B,0x9A,0x5A,0xD6,0x01,0x72,0x31,0x5D,0x85,0x91,0xBC,0x93,0x9B,0x65,0x60,0x25,0xC6,0x1F,0xBC,0xDD,0x69,0x44,0x62,0xC2,0xB2,0x6F,0x46,0xAB,0x2F,0x20,0xA5,0x6F,0xDA,0x48,0x6C,0x9C };

	X509Certificate x509 = new X509Certificate (cert);
	X509Certificate clone = new X509Certificate (x509);

	Assertion.Assert ("Equals", x509.Equals (clone));
	Assertion.Assert ("Equals", clone.Equals (x509));

	FileStream fs = File.OpenWrite ("temp.cer");
	fs.Write (cert, 0, cert.Length);
	fs.Close ();

	X509Certificate disk = X509Certificate.CreateFromCertFile ("temp.cer");
	Assertion.Assert ("Equals", disk.Equals (clone));
	Assertion.Assert ("Equals", disk.Equals (x509));
	byte[] hash = { 0x52,0xD1,0x84,0x8F,0x6A,0x16,0x1B,0xA5,0xC6,0xC1,0x90,0x07,0xF8,0x16,0x46,0x68,0xF5,0xA3,0xB2,0xCF };
	AssertEquals ("GetCertHash", hash, x509.GetCertHash ());
	Assertion.AssertEquals ("GetCertHashString", "52D1848F6A161BA5C6C19007F8164668F5A3B2CF", x509.GetCertHashString ());
#if NET_2_0
	DateTime from = DateTime.ParseExact (x509.GetEffectiveDateString (), "MM/dd/yyyy HH:mm:ss", null).ToUniversalTime ();
	Assert.AreEqual ("08/21/1996 00:00:00", from.ToString (), "GetEffectiveDateString");
	DateTime until = DateTime.ParseExact (x509.GetExpirationDateString (), "MM/dd/yyyy HH:mm:ss", null).ToUniversalTime ();
	Assert.AreEqual ("08/20/1997 23:59:59", until.ToString (), "GetExpirationDateString");
	Assert.AreEqual ("200B355ECEC4B063B7DEC634B9703444", x509.GetSerialNumberString (), "GetSerialNumberString");
#else
	// fx 1.x has a bug where the returned dates were always in the Seattle time zone
	Assertion.AssertEquals ("GetEffectiveDateString", "08/20/1996 16:00:00", x509.GetEffectiveDateString ());
	Assertion.AssertEquals ("GetExpirationDateString", "08/20/1997 15:59:59", x509.GetExpirationDateString ());
	// which was making it easier to test the dates ;-)
	Assert.AreEqual ("443470B934C6DEB763B0C4CE5E350B20", x509.GetSerialNumberString (), "GetSerialNumberString");
#endif
	Assertion.AssertEquals ("GetFormat", "X509", x509.GetFormat ());
	Assertion.AssertEquals ("GetHashCode", 1389462671, x509.GetHashCode ());
	Assertion.AssertEquals ("GetIssuerName", "L=Internet, O=\"VeriSign, Inc.\", OU=VeriSign Class 1 CA - Individual Subscriber", x509.GetIssuerName ());
	Assertion.AssertEquals ("GetKeyAlgorithm", "1.2.840.113549.1.1.1", x509.GetKeyAlgorithm ());
	byte[] keyparams = { 0x05,0x00 };
	AssertEquals ("GetKeyAlgorithmParameters", keyparams, x509.GetKeyAlgorithmParameters ());
	Assertion.AssertEquals ("GetKeyAlgorithmParametersString", "0500", x509.GetKeyAlgorithmParametersString ());
	Assertion.AssertEquals ("GetName", "L=Internet, O=\"VeriSign, Inc.\", OU=VeriSign Class 1 CA - Individual Subscriber, OU=\"www.verisign.com/repository/CPS Incorp. by Ref.,LIAB.LTD(c)96\", OU=Digital ID Class 1 - Netscape, CN=David T. Gray, E=david@formal.ie", x509.GetName ());
	byte[] pubkey = { 0x30,0x48,0x02,0x41,0x00,0xC5,0x81,0x07,0xA2,0xEB,0x0F,0xB8,0xFF,0xF8,0xF8,0x1C,0xEE,0x32,0xFF,0xBF,0x12,0x35,0x6A,0xF9,0x6B,0xC8,0xBE,0x2F,0xFB,0x3E,0xAF,0x04,0x51,0x4A,0xAC,0xDD,0x10,0x29,0xA8,0xCD,0x40,0x5B,0x66,0x1E,0x98,0xEF,0xF2,0x4C,0x77,0xFA,0x8F,0x86,0xD1,0x21,0x67,0x92,0x44,0x4A,0xC4,0x89,0xC9,0x83,0xCF,0x88,0x9F,0x6F,0xE2,0x32,0x35,0x02,0x03,0x01,0x00,0x01 };
	AssertEquals ("GetPublicKey", pubkey, x509.GetPublicKey ());
	Assertion.AssertEquals ("GetPublicKeyString", "3048024100C58107A2EB0FB8FFF8F81CEE32FFBF12356AF96BC8BE2FFB3EAF04514AACDD1029A8CD405B661E98EFF24C77FA8F86D1216792444AC489C983CF889F6FE232350203010001", x509.GetPublicKeyString ());
	AssertEquals ("GetRawCertData", cert, x509.GetRawCertData ());
	Assertion.AssertEquals ("GetRawCertDataString", ToString (cert), x509.GetRawCertDataString ());
	byte[] serial = { 0x44,0x34,0x70,0xB9,0x34,0xC6,0xDE,0xB7,0x63,0xB0,0xC4,0xCE,0x5E,0x35,0x0B,0x20 };
	AssertEquals ("GetSerialNumber", serial, x509.GetSerialNumber ());
	byte[] ToStringArray = { 0x43,0x45,0x52,0x54,0x49,0x46,0x49,0x43,0x41,0x54,0x45,0x3A,0x0D,0x0A,0x09,0x46,0x6F,0x72,0x6D,0x61,0x74,0x3A,0x20,0x20,0x58,0x35,0x30,0x39,0x0D,0x0A,0x09,0x4E,0x61,0x6D,0x65,0x3A,0x20,0x20,0x4C,0x3D,0x49,0x6E,0x74,0x65,0x72,0x6E,0x65,0x74,0x2C,0x20,0x4F,0x3D,0x22,0x56,0x65,0x72,0x69,0x53,0x69,0x67,0x6E,0x2C,0x20,0x49,0x6E,0x63,0x2E,0x22,0x2C,0x20,0x4F,0x55,0x3D,0x56,0x65,0x72,0x69,0x53,0x69,0x67,0x6E,0x20,0x43,0x6C,0x61,0x73,0x73,0x20,0x31,0x20,0x43,0x41,0x20,0x2D,0x20,0x49,0x6E,0x64,0x69,0x76,0x69,
		0x64,0x75,0x61,0x6C,0x20,0x53,0x75,0x62,0x73,0x63,0x72,0x69,0x62,0x65,0x72,0x2C,0x20,0x4F,0x55,0x3D,0x22,0x77,0x77,0x77,0x2E,0x76,0x65,0x72,0x69,0x73,0x69,0x67,0x6E,0x2E,0x63,0x6F,0x6D,0x2F,0x72,0x65,0x70,0x6F,0x73,0x69,0x74,0x6F,0x72,0x79,0x2F,0x43,0x50,0x53,0x20,0x49,0x6E,0x63,0x6F,0x72,0x70,0x2E,0x20,0x62,0x79,0x20,0x52,0x65,0x66,0x2E,0x2C,0x4C,0x49,0x41,0x42,0x2E,0x4C,0x54,0x44,0x28,0x63,0x29,0x39,0x36,0x22,0x2C,0x20,0x4F,0x55,0x3D,0x44,0x69,0x67,0x69,0x74,0x61,0x6C,0x20,0x49,0x44,0x20,0x43,
		0x6C,0x61,0x73,0x73,0x20,0x31,0x20,0x2D,0x20,0x4E,0x65,0x74,0x73,0x63,0x61,0x70,0x65,0x2C,0x20,0x43,0x4E,0x3D,0x44,0x61,0x76,0x69,0x64,0x20,0x54,0x2E,0x20,0x47,0x72,0x61,0x79,0x2C,0x20,0x45,0x3D,0x64,0x61,0x76,0x69,0x64,0x40,0x66,0x6F,0x72,0x6D,0x61,0x6C,0x2E,0x69,0x65,0x0D,0x0A,0x09,0x49,0x73,0x73,0x75,0x69,0x6E,0x67,0x20,0x43,0x41,0x3A,0x20,0x20,0x4C,0x3D,0x49,0x6E,0x74,0x65,0x72,0x6E,0x65,0x74,0x2C,0x20,0x4F,0x3D,0x22,0x56,0x65,0x72,0x69,0x53,0x69,0x67,0x6E,0x2C,0x20,0x49,0x6E,0x63,0x2E,0x22,
		0x2C,0x20,0x4F,0x55,0x3D,0x56,0x65,0x72,0x69,0x53,0x69,0x67,0x6E,0x20,0x43,0x6C,0x61,0x73,0x73,0x20,0x31,0x20,0x43,0x41,0x20,0x2D,0x20,0x49,0x6E,0x64,0x69,0x76,0x69,0x64,0x75,0x61,0x6C,0x20,0x53,0x75,0x62,0x73,0x63,0x72,0x69,0x62,0x65,0x72,0x0D,0x0A,0x09,0x4B,0x65,0x79,0x20,0x41,0x6C,0x67,0x6F,0x72,0x69,0x74,0x68,0x6D,0x3A,0x20,0x20,0x31,0x2E,0x32,0x2E,0x38,0x34,0x30,0x2E,0x31,0x31,0x33,0x35,0x34,0x39,0x2E,0x31,0x2E,0x31,0x2E,0x31,0x0D,0x0A,0x09,0x53,0x65,0x72,0x69,0x61,0x6C,0x20,0x4E,0x75,0x6D,
		0x62,0x65,0x72,0x3A,0x20,0x20,0x34,0x34,0x33,0x34,0x37,0x30,0x42,0x39,0x33,0x34,0x43,0x36,0x44,0x45,0x42,0x37,0x36,0x33,0x42,0x30,0x43,0x34,0x43,0x45,0x35,0x45,0x33,0x35,0x30,0x42,0x32,0x30,0x0D,0x0A,0x09,0x4B,0x65,0x79,0x20,0x41,0x6C,0x6F,0x67,0x72,0x69,0x74,0x68,0x6D,0x20,0x50,0x61,0x72,0x61,0x6D,0x65,0x74,0x65,0x72,0x73,0x3A,0x20,0x20,0x30,0x35,0x30,0x30,0x0D,0x0A,0x09,0x50,0x75,0x62,0x6C,0x69,0x63,0x20,0x4B,0x65,0x79,0x3A,0x20,0x20,0x33,0x30,0x34,0x38,0x30,0x32,0x34,0x31,0x30,0x30,0x43,0x35,
		0x38,0x31,0x30,0x37,0x41,0x32,0x45,0x42,0x30,0x46,0x42,0x38,0x46,0x46,0x46,0x38,0x46,0x38,0x31,0x43,0x45,0x45,0x33,0x32,0x46,0x46,0x42,0x46,0x31,0x32,0x33,0x35,0x36,0x41,0x46,0x39,0x36,0x42,0x43,0x38,0x42,0x45,0x32,0x46,0x46,0x42,0x33,0x45,0x41,0x46,0x30,0x34,0x35,0x31,0x34,0x41,0x41,0x43,0x44,0x44,0x31,0x30,0x32,0x39,0x41,0x38,0x43,0x44,0x34,0x30,0x35,0x42,0x36,0x36,0x31,0x45,0x39,0x38,0x45,0x46,0x46,0x32,0x34,0x43,0x37,0x37,0x46,0x41,0x38,0x46,0x38,0x36,0x44,0x31,0x32,0x31,0x36,0x37,0x39,0x32,
		0x34,0x34,0x34,0x41,0x43,0x34,0x38,0x39,0x43,0x39,0x38,0x33,0x43,0x46,0x38,0x38,0x39,0x46,0x36,0x46,0x45,0x32,0x33,0x32,0x33,0x35,0x30,0x32,0x30,0x33,0x30,0x31,0x30,0x30,0x30,0x31,0x0D,0x0A,0x0D,0x0A };
	string tsa = Encoding.Default.GetString (ToStringArray).Replace ("\r\n", Environment.NewLine);
	//Assertion.AssertEquals ("ToString(true)", tsa, x509.ToString (true));
}


// Certificate: basic\KEY.cer
// - X.509v1 self-signed certificate
// - md5RSA, RSA 1024 bits
// - has a unknown OID (looks like email) in it's DN (1.3.6.1.4.1.42.2.11.2.1)
[Test]
public void Certificate10 ()
{
	// certificate can be loaded by constructor
	byte[] cert = { 0x30,0x82,0x02,0x1D,0x30,0x82,0x01,0x86,0x02,0x01,0x14,0x30,0x0D,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x01,0x04,0x05,0x00,0x30,0x58,0x31,0x0B,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x43,0x41,0x31,0x1F,0x30,0x1D,0x06,0x03,0x55,0x04,0x03,0x13,0x16,0x4B,0x65,0x79,0x77,0x69,0x74,0x6E,0x65,0x73,0x73,0x20,0x43,0x61,0x6E,0x61,0x64,0x61,0x20,0x49,0x6E,0x63,0x2E,0x31,0x28,0x30,0x26,0x06,0x0A,0x2B,0x06,0x01,0x04,0x01,0x2A,0x02,0x0B,0x02,0x01,0x13,0x18,0x6B,0x65,0x79,0x77,0x69,0x74,0x6E,0x65,0x73,
		0x73,0x40,0x6B,0x65,0x79,0x77,0x69,0x74,0x6E,0x65,0x73,0x73,0x2E,0x63,0x61,0x30,0x1E,0x17,0x0D,0x39,0x36,0x30,0x35,0x30,0x37,0x30,0x30,0x30,0x30,0x30,0x30,0x5A,0x17,0x0D,0x39,0x39,0x30,0x35,0x30,0x37,0x30,0x30,0x30,0x30,0x30,0x30,0x5A,0x30,0x58,0x31,0x0B,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x43,0x41,0x31,0x1F,0x30,0x1D,0x06,0x03,0x55,0x04,0x03,0x13,0x16,0x4B,0x65,0x79,0x77,0x69,0x74,0x6E,0x65,0x73,0x73,0x20,0x43,0x61,0x6E,0x61,0x64,0x61,0x20,0x49,0x6E,0x63,0x2E,0x31,0x28,0x30,0x26,0x06,
		0x0A,0x2B,0x06,0x01,0x04,0x01,0x2A,0x02,0x0B,0x02,0x01,0x13,0x18,0x6B,0x65,0x79,0x77,0x69,0x74,0x6E,0x65,0x73,0x73,0x40,0x6B,0x65,0x79,0x77,0x69,0x74,0x6E,0x65,0x73,0x73,0x2E,0x63,0x61,0x30,0x81,0x9D,0x30,0x0D,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x01,0x01,0x05,0x00,0x03,0x81,0x8B,0x00,0x30,0x81,0x87,0x02,0x81,0x81,0x00,0xCD,0x23,0xFA,0x2A,0xE1,0xED,0x98,0xF4,0xE9,0xD0,0x93,0x3E,0xD7,0x7A,0x80,0x02,0x4C,0xCC,0xC1,0x02,0xAF,0x5C,0xB6,0x1F,0x7F,0xFA,0x57,0x42,0x6F,0x30,0xD1,0x20,0xC5,0xB5,
		0x21,0x07,0x40,0x2C,0xA9,0x86,0xC2,0xF3,0x64,0x84,0xAE,0x3D,0x85,0x2E,0xED,0x85,0xBD,0x54,0xB0,0x18,0x28,0xEF,0x6A,0xF8,0x1B,0xE7,0x0B,0x16,0x1F,0x93,0x25,0x4F,0xC7,0xF8,0x8E,0xC3,0xB9,0xCA,0x98,0x84,0x0E,0x55,0xD0,0x2F,0xEF,0x78,0x77,0xC5,0x72,0x28,0x5F,0x60,0xBF,0x19,0x2B,0xD1,0x72,0xA2,0xB7,0xD8,0x3F,0xE0,0x97,0x34,0x5A,0x01,0xBD,0x04,0x9C,0xC8,0x78,0x45,0xCD,0x93,0x8D,0x15,0xF2,0x76,0x10,0x11,0xAB,0xB8,0x5B,0x2E,0x9E,0x52,0xDD,0x81,0x3E,0x9C,0x64,0xC8,0x29,0x93,0x02,0x01,0x03,0x30,0x0D,0x06,
		0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x01,0x04,0x05,0x00,0x03,0x81,0x81,0x00,0x32,0x1A,0x35,0xBA,0xBF,0x43,0x27,0xD6,0xB4,0xD4,0xB8,0x76,0xE5,0xE3,0x9B,0x4D,0x6C,0xC0,0x86,0xC9,0x77,0x35,0xBA,0x6B,0x16,0x2D,0x13,0x46,0x4A,0xB0,0x32,0x53,0xA1,0x5B,0x5A,0xE9,0x99,0xE2,0x0C,0x86,0x88,0x17,0x4E,0x0D,0xFE,0x82,0xAC,0x4E,0x47,0xEF,0xFB,0xFF,0x39,0xAC,0xEE,0x35,0xC8,0xFA,0x52,0x37,0x0A,0x49,0xAD,0x59,0xAD,0xE2,0x8A,0xA9,0x1C,0xC6,0x5F,0x1F,0xF8,0x6F,0x73,0x7E,0xCD,0xA0,0x31,0xE8,0x0C,0xBE,0xF5,0x4D,
		0xD9,0xB2,0xAB,0x8A,0x12,0xB6,0x30,0x78,0x68,0x11,0x7C,0x0D,0xF1,0x49,0x4D,0xA3,0xFD,0xB2,0xE9,0xFF,0x1D,0xF0,0x91,0xFA,0x54,0x85,0xFF,0x33,0x90,0xE8,0xC1,0xBF,0xA4,0x9B,0xA4,0x62,0x46,0xBD,0x61,0x12,0x59,0x98,0x41,0x89 };

	X509Certificate x509 = new X509Certificate (cert);
	X509Certificate clone = new X509Certificate (x509);

	Assertion.Assert ("Equals", x509.Equals (clone));
	Assertion.Assert ("Equals", clone.Equals (x509));

	FileStream fs = File.OpenWrite ("temp.cer");
	fs.Write (cert, 0, cert.Length);
	fs.Close ();

	X509Certificate disk = X509Certificate.CreateFromCertFile ("temp.cer");
	Assertion.Assert ("Equals", disk.Equals (clone));
	Assertion.Assert ("Equals", disk.Equals (x509));
	byte[] hash = { 0xE0,0xA0,0x0A,0xBE,0xB2,0xCC,0xA8,0x45,0x40,0xC2,0x1A,0x75,0xE0,0xB3,0xA4,0x06,0xF9,0xA6,0x24,0xAC };
	AssertEquals ("GetCertHash", hash, x509.GetCertHash ());
	Assertion.AssertEquals ("GetCertHashString", "E0A00ABEB2CCA84540C21A75E0B3A406F9A624AC", x509.GetCertHashString ());
#if NET_2_0
	DateTime from = DateTime.ParseExact (x509.GetEffectiveDateString (), "MM/dd/yyyy HH:mm:ss", null).ToUniversalTime ();
	Assert.AreEqual ("05/07/1996 00:00:00", from.ToString (), "GetEffectiveDateString");
	DateTime until = DateTime.ParseExact (x509.GetExpirationDateString (), "MM/dd/yyyy HH:mm:ss", null).ToUniversalTime ();
	Assert.AreEqual ("05/07/1999 00:00:00", until.ToString (), "GetExpirationDateString");
#else
	// fx 1.x has a bug where the returned dates were always in the Seattle time zone
	Assertion.AssertEquals ("GetEffectiveDateString", "05/06/1996 16:00:00", x509.GetEffectiveDateString ());
	Assertion.AssertEquals ("GetExpirationDateString", "05/06/1999 16:00:00", x509.GetExpirationDateString ());
	// which was making it easier to test the dates ;-)
#endif
	Assertion.AssertEquals ("GetFormat", "X509", x509.GetFormat ());
	Assertion.AssertEquals ("GetHashCode", -526382402, x509.GetHashCode ());
	Assertion.AssertEquals ("GetIssuerName", "C=CA, CN=Keywitness Canada Inc., OID.1.3.6.1.4.1.42.2.11.2.1=keywitness@keywitness.ca", x509.GetIssuerName ());
	Assertion.AssertEquals ("GetKeyAlgorithm", "1.2.840.113549.1.1.1", x509.GetKeyAlgorithm ());
	byte[] keyparams = { 0x05,0x00 };
	AssertEquals ("GetKeyAlgorithmParameters", keyparams, x509.GetKeyAlgorithmParameters ());
	Assertion.AssertEquals ("GetKeyAlgorithmParametersString", "0500", x509.GetKeyAlgorithmParametersString ());
	Assertion.AssertEquals ("GetName", "C=CA, CN=Keywitness Canada Inc., OID.1.3.6.1.4.1.42.2.11.2.1=keywitness@keywitness.ca", x509.GetName ());
	byte[] pubkey = { 0x30,0x81,0x87,0x02,0x81,0x81,0x00,0xCD,0x23,0xFA,0x2A,0xE1,0xED,0x98,0xF4,0xE9,0xD0,0x93,0x3E,0xD7,0x7A,0x80,0x02,0x4C,0xCC,0xC1,0x02,0xAF,0x5C,0xB6,0x1F,0x7F,0xFA,0x57,0x42,0x6F,0x30,0xD1,0x20,0xC5,0xB5,0x21,0x07,0x40,0x2C,0xA9,0x86,0xC2,0xF3,0x64,0x84,0xAE,0x3D,0x85,0x2E,0xED,0x85,0xBD,0x54,0xB0,0x18,0x28,0xEF,0x6A,0xF8,0x1B,0xE7,0x0B,0x16,0x1F,0x93,0x25,0x4F,0xC7,0xF8,0x8E,0xC3,0xB9,0xCA,0x98,0x84,0x0E,0x55,0xD0,0x2F,0xEF,0x78,0x77,0xC5,0x72,0x28,0x5F,0x60,0xBF,0x19,0x2B,0xD1,0x72,0xA2,0xB7,0xD8,
		0x3F,0xE0,0x97,0x34,0x5A,0x01,0xBD,0x04,0x9C,0xC8,0x78,0x45,0xCD,0x93,0x8D,0x15,0xF2,0x76,0x10,0x11,0xAB,0xB8,0x5B,0x2E,0x9E,0x52,0xDD,0x81,0x3E,0x9C,0x64,0xC8,0x29,0x93,0x02,0x01,0x03 };
	AssertEquals ("GetPublicKey", pubkey, x509.GetPublicKey ());
	Assertion.AssertEquals ("GetPublicKeyString", "30818702818100CD23FA2AE1ED98F4E9D0933ED77A80024CCCC102AF5CB61F7FFA57426F30D120C5B52107402CA986C2F36484AE3D852EED85BD54B01828EF6AF81BE70B161F93254FC7F88EC3B9CA98840E55D02FEF7877C572285F60BF192BD172A2B7D83FE097345A01BD049CC87845CD938D15F2761011ABB85B2E9E52DD813E9C64C82993020103", x509.GetPublicKeyString ());
	AssertEquals ("GetRawCertData", cert, x509.GetRawCertData ());
	Assertion.AssertEquals ("GetRawCertDataString", ToString (cert), x509.GetRawCertDataString ());
	byte[] serial = { 0x14 };
	AssertEquals ("GetSerialNumber", serial, x509.GetSerialNumber ());
	Assertion.AssertEquals ("GetSerialNumberString", "14", x509.GetSerialNumberString ());
	byte[] ToStringArray = { 0x43,0x45,0x52,0x54,0x49,0x46,0x49,0x43,0x41,0x54,0x45,0x3A,0x0D,0x0A,0x09,0x46,0x6F,0x72,0x6D,0x61,0x74,0x3A,0x20,0x20,0x58,0x35,0x30,0x39,0x0D,0x0A,0x09,0x4E,0x61,0x6D,0x65,0x3A,0x20,0x20,0x43,0x3D,0x43,0x41,0x2C,0x20,0x43,0x4E,0x3D,0x4B,0x65,0x79,0x77,0x69,0x74,0x6E,0x65,0x73,0x73,0x20,0x43,0x61,0x6E,0x61,0x64,0x61,0x20,0x49,0x6E,0x63,0x2E,0x2C,0x20,0x4F,0x49,0x44,0x2E,0x31,0x2E,0x33,0x2E,0x36,0x2E,0x31,0x2E,0x34,0x2E,0x31,0x2E,0x34,0x32,0x2E,0x32,0x2E,0x31,0x31,0x2E,0x32,0x2E,0x31,0x3D,0x6B,0x65,
		0x79,0x77,0x69,0x74,0x6E,0x65,0x73,0x73,0x40,0x6B,0x65,0x79,0x77,0x69,0x74,0x6E,0x65,0x73,0x73,0x2E,0x63,0x61,0x0D,0x0A,0x09,0x49,0x73,0x73,0x75,0x69,0x6E,0x67,0x20,0x43,0x41,0x3A,0x20,0x20,0x43,0x3D,0x43,0x41,0x2C,0x20,0x43,0x4E,0x3D,0x4B,0x65,0x79,0x77,0x69,0x74,0x6E,0x65,0x73,0x73,0x20,0x43,0x61,0x6E,0x61,0x64,0x61,0x20,0x49,0x6E,0x63,0x2E,0x2C,0x20,0x4F,0x49,0x44,0x2E,0x31,0x2E,0x33,0x2E,0x36,0x2E,0x31,0x2E,0x34,0x2E,0x31,0x2E,0x34,0x32,0x2E,0x32,0x2E,0x31,0x31,0x2E,0x32,0x2E,0x31,0x3D,0x6B,
		0x65,0x79,0x77,0x69,0x74,0x6E,0x65,0x73,0x73,0x40,0x6B,0x65,0x79,0x77,0x69,0x74,0x6E,0x65,0x73,0x73,0x2E,0x63,0x61,0x0D,0x0A,0x09,0x4B,0x65,0x79,0x20,0x41,0x6C,0x67,0x6F,0x72,0x69,0x74,0x68,0x6D,0x3A,0x20,0x20,0x31,0x2E,0x32,0x2E,0x38,0x34,0x30,0x2E,0x31,0x31,0x33,0x35,0x34,0x39,0x2E,0x31,0x2E,0x31,0x2E,0x31,0x0D,0x0A,0x09,0x53,0x65,0x72,0x69,0x61,0x6C,0x20,0x4E,0x75,0x6D,0x62,0x65,0x72,0x3A,0x20,0x20,0x31,0x34,0x0D,0x0A,0x09,0x4B,0x65,0x79,0x20,0x41,0x6C,0x6F,0x67,0x72,0x69,0x74,0x68,0x6D,0x20,
		0x50,0x61,0x72,0x61,0x6D,0x65,0x74,0x65,0x72,0x73,0x3A,0x20,0x20,0x30,0x35,0x30,0x30,0x0D,0x0A,0x09,0x50,0x75,0x62,0x6C,0x69,0x63,0x20,0x4B,0x65,0x79,0x3A,0x20,0x20,0x33,0x30,0x38,0x31,0x38,0x37,0x30,0x32,0x38,0x31,0x38,0x31,0x30,0x30,0x43,0x44,0x32,0x33,0x46,0x41,0x32,0x41,0x45,0x31,0x45,0x44,0x39,0x38,0x46,0x34,0x45,0x39,0x44,0x30,0x39,0x33,0x33,0x45,0x44,0x37,0x37,0x41,0x38,0x30,0x30,0x32,0x34,0x43,0x43,0x43,0x43,0x31,0x30,0x32,0x41,0x46,0x35,0x43,0x42,0x36,0x31,0x46,0x37,0x46,0x46,0x41,0x35,
		0x37,0x34,0x32,0x36,0x46,0x33,0x30,0x44,0x31,0x32,0x30,0x43,0x35,0x42,0x35,0x32,0x31,0x30,0x37,0x34,0x30,0x32,0x43,0x41,0x39,0x38,0x36,0x43,0x32,0x46,0x33,0x36,0x34,0x38,0x34,0x41,0x45,0x33,0x44,0x38,0x35,0x32,0x45,0x45,0x44,0x38,0x35,0x42,0x44,0x35,0x34,0x42,0x30,0x31,0x38,0x32,0x38,0x45,0x46,0x36,0x41,0x46,0x38,0x31,0x42,0x45,0x37,0x30,0x42,0x31,0x36,0x31,0x46,0x39,0x33,0x32,0x35,0x34,0x46,0x43,0x37,0x46,0x38,0x38,0x45,0x43,0x33,0x42,0x39,0x43,0x41,0x39,0x38,0x38,0x34,0x30,0x45,0x35,0x35,0x44,
		0x30,0x32,0x46,0x45,0x46,0x37,0x38,0x37,0x37,0x43,0x35,0x37,0x32,0x32,0x38,0x35,0x46,0x36,0x30,0x42,0x46,0x31,0x39,0x32,0x42,0x44,0x31,0x37,0x32,0x41,0x32,0x42,0x37,0x44,0x38,0x33,0x46,0x45,0x30,0x39,0x37,0x33,0x34,0x35,0x41,0x30,0x31,0x42,0x44,0x30,0x34,0x39,0x43,0x43,0x38,0x37,0x38,0x34,0x35,0x43,0x44,0x39,0x33,0x38,0x44,0x31,0x35,0x46,0x32,0x37,0x36,0x31,0x30,0x31,0x31,0x41,0x42,0x42,0x38,0x35,0x42,0x32,0x45,0x39,0x45,0x35,0x32,0x44,0x44,0x38,0x31,0x33,0x45,0x39,0x43,0x36,0x34,0x43,0x38,0x32,
		0x39,0x39,0x33,0x30,0x32,0x30,0x31,0x30,0x33,0x0D,0x0A,0x0D,0x0A };
	string tsa = Encoding.Default.GetString (ToStringArray).Replace ("\r\n", Environment.NewLine);
	//Assertion.AssertEquals ("ToString(true)", tsa, x509.ToString (true));
}


// Certificate: basic\GTE.cer
// - X.509v1 self-signed certificate
// - md5RSA, RSA 1024 bits
// - Valid from/to doesn't include seconds!
[Test]
public void Certificate11 ()
{
	// certificate can be loaded by constructor
	byte[] cert = { 0x30,0x82,0x01,0xF5,0x30,0x82,0x01,0x5E,0x02,0x01,0x00,0x30,0x0D,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x01,0x04,0x05,0x00,0x30,0x45,0x31,0x0B,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x55,0x53,0x31,0x18,0x30,0x16,0x06,0x03,0x55,0x04,0x0A,0x13,0x0F,0x47,0x54,0x45,0x20,0x43,0x6F,0x72,0x70,0x6F,0x72,0x61,0x74,0x69,0x6F,0x6E,0x31,0x1C,0x30,0x1A,0x06,0x03,0x55,0x04,0x03,0x13,0x13,0x47,0x54,0x45,0x20,0x43,0x79,0x62,0x65,0x72,0x54,0x72,0x75,0x73,0x74,0x20,0x52,0x6F,0x6F,0x74,0x30,0x1A,0x17,0x0B,
		0x39,0x36,0x30,0x32,0x32,0x33,0x31,0x39,0x31,0x35,0x5A,0x17,0x0B,0x39,0x39,0x31,0x32,0x33,0x31,0x32,0x33,0x35,0x39,0x5A,0x30,0x45,0x31,0x0B,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x55,0x53,0x31,0x18,0x30,0x16,0x06,0x03,0x55,0x04,0x0A,0x13,0x0F,0x47,0x54,0x45,0x20,0x43,0x6F,0x72,0x70,0x6F,0x72,0x61,0x74,0x69,0x6F,0x6E,0x31,0x1C,0x30,0x1A,0x06,0x03,0x55,0x04,0x03,0x13,0x13,0x47,0x54,0x45,0x20,0x43,0x79,0x62,0x65,0x72,0x54,0x72,0x75,0x73,0x74,0x20,0x52,0x6F,0x6F,0x74,0x30,0x81,0x9F,0x30,0x0D,
		0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x01,0x01,0x05,0x00,0x03,0x81,0x8D,0x00,0x30,0x81,0x89,0x02,0x81,0x81,0x00,0xB8,0xE6,0x4F,0xBA,0xDB,0x98,0x7C,0x71,0x7C,0xAF,0x44,0xB7,0xD3,0x0F,0x46,0xD9,0x64,0xE5,0x93,0xC1,0x42,0x8E,0xC7,0xBA,0x49,0x8D,0x35,0x2D,0x7A,0xE7,0x8B,0xBD,0xE5,0x05,0x31,0x59,0xC6,0xB1,0x2F,0x0A,0x0C,0xFB,0x9F,0xA7,0x3F,0xA2,0x09,0x66,0x84,0x56,0x1E,0x37,0x29,0x1B,0x87,0xE9,0x7E,0x0C,0xCA,0x9A,0x9F,0xA5,0x7F,0xF5,0x15,0x94,0xA3,0xD5,0xA2,0x46,0x82,0xD8,0x68,0x4C,0xD1,0x37,
		0x15,0x06,0x68,0xAF,0xBD,0xF8,0xB0,0xB3,0xF0,0x29,0xF5,0x95,0x5A,0x09,0x16,0x61,0x77,0x0A,0x22,0x25,0xD4,0x4F,0x45,0xAA,0xC7,0xBD,0xE5,0x96,0xDF,0xF9,0xD4,0xA8,0x8E,0x42,0xCC,0x24,0xC0,0x1E,0x91,0x27,0x4A,0xB5,0x6D,0x06,0x80,0x63,0x39,0xC4,0xA2,0x5E,0x38,0x03,0x02,0x03,0x01,0x00,0x01,0x30,0x0D,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x01,0x04,0x05,0x00,0x03,0x81,0x81,0x00,0x34,0x9B,0x09,0x05,0x9D,0x42,0x48,0x7F,0x7C,0x24,0x8E,0xC2,0x88,0x65,0x78,0x6B,0x4D,0x2A,0x0E,0xAF,0xED,0xED,0xF0,0x6A,
		0x2D,0x37,0xEB,0x57,0x51,0x0E,0x8A,0xAD,0x2D,0x13,0x02,0xA1,0x40,0x6A,0x41,0xF6,0x69,0x72,0xBF,0x0F,0xA0,0x4E,0x55,0xE7,0x82,0xA6,0x57,0x91,0x8B,0x5D,0x43,0x08,0xC9,0xF8,0x71,0x07,0x82,0xE3,0x3E,0x14,0x01,0x24,0x0C,0x39,0xEB,0x61,0x22,0xF7,0xA2,0x0B,0x42,0x23,0x14,0xCA,0x10,0x18,0x36,0x0A,0x05,0xD1,0xC8,0x4A,0xF1,0x0E,0xB3,0xFD,0x00,0xD0,0xC6,0xEE,0x55,0xF2,0x4C,0x62,0xB4,0x19,0x73,0x1F,0x48,0x18,0xD8,0xB0,0x4E,0xB0,0x75,0xBC,0xB9,0x84,0x8B,0x1F,0x96,0x8A,0x83,0xF1,0xA0,0x1A,0xA2,0xED,0xFF,0x87,
		0x8C,0xA3,0x4F,0xA2 };

	X509Certificate x509 = new X509Certificate (cert);
	X509Certificate clone = new X509Certificate (x509);

	Assertion.Assert ("Equals", x509.Equals (clone));
	Assertion.Assert ("Equals", clone.Equals (x509));

	FileStream fs = File.OpenWrite ("temp.cer");
	fs.Write (cert, 0, cert.Length);
	fs.Close ();

	X509Certificate disk = X509Certificate.CreateFromCertFile ("temp.cer");
	Assertion.Assert ("Equals", disk.Equals (clone));
	Assertion.Assert ("Equals", disk.Equals (x509));
	byte[] hash = { 0xA2,0x75,0xE0,0x26,0xAC,0xD8,0x54,0x79,0x4A,0x4A,0xA2,0xCB,0x53,0xF6,0x62,0x33,0x12,0x9C,0x55,0xB6 };
	AssertEquals ("GetCertHash", hash, x509.GetCertHash ());
	Assertion.AssertEquals ("GetCertHashString", "A275E026ACD854794A4AA2CB53F66233129C55B6", x509.GetCertHashString ());
#if NET_2_0
	DateTime from = DateTime.ParseExact (x509.GetEffectiveDateString (), "MM/dd/yyyy HH:mm:ss", null).ToUniversalTime ();
	Assert.AreEqual ("02/23/1996 19:15:00", from.ToString (), "GetEffectiveDateString");
	DateTime until = DateTime.ParseExact (x509.GetExpirationDateString (), "MM/dd/yyyy HH:mm:ss", null).ToUniversalTime ();
	Assert.AreEqual ("12/31/1999 23:59:00", until.ToString (), "GetExpirationDateString");
#else
	// fx 1.x has a bug where the returned dates were always in the Seattle time zone
	Assertion.AssertEquals ("GetEffectiveDateString", "02/23/1996 11:15:00", x509.GetEffectiveDateString ());
	Assertion.AssertEquals ("GetExpirationDateString", "12/31/1999 15:59:00", x509.GetExpirationDateString ());
	// which was making it easier to test the dates ;-)
#endif
	Assertion.AssertEquals ("GetFormat", "X509", x509.GetFormat ());
	Assertion.AssertEquals ("GetHashCode", -1569333210, x509.GetHashCode ());
	Assertion.AssertEquals ("GetIssuerName", "C=US, O=GTE Corporation, CN=GTE CyberTrust Root", x509.GetIssuerName ());
	Assertion.AssertEquals ("GetKeyAlgorithm", "1.2.840.113549.1.1.1", x509.GetKeyAlgorithm ());
	byte[] keyparams = { 0x05,0x00 };
	AssertEquals ("GetKeyAlgorithmParameters", keyparams, x509.GetKeyAlgorithmParameters ());
	Assertion.AssertEquals ("GetKeyAlgorithmParametersString", "0500", x509.GetKeyAlgorithmParametersString ());
	Assertion.AssertEquals ("GetName", "C=US, O=GTE Corporation, CN=GTE CyberTrust Root", x509.GetName ());
	byte[] pubkey = { 0x30,0x81,0x89,0x02,0x81,0x81,0x00,0xB8,0xE6,0x4F,0xBA,0xDB,0x98,0x7C,0x71,0x7C,0xAF,0x44,0xB7,0xD3,0x0F,0x46,0xD9,0x64,0xE5,0x93,0xC1,0x42,0x8E,0xC7,0xBA,0x49,0x8D,0x35,0x2D,0x7A,0xE7,0x8B,0xBD,0xE5,0x05,0x31,0x59,0xC6,0xB1,0x2F,0x0A,0x0C,0xFB,0x9F,0xA7,0x3F,0xA2,0x09,0x66,0x84,0x56,0x1E,0x37,0x29,0x1B,0x87,0xE9,0x7E,0x0C,0xCA,0x9A,0x9F,0xA5,0x7F,0xF5,0x15,0x94,0xA3,0xD5,0xA2,0x46,0x82,0xD8,0x68,0x4C,0xD1,0x37,0x15,0x06,0x68,0xAF,0xBD,0xF8,0xB0,0xB3,0xF0,0x29,0xF5,0x95,0x5A,0x09,0x16,0x61,0x77,0x0A,
		0x22,0x25,0xD4,0x4F,0x45,0xAA,0xC7,0xBD,0xE5,0x96,0xDF,0xF9,0xD4,0xA8,0x8E,0x42,0xCC,0x24,0xC0,0x1E,0x91,0x27,0x4A,0xB5,0x6D,0x06,0x80,0x63,0x39,0xC4,0xA2,0x5E,0x38,0x03,0x02,0x03,0x01,0x00,0x01 };
	AssertEquals ("GetPublicKey", pubkey, x509.GetPublicKey ());
	Assertion.AssertEquals ("GetPublicKeyString", "30818902818100B8E64FBADB987C717CAF44B7D30F46D964E593C1428EC7BA498D352D7AE78BBDE5053159C6B12F0A0CFB9FA73FA2096684561E37291B87E97E0CCA9A9FA57FF51594A3D5A24682D8684CD137150668AFBDF8B0B3F029F5955A091661770A2225D44F45AAC7BDE596DFF9D4A88E42CC24C01E91274AB56D06806339C4A25E38030203010001", x509.GetPublicKeyString ());
	AssertEquals ("GetRawCertData", cert, x509.GetRawCertData ());
	Assertion.AssertEquals ("GetRawCertDataString", ToString (cert), x509.GetRawCertDataString ());
	byte[] serial = { 0x00 };
	AssertEquals ("GetSerialNumber", serial, x509.GetSerialNumber ());
	Assertion.AssertEquals ("GetSerialNumberString", "00", x509.GetSerialNumberString ());
	byte[] ToStringArray = { 0x43,0x45,0x52,0x54,0x49,0x46,0x49,0x43,0x41,0x54,0x45,0x3A,0x0D,0x0A,0x09,0x46,0x6F,0x72,0x6D,0x61,0x74,0x3A,0x20,0x20,0x58,0x35,0x30,0x39,0x0D,0x0A,0x09,0x4E,0x61,0x6D,0x65,0x3A,0x20,0x20,0x43,0x3D,0x55,0x53,0x2C,0x20,0x4F,0x3D,0x47,0x54,0x45,0x20,0x43,0x6F,0x72,0x70,0x6F,0x72,0x61,0x74,0x69,0x6F,0x6E,0x2C,0x20,0x43,0x4E,0x3D,0x47,0x54,0x45,0x20,0x43,0x79,0x62,0x65,0x72,0x54,0x72,0x75,0x73,0x74,0x20,0x52,0x6F,0x6F,0x74,0x0D,0x0A,0x09,0x49,0x73,0x73,0x75,0x69,0x6E,0x67,0x20,0x43,0x41,0x3A,0x20,0x20,
		0x43,0x3D,0x55,0x53,0x2C,0x20,0x4F,0x3D,0x47,0x54,0x45,0x20,0x43,0x6F,0x72,0x70,0x6F,0x72,0x61,0x74,0x69,0x6F,0x6E,0x2C,0x20,0x43,0x4E,0x3D,0x47,0x54,0x45,0x20,0x43,0x79,0x62,0x65,0x72,0x54,0x72,0x75,0x73,0x74,0x20,0x52,0x6F,0x6F,0x74,0x0D,0x0A,0x09,0x4B,0x65,0x79,0x20,0x41,0x6C,0x67,0x6F,0x72,0x69,0x74,0x68,0x6D,0x3A,0x20,0x20,0x31,0x2E,0x32,0x2E,0x38,0x34,0x30,0x2E,0x31,0x31,0x33,0x35,0x34,0x39,0x2E,0x31,0x2E,0x31,0x2E,0x31,0x0D,0x0A,0x09,0x53,0x65,0x72,0x69,0x61,0x6C,0x20,0x4E,0x75,0x6D,0x62,
		0x65,0x72,0x3A,0x20,0x20,0x30,0x30,0x0D,0x0A,0x09,0x4B,0x65,0x79,0x20,0x41,0x6C,0x6F,0x67,0x72,0x69,0x74,0x68,0x6D,0x20,0x50,0x61,0x72,0x61,0x6D,0x65,0x74,0x65,0x72,0x73,0x3A,0x20,0x20,0x30,0x35,0x30,0x30,0x0D,0x0A,0x09,0x50,0x75,0x62,0x6C,0x69,0x63,0x20,0x4B,0x65,0x79,0x3A,0x20,0x20,0x33,0x30,0x38,0x31,0x38,0x39,0x30,0x32,0x38,0x31,0x38,0x31,0x30,0x30,0x42,0x38,0x45,0x36,0x34,0x46,0x42,0x41,0x44,0x42,0x39,0x38,0x37,0x43,0x37,0x31,0x37,0x43,0x41,0x46,0x34,0x34,0x42,0x37,0x44,0x33,0x30,0x46,0x34,
		0x36,0x44,0x39,0x36,0x34,0x45,0x35,0x39,0x33,0x43,0x31,0x34,0x32,0x38,0x45,0x43,0x37,0x42,0x41,0x34,0x39,0x38,0x44,0x33,0x35,0x32,0x44,0x37,0x41,0x45,0x37,0x38,0x42,0x42,0x44,0x45,0x35,0x30,0x35,0x33,0x31,0x35,0x39,0x43,0x36,0x42,0x31,0x32,0x46,0x30,0x41,0x30,0x43,0x46,0x42,0x39,0x46,0x41,0x37,0x33,0x46,0x41,0x32,0x30,0x39,0x36,0x36,0x38,0x34,0x35,0x36,0x31,0x45,0x33,0x37,0x32,0x39,0x31,0x42,0x38,0x37,0x45,0x39,0x37,0x45,0x30,0x43,0x43,0x41,0x39,0x41,0x39,0x46,0x41,0x35,0x37,0x46,0x46,0x35,0x31,
		0x35,0x39,0x34,0x41,0x33,0x44,0x35,0x41,0x32,0x34,0x36,0x38,0x32,0x44,0x38,0x36,0x38,0x34,0x43,0x44,0x31,0x33,0x37,0x31,0x35,0x30,0x36,0x36,0x38,0x41,0x46,0x42,0x44,0x46,0x38,0x42,0x30,0x42,0x33,0x46,0x30,0x32,0x39,0x46,0x35,0x39,0x35,0x35,0x41,0x30,0x39,0x31,0x36,0x36,0x31,0x37,0x37,0x30,0x41,0x32,0x32,0x32,0x35,0x44,0x34,0x34,0x46,0x34,0x35,0x41,0x41,0x43,0x37,0x42,0x44,0x45,0x35,0x39,0x36,0x44,0x46,0x46,0x39,0x44,0x34,0x41,0x38,0x38,0x45,0x34,0x32,0x43,0x43,0x32,0x34,0x43,0x30,0x31,0x45,0x39,
		0x31,0x32,0x37,0x34,0x41,0x42,0x35,0x36,0x44,0x30,0x36,0x38,0x30,0x36,0x33,0x33,0x39,0x43,0x34,0x41,0x32,0x35,0x45,0x33,0x38,0x30,0x33,0x30,0x32,0x30,0x33,0x30,0x31,0x30,0x30,0x30,0x31,0x0D,0x0A,0x0D,0x0A };
	string tsa = Encoding.Default.GetString (ToStringArray).Replace ("\r\n", Environment.NewLine);
	//Assertion.AssertEquals ("ToString(true)", tsa, x509.ToString (true));
}


// Certificate: basic\ms_spc.cer
// no comments
[Test]
public void Certificate12 ()
{
	// certificate can be loaded by constructor
	byte[] cert = { 0x30,0x82,0x05,0x0F,0x30,0x82,0x03,0xF7,0xA0,0x03,0x02,0x01,0x02,0x02,0x0A,0x61,0x07,0x11,0x43,0x00,0x00,0x00,0x00,0x00,0x34,0x30,0x0D,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x01,0x05,0x05,0x00,0x30,0x81,0xA6,0x31,0x0B,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x55,0x53,0x31,0x13,0x30,0x11,0x06,0x03,0x55,0x04,0x08,0x13,0x0A,0x57,0x61,0x73,0x68,0x69,0x6E,0x67,0x74,0x6F,0x6E,0x31,0x10,0x30,0x0E,0x06,0x03,0x55,0x04,0x07,0x13,0x07,0x52,0x65,0x64,0x6D,0x6F,0x6E,0x64,0x31,0x1E,0x30,0x1C,0x06,0x03,
		0x55,0x04,0x0A,0x13,0x15,0x4D,0x69,0x63,0x72,0x6F,0x73,0x6F,0x66,0x74,0x20,0x43,0x6F,0x72,0x70,0x6F,0x72,0x61,0x74,0x69,0x6F,0x6E,0x31,0x2B,0x30,0x29,0x06,0x03,0x55,0x04,0x0B,0x13,0x22,0x43,0x6F,0x70,0x79,0x72,0x69,0x67,0x68,0x74,0x20,0x28,0x63,0x29,0x20,0x32,0x30,0x30,0x30,0x20,0x4D,0x69,0x63,0x72,0x6F,0x73,0x6F,0x66,0x74,0x20,0x43,0x6F,0x72,0x70,0x2E,0x31,0x23,0x30,0x21,0x06,0x03,0x55,0x04,0x03,0x13,0x1A,0x4D,0x69,0x63,0x72,0x6F,0x73,0x6F,0x66,0x74,0x20,0x43,0x6F,0x64,0x65,0x20,0x53,0x69,0x67,
		0x6E,0x69,0x6E,0x67,0x20,0x50,0x43,0x41,0x30,0x1E,0x17,0x0D,0x30,0x32,0x30,0x35,0x32,0x35,0x30,0x30,0x35,0x35,0x34,0x38,0x5A,0x17,0x0D,0x30,0x33,0x31,0x31,0x32,0x35,0x30,0x31,0x30,0x35,0x34,0x38,0x5A,0x30,0x81,0xA1,0x31,0x0B,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x55,0x53,0x31,0x13,0x30,0x11,0x06,0x03,0x55,0x04,0x08,0x13,0x0A,0x57,0x61,0x73,0x68,0x69,0x6E,0x67,0x74,0x6F,0x6E,0x31,0x10,0x30,0x0E,0x06,0x03,0x55,0x04,0x07,0x13,0x07,0x52,0x65,0x64,0x6D,0x6F,0x6E,0x64,0x31,0x1E,0x30,0x1C,0x06,
		0x03,0x55,0x04,0x0A,0x13,0x15,0x4D,0x69,0x63,0x72,0x6F,0x73,0x6F,0x66,0x74,0x20,0x43,0x6F,0x72,0x70,0x6F,0x72,0x61,0x74,0x69,0x6F,0x6E,0x31,0x2B,0x30,0x29,0x06,0x03,0x55,0x04,0x0B,0x13,0x22,0x43,0x6F,0x70,0x79,0x72,0x69,0x67,0x68,0x74,0x20,0x28,0x63,0x29,0x20,0x32,0x30,0x30,0x32,0x20,0x4D,0x69,0x63,0x72,0x6F,0x73,0x6F,0x66,0x74,0x20,0x43,0x6F,0x72,0x70,0x2E,0x31,0x1E,0x30,0x1C,0x06,0x03,0x55,0x04,0x03,0x13,0x15,0x4D,0x69,0x63,0x72,0x6F,0x73,0x6F,0x66,0x74,0x20,0x43,0x6F,0x72,0x70,0x6F,0x72,0x61,
		0x74,0x69,0x6F,0x6E,0x30,0x82,0x01,0x22,0x30,0x0D,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x01,0x01,0x05,0x00,0x03,0x82,0x01,0x0F,0x00,0x30,0x82,0x01,0x0A,0x02,0x82,0x01,0x01,0x00,0xAA,0x99,0xBD,0x39,0xA8,0x18,0x27,0xF4,0x2B,0x3D,0x0B,0x4C,0x3F,0x7C,0x77,0x2E,0xA7,0xCB,0xB5,0xD1,0x8C,0x0D,0xC2,0x3A,0x74,0xD7,0x93,0xB5,0xE0,0xA0,0x4B,0x3F,0x59,0x5E,0xCE,0x45,0x4F,0x9A,0x79,0x29,0xF1,0x49,0xCC,0x1A,0x47,0xEE,0x55,0xC2,0x08,0x3E,0x12,0x20,0xF8,0x55,0xF2,0xEE,0x5F,0xD3,0xE0,0xCA,0x96,0xBC,0x30,
		0xDE,0xFE,0x58,0xC8,0x27,0x32,0xD0,0x85,0x54,0xE8,0xF0,0x91,0x10,0xBB,0xF3,0x2B,0xBE,0x19,0xE5,0x03,0x9B,0x0B,0x86,0x1D,0xF3,0xB0,0x39,0x8C,0xB8,0xFD,0x0B,0x1D,0x3C,0x73,0x26,0xAC,0x57,0x2B,0xCA,0x29,0xA2,0x15,0x90,0x82,0x15,0xE2,0x77,0xA3,0x40,0x52,0x03,0x8B,0x9D,0xC2,0x70,0xBA,0x1F,0xE9,0x34,0xF6,0xF3,0x35,0x92,0x4E,0x55,0x83,0xF8,0xDA,0x30,0xB6,0x20,0xDE,0x57,0x06,0xB5,0x5A,0x42,0x06,0xDE,0x59,0xCB,0xF2,0xDF,0xA6,0xBD,0x15,0x47,0x71,0x19,0x25,0x23,0xD2,0xCB,0x6F,0x9B,0x19,0x79,0xDF,0x6A,0x5B,
		0xF1,0x76,0x05,0x79,0x29,0xFC,0xC3,0x56,0xCA,0x8F,0x44,0x08,0x85,0x55,0x8A,0xCB,0xC8,0x0F,0x46,0x4B,0x55,0xCB,0x8C,0x96,0x77,0x4A,0x87,0xE8,0xA9,0x41,0x06,0xC7,0xFF,0x0D,0xE9,0x68,0x57,0x63,0x72,0xC3,0x69,0x57,0xB4,0x43,0xCF,0x32,0x3A,0x30,0xDC,0x1B,0xE9,0xD5,0x43,0x26,0x2A,0x79,0xFE,0x95,0xDB,0x22,0x67,0x24,0xC9,0x2F,0xD0,0x34,0xE3,0xE6,0xFB,0x51,0x49,0x86,0xB8,0x3C,0xD0,0x25,0x5F,0xD6,0xEC,0x9E,0x03,0x61,0x87,0xA9,0x68,0x40,0xC7,0xF8,0xE2,0x03,0xE6,0xCF,0x05,0x02,0x03,0x01,0x00,0x01,0xA3,0x82,
		0x01,0x40,0x30,0x82,0x01,0x3C,0x30,0x0E,0x06,0x03,0x55,0x1D,0x0F,0x01,0x01,0xFF,0x04,0x04,0x03,0x02,0x06,0xC0,0x30,0x13,0x06,0x03,0x55,0x1D,0x25,0x04,0x0C,0x30,0x0A,0x06,0x08,0x2B,0x06,0x01,0x05,0x05,0x07,0x03,0x03,0x30,0x1D,0x06,0x03,0x55,0x1D,0x0E,0x04,0x16,0x04,0x14,0x6B,0xC8,0xC6,0x51,0x20,0xF0,0xB4,0x2F,0xD3,0xA0,0xB6,0xAE,0x7F,0x5E,0x26,0xB2,0xB8,0x87,0x52,0x29,0x30,0x81,0xA9,0x06,0x03,0x55,0x1D,0x23,0x04,0x81,0xA1,0x30,0x81,0x9E,0x80,0x14,0x29,0x5C,0xB9,0x1B,0xB6,0xCD,0x33,0xEE,0xBB,0x9E,
		0x59,0x7D,0xF7,0xE5,0xCA,0x2E,0xC4,0x0D,0x34,0x28,0xA1,0x74,0xA4,0x72,0x30,0x70,0x31,0x2B,0x30,0x29,0x06,0x03,0x55,0x04,0x0B,0x13,0x22,0x43,0x6F,0x70,0x79,0x72,0x69,0x67,0x68,0x74,0x20,0x28,0x63,0x29,0x20,0x31,0x39,0x39,0x37,0x20,0x4D,0x69,0x63,0x72,0x6F,0x73,0x6F,0x66,0x74,0x20,0x43,0x6F,0x72,0x70,0x2E,0x31,0x1E,0x30,0x1C,0x06,0x03,0x55,0x04,0x0B,0x13,0x15,0x4D,0x69,0x63,0x72,0x6F,0x73,0x6F,0x66,0x74,0x20,0x43,0x6F,0x72,0x70,0x6F,0x72,0x61,0x74,0x69,0x6F,0x6E,0x31,0x21,0x30,0x1F,0x06,0x03,0x55,
		0x04,0x03,0x13,0x18,0x4D,0x69,0x63,0x72,0x6F,0x73,0x6F,0x66,0x74,0x20,0x52,0x6F,0x6F,0x74,0x20,0x41,0x75,0x74,0x68,0x6F,0x72,0x69,0x74,0x79,0x82,0x10,0x6A,0x0B,0x99,0x4F,0xC0,0x00,0xDE,0xAA,0x11,0xD4,0xD8,0x40,0x9A,0xA8,0xBE,0xE6,0x30,0x4A,0x06,0x03,0x55,0x1D,0x1F,0x04,0x43,0x30,0x41,0x30,0x3F,0xA0,0x3D,0xA0,0x3B,0x86,0x39,0x68,0x74,0x74,0x70,0x3A,0x2F,0x2F,0x63,0x72,0x6C,0x2E,0x6D,0x69,0x63,0x72,0x6F,0x73,0x6F,0x66,0x74,0x2E,0x63,0x6F,0x6D,0x2F,0x70,0x6B,0x69,0x2F,0x63,0x72,0x6C,0x2F,0x70,0x72,
		0x6F,0x64,0x75,0x63,0x74,0x73,0x2F,0x43,0x6F,0x64,0x65,0x53,0x69,0x67,0x6E,0x50,0x43,0x41,0x2E,0x63,0x72,0x6C,0x30,0x0D,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x01,0x05,0x05,0x00,0x03,0x82,0x01,0x01,0x00,0x35,0x23,0xFD,0x13,0x54,0xFC,0xE9,0xDC,0xF0,0xDD,0x0C,0x14,0x7A,0xFA,0xA7,0xB3,0xCE,0xFD,0xA7,0x3A,0xC8,0xBA,0xE5,0xE7,0xF6,0x03,0xFB,0x53,0xDB,0xA7,0x99,0xA9,0xA0,0x9B,0x36,0x9C,0x03,0xEB,0x82,0x47,0x1C,0x21,0xBD,0x14,0xCB,0xE7,0x67,0x40,0x09,0xC7,0x16,0x91,0x02,0x55,0xCE,0x43,0x42,0xB4,
		0xCD,0x1B,0x5D,0xB0,0xF3,0x32,0x04,0x3D,0x12,0xE5,0x1D,0xA7,0x07,0xA7,0x8F,0xA3,0x7E,0x45,0x55,0x76,0x1B,0x96,0x95,0x91,0x69,0xF0,0xDD,0x38,0xF3,0x48,0x89,0xEF,0x70,0x40,0xB7,0xDB,0xB5,0x55,0x80,0xC0,0x03,0xC4,0x2E,0xB6,0x28,0xDC,0x0A,0x82,0x0E,0xC7,0x43,0xE3,0x7A,0x48,0x5D,0xB8,0x06,0x89,0x92,0x40,0x6C,0x6E,0xC5,0xDC,0xF8,0x9A,0xEF,0x0B,0xBE,0x21,0x0A,0x8C,0x2F,0x3A,0xB5,0xED,0xA7,0xCE,0x71,0x87,0x68,0x23,0xE1,0xB3,0xE4,0x18,0x7D,0xB8,0x47,0x01,0xA5,0x2B,0xC4,0x58,0xCB,0xB2,0x89,0x6C,0x5F,0xFD,
		0xD3,0x2C,0xC4,0x6F,0xB8,0x23,0xB2,0x0D,0xFF,0x3C,0xF2,0x11,0x45,0x74,0xF2,0x09,0x06,0x99,0x18,0xDD,0x6F,0xC0,0x86,0x01,0x18,0x12,0x1D,0x2B,0x16,0xAF,0x56,0xEF,0x65,0x33,0xA1,0xEA,0x67,0x4E,0xF4,0x4B,0x82,0xAB,0xE9,0x0F,0xDC,0x01,0xFA,0xDF,0x60,0x7F,0x66,0x47,0x5D,0xCB,0x2C,0x70,0xCC,0x7B,0x4E,0xD9,0x06,0xB8,0x6E,0x8C,0x0C,0xFE,0x62,0x1E,0x42,0xF9,0x93,0x7C,0xA2,0xAB,0x0A,0x9E,0xD0,0x23,0x10,0xAE,0x4D,0x7B,0x27,0x91,0x6F,0x26,0xBE,0x68,0xFA,0xA6,0x3F,0x9F,0x23,0xEB,0xC8,0x9D,0xBB,0x87 };

	X509Certificate x509 = new X509Certificate (cert);
	X509Certificate clone = new X509Certificate (x509);

	Assertion.Assert ("Equals", x509.Equals (clone));
	Assertion.Assert ("Equals", clone.Equals (x509));

	FileStream fs = File.OpenWrite ("temp.cer");
	fs.Write (cert, 0, cert.Length);
	fs.Close ();

	X509Certificate disk = X509Certificate.CreateFromCertFile ("temp.cer");
	Assertion.Assert ("Equals", disk.Equals (clone));
	Assertion.Assert ("Equals", disk.Equals (x509));
	byte[] hash = { 0x28,0x2D,0x98,0x06,0xC3,0xDF,0x73,0x45,0x92,0x9F,0x64,0xF5,0x89,0x5E,0xF2,0xEA,0x4A,0xC2,0x93,0x02 };
	AssertEquals ("GetCertHash", hash, x509.GetCertHash ());
	Assertion.AssertEquals ("GetCertHashString", "282D9806C3DF7345929F64F5895EF2EA4AC29302", x509.GetCertHashString ());
#if NET_2_0
	DateTime from = DateTime.ParseExact (x509.GetEffectiveDateString (), "MM/dd/yyyy HH:mm:ss", null).ToUniversalTime ();
	Assert.AreEqual ("05/25/2002 00:55:48", from.ToString (), "GetEffectiveDateString");
	DateTime until = DateTime.ParseExact (x509.GetExpirationDateString (), "MM/dd/yyyy HH:mm:ss", null).ToUniversalTime ();
	Assert.AreEqual ("11/25/2003 01:05:48", until.ToString (), "GetExpirationDateString");
	Assert.AreEqual ("61071143000000000034", x509.GetSerialNumberString (), "GetSerialNumberString");
#else
	// fx 1.x has a bug where the returned dates were always in the Seattle time zone
	Assertion.AssertEquals ("GetEffectiveDateString", "05/24/2002 16:55:48", x509.GetEffectiveDateString ());
	Assertion.AssertEquals ("GetExpirationDateString", "11/24/2003 17:05:48", x509.GetExpirationDateString ());
	// which was making it easier to test the dates ;-)
	Assert.AreEqual ("34000000000043110761", x509.GetSerialNumberString (), "GetSerialNumberString");
#endif
	Assertion.AssertEquals ("GetFormat", "X509", x509.GetFormat ());
	Assertion.AssertEquals ("GetHashCode", 674076678, x509.GetHashCode ());
	Assertion.AssertEquals ("GetIssuerName", "C=US, S=Washington, L=Redmond, O=Microsoft Corporation, OU=Copyright (c) 2000 Microsoft Corp., CN=Microsoft Code Signing PCA", x509.GetIssuerName ());
	Assertion.AssertEquals ("GetKeyAlgorithm", "1.2.840.113549.1.1.1", x509.GetKeyAlgorithm ());
	byte[] keyparams = { 0x05,0x00 };
	AssertEquals ("GetKeyAlgorithmParameters", keyparams, x509.GetKeyAlgorithmParameters ());
	Assertion.AssertEquals ("GetKeyAlgorithmParametersString", "0500", x509.GetKeyAlgorithmParametersString ());
	Assertion.AssertEquals ("GetName", "C=US, S=Washington, L=Redmond, O=Microsoft Corporation, OU=Copyright (c) 2002 Microsoft Corp., CN=Microsoft Corporation", x509.GetName ());
	byte[] pubkey = { 0x30,0x82,0x01,0x0A,0x02,0x82,0x01,0x01,0x00,0xAA,0x99,0xBD,0x39,0xA8,0x18,0x27,0xF4,0x2B,0x3D,0x0B,0x4C,0x3F,0x7C,0x77,0x2E,0xA7,0xCB,0xB5,0xD1,0x8C,0x0D,0xC2,0x3A,0x74,0xD7,0x93,0xB5,0xE0,0xA0,0x4B,0x3F,0x59,0x5E,0xCE,0x45,0x4F,0x9A,0x79,0x29,0xF1,0x49,0xCC,0x1A,0x47,0xEE,0x55,0xC2,0x08,0x3E,0x12,0x20,0xF8,0x55,0xF2,0xEE,0x5F,0xD3,0xE0,0xCA,0x96,0xBC,0x30,0xDE,0xFE,0x58,0xC8,0x27,0x32,0xD0,0x85,0x54,0xE8,0xF0,0x91,0x10,0xBB,0xF3,0x2B,0xBE,0x19,0xE5,0x03,0x9B,0x0B,0x86,0x1D,0xF3,0xB0,0x39,0x8C,0xB8,
		0xFD,0x0B,0x1D,0x3C,0x73,0x26,0xAC,0x57,0x2B,0xCA,0x29,0xA2,0x15,0x90,0x82,0x15,0xE2,0x77,0xA3,0x40,0x52,0x03,0x8B,0x9D,0xC2,0x70,0xBA,0x1F,0xE9,0x34,0xF6,0xF3,0x35,0x92,0x4E,0x55,0x83,0xF8,0xDA,0x30,0xB6,0x20,0xDE,0x57,0x06,0xB5,0x5A,0x42,0x06,0xDE,0x59,0xCB,0xF2,0xDF,0xA6,0xBD,0x15,0x47,0x71,0x19,0x25,0x23,0xD2,0xCB,0x6F,0x9B,0x19,0x79,0xDF,0x6A,0x5B,0xF1,0x76,0x05,0x79,0x29,0xFC,0xC3,0x56,0xCA,0x8F,0x44,0x08,0x85,0x55,0x8A,0xCB,0xC8,0x0F,0x46,0x4B,0x55,0xCB,0x8C,0x96,0x77,0x4A,0x87,0xE8,0xA9,
		0x41,0x06,0xC7,0xFF,0x0D,0xE9,0x68,0x57,0x63,0x72,0xC3,0x69,0x57,0xB4,0x43,0xCF,0x32,0x3A,0x30,0xDC,0x1B,0xE9,0xD5,0x43,0x26,0x2A,0x79,0xFE,0x95,0xDB,0x22,0x67,0x24,0xC9,0x2F,0xD0,0x34,0xE3,0xE6,0xFB,0x51,0x49,0x86,0xB8,0x3C,0xD0,0x25,0x5F,0xD6,0xEC,0x9E,0x03,0x61,0x87,0xA9,0x68,0x40,0xC7,0xF8,0xE2,0x03,0xE6,0xCF,0x05,0x02,0x03,0x01,0x00,0x01 };
	AssertEquals ("GetPublicKey", pubkey, x509.GetPublicKey ());
	Assertion.AssertEquals ("GetPublicKeyString", "3082010A0282010100AA99BD39A81827F42B3D0B4C3F7C772EA7CBB5D18C0DC23A74D793B5E0A04B3F595ECE454F9A7929F149CC1A47EE55C2083E1220F855F2EE5FD3E0CA96BC30DEFE58C82732D08554E8F09110BBF32BBE19E5039B0B861DF3B0398CB8FD0B1D3C7326AC572BCA29A215908215E277A34052038B9DC270BA1FE934F6F335924E5583F8DA30B620DE5706B55A4206DE59CBF2DFA6BD154771192523D2CB6F9B1979DF6A5BF176057929FCC356CA8F440885558ACBC80F464B55CB8C96774A87E8A94106C7FF0DE968576372C36957B443CF323A30DC1BE9D543262A79FE95DB226724C92FD034E3E6FB514986B83CD0255FD6EC9E036187A96840C7F8E203E6CF050203010001", x509.GetPublicKeyString ());
	AssertEquals ("GetRawCertData", cert, x509.GetRawCertData ());
	Assertion.AssertEquals ("GetRawCertDataString", ToString (cert), x509.GetRawCertDataString ());
	byte[] serial = { 0x34,0x00,0x00,0x00,0x00,0x00,0x43,0x11,0x07,0x61 };
	AssertEquals ("GetSerialNumber", serial, x509.GetSerialNumber ());
	byte[] ToStringArray = { 0x43,0x45,0x52,0x54,0x49,0x46,0x49,0x43,0x41,0x54,0x45,0x3A,0x0D,0x0A,0x09,0x46,0x6F,0x72,0x6D,0x61,0x74,0x3A,0x20,0x20,0x58,0x35,0x30,0x39,0x0D,0x0A,0x09,0x4E,0x61,0x6D,0x65,0x3A,0x20,0x20,0x43,0x3D,0x55,0x53,0x2C,0x20,0x53,0x3D,0x57,0x61,0x73,0x68,0x69,0x6E,0x67,0x74,0x6F,0x6E,0x2C,0x20,0x4C,0x3D,0x52,0x65,0x64,0x6D,0x6F,0x6E,0x64,0x2C,0x20,0x4F,0x3D,0x4D,0x69,0x63,0x72,0x6F,0x73,0x6F,0x66,0x74,0x20,0x43,0x6F,0x72,0x70,0x6F,0x72,0x61,0x74,0x69,0x6F,0x6E,0x2C,0x20,0x4F,0x55,0x3D,0x43,0x6F,0x70,0x79,
		0x72,0x69,0x67,0x68,0x74,0x20,0x28,0x63,0x29,0x20,0x32,0x30,0x30,0x32,0x20,0x4D,0x69,0x63,0x72,0x6F,0x73,0x6F,0x66,0x74,0x20,0x43,0x6F,0x72,0x70,0x2E,0x2C,0x20,0x43,0x4E,0x3D,0x4D,0x69,0x63,0x72,0x6F,0x73,0x6F,0x66,0x74,0x20,0x43,0x6F,0x72,0x70,0x6F,0x72,0x61,0x74,0x69,0x6F,0x6E,0x0D,0x0A,0x09,0x49,0x73,0x73,0x75,0x69,0x6E,0x67,0x20,0x43,0x41,0x3A,0x20,0x20,0x43,0x3D,0x55,0x53,0x2C,0x20,0x53,0x3D,0x57,0x61,0x73,0x68,0x69,0x6E,0x67,0x74,0x6F,0x6E,0x2C,0x20,0x4C,0x3D,0x52,0x65,0x64,0x6D,0x6F,0x6E,
		0x64,0x2C,0x20,0x4F,0x3D,0x4D,0x69,0x63,0x72,0x6F,0x73,0x6F,0x66,0x74,0x20,0x43,0x6F,0x72,0x70,0x6F,0x72,0x61,0x74,0x69,0x6F,0x6E,0x2C,0x20,0x4F,0x55,0x3D,0x43,0x6F,0x70,0x79,0x72,0x69,0x67,0x68,0x74,0x20,0x28,0x63,0x29,0x20,0x32,0x30,0x30,0x30,0x20,0x4D,0x69,0x63,0x72,0x6F,0x73,0x6F,0x66,0x74,0x20,0x43,0x6F,0x72,0x70,0x2E,0x2C,0x20,0x43,0x4E,0x3D,0x4D,0x69,0x63,0x72,0x6F,0x73,0x6F,0x66,0x74,0x20,0x43,0x6F,0x64,0x65,0x20,0x53,0x69,0x67,0x6E,0x69,0x6E,0x67,0x20,0x50,0x43,0x41,0x0D,0x0A,0x09,0x4B,
		0x65,0x79,0x20,0x41,0x6C,0x67,0x6F,0x72,0x69,0x74,0x68,0x6D,0x3A,0x20,0x20,0x31,0x2E,0x32,0x2E,0x38,0x34,0x30,0x2E,0x31,0x31,0x33,0x35,0x34,0x39,0x2E,0x31,0x2E,0x31,0x2E,0x31,0x0D,0x0A,0x09,0x53,0x65,0x72,0x69,0x61,0x6C,0x20,0x4E,0x75,0x6D,0x62,0x65,0x72,0x3A,0x20,0x20,0x33,0x34,0x30,0x30,0x30,0x30,0x30,0x30,0x30,0x30,0x30,0x30,0x34,0x33,0x31,0x31,0x30,0x37,0x36,0x31,0x0D,0x0A,0x09,0x4B,0x65,0x79,0x20,0x41,0x6C,0x6F,0x67,0x72,0x69,0x74,0x68,0x6D,0x20,0x50,0x61,0x72,0x61,0x6D,0x65,0x74,0x65,0x72,
		0x73,0x3A,0x20,0x20,0x30,0x35,0x30,0x30,0x0D,0x0A,0x09,0x50,0x75,0x62,0x6C,0x69,0x63,0x20,0x4B,0x65,0x79,0x3A,0x20,0x20,0x33,0x30,0x38,0x32,0x30,0x31,0x30,0x41,0x30,0x32,0x38,0x32,0x30,0x31,0x30,0x31,0x30,0x30,0x41,0x41,0x39,0x39,0x42,0x44,0x33,0x39,0x41,0x38,0x31,0x38,0x32,0x37,0x46,0x34,0x32,0x42,0x33,0x44,0x30,0x42,0x34,0x43,0x33,0x46,0x37,0x43,0x37,0x37,0x32,0x45,0x41,0x37,0x43,0x42,0x42,0x35,0x44,0x31,0x38,0x43,0x30,0x44,0x43,0x32,0x33,0x41,0x37,0x34,0x44,0x37,0x39,0x33,0x42,0x35,0x45,0x30,
		0x41,0x30,0x34,0x42,0x33,0x46,0x35,0x39,0x35,0x45,0x43,0x45,0x34,0x35,0x34,0x46,0x39,0x41,0x37,0x39,0x32,0x39,0x46,0x31,0x34,0x39,0x43,0x43,0x31,0x41,0x34,0x37,0x45,0x45,0x35,0x35,0x43,0x32,0x30,0x38,0x33,0x45,0x31,0x32,0x32,0x30,0x46,0x38,0x35,0x35,0x46,0x32,0x45,0x45,0x35,0x46,0x44,0x33,0x45,0x30,0x43,0x41,0x39,0x36,0x42,0x43,0x33,0x30,0x44,0x45,0x46,0x45,0x35,0x38,0x43,0x38,0x32,0x37,0x33,0x32,0x44,0x30,0x38,0x35,0x35,0x34,0x45,0x38,0x46,0x30,0x39,0x31,0x31,0x30,0x42,0x42,0x46,0x33,0x32,0x42,
		0x42,0x45,0x31,0x39,0x45,0x35,0x30,0x33,0x39,0x42,0x30,0x42,0x38,0x36,0x31,0x44,0x46,0x33,0x42,0x30,0x33,0x39,0x38,0x43,0x42,0x38,0x46,0x44,0x30,0x42,0x31,0x44,0x33,0x43,0x37,0x33,0x32,0x36,0x41,0x43,0x35,0x37,0x32,0x42,0x43,0x41,0x32,0x39,0x41,0x32,0x31,0x35,0x39,0x30,0x38,0x32,0x31,0x35,0x45,0x32,0x37,0x37,0x41,0x33,0x34,0x30,0x35,0x32,0x30,0x33,0x38,0x42,0x39,0x44,0x43,0x32,0x37,0x30,0x42,0x41,0x31,0x46,0x45,0x39,0x33,0x34,0x46,0x36,0x46,0x33,0x33,0x35,0x39,0x32,0x34,0x45,0x35,0x35,0x38,0x33,
		0x46,0x38,0x44,0x41,0x33,0x30,0x42,0x36,0x32,0x30,0x44,0x45,0x35,0x37,0x30,0x36,0x42,0x35,0x35,0x41,0x34,0x32,0x30,0x36,0x44,0x45,0x35,0x39,0x43,0x42,0x46,0x32,0x44,0x46,0x41,0x36,0x42,0x44,0x31,0x35,0x34,0x37,0x37,0x31,0x31,0x39,0x32,0x35,0x32,0x33,0x44,0x32,0x43,0x42,0x36,0x46,0x39,0x42,0x31,0x39,0x37,0x39,0x44,0x46,0x36,0x41,0x35,0x42,0x46,0x31,0x37,0x36,0x30,0x35,0x37,0x39,0x32,0x39,0x46,0x43,0x43,0x33,0x35,0x36,0x43,0x41,0x38,0x46,0x34,0x34,0x30,0x38,0x38,0x35,0x35,0x35,0x38,0x41,0x43,0x42,
		0x43,0x38,0x30,0x46,0x34,0x36,0x34,0x42,0x35,0x35,0x43,0x42,0x38,0x43,0x39,0x36,0x37,0x37,0x34,0x41,0x38,0x37,0x45,0x38,0x41,0x39,0x34,0x31,0x30,0x36,0x43,0x37,0x46,0x46,0x30,0x44,0x45,0x39,0x36,0x38,0x35,0x37,0x36,0x33,0x37,0x32,0x43,0x33,0x36,0x39,0x35,0x37,0x42,0x34,0x34,0x33,0x43,0x46,0x33,0x32,0x33,0x41,0x33,0x30,0x44,0x43,0x31,0x42,0x45,0x39,0x44,0x35,0x34,0x33,0x32,0x36,0x32,0x41,0x37,0x39,0x46,0x45,0x39,0x35,0x44,0x42,0x32,0x32,0x36,0x37,0x32,0x34,0x43,0x39,0x32,0x46,0x44,0x30,0x33,0x34,
		0x45,0x33,0x45,0x36,0x46,0x42,0x35,0x31,0x34,0x39,0x38,0x36,0x42,0x38,0x33,0x43,0x44,0x30,0x32,0x35,0x35,0x46,0x44,0x36,0x45,0x43,0x39,0x45,0x30,0x33,0x36,0x31,0x38,0x37,0x41,0x39,0x36,0x38,0x34,0x30,0x43,0x37,0x46,0x38,0x45,0x32,0x30,0x33,0x45,0x36,0x43,0x46,0x30,0x35,0x30,0x32,0x30,0x33,0x30,0x31,0x30,0x30,0x30,0x31,0x0D,0x0A,0x0D,0x0A };
	string tsa = Encoding.Default.GetString (ToStringArray).Replace ("\r\n", Environment.NewLine);
	//Assertion.AssertEquals ("ToString(true)", tsa, x509.ToString (true));
}


// Certificate: basic\DSAParametersInheritedCACert.cer
// - X.509v3 end-entity certificate
// - sha1DSA, DSA 1024 bits
// - has NO parameters (shared group is present in issuer certificate)
// !!! MS bugs when using constructor X509Certificate(X509Certificate) !!!
[Test]
#if NET_2_0
[ExpectedException (typeof (CryptographicException))]
#endif
public void Certificate13 ()
{
	// certificate can be loaded by constructor
	byte[] cert = { 0x30,0x82,0x02,0x14,0x30,0x82,0x01,0xD3,0xA0,0x03,0x02,0x01,0x02,0x02,0x01,0x02,0x30,0x09,0x06,0x07,0x2A,0x86,0x48,0xCE,0x38,0x04,0x03,0x30,0x3A,0x31,0x0B,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x55,0x53,0x31,0x1A,0x30,0x18,0x06,0x03,0x55,0x04,0x0A,0x13,0x11,0x54,0x65,0x73,0x74,0x20,0x43,0x65,0x72,0x74,0x69,0x66,0x69,0x63,0x61,0x74,0x65,0x73,0x31,0x0F,0x30,0x0D,0x06,0x03,0x55,0x04,0x03,0x13,0x06,0x44,0x53,0x41,0x20,0x43,0x41,0x30,0x1E,0x17,0x0D,0x30,0x31,0x30,0x34,0x31,0x39,0x31,0x34,0x35,0x37,
		0x32,0x30,0x5A,0x17,0x0D,0x31,0x31,0x30,0x34,0x31,0x39,0x31,0x34,0x35,0x37,0x32,0x30,0x5A,0x30,0x4F,0x31,0x0B,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x55,0x53,0x31,0x1A,0x30,0x18,0x06,0x03,0x55,0x04,0x0A,0x13,0x11,0x54,0x65,0x73,0x74,0x20,0x43,0x65,0x72,0x74,0x69,0x66,0x69,0x63,0x61,0x74,0x65,0x73,0x31,0x24,0x30,0x22,0x06,0x03,0x55,0x04,0x03,0x13,0x1B,0x44,0x53,0x41,0x20,0x50,0x61,0x72,0x61,0x6D,0x65,0x74,0x65,0x72,0x73,0x20,0x49,0x6E,0x68,0x65,0x72,0x69,0x74,0x65,0x64,0x20,0x43,0x41,0x30,
		0x81,0x92,0x30,0x09,0x06,0x07,0x2A,0x86,0x48,0xCE,0x38,0x04,0x01,0x03,0x81,0x84,0x00,0x02,0x81,0x80,0x67,0x8C,0x47,0xDA,0x0C,0x36,0x81,0x64,0x39,0xF8,0x1A,0x37,0x0B,0xF0,0xBB,0xD4,0x13,0xFD,0x67,0x3D,0x40,0xC8,0x9C,0x92,0xE1,0x3A,0x89,0x54,0xB0,0xF2,0x97,0xA5,0x70,0xD0,0x64,0x17,0xA4,0xA7,0x7C,0xA9,0xE9,0x27,0x09,0x65,0xDF,0xA0,0xA5,0x1C,0xFC,0x04,0xDA,0x3E,0xDD,0x62,0x6B,0xF0,0x0C,0xED,0x81,0x57,0x4F,0x5D,0x29,0xC8,0x08,0x59,0x19,0x0C,0x5F,0x8D,0x88,0xF0,0xFF,0x8C,0xB6,0xE8,0x75,0x01,0xBE,0x10,
		0x0A,0x64,0xDF,0x57,0x7B,0x8B,0x6C,0x56,0xF0,0x53,0xCE,0x43,0x88,0xB9,0x09,0x99,0x08,0x3C,0x90,0xAF,0x01,0xE8,0xD9,0x32,0x5D,0xED,0x56,0x02,0xCF,0x60,0xD2,0x01,0x3A,0xE3,0x0A,0x2B,0x9E,0x91,0x41,0xF6,0xC7,0x7C,0xDE,0x6C,0x99,0x63,0x00,0x0E,0xA3,0x7C,0x30,0x7A,0x30,0x1D,0x06,0x03,0x55,0x1D,0x0E,0x04,0x16,0x04,0x14,0x5D,0x24,0xEE,0x8A,0x55,0x1A,0xF2,0xC6,0xC9,0xB2,0xC2,0xBF,0x8A,0xF0,0xB2,0x49,0x4F,0x3A,0xB3,0x1B,0x30,0x1F,0x06,0x03,0x55,0x1D,0x23,0x04,0x18,0x30,0x16,0x80,0x14,0x74,0x15,0xD5,0x24,
		0x1C,0xBD,0x5E,0x65,0x88,0x1F,0xE1,0x8B,0x09,0x7E,0x7F,0xEA,0x19,0x48,0x4E,0x61,0x30,0x17,0x06,0x03,0x55,0x1D,0x20,0x04,0x10,0x30,0x0E,0x30,0x0C,0x06,0x0A,0x60,0x86,0x48,0x01,0x65,0x03,0x02,0x01,0x30,0x01,0x30,0x0E,0x06,0x03,0x55,0x1D,0x0F,0x01,0x01,0xFF,0x04,0x04,0x03,0x02,0x01,0x06,0x30,0x0F,0x06,0x03,0x55,0x1D,0x13,0x01,0x01,0xFF,0x04,0x05,0x30,0x03,0x01,0x01,0xFF,0x30,0x09,0x06,0x07,0x2A,0x86,0x48,0xCE,0x38,0x04,0x03,0x03,0x30,0x00,0x30,0x2D,0x02,0x15,0x00,0xA8,0x59,0x6F,0x31,0x77,0xB6,0x20,
		0xEC,0x36,0x9B,0xEB,0x4B,0x61,0x0A,0xAF,0x44,0xED,0x72,0xBA,0x29,0x02,0x14,0x6D,0x22,0xE1,0xBD,0x4D,0x27,0xF6,0x2E,0x3B,0x1F,0xD7,0x9D,0xD6,0x59,0x5E,0xCB,0x25,0x86,0x22,0xD8 };

	X509Certificate x509 = new X509Certificate (cert);
//	X509Certificate clone = new X509Certificate (x509);

//	Assertion.Assert ("Equals", x509.Equals (clone));
//	Assertion.Assert ("Equals", clone.Equals (x509));

	FileStream fs = File.OpenWrite ("temp.cer");
	fs.Write (cert, 0, cert.Length);
	fs.Close ();

	X509Certificate disk = X509Certificate.CreateFromCertFile ("temp.cer");
//	Assertion.Assert ("Equals", disk.Equals (clone));
	Assertion.Assert ("Equals", disk.Equals (x509));
	byte[] hash = { 0x0E,0x29,0xCD,0xDC,0xA5,0xE7,0x32,0xA2,0x0D,0xCE,0xD8,0x7B,0x3D,0x00,0x24,0x46,0x85,0x3E,0xBB,0xD1 };
	AssertEquals ("GetCertHash", hash, x509.GetCertHash ());
	Assertion.AssertEquals ("GetCertHashString", "0E29CDDCA5E732A20DCED87B3D002446853EBBD1", x509.GetCertHashString ());
#if NET_2_0
	DateTime from = DateTime.ParseExact (x509.GetEffectiveDateString (), "MM/dd/yyyy HH:mm:ss", null).ToUniversalTime ();
	Assert.AreEqual ("04/19/2001 14:57:20", from.ToString (), "GetEffectiveDateString");
	DateTime until = DateTime.ParseExact (x509.GetExpirationDateString (), "MM/dd/yyyy HH:mm:ss", null).ToUniversalTime ();
	Assert.AreEqual ("04/19/2011 14:57:20", until.ToString (), "GetExpirationDateString");
#else
	// fx 1.x has a bug where the returned dates were always in the Seattle time zone
	Assertion.AssertEquals ("GetEffectiveDateString", DateTime.Parse ("04/19/2001 06:57:20"), DateTime.Parse (x509.GetEffectiveDateString ()));
	Assertion.AssertEquals ("GetExpirationDateString", DateTime.Parse ("04/19/2011 06:57:20"), DateTime.Parse (x509.GetExpirationDateString ()));
	// which was making it easier to test the dates ;-)
#endif
	Assertion.AssertEquals ("GetFormat", "X509", x509.GetFormat ());
	Assertion.AssertEquals ("GetHashCode", 237620700, x509.GetHashCode ());
	Assertion.AssertEquals ("GetIssuerName", "C=US, O=Test Certificates, CN=DSA CA", x509.GetIssuerName ());
	Assertion.AssertEquals ("GetKeyAlgorithm", "1.2.840.10040.4.1", x509.GetKeyAlgorithm ());
	// this fails under 2.0
	Assertion.AssertNull ("GetKeyAlgorithmParameters", x509.GetKeyAlgorithmParameters ());
	Assertion.AssertNull ("GetKeyAlgorithmParametersString", x509.GetKeyAlgorithmParametersString ());
	Assertion.AssertEquals ("GetName", "C=US, O=Test Certificates, CN=DSA Parameters Inherited CA", x509.GetName ());
	byte[] pubkey = { 0x02,0x81,0x80,0x67,0x8C,0x47,0xDA,0x0C,0x36,0x81,0x64,0x39,0xF8,0x1A,0x37,0x0B,0xF0,0xBB,0xD4,0x13,0xFD,0x67,0x3D,0x40,0xC8,0x9C,0x92,0xE1,0x3A,0x89,0x54,0xB0,0xF2,0x97,0xA5,0x70,0xD0,0x64,0x17,0xA4,0xA7,0x7C,0xA9,0xE9,0x27,0x09,0x65,0xDF,0xA0,0xA5,0x1C,0xFC,0x04,0xDA,0x3E,0xDD,0x62,0x6B,0xF0,0x0C,0xED,0x81,0x57,0x4F,0x5D,0x29,0xC8,0x08,0x59,0x19,0x0C,0x5F,0x8D,0x88,0xF0,0xFF,0x8C,0xB6,0xE8,0x75,0x01,0xBE,0x10,0x0A,0x64,0xDF,0x57,0x7B,0x8B,0x6C,0x56,0xF0,0x53,0xCE,0x43,0x88,0xB9,0x09,0x99,0x08,0x3C,
		0x90,0xAF,0x01,0xE8,0xD9,0x32,0x5D,0xED,0x56,0x02,0xCF,0x60,0xD2,0x01,0x3A,0xE3,0x0A,0x2B,0x9E,0x91,0x41,0xF6,0xC7,0x7C,0xDE,0x6C,0x99,0x63,0x00,0x0E };
	AssertEquals ("GetPublicKey", pubkey, x509.GetPublicKey ());
	Assertion.AssertEquals ("GetPublicKeyString", "028180678C47DA0C36816439F81A370BF0BBD413FD673D40C89C92E13A8954B0F297A570D06417A4A77CA9E9270965DFA0A51CFC04DA3EDD626BF00CED81574F5D29C80859190C5F8D88F0FF8CB6E87501BE100A64DF577B8B6C56F053CE4388B90999083C90AF01E8D9325DED5602CF60D2013AE30A2B9E9141F6C77CDE6C9963000E", x509.GetPublicKeyString ());
	AssertEquals ("GetRawCertData", cert, x509.GetRawCertData ());
	Assertion.AssertEquals ("GetRawCertDataString", ToString (cert), x509.GetRawCertDataString ());
	byte[] serial = { 0x02 };
	AssertEquals ("GetSerialNumber", serial, x509.GetSerialNumber ());
	Assertion.AssertEquals ("GetSerialNumberString", "02", x509.GetSerialNumberString ());
	byte[] ToStringArray = { 0x43,0x45,0x52,0x54,0x49,0x46,0x49,0x43,0x41,0x54,0x45,0x3A,0x0D,0x0A,0x09,0x46,0x6F,0x72,0x6D,0x61,0x74,0x3A,0x20,0x20,0x58,0x35,0x30,0x39,0x0D,0x0A,0x09,0x4E,0x61,0x6D,0x65,0x3A,0x20,0x20,0x43,0x3D,0x55,0x53,0x2C,0x20,0x4F,0x3D,0x54,0x65,0x73,0x74,0x20,0x43,0x65,0x72,0x74,0x69,0x66,0x69,0x63,0x61,0x74,0x65,0x73,0x2C,0x20,0x43,0x4E,0x3D,0x44,0x53,0x41,0x20,0x50,0x61,0x72,0x61,0x6D,0x65,0x74,0x65,0x72,0x73,0x20,0x49,0x6E,0x68,0x65,0x72,0x69,0x74,0x65,0x64,0x20,0x43,0x41,0x0D,0x0A,0x09,0x49,0x73,0x73,
		0x75,0x69,0x6E,0x67,0x20,0x43,0x41,0x3A,0x20,0x20,0x43,0x3D,0x55,0x53,0x2C,0x20,0x4F,0x3D,0x54,0x65,0x73,0x74,0x20,0x43,0x65,0x72,0x74,0x69,0x66,0x69,0x63,0x61,0x74,0x65,0x73,0x2C,0x20,0x43,0x4E,0x3D,0x44,0x53,0x41,0x20,0x43,0x41,0x0D,0x0A,0x09,0x4B,0x65,0x79,0x20,0x41,0x6C,0x67,0x6F,0x72,0x69,0x74,0x68,0x6D,0x3A,0x20,0x20,0x31,0x2E,0x32,0x2E,0x38,0x34,0x30,0x2E,0x31,0x30,0x30,0x34,0x30,0x2E,0x34,0x2E,0x31,0x0D,0x0A,0x09,0x53,0x65,0x72,0x69,0x61,0x6C,0x20,0x4E,0x75,0x6D,0x62,0x65,0x72,0x3A,0x20,
		0x20,0x30,0x32,0x0D,0x0A,0x09,0x50,0x75,0x62,0x6C,0x69,0x63,0x20,0x4B,0x65,0x79,0x3A,0x20,0x20,0x30,0x32,0x38,0x31,0x38,0x30,0x36,0x37,0x38,0x43,0x34,0x37,0x44,0x41,0x30,0x43,0x33,0x36,0x38,0x31,0x36,0x34,0x33,0x39,0x46,0x38,0x31,0x41,0x33,0x37,0x30,0x42,0x46,0x30,0x42,0x42,0x44,0x34,0x31,0x33,0x46,0x44,0x36,0x37,0x33,0x44,0x34,0x30,0x43,0x38,0x39,0x43,0x39,0x32,0x45,0x31,0x33,0x41,0x38,0x39,0x35,0x34,0x42,0x30,0x46,0x32,0x39,0x37,0x41,0x35,0x37,0x30,0x44,0x30,0x36,0x34,0x31,0x37,0x41,0x34,0x41,
		0x37,0x37,0x43,0x41,0x39,0x45,0x39,0x32,0x37,0x30,0x39,0x36,0x35,0x44,0x46,0x41,0x30,0x41,0x35,0x31,0x43,0x46,0x43,0x30,0x34,0x44,0x41,0x33,0x45,0x44,0x44,0x36,0x32,0x36,0x42,0x46,0x30,0x30,0x43,0x45,0x44,0x38,0x31,0x35,0x37,0x34,0x46,0x35,0x44,0x32,0x39,0x43,0x38,0x30,0x38,0x35,0x39,0x31,0x39,0x30,0x43,0x35,0x46,0x38,0x44,0x38,0x38,0x46,0x30,0x46,0x46,0x38,0x43,0x42,0x36,0x45,0x38,0x37,0x35,0x30,0x31,0x42,0x45,0x31,0x30,0x30,0x41,0x36,0x34,0x44,0x46,0x35,0x37,0x37,0x42,0x38,0x42,0x36,0x43,0x35,
		0x36,0x46,0x30,0x35,0x33,0x43,0x45,0x34,0x33,0x38,0x38,0x42,0x39,0x30,0x39,0x39,0x39,0x30,0x38,0x33,0x43,0x39,0x30,0x41,0x46,0x30,0x31,0x45,0x38,0x44,0x39,0x33,0x32,0x35,0x44,0x45,0x44,0x35,0x36,0x30,0x32,0x43,0x46,0x36,0x30,0x44,0x32,0x30,0x31,0x33,0x41,0x45,0x33,0x30,0x41,0x32,0x42,0x39,0x45,0x39,0x31,0x34,0x31,0x46,0x36,0x43,0x37,0x37,0x43,0x44,0x45,0x36,0x43,0x39,0x39,0x36,0x33,0x30,0x30,0x30,0x45,0x0D,0x0A,0x0D,0x0A };
	string tsa = Encoding.Default.GetString (ToStringArray).Replace ("\r\n", Environment.NewLine);
	//Assertion.AssertEquals ("ToString(true)", tsa, x509.ToString (true));
}


// Number of certificates: 12
// Number of invalid certificates: 1
}
}

