'''
Base class for network related tests.

This provides fake wifi devices with mac80211_hwsim and hostapd, and some
utility functions.
'''

__author__ = 'Martin Pitt <martin.pitt@ubuntu.com>'
__copyright__ = '(C) 2013 Canonical Ltd.'
__license__ = 'GPL v2 or later'

import sys
import os
import os.path
import time
import tempfile
import subprocess
import re
import unittest
from glob import glob

# check availability of programs, and cleanly skip test if they are not
# available
for program in ['wpa_supplicant', 'hostapd', 'dnsmasq', 'dhclient']:
    if subprocess.call(['which', program], stdout=subprocess.PIPE) != 0:
        sys.stderr.write('%s is required for this test suite, but not available. Skipping\n' % program)
        sys.exit(0)


class NetworkTestBase(unittest.TestCase):
    '''Common functionality for network test cases

    Create two test wlan devices, one for a simulated access point
    (self.dev_ap), the other for a simulated client device (self.dev_client)

    Each test should call self.setup_ap() with the desired configuration.
    '''
    @classmethod
    def setUpClass(klass):
        klass.create_devices()
        klass.workdir_obj = tempfile.TemporaryDirectory()
        klass.workdir = klass.workdir_obj.name

        # stop system-wide NetworkManager to avoid interfering with tests
        klass.nm_running = subprocess.call('service network-manager stop 2>&1',
                                           shell=True) == 0

        # static entropy file to avoid draining/blocking on /dev/random
        klass.entropy_file = os.path.join(klass.workdir, 'entropy')
        with open(klass.entropy_file, 'wb') as f:
            f.write(b'0123456789012345678901')

        # set regulatory domain "EU", so that we can use 80211.a 5 GHz channels
        out = subprocess.check_output(['iw', 'reg', 'get'], universal_newlines=True)
        m = re.match('^country (\S+):', out)
        assert m
        klass.orig_country = m.group(1)
        subprocess.check_call(['iw', 'reg', 'set', 'EU'])

    @classmethod
    def tearDownClass(klass):
        subprocess.check_call(['iw', 'reg', 'set', klass.orig_country])
        klass.shutdown_devices()
        klass.workdir_obj.cleanup()

        # restart system-wide NetworkManager if we stopped it
        if klass.nm_running:
            subprocess.call('service network-manager start 2>&1', shell=True) == 0

    @classmethod
    def create_devices(klass):
        '''Create Access Point and Client devices with mac80211_hwsim'''

        if os.path.exists('/sys/module/mac80211_hwsim'):
            raise SystemError('mac80211_hwsim module already loaded')

        before = set([c for c in os.listdir('/sys/class/net') if c.startswith('wlan')])
        subprocess.check_call(['modprobe', 'mac80211_hwsim'])
        # wait 5 seconds for fake devices to appear
        timeout = 50
        while timeout > 0:
            after = set([c for c in os.listdir('/sys/class/net') if c.startswith('wlan')])
            if len(after) - len(before) >= 2:
                break
            timeout -= 1
            time.sleep(0.1)
        else:
            raise SystemError('timed out waiting for fake wlan devices to appear')
        devs = list(after - before)
        klass.dev_ap = devs[0]
        klass.dev_client = devs[1]

        with open('/sys/class/net/%s/address' % klass.dev_ap) as f:
            klass.mac_ap = f.read().strip()
        with open('/sys/class/net/%s/address' % klass.dev_client) as f:
            klass.mac_client = f.read().strip()
        print('Created fake devices: AP: %s, client: %s' % (klass.dev_ap, klass.dev_client))

    @classmethod
    def shutdown_devices(klass):
        '''Remove test wlan devices'''

        subprocess.check_call(['rmmod', 'mac80211_hwsim'])
        klass.dev_ap = None
        klass.dev_client = None

    def run(self, result=None):
        '''Show log files on failed tests'''

        if result:
            orig_err_fail = len(result.errors) + len(result.failures)
        super().run(result)
        logs = glob(os.path.join(self.workdir, '*.log'))
        if result and len(result.errors) + len(result.failures) > orig_err_fail:
            for log_file in logs:
                with open(log_file) as f:
                    print('\n----- %s -----\n%s\n------\n'
                          % (os.path.basename(log_file), f.read()))

        # clean up log files, so that we don't see ones from previous tests
        for log_file in logs:
            os.unlink(log_file)

    def setup_ap(self, hostapd_conf, ipv6_mode):
        '''Set up simulated access point

        On self.dev_ap, run hostapd with given configuration. Setup dnsmasq
        according to ipv6_mode, see start_dnsmasq().

        This is torn down automatically at the end of the test.
        '''
        # give our AP an IP
        subprocess.check_call(['ip', 'a', 'flush', 'dev', self.dev_ap])
        if ipv6_mode is not None:
            subprocess.check_call(['ip', 'a', 'add', '2600::1/64', 'dev', self.dev_ap])
        else:
            subprocess.check_call(['ip', 'a', 'add', '192.168.5.1/24', 'dev', self.dev_ap])

        self.start_hostapd(hostapd_conf)
        self.start_dnsmasq(ipv6_mode)

    def start_wpasupp(self, conf):
        '''Start wpa_supplicant on client interface'''

        w_conf = os.path.join(self.workdir, 'wpasupplicant.conf')
        with open(w_conf, 'w') as f:
            f.write('ctrl_interface=%s\nnetwork={\n%s\n}\n' % (self.workdir, conf))
        log = os.path.join(self.workdir, 'wpasupp.log')
        p = subprocess.Popen(['wpa_supplicant', '-Dwext', '-i', self.dev_client,
                              '-e', self.entropy_file, '-c', w_conf, '-f', log],
                             stderr=subprocess.PIPE)
        self.addCleanup(p.wait)
        self.addCleanup(p.terminate)
        # TODO: why does this sometimes take so long?
        self.poll_text(log, 'CTRL-EVENT-CONNECTED', timeout=200)

    #
    # Internal implementation details
    #

    @classmethod
    def poll_text(klass, logpath, string, timeout=50):
        '''Poll log file for a given string with a timeout.

        Timeout is given in deciseconds.
        '''
        log = ''
        while timeout > 0:
            if os.path.exists(logpath):
                break
            timeout -= 1
            time.sleep(0.1)
        assert timeout > 0, 'Timed out waiting for file %s to appear' % logpath

        with open(logpath) as f:
            while timeout > 0:
                line = f.readline()
                if line:
                    log += line
                    if string in line:
                        break
                    continue
                timeout -= 1
                time.sleep(0.1)

        assert timeout > 0, 'Timed out waiting for "%s":\n------------\n%s\n-------\n' % (string, log)

    def start_hostapd(self, conf):
        hostapd_conf = os.path.join(self.workdir, 'hostapd.conf')
        with open(hostapd_conf, 'w') as f:
            f.write('interface=%s\ndriver=nl80211\n' % self.dev_ap)
            f.write(conf)

        log = os.path.join(self.workdir, 'hostapd.log')
        p = subprocess.Popen(['hostapd', '-e', self.entropy_file, '-f', log, hostapd_conf],
                             stdout=subprocess.PIPE)
        self.addCleanup(p.wait)
        self.addCleanup(p.terminate)
        self.poll_text(log, 'Using interface ' + self.dev_ap)

    def start_dnsmasq(self, ipv6_mode):
        '''Start dnsmasq.

        If ipv6_mode is None, IPv4 is set up with DHCP. If it is not None, it
        must be a valid dnsmasq mode, i. e. a combination of "ra-only",
        "slaac", "ra-stateless", and "ra-names". See dnsmasq(8).
        '''
        if ipv6_mode is None:
            dhcp_range = '192.168.5.10,192.168.5.200'
        else:
            dhcp_range = '2600::10,2600::20'
            if ipv6_mode:
                dhcp_range += ',' + ipv6_mode

        self.dnsmasq_log = os.path.join(self.workdir, 'dnsmasq.log')

        p = subprocess.Popen(['dnsmasq', '--keep-in-foreground', '--log-queries',
                              '--log-facility=' + self.dnsmasq_log,
                              '--conf-file=/dev/null',
                              '--bind-interfaces',
                              '--interface=' + self.dev_ap,
                              '--except-interface=lo',
                              '--enable-ra',
                              '--dhcp-range=' + dhcp_range])
        self.addCleanup(p.wait)
        self.addCleanup(p.terminate)

        if ipv6_mode is not None:
            self.poll_text(self.dnsmasq_log, 'IPv6 router advertisement enabled')
        else:
            self.poll_text(self.dnsmasq_log, 'DHCP, IP range')
