# Copyright (C) 2011 Midokura KK
# All Rights Reserved.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.

"""The virtual interfaces extension."""

import webob

from nova.api.openstack import api_version_request
from nova.api.openstack import common
from nova.api.openstack import extensions
from nova.api.openstack import wsgi
from nova import compute
from nova.i18n import _
from nova import network


ALIAS = 'os-virtual-interfaces'
authorize = extensions.os_compute_authorizer(ALIAS)


def _translate_vif_summary_view(req, vif):
    """Maps keys for VIF summary view."""
    d = {}
    d['id'] = vif.uuid
    d['mac_address'] = vif.address
    if (req.api_version_request >=
        api_version_request.APIVersionRequest("2.12")):
        d['net_id'] = vif.net_uuid
    # NOTE(gmann): This is for v2.1 compatible mode where response should be
    # same as v2 one.
    if req.is_legacy_v2():
        d['OS-EXT-VIF-NET:net_id'] = vif.net_uuid
    return d


class ServerVirtualInterfaceController(wsgi.Controller):
    """The instance VIF API controller for the OpenStack API.
    """

    def __init__(self):
        self.compute_api = compute.API(skip_policy_check=True)
        self.network_api = network.API(skip_policy_check=True)
        super(ServerVirtualInterfaceController, self).__init__()

    def _items(self, req, server_id, entity_maker):
        """Returns a list of VIFs, transformed through entity_maker."""
        context = req.environ['nova.context']
        authorize(context)
        instance = common.get_instance(self.compute_api, context, server_id)

        try:
            vifs = self.network_api.get_vifs_by_instance(context, instance)
        except NotImplementedError:
            msg = _('Listing virtual interfaces is not supported by this '
                    'cloud.')
            raise webob.exc.HTTPBadRequest(explanation=msg)
        limited_list = common.limited(vifs, req)
        res = [entity_maker(req, vif) for vif in limited_list]
        return {'virtual_interfaces': res}

    @extensions.expected_errors((400, 404))
    def index(self, req, server_id):
        """Returns the list of VIFs for a given instance."""
        return self._items(req, server_id,
                           entity_maker=_translate_vif_summary_view)


class VirtualInterfaces(extensions.V21APIExtensionBase):
    """Virtual interface support."""

    name = "VirtualInterfaces"
    alias = ALIAS
    version = 1

    def get_resources(self):
        resources = []

        res = extensions.ResourceExtension(
            ALIAS,
            controller=ServerVirtualInterfaceController(),
            parent=dict(member_name='server', collection_name='servers'))
        resources.append(res)

        return resources

    def get_controller_extensions(self):
        return []
