/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: confighelper.cxx,v $
 *
 *  $Revision: 1.6 $
 *
 *  last change: $Author: hr $ $Date: 2005/09/23 15:44:38 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/

#ifndef _COM_SUN_STAR_BEANS_PROPERTYVALUE_HPP_
#include <com/sun/star/beans/PropertyValue.hpp>
#endif

#ifndef _COM_SUN_STAR_CONTAINER_XCONTAINERQUERY_HPP_
#include <com/sun/star/container/XContainerQuery.hpp>
#endif

#ifndef _COM_SUN_STAR_DOCUMENT_XTYPEDETECTION_HPP_
#include <com/sun/star/document/XTypeDetection.hpp>
#endif

#include <convert.hxx>

#include <confighelper.hxx>
#include <comphelper/classids.hxx>

using namespace ::com::sun::star;

uno::Sequence< sal_Int8 > GetSequenceClassID( sal_uInt32 n1, sal_uInt16 n2, sal_uInt16 n3,
												sal_uInt8 b8, sal_uInt8 b9, sal_uInt8 b10, sal_uInt8 b11,
                                                sal_uInt8 b12, sal_uInt8 b13, sal_uInt8 b14, sal_uInt8 b15 );

//-----------------------------------------------------------------------
ConfigurationHelper::ConfigurationHelper( const uno::Reference< lang::XMultiServiceFactory >& xFactory )
: m_xFactory( xFactory )
{
	if ( !m_xFactory.is() )
		throw uno::RuntimeException();
}

//-----------------------------------------------------------------------
::rtl::OUString ConfigurationHelper::GetStringClassIDRepresentation( const uno::Sequence< sal_Int8 >& aClassID )
{
	::rtl::OUString aResult;

	if ( aClassID.getLength() == 16 )
	{
		for ( sal_Int32 nInd = 0; nInd < aClassID.getLength(); nInd++ )
		{
			if ( nInd == 4 || nInd == 6 || nInd == 8 || nInd == 10 )
				aResult += ::rtl::OUString::createFromAscii( "-" );

			sal_Int32 nDigit1 = (sal_Int32)( (sal_uInt8)aClassID[nInd] / 16 );
			sal_Int32 nDigit2 = (sal_uInt8)aClassID[nInd] % 16;
			aResult += ::rtl::OUString::valueOf( nDigit1, 16 );
			aResult += ::rtl::OUString::valueOf( nDigit2, 16 );
		}
	}

	return aResult;
}

//-----------------------------------------------------------------------
sal_uInt8 GetDigit_Impl( sal_Char aChar )
{
	if ( aChar >= '0' && aChar <= '9' )
		return aChar - '0';
	else if ( aChar >= 'a' && aChar <= 'f' )
		return aChar - 'a' + 10;
	else if ( aChar >= 'A' && aChar <= 'F' )
		return aChar - 'A' + 10;
	else
		return 16;
}

//-----------------------------------------------------------------------
uno::Sequence< sal_Int8 > ConfigurationHelper::GetSequenceClassIDRepresentation( const ::rtl::OUString& aClassID )
{
	sal_Int32 nLength = aClassID.getLength();
	if ( nLength == 36 )
	{
		::rtl::OString aCharClassID = ::rtl::OUStringToOString( aClassID, RTL_TEXTENCODING_ASCII_US );
		const sal_Char* pString = aCharClassID.getStr();
		if ( pString )
		{
			uno::Sequence< sal_Int8 > aResult( 16 );

			sal_Int32 nStrPointer = 0;
			sal_Int32 nSeqInd = 0;
			while( nSeqInd < 16 && nStrPointer + 1 < nLength )
			{
				sal_uInt8 nDigit1 = GetDigit_Impl( pString[nStrPointer++] );
				sal_uInt8 nDigit2 = GetDigit_Impl( pString[nStrPointer++] );

				if ( nDigit1 > 15 || nDigit2 > 15 )
					break;

				aResult[nSeqInd++] = (sal_Int8)( nDigit1 * 16 + nDigit2 );

				if ( nStrPointer < nLength && pString[nStrPointer] == '-' )
					nStrPointer++;
			}

			if ( nSeqInd == 16 && nStrPointer == nLength )
				return aResult;
		}
	}

	return uno::Sequence< sal_Int8 >();
}

//-----------------------------------------------------------------------
uno::Reference< container::XNameAccess > ConfigurationHelper::GetConfigurationByPath( const ::rtl::OUString& aPath )
{
	osl::MutexGuard aGuard( m_aMutex );

	uno::Reference< container::XNameAccess > xConfig;

	try
	{
		if ( !m_xConfigProvider.is() )
			m_xConfigProvider = uno::Reference< lang::XMultiServiceFactory >(
				m_xFactory->createInstance(
					::rtl::OUString::createFromAscii( "com.sun.star.configuration.ConfigurationProvider" ) ),
				uno::UNO_QUERY_THROW );

		uno::Sequence< uno::Any > aArgs( 1 );
		beans::PropertyValue aPathProp;
		aPathProp.Name = ::rtl::OUString::createFromAscii( "nodepath" );
		aPathProp.Value <<= aPath;
		aArgs[0] <<= aPathProp;

		xConfig = uno::Reference< container::XNameAccess >(
							m_xConfigProvider->createInstanceWithArguments(
								::rtl::OUString::createFromAscii( "com.sun.star.configuration.ConfigurationAccess" ),
								aArgs ),
							uno::UNO_QUERY );
	}
	catch( uno::Exception& )
	{}

	return xConfig;
}

//-----------------------------------------------------------------------
uno::Reference< container::XNameAccess > ConfigurationHelper::GetObjConfiguration()
{
	osl::MutexGuard aGuard( m_aMutex );

	if ( !m_xObjectConfig.is() )
		m_xObjectConfig = GetConfigurationByPath(
										::rtl::OUString::createFromAscii( "/org.openoffice.Office.Embedding/Objects" ) );

	return m_xObjectConfig;
}

//-----------------------------------------------------------------------
uno::Reference< container::XNameAccess > ConfigurationHelper::GetVerbsConfiguration()
{
	osl::MutexGuard aGuard( m_aMutex );

	if ( !m_xVerbsConfig.is() )
		m_xVerbsConfig = GetConfigurationByPath(
										::rtl::OUString::createFromAscii( "/org.openoffice.Office.Embedding/Verbs" ) );

	return m_xVerbsConfig;
}

//-----------------------------------------------------------------------
uno::Reference< container::XNameAccess > ConfigurationHelper::GetMediaTypeConfiguration()
{
	osl::MutexGuard aGuard( m_aMutex );

	if ( !m_xMediaTypeConfig.is() )
		m_xMediaTypeConfig = GetConfigurationByPath(
					::rtl::OUString::createFromAscii( "/org.openoffice.Office.Embedding/MimeTypeClassIDRelations" ) );

	return m_xMediaTypeConfig;
}
//-------------------------------------------------------------------------
::rtl::OUString ConfigurationHelper::GetDocServiceNameFromFilter( const ::rtl::OUString& aFilterName )
{
	::rtl::OUString aDocServiceName;

	try
	{
		uno::Reference< container::XNameAccess > xFilterFactory(
			m_xFactory->createInstance( ::rtl::OUString::createFromAscii( "com.sun.star.document.FilterFactory" ) ),
			uno::UNO_QUERY_THROW );

		uno::Any aFilterAnyData = xFilterFactory->getByName( aFilterName );
		uno::Sequence< beans::PropertyValue > aFilterData;
		if ( aFilterAnyData >>= aFilterData )
		{
			for ( sal_Int32 nInd = 0; nInd < aFilterData.getLength(); nInd++ )
				if ( aFilterData[nInd].Name.equalsAscii( "DocumentService" ) )
					aFilterData[nInd].Value >>= aDocServiceName;

			if ( aDocServiceName.getLength() )
				aDocServiceName = ConvertServiceToImplementationName( aDocServiceName );
		}
	}
	catch( uno::Exception& )
	{}

	return aDocServiceName;
}

//-------------------------------------------------------------------------
::rtl::OUString ConfigurationHelper::GetDocServiceNameFromMediaType( const ::rtl::OUString& aMediaType )
{
    uno::Reference< container::XContainerQuery > xTypeCFG(
			m_xFactory->createInstance(
				::rtl::OUString::createFromAscii( "com.sun.star.document.TypeDetection" ) ),
			uno::UNO_QUERY );

    if ( xTypeCFG.is() )
    {
		try
		{
        	// make query for all types matching the properties
            uno::Sequence < beans::NamedValue > aSeq( 1 );
			aSeq[0].Name = ::rtl::OUString::createFromAscii( "MediaType" );
			aSeq[0].Value <<= aMediaType;

        	uno::Reference < container::XEnumeration > xEnum = xTypeCFG->createSubSetEnumerationByProperties( aSeq );
        	while ( xEnum->hasMoreElements() )
        	{
                uno::Sequence< beans::PropertyValue > aType;
				if ( xEnum->nextElement() >>= aType )
				{
					for ( sal_Int32 nInd = 0; nInd < aType.getLength(); nInd++ )
					{
						::rtl::OUString aFilterName;
                        if ( aType[nInd].Name.equalsAscii( "PreferredFilter" )
						  && ( aType[nInd].Value >>= aFilterName ) && aFilterName.getLength() )
						{
							::rtl::OUString aDocumentName = GetDocServiceNameFromFilter( aFilterName );
							if ( aDocumentName.getLength() )
								return aDocumentName;
						}
					}
				}
			}
        }
		catch( uno::Exception& )
		{}
    }

    return ::rtl::OUString();
}

//-------------------------------------------------------------------------
sal_Bool ConfigurationHelper::GetVerbByShortcut( const ::rtl::OUString& aVerbShortcut,
												embed::VerbDescriptor& aDescriptor )
{
	sal_Bool bResult = sal_False;

	uno::Reference< container::XNameAccess > xVerbsConfig = GetVerbsConfiguration();
	uno::Reference< container::XNameAccess > xVerbsProps;
	try
	{
		if ( xVerbsConfig.is() && ( xVerbsConfig->getByName( aVerbShortcut ) >>= xVerbsProps ) && xVerbsProps.is() )
		{
			embed::VerbDescriptor aTempDescr;
			if ( ( xVerbsProps->getByName( ::rtl::OUString::createFromAscii( "VerbID" ) ) >>= aTempDescr.VerbID )
			  && ( xVerbsProps->getByName( ::rtl::OUString::createFromAscii( "VerbUIName" ) ) >>= aTempDescr.VerbName )
			  && ( xVerbsProps->getByName( ::rtl::OUString::createFromAscii( "VerbFlags" ) ) >>= aTempDescr.VerbFlags )
			  && ( xVerbsProps->getByName( ::rtl::OUString::createFromAscii( "VerbAttributes" ) ) >>= aTempDescr.VerbAttributes ) )
			{
			  	aDescriptor = aTempDescr;
				bResult = sal_True;
			}
		}
	}
	catch( uno::Exception& )
	{
	}

	return bResult;
}

//-------------------------------------------------------------------------
uno::Sequence< beans::NamedValue > ConfigurationHelper::GetObjPropsFromConfigEntry(
											const uno::Sequence< sal_Int8 >& aClassID,
											const uno::Reference< container::XNameAccess >& xObjectProps )
{
	uno::Sequence< beans::NamedValue > aResult;

	if ( aClassID.getLength() == 16 )
	{
		try
		{
			uno::Sequence< ::rtl::OUString > aObjPropNames = xObjectProps->getElementNames();

			aResult.realloc( aObjPropNames.getLength() + 1 );
			aResult[0].Name = ::rtl::OUString::createFromAscii( "ClassID" );
			aResult[0].Value <<= aClassID;

			for ( sal_Int32 nInd = 0; nInd < aObjPropNames.getLength(); nInd++ )
			{
				aResult[nInd + 1].Name = aObjPropNames[nInd];

				if ( aObjPropNames[nInd].equalsAscii( "ObjectVerbs" ) )
				{
					uno::Sequence< ::rtl::OUString > aVerbShortcuts;
					if ( xObjectProps->getByName( aObjPropNames[nInd] ) >>= aVerbShortcuts )
					{
						uno::Sequence< embed::VerbDescriptor > aVerbDescriptors( aVerbShortcuts.getLength() );
						for ( sal_Int32 nVerbI = 0; nVerbI < aVerbShortcuts.getLength(); nVerbI++ )
							if ( !GetVerbByShortcut( aVerbShortcuts[nVerbI], aVerbDescriptors[nVerbI] ) )
								throw uno::RuntimeException();

						aResult[nInd+1].Value <<= aVerbDescriptors;
					}
					else
						throw uno::RuntimeException();
				}
				else
					aResult[nInd+1].Value = xObjectProps->getByName( aObjPropNames[nInd] );
			}
		}
		catch( uno::Exception& )
		{
			aResult.realloc( 0 );
		}
	}

	return aResult;
}

//-----------------------------------------------------------------------
::rtl::OUString ConfigurationHelper::GetExplicitlyRegisteredObjClassID( const ::rtl::OUString& aMediaType )
{
	::rtl::OUString aStringClassID;

	uno::Reference< container::XNameAccess > xMediaTypeConfig = GetMediaTypeConfiguration();
	try
	{
		if ( xMediaTypeConfig.is() )
			xMediaTypeConfig->getByName( aMediaType ) >>= aStringClassID;
	}
	catch( uno::Exception& )
	{
	}

	return aStringClassID;

}

//-----------------------------------------------------------------------
uno::Sequence< beans::NamedValue > ConfigurationHelper::GetObjectPropsByStringClassID(
																const ::rtl::OUString& aStringClassID )
{
	uno::Sequence< beans::NamedValue > aObjProps;

	uno::Sequence< sal_Int8 > aClassID = GetSequenceClassIDRepresentation( aStringClassID );
    if ( ClassIDsEqual( aClassID, GetSequenceClassID( SO3_DUMMY_CLASSID ) ) )
    {
        aObjProps.realloc(2);
        aObjProps[0].Name = ::rtl::OUString::createFromAscii("ObjectFactory");
        aObjProps[0].Value <<= ::rtl::OUString::createFromAscii("com.sun.star.embed.OOoSpecialEmbeddedObjectFactory");
        aObjProps[1].Name = ::rtl::OUString::createFromAscii("ClassID");
        aObjProps[1].Value <<= aClassID;
        return aObjProps;
    }

	if ( aClassID.getLength() == 16 )
	{
		uno::Reference< container::XNameAccess > xObjConfig = GetObjConfiguration();
		uno::Reference< container::XNameAccess > xObjectProps;
		try
		{
			// TODO/LATER: allow to provide ClassID string in any format, only digits are counted
			if ( xObjConfig.is() && ( xObjConfig->getByName( aStringClassID.toAsciiUpperCase() ) >>= xObjectProps ) && xObjectProps.is() )
				aObjProps = GetObjPropsFromConfigEntry( aClassID, xObjectProps );
		}
		catch( uno::Exception& )
		{
		}
	}

	return aObjProps;
}

//-----------------------------------------------------------------------
uno::Sequence< beans::NamedValue > ConfigurationHelper::GetObjectPropsByClassID(
																const uno::Sequence< sal_Int8 >& aClassID )
{
	uno::Sequence< beans::NamedValue > aObjProps;
    if ( ClassIDsEqual( aClassID, GetSequenceClassID( SO3_DUMMY_CLASSID ) ) )
    {
        aObjProps.realloc(2);
        aObjProps[0].Name = ::rtl::OUString::createFromAscii("ObjectFactory");
        aObjProps[0].Value <<= ::rtl::OUString::createFromAscii("com.sun.star.embed.OOoSpecialEmbeddedObjectFactory");
        aObjProps[1].Name = ::rtl::OUString::createFromAscii("ClassID");
        aObjProps[1].Value <<= aClassID;
    }

	::rtl::OUString aStringClassID = GetStringClassIDRepresentation( aClassID );
	if ( aStringClassID.getLength() )
	{
		uno::Reference< container::XNameAccess > xObjConfig = GetObjConfiguration();
		uno::Reference< container::XNameAccess > xObjectProps;
		try
		{
			if ( xObjConfig.is() && ( xObjConfig->getByName( aStringClassID.toAsciiUpperCase() ) >>= xObjectProps ) && xObjectProps.is() )
				aObjProps = GetObjPropsFromConfigEntry( aClassID, xObjectProps );
		}
		catch( uno::Exception& )
		{
		}
	}

	return aObjProps;
}

//-----------------------------------------------------------------------
uno::Sequence< beans::NamedValue > ConfigurationHelper::GetObjectPropsByMediaType( const ::rtl::OUString& aMediaType )
{
	uno::Sequence< beans::NamedValue > aObject =
									GetObjectPropsByStringClassID( GetExplicitlyRegisteredObjClassID( aMediaType ) );
	if ( aObject.getLength() )
		return aObject;

	::rtl::OUString aDocumentName = GetDocServiceNameFromMediaType( aMediaType );
	if ( aDocumentName.getLength() )
		return GetObjectPropsByDocumentName( aDocumentName );

	return uno::Sequence< beans::NamedValue >();
}

//-----------------------------------------------------------------------
uno::Sequence< beans::NamedValue > ConfigurationHelper::GetObjectPropsByFilter( const ::rtl::OUString& aFilterName )
{
	::rtl::OUString aDocumentName = GetDocServiceNameFromFilter( aFilterName );
	if ( aDocumentName.getLength() )
		return GetObjectPropsByDocumentName( aDocumentName );

	return uno::Sequence< beans::NamedValue >();
}

//-----------------------------------------------------------------------
uno::Sequence< beans::NamedValue > ConfigurationHelper::GetObjectPropsByDocumentName( const ::rtl::OUString& aDocName )
{
	if ( aDocName.getLength() )
	{
		uno::Reference< container::XNameAccess > xObjConfig = GetObjConfiguration();
		if ( xObjConfig.is() )
		{
			try
			{
				uno::Sequence< ::rtl::OUString > aClassIDs = xObjConfig->getElementNames();
				for ( sal_Int32 nInd = 0; nInd < aClassIDs.getLength(); nInd++ )
				{
					uno::Reference< container::XNameAccess > xObjectProps;
					::rtl::OUString aEntryDocName;

					if ( ( xObjConfig->getByName( aClassIDs[nInd] ) >>= xObjectProps ) && xObjectProps.is()
					  && ( xObjectProps->getByName(
			  					::rtl::OUString::createFromAscii( "ObjectDocumentServiceName" ) ) >>= aEntryDocName )
					  && aEntryDocName.equals( aDocName ) )
					{
						return GetObjPropsFromConfigEntry( GetSequenceClassIDRepresentation( aClassIDs[nInd] ),
															xObjectProps );
					}
				}
			}
			catch( uno::Exception& )
			{}
		}
	}

	return uno::Sequence< beans::NamedValue >();
}

//-----------------------------------------------------------------------
::rtl::OUString ConfigurationHelper::GetFactoryNameByClassID( const uno::Sequence< sal_Int8 >& aClassID )
{
	return GetFactoryNameByStringClassID( GetStringClassIDRepresentation( aClassID ) );
}

//-----------------------------------------------------------------------
::rtl::OUString ConfigurationHelper::GetFactoryNameByStringClassID( const ::rtl::OUString& aStringClassID )
{
	::rtl::OUString aResult;

	if ( aStringClassID.getLength() )
	{
		uno::Reference< container::XNameAccess > xObjConfig = GetObjConfiguration();
		uno::Reference< container::XNameAccess > xObjectProps;
		try
		{
			if ( xObjConfig.is() && ( xObjConfig->getByName( aStringClassID.toAsciiUpperCase() ) >>= xObjectProps ) && xObjectProps.is() )
				xObjectProps->getByName( ::rtl::OUString::createFromAscii( "ObjectFactory" ) ) >>= aResult;
		}
		catch( uno::Exception& )
		{
            uno::Sequence< sal_Int8 > aClassID = GetSequenceClassIDRepresentation( aStringClassID );
            if ( ClassIDsEqual( aClassID, GetSequenceClassID( SO3_DUMMY_CLASSID ) ) )
                return ::rtl::OUString::createFromAscii("com.sun.star.embed.OOoSpecialEmbeddedObjectFactory");
		}
	}

	return aResult;
}

//-----------------------------------------------------------------------
::rtl::OUString ConfigurationHelper::GetFactoryNameByDocumentName( const ::rtl::OUString& aDocName )
{
	::rtl::OUString aResult;

	if ( aDocName.getLength() )
	{
		uno::Reference< container::XNameAccess > xObjConfig = GetObjConfiguration();
		if ( xObjConfig.is() )
		{
			try
			{
				uno::Sequence< ::rtl::OUString > aClassIDs = xObjConfig->getElementNames();
				for ( sal_Int32 nInd = 0; nInd < aClassIDs.getLength(); nInd++ )
				{
					uno::Reference< container::XNameAccess > xObjectProps;
					::rtl::OUString aEntryDocName;

					if ( ( xObjConfig->getByName( aClassIDs[nInd] ) >>= xObjectProps ) && xObjectProps.is()
					  && ( xObjectProps->getByName(
			  					::rtl::OUString::createFromAscii( "ObjectDocumentServiceName" ) ) >>= aEntryDocName )
					  && aEntryDocName.equals( aDocName ) )
					{
						xObjectProps->getByName( ::rtl::OUString::createFromAscii( "ObjectFactory" ) ) >>= aResult;
                        break;
					}
				}
			}
			catch( uno::Exception& )
			{}
		}
	}

	return aResult;
}


//-----------------------------------------------------------------------
::rtl::OUString ConfigurationHelper::GetFactoryNameByMediaType( const ::rtl::OUString& aMediaType )
{
	::rtl::OUString aResult = GetFactoryNameByStringClassID( GetExplicitlyRegisteredObjClassID( aMediaType ) );

	if ( !aResult.getLength() )
	{
		::rtl::OUString aDocumentName = GetDocServiceNameFromMediaType( aMediaType );
		if ( aDocumentName.getLength() )
			aResult = GetFactoryNameByDocumentName( aDocumentName );
	}

	return aResult;
}

//-----------------------------------------------------------------------
::rtl::OUString ConfigurationHelper::UpdateMediaDescriptorWithFilterName(
										uno::Sequence< beans::PropertyValue >& aMediaDescr,
										sal_Bool bIgnoreType )
{
	::rtl::OUString aFilterName;

	for ( sal_Int32 nInd = 0; nInd < aMediaDescr.getLength(); nInd++ )
		if ( aMediaDescr[nInd].Name.equalsAscii( "FilterName" ) )
			aMediaDescr[nInd].Value >>= aFilterName;

	if ( !aFilterName.getLength() )
	{
		// filter name is not specified, so type detection should be done

		uno::Reference< document::XTypeDetection > xTypeDetection(
				m_xFactory->createInstance( ::rtl::OUString::createFromAscii( "com.sun.star.document.TypeDetection" ) ),
				uno::UNO_QUERY );

		if ( !xTypeDetection.is() )
			throw uno::RuntimeException(); // TODO

		// typedetection can change the mode, add a stream and so on, thus a copy should be used
		uno::Sequence< beans::PropertyValue > aTempMD( aMediaDescr );

		// get TypeName
		::rtl::OUString aTypeName = xTypeDetection->queryTypeByDescriptor( aTempMD, sal_True );

		// get FilterName
		for ( sal_Int32 nInd = 0; nInd < aTempMD.getLength(); nInd++ )
			if ( aTempMD[nInd].Name.equalsAscii( "FilterName" ) )
				aTempMD[nInd].Value >>= aFilterName;

		if ( aFilterName.getLength() )
		{
			sal_Int32 nOldLen = aMediaDescr.getLength();
			aMediaDescr.realloc( nOldLen + 1 );
			aMediaDescr[nOldLen].Name = ::rtl::OUString::createFromAscii( "FilterName" );
			aMediaDescr[ nOldLen ].Value <<= aFilterName;

		}
		else if ( aTypeName.getLength() && !bIgnoreType )
		{
			uno::Reference< container::XNameAccess > xNameAccess( xTypeDetection, uno::UNO_QUERY );
			uno::Sequence< beans::PropertyValue > aTypes;

			if ( xNameAccess.is() && ( xNameAccess->getByName( aTypeName ) >>= aTypes ) )
			{
				for ( sal_Int32 nInd = 0; nInd < aTypes.getLength(); nInd++ )
				{
					if ( aTypes[nInd].Name.equalsAscii( "PreferredFilter" ) && ( aTypes[nInd].Value >>= aFilterName ) )
					{
						sal_Int32 nOldLen = aMediaDescr.getLength();
						aMediaDescr.realloc( nOldLen + 1 );
						aMediaDescr[nOldLen].Name = ::rtl::OUString::createFromAscii( "FilterName" );
						aMediaDescr[ nOldLen ].Value = aTypes[nInd].Value;
						break;
					}
				}
			}
		}
	}

	return aFilterName;
}

::rtl::OUString ConfigurationHelper::UpdateMediaDescriptorWithFilterName(
						uno::Sequence< beans::PropertyValue >& aMediaDescr,
						uno::Sequence< beans::NamedValue >& aObject )
{
	::rtl::OUString aDocName;
	for ( sal_Int32 nInd = 0; nInd < aObject.getLength(); nInd++ )
		if ( aObject[nInd].Name.equalsAscii( "ObjectDocumentServiceName" ) )
		{
			aObject[nInd].Value >>= aDocName;
			break;
		}

	OSL_ENSURE( aDocName.getLength(), "The name must exist at this point!\n" );


	sal_Bool bNeedsAddition = sal_True;
	for ( sal_Int32 nMedInd = 0; nMedInd < aMediaDescr.getLength(); nMedInd++ )
		if ( aMediaDescr[nMedInd].Name.equalsAscii( "DocumentService" ) )
		{
			aMediaDescr[nMedInd].Value <<= aDocName;
			bNeedsAddition = sal_False;
			break;
		}

	if ( bNeedsAddition )
	{
		sal_Int32 nOldLen = aMediaDescr.getLength();
		aMediaDescr.realloc( nOldLen + 1 );
		aMediaDescr[nOldLen].Name = ::rtl::OUString::createFromAscii( "DocumentService" );
		aMediaDescr[nOldLen].Value <<= aDocName;
	}

	return UpdateMediaDescriptorWithFilterName( aMediaDescr, sal_True );
}

sal_Bool ConfigurationHelper::AddFilterNameCheckOwnFile(
						uno::Sequence< beans::PropertyValue >& aMediaDescr )
{
	::rtl::OUString aFilterName = UpdateMediaDescriptorWithFilterName( aMediaDescr, sal_False );
	if ( aFilterName.getLength() )
	{
		try
		{
			uno::Reference< container::XNameAccess > xFilterFactory(
				m_xFactory->createInstance( ::rtl::OUString::createFromAscii( "com.sun.star.document.FilterFactory" ) ),
				uno::UNO_QUERY_THROW );

			uno::Any aFilterAnyData = xFilterFactory->getByName( aFilterName );
			uno::Sequence< beans::PropertyValue > aFilterData;
			if ( aFilterAnyData >>= aFilterData )
			{
				for ( sal_Int32 nInd = 0; nInd < aFilterData.getLength(); nInd++ )
					if ( aFilterData[nInd].Name.equalsAscii( "Flags" ) )
					{
						uno::Any aVal = aFilterData[nInd].Value;
						sal_Int32 nFlags = 0;
						// check the OWN flag
						if ( ( aFilterData[nInd].Value >>= nFlags ) && ( nFlags & 0x20 ) )
							return sal_True;
						break;
					}
			}
		}
		catch( uno::Exception& )
		{}
	}

	return sal_False;
}

