/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: redrawmanagement.hxx,v $
 *
 *  $Revision: 1.4 $
 *
 *  last change: $Author: rt $ $Date: 2005/09/07 23:05:41 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/

#ifndef _CANVAS_INTERNAL_REDRAWMANAGEMENT_HXX
#define _CANVAS_INTERNAL_REDRAWMANAGEMENT_HXX

#include <canvas/debug.hxx>

#ifndef _BGFX_POINT_B2DPOINT_HXX
#include <basegfx/point/b2dpoint.hxx>
#endif
#ifndef _BGFX_VECTOR_B2DSIZE_HXX
#include <basegfx/vector/b2dsize.hxx>
#endif
#ifndef _BGFX_RANGE_B2DRECTANGLE_HXX
#include <basegfx/range/b2drectangle.hxx>
#endif

#include <algorithm>
#include <vector>


namespace canvas
{

    namespace internal
    {
        /** Helper struct for SpriteTracer template

        	This templated struct stores change information to a
        	sprite's visual appearance (move, content updated, and the like).

            @tpl SpriteRefType
            Reference to sprite. Simply plug in the corresponding
            (smart) pointer for your sprite class type here.
         */
        template< typename SpriteRefType > struct SpriteChangeRecord
        {
            typedef enum{ none=0, move, update } ChangeType;

            SpriteChangeRecord() :
                meChangeType( none ),
                mpAffectedSprite(),
                maOldPos(),
                maUpdateArea()
            {
            }

            SpriteChangeRecord( const SpriteRefType& 		rSprite, 
                                const ::basegfx::B2DPoint& 	rOldPos,
                                const ::basegfx::B2DPoint& 	rNewPos,
                                const ::basegfx::B2DSize& 	rSpriteSize ) :
                meChangeType( move ),
                mpAffectedSprite( rSprite ),
                maOldPos( rOldPos ),
                maUpdateArea( rNewPos.getX(),
                              rNewPos.getY(),
                              rNewPos.getX() + rSpriteSize.getX(),
                              rNewPos.getY() + rSpriteSize.getY() )
            {
            }

            SpriteChangeRecord( const SpriteRefType& 			rSprite, 
                                const ::basegfx::B2DPoint& 		rPos,
                                const ::basegfx::B2DRectangle& 	rUpdateArea ) :
                meChangeType( update ),
                mpAffectedSprite( rSprite ),
                maOldPos( rPos ),
                maUpdateArea( rUpdateArea )
            {
            }

            SpriteRefType getSprite() const { return mpAffectedSprite; }

            ChangeType				meChangeType;
            SpriteRefType			mpAffectedSprite;
            ::basegfx::B2DPoint		maOldPos;
            ::basegfx::B2DRectangle	maUpdateArea;
        };

        /** Helper class to condense sprite updates into a single action

        	This template class tracks the sprite changes over the
        	recorded change list, and generates a single update action
        	from that (note that per screen update, several moves,
        	visibility changes and content updates might happen)

            @tpl SpriteRefType
            Reference to sprite. Simply plug in the corresponding
            (smart) pointer for your sprite class type here. The
            dereferenced SpriteRefType object must provide a getSize()
            method, returning something that is convertible to
            ::basegfx::B2DSize.

            @tpl UpdateCollector
            This is the sprite update range collector type to operate
            with. The type given here must provide an addRange( const
            ::basegfx::B2DRange& rRect, const SpriteRefType& rSprite)
            method, where rRect denotes the area that needs update,
            and rSprite the sprite that is to be drawn.
         */
        template< typename SpriteRefType, typename UpdateCollector > class SpriteTracer
        {
        public:
            typedef SpriteChangeRecord< SpriteRefType > SpriteChangeRecordType;

            SpriteTracer( const SpriteRefType& rAffectedSprite ) :
                mpAffectedSprite(rAffectedSprite),
                maMoveStartArea(),
                maMoveEndArea(),
                mbFirstMove( true ),
                mbIsMove( false )
            {
            }

            void operator()( const SpriteChangeRecordType& rSpriteRecord )
            {
                // only deal with change events from the currently
                // affected sprite
                if( rSpriteRecord.mpAffectedSprite == mpAffectedSprite )
                {
                    switch( rSpriteRecord.meChangeType )
                    {
                        case SpriteChangeRecordType::move:
                            if( mbFirstMove )
                            {
                                maMoveStartArea = ::basegfx::B2DRectangle(
                                    rSpriteRecord.maOldPos,
                                    rSpriteRecord.maOldPos + rSpriteRecord.maUpdateArea.getRange() );
                                mbFirstMove 	= false;
                            }

                            maMoveEndArea	= rSpriteRecord.maUpdateArea;
                            mbIsMove 		= true;
                            break;

                        case SpriteChangeRecordType::update:
                            // update end update area of the
                            // sprite. Thus, every update() action
                            // _after_ the last move will correctly
                            // update the final repaint area. And this
                            // does not interfere with subsequent
                            // moves, because moves always perform a
                            // hard set of maMoveEndArea to their
                            // stored value
                            maMoveEndArea.expand( rSpriteRecord.maUpdateArea );
                            break;

                        default:
                            ENSURE_AND_THROW( false, 
                                              "Unexpected case in SpriteUpdater::operator()" );
                            break;
                    }
                }
            }

            void commit( UpdateCollector& rUpdateCollector ) const
            {
                if( mbIsMove )
                {
                    // do new and old sprite area overlap? if yes, we have
                    // to prepare the whole area
                    if( maMoveStartArea.overlaps( maMoveEndArea ) )
                    {
                        ::basegfx::B2DRectangle aTemp( maMoveStartArea );
                        aTemp.expand( maMoveEndArea );

                        rUpdateCollector.addRange( aTemp,
                                                   mpAffectedSprite );
                    }
                    else
                    {
                        // we can update new and old position separately

                        // first, draw the new sprite position
                        rUpdateCollector.addRange( maMoveEndArea, 
                                                   mpAffectedSprite );

                        // then, clear the old place (looks smoother
                        // this way)
                        rUpdateCollector.addRange( maMoveStartArea, 
                                                   SpriteRefType() );
                    }
                }
                else if( !maMoveEndArea.isEmpty() )
                {
                    rUpdateCollector.addRange( maMoveEndArea, 
                                               mpAffectedSprite );
                }
            }

        private:
            SpriteRefType			mpAffectedSprite;
            ::basegfx::B2DRectangle	maMoveStartArea;
            ::basegfx::B2DRectangle	maMoveEndArea;
            bool					mbFirstMove;
            bool					mbIsMove;        
        };


        /** SpriteChecker functor, which for every sprite checks the
            given update vector for necessary screen updates

            @tpl SpriteRefType
            Reference to sprite. Simply plug in the corresponding
            (smart) pointer for your sprite class type here. The
            dereferenced SpriteRefType object must provide a getSize()
            method, returning something that is convertible to
            ::basegfx::B2DSize.

            @tpl RecordContainerType
            An STL container (or something concept-compatible), which
            contains the SpriteChangeRecord entries

            @tpl UpdaterType
            Update target type. This class must provide an addRange()
            method, see SpriteTracer template.
            
         */
        template< typename SpriteRefType, class RecordContainerType, class UpdateCollector > class SpriteUpdater
        {
        public:
            typedef SpriteTracer< SpriteRefType, UpdateCollector >	SpriteTracerType;

            /** Generate update area list
                
            	@param rUpdater
                Reference to an updater object, which will receive the
                update areas.

                @param rChangeContainer
                Container with all sprite change requests

             */
            SpriteUpdater( UpdateCollector&				rUpdater,
                           const RecordContainerType&	rChangeContainer ) :
                mrUpdater( rUpdater ),
                mrChangeContainer( rChangeContainer )
            {
            }

            /** Call this method for every sprite on your screen

            	This method scans the change container, collecting all
            	update info for the given sprite into one or two
            	update operations, which in turn are inserted into the
            	connected ranges processor.

                @param rSprite
                Current sprite to collect update info for.
             */
            void operator()( const SpriteRefType& rSprite )
            {
                const SpriteTracerType aSpriteTracer( 
                    ::std::for_each( mrChangeContainer.begin(), 
                                     mrChangeContainer.end(), 
                                     SpriteTracerType( rSprite ) ) );

                aSpriteTracer.commit( mrUpdater );
            }

        private:
            UpdateCollector&			mrUpdater;
            const RecordContainerType&	mrChangeContainer;
        };

    }
}

#endif /* _CANVAS_INTERNAL_REDRAWMANAGEMENT_HXX */
