/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: dx_canvascustomsprite.cxx,v $
 *
 *  $Revision: 1.5 $
 *
 *  last change: $Author: rt $ $Date: 2005/09/07 23:25:33 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/

#include <canvas/debug.hxx>
#include <canvas/verbosetrace.hxx>

#ifndef _RTL_LOGFILE_HXX_
#include <rtl/logfile.hxx>
#endif
#ifndef INCLUDED_RTL_MATH_HXX
#include <rtl/math.hxx>
#endif

#ifndef _RTL_ALLOC_H_
#include <rtl/alloc.h>
#endif

#include <canvas/canvastools.hxx>

#include <dx_canvascustomsprite.hxx>
#include <dx_spritecanvas.hxx>
#include <dx_impltools.hxx>
#include <dx_surfacemanager.hxx>
#include <dx_gdisurface.hxx>
#include <dx_polypolygonrasterizer.hxx>

#ifndef _BGFX_MATRIX_B2DHOMMATRIX_HXX
#include <basegfx/matrix/b2dhommatrix.hxx>
#endif
#ifndef _BGFX_POINT_B2DPOINT_HXX
#include <basegfx/point/b2dpoint.hxx>
#endif
#ifndef _BGFX_TOOLS_CANVASTOOLS_HXX
#include <basegfx/tools/canvastools.hxx>
#endif
#ifndef _BGFX_NUMERIC_FTOOLS_HXX
#include <basegfx/numeric/ftools.hxx>
#endif
#ifndef _BGFX_POLYGON_B2DPOLYPOLYGONRASTERCONVERTER_HXX
#include <basegfx/polygon/b2dpolypolygonrasterconverter.hxx>
#endif


using namespace ::com::sun::star;

namespace dxcanvas
{
    /** A border of width ANTIALIASING_BORDER is put around every
        sprite, to account for alpha effects at the border of objects
    */
    enum{ ANTIALIASING_BORDER=2 };
    

    CanvasCustomSprite::CanvasCustomSprite( const geometry::RealSize2D& 		rSpriteSize,
                                            const WindowGraphicDevice::ImplRef&	rRefDevice,
                                            const SpriteCanvas::ImplRef&		rSpriteCanvas,
                                            const DeviceSharedPtr&				rDevice ) :
        mxRefDevice( rRefDevice ),
        mpSpriteCanvas( rSpriteCanvas ),
        mpBitmap(),
        mpDXDevice( rDevice ),
        mpDXTexture(),
        // subtract AA border back from sprite pos, to cancel the
        // effect of the shifted output (see CanvasHelper::setBitmap
        // below)
        maOutputOffset( -ANTIALIASING_BORDER,
                        -ANTIALIASING_BORDER ),
        // round to next larger integer, to make fractional pixel
        // coverage visible
        maOriginalSize( static_cast<sal_Int32>(rSpriteSize.Width + 1.0),
                        static_cast<sal_Int32>(rSpriteSize.Height + 1.0) ),
        // Add AA border to original size
        maSize( maOriginalSize.getX() + 2*ANTIALIASING_BORDER,
                maOriginalSize.getY() + 2*ANTIALIASING_BORDER ),
        maPosition(),
        mxClipPoly(),
        mfAlpha(0.0),
        mbActive( false ),
        mbClipDirty( true ),
        mbIsContentFullyOpaque( false )
    {
        maCanvasHelper.setGraphicDevice( rRefDevice );

        mpBitmap.reset( 
            new Gdiplus::Bitmap( 
                maSize.getX(),
                maSize.getY(),
                PixelFormat32bppARGB 
                ) );

        // set larger bitmap target. Shift output ANTIALIASING_BORDER
        // down-right, to allow antialiased pixel at the top and left
        // border appear.
        maCanvasHelper.setBitmap( mpBitmap,
                                  ::basegfx::B2ISize(ANTIALIASING_BORDER,
                                                     ANTIALIASING_BORDER) );
    }

    CanvasCustomSprite::~CanvasCustomSprite()
    {
    }
    
    void SAL_CALL CanvasCustomSprite::disposing()
    {
        ::osl::MutexGuard aGuard( m_aMutex );

        mxRefDevice.reset();
        mpSpriteCanvas.clear();
        mpBitmap.reset();
        mpDXDevice.reset();
        mpDXTexture.reset();

        // forward to parent
        CanvasCustomSprite_Base::disposing();
    }

    void SAL_CALL CanvasCustomSprite::setAlpha( double alpha ) throw (uno::RuntimeException)
    {
        ::osl::MutexGuard aGuard( m_aMutex );

        ::canvas::tools::checkRange(alpha, 0.0, 1.0);

        if( alpha != mfAlpha )
        {
            if( mbActive )
                mpSpriteCanvas->updateSprite( Sprite::ImplRef( this ), 
                                              maPosition,
                                              getSpriteRect() );
            mfAlpha = alpha;        
        }
    }

    void SAL_CALL CanvasCustomSprite::move( const geometry::RealPoint2D& 	aNewPos, 
                                            const rendering::ViewState& 	viewState, 
                                            const rendering::RenderState& 	renderState ) throw (uno::RuntimeException)
    {
        ::osl::MutexGuard aGuard( m_aMutex );

        ::basegfx::B2DHomMatrix aTransform;
        ::canvas::tools::mergeViewAndRenderTransform(aTransform,
                                                     viewState, 
                                                     renderState);

        // subtract output offset
        if( !maOutputOffset.equalZero() )
        {
            ::basegfx::B2DHomMatrix aOutputOffset;
            aOutputOffset.translate( maOutputOffset.getX(), 
                                     maOutputOffset.getY() );
            
            aTransform = aOutputOffset * aTransform;
        }

        ::basegfx::B2DPoint aPoint( ::basegfx::unotools::b2DPointFromRealPoint2D(aNewPos) );
        aPoint *= aTransform;

        if( aPoint != maPosition )
        {
            if( mbActive )
                mpSpriteCanvas->moveSprite( Sprite::ImplRef( this ), 
                                            maPosition,
                                            aPoint );
            maPosition = aPoint;
        }
    }

    void SAL_CALL CanvasCustomSprite::transform( const geometry::AffineMatrix2D& aTransformation ) throw (lang::IllegalArgumentException, 
                                                                                                          uno::RuntimeException)
    {
        ::osl::MutexGuard aGuard( m_aMutex );

        // TODO(P3): Implement transformed sprites
        if( mbActive )
            mpSpriteCanvas->updateSprite( Sprite::ImplRef( this ), 
                                          maPosition,
                                          getSpriteRect() );
    }

	void SAL_CALL CanvasCustomSprite::clip( const uno::Reference< rendering::XPolyPolygon2D >& xClip ) throw (uno::RuntimeException)
    {
        ::osl::MutexGuard aGuard( m_aMutex );

        mxClipPoly = xClip;

        if( mbActive )
        {
            mpSpriteCanvas->updateSprite( Sprite::ImplRef( this ), 
                                          maPosition,
                                          getSpriteRect() );

            // when DX is active, changing the sprite clip makes a
            // texture update necessary, because we (mis)use the alpha
            // channel as the clip mask.
            if( mpDXTexture.get() != NULL )
                mbClipDirty = true;
        }
    }

    void SAL_CALL CanvasCustomSprite::setPriority( double nPriority ) throw (uno::RuntimeException)
    {
        ::osl::MutexGuard aGuard( m_aMutex );

        // TODO(F2): Implement sprite priority
    }

    void SAL_CALL CanvasCustomSprite::show(  ) throw (uno::RuntimeException)
    {
        ::osl::MutexGuard aGuard( m_aMutex );

        if( !mbActive )
        {
            mpSpriteCanvas->showSprite( Sprite::ImplRef( this ) );
            mbActive = true;

            if( mfAlpha != 0.0 )
            {
                mpSpriteCanvas->updateSprite( Sprite::ImplRef( this ), 
                                              maPosition,
                                              getSpriteRect() );
            }
        }
    }

    void SAL_CALL CanvasCustomSprite::hide(  ) throw (uno::RuntimeException)
    {
        ::osl::MutexGuard aGuard( m_aMutex );

        if( mbActive )
        {
            mpSpriteCanvas->hideSprite( Sprite::ImplRef( this ) );
            mbActive = false;

            if( mfAlpha != 0.0 )
            {
                mpSpriteCanvas->updateSprite( Sprite::ImplRef( this ), 
                                              maPosition,
                                              getSpriteRect() );
            }
        }
    }

    uno::Reference< rendering::XCanvas > SAL_CALL CanvasCustomSprite::getContentCanvas() throw (uno::RuntimeException)
    {
        ::osl::MutexGuard aGuard( m_aMutex );

        if( mbActive )
        {
            // the client is about to render into the sprite. Thus,
            // potentially the whole sprite area has changed.
            mpSpriteCanvas->updateSprite( Sprite::ImplRef( this ), 
                                          maPosition,
                                          getSpriteRect() );
        }

        // clear surface
        SurfaceGraphicsSharedPtr aGraphics( maCanvasHelper.getSurface()->getGraphics() );

        Gdiplus::Color transColor(0, 255, 255, 255);
        (*aGraphics.get())->SetCompositingMode( Gdiplus::CompositingModeSourceCopy ); // force set, don't blend
        (*aGraphics.get())->Clear( transColor );

        // surface content has changed (we cleared it, at least)
        mbSurfaceDirty = true;

        // clients will potentially render into bitmap, thus, cannot
        // be sure that content will remain fully opaque
        mbIsContentFullyOpaque = false;

        return this;
    }

#define SERVICE_NAME "com.sun.star.rendering.CanvasCustomSprite"

    ::rtl::OUString SAL_CALL CanvasCustomSprite::getImplementationName() throw( uno::RuntimeException )
    {
        return ::rtl::OUString( RTL_CONSTASCII_USTRINGPARAM( CANVASCUSTOMSPRITE_IMPLEMENTATION_NAME ) );
    }

    sal_Bool SAL_CALL CanvasCustomSprite::supportsService( const ::rtl::OUString& ServiceName ) throw( uno::RuntimeException )
    {
        return ServiceName.equalsAsciiL( RTL_CONSTASCII_STRINGPARAM ( SERVICE_NAME ) );
    }

    uno::Sequence< ::rtl::OUString > SAL_CALL CanvasCustomSprite::getSupportedServiceNames()  throw( uno::RuntimeException )
    {
        uno::Sequence< ::rtl::OUString > aRet(1);
        aRet[0] = ::rtl::OUString( RTL_CONSTASCII_USTRINGPARAM ( SERVICE_NAME ) );
        
        return aRet;
    }

    // Sprite
    void CanvasCustomSprite::redraw( const DeviceSharedPtr& rTarget ) const
    {
        ::osl::MutexGuard aGuard( m_aMutex );

        redraw( rTarget, 
                maPosition );
    }

    namespace
    {
        bool checkAlphaOpacity( const Gdiplus::BitmapData& rBitmapData )
        {
            // be a little bit tolerant at the border of the sprite:
            // due to antialiasing, those areas might not contain
            // fully opaque pixel, although mathematically, the whole
            // sprite is filled with an opaque object.
            const int nBorder( 2*ANTIALIASING_BORDER );
            const ::std::size_t nHeight( rBitmapData.Height > nBorder ? 
                                         rBitmapData.Height - nBorder : 0 );
            const ::std::size_t nWidth( rBitmapData.Width > nBorder ? 
                                        rBitmapData.Width - nBorder : 0 );
            sal_uInt8* pDest = (sal_uInt8*)rBitmapData.Scan0 + 
                rBitmapData.Stride*nBorder + 4*nBorder;
            sal_uInt8* pCurrPix = pDest+3;

            for( ::std::size_t y=nBorder; y<nHeight; ++y )
            {
                for( ::std::size_t x=nBorder; x<nWidth; ++x, pCurrPix += 4 )
                    if( *pCurrPix != 255 )
                        return false;

                pDest += rBitmapData.Stride;
                pCurrPix = pDest+3;
            }

            return true; // all bitmap's alpha values are 255 -> opaque
        }
    }

    void CanvasCustomSprite::redraw( const DeviceSharedPtr& 	rTarget, 
                                     const ::basegfx::B2DPoint& rOutputPosition ) const
    {
        RTL_LOGFILE_CONTEXT( aLog, "::dxcanvas::CanvasCustomSprite::redraw()" );

        ::osl::MutexGuard aGuard( m_aMutex );

        if( !::basegfx::fTools::equalZero( mfAlpha ) )
        {
            if( mpDXTexture.get() == NULL ||
                mpDXTexture->contentLost()		)
            {
                // (re-)create DirectX texture
                mpDXTexture.reset();
                mpDXTexture = SurfaceManager::getInstance().createTexture( maSize,
                                                                           0.5, 
                                                                           mpDXDevice );
                mbSurfaceDirty = true;
            }

            // render sprite content to primary surface
            if( mpDXTexture.get() != NULL )
            {
                // does the DirectX texture need an update?
                if( mbSurfaceDirty )
                {
                    // yes -> transfer sprite bitmap content to
                    // texture
                    mpDXDevice->clearSurface( mpDXTexture );

                    Gdiplus::BitmapData aBitmapData;

                    aBitmapData.Width = maSize.getX();
                    aBitmapData.Height = maSize.getY();
                    aBitmapData.Stride = 4*aBitmapData.Width; // implicitely DWORD-aligned
                    aBitmapData.PixelFormat = PixelFormat32bppARGB;

                    try
                    {
                        // TODO(P2): There exist a case where we can
                        // avoid this temp mem copy here, but pass a
                        // texture pointer directly as the Scan0 mem:
                        // bitmap and texture layout are exactly the
                        // same (which also means same size and
                        // padding), and we don't need no clip alpha
                        // buffer.
                        aBitmapData.Scan0 = rtl_allocateMemory( aBitmapData.Stride*aBitmapData.Height );
                        aBitmapData.Reserved = NULL;
                        
                        if( Gdiplus::Ok == mpBitmap->LockBits(
                                NULL,
                                Gdiplus::ImageLockModeRead|Gdiplus::ImageLockModeUserInputBuf,
                                PixelFormat32bppARGB,
                                &aBitmapData) )
                        {
                            mpBitmap->UnlockBits(&aBitmapData);
                        }

                        // apply clip (if any)
                        if( mxClipPoly.is() )
                        {
                            // since DirectX does not provide appropriate
                            // clipping support, we (mis)use the alpha
                            // channel for that: we render the clip
                            // polygon into a separate bitmap, and use
                            // that to stencil the sprite alpha channel
                            ::basegfx::B2DPolyPolygon aClipPath( tools::polyPolygonFromXPolyPolygon2D( mxClipPoly ) );

                            if( aClipPath.count() )
                            {
                                RTL_LOGFILE_CONTEXT( aLog, "::dxcanvas::CanvasCustomSprite::redraw() clip rendering" );

                                // check whether alpha channel is
                                // fully opaque. This is a
                                // sufficiently common case, which is
                                // significant especially for slide
                                // transitions (large, fully opaque
                                // bitmaps).
                                mbIsContentFullyOpaque = checkAlphaOpacity( aBitmapData );

                                // move clip by output offset
                                ::basegfx::B2DHomMatrix aTranslate;
                                aTranslate.translate( ANTIALIASING_BORDER,
                                                      ANTIALIASING_BORDER );
                                aClipPath.transform( aTranslate );

                                // render clip poly-polygon into
                                // bitmap's alpha channel.  This is
                                // performed by leaving the 'inside'
                                // areas of the polygon intact, and
                                // overwriting the 'outside' area of
                                // the polygon with alpha values of 0.
                                PolyPolygonRasterizer aRasterizer( aClipPath,
                                                                   PolyPolygonRasterizer::OFF_SPANS,
                                                                   aBitmapData.Width,
                                                                   aBitmapData.Height,
                                                                   aBitmapData.Stride,
                                                                   aBitmapData.Scan0 );

                                aRasterizer.rasterConvert( ::basegfx::FillRule_EVEN_ODD );
                            }
                        }

                        // TODO(F1): copyBits currently does not perform format conversion
                        mpDXDevice->copyBits( mpDXTexture, maSize, (sal_uInt8*)aBitmapData.Scan0 );
                        
                        rtl_freeMemory( aBitmapData.Scan0 );
                    }
                    catch(...)
                    {
                        rtl_freeMemory( aBitmapData.Scan0 );
                        throw;
                    }
                }
                else if( mbClipDirty )
                {
                    // apply clip (if any)
                    if( mxClipPoly.is() )
                    {
                        // since DirectX does not provide appropriate
                        // clipping support, we (mis)use the alpha
                        // channel for that: we render the clip
                        // polygon into a separate bitmap, and use
                        // that to stencil the sprite alpha channel
                        ::basegfx::B2DPolyPolygon aClipPath( tools::polyPolygonFromXPolyPolygon2D( mxClipPoly ) );

                        if( aClipPath.count() )
                        {
                            if( mbIsContentFullyOpaque )
                            {
                                // Optimized case: alpha channel of
                                // bitmap is fully opaque, no need to
                                // read original alpha channel again.

                                // TODO(F1): applyClipMask currently does not perform format conversion
                                mpDXDevice->applyClipMask( mpDXTexture, maSize, aClipPath );
                            }
                            else
                            {
                                // TODO(P2): There's quite possibly
                                // significant performance
                                // improvements possible here. First
                                // and foremost, we basically don't
                                // have to touch the full bitmap, the
                                // alpha channel would
                                // suffice. Secondly, we could use
                                // multi-texturing, to leave the
                                // original texture intact. And third,
                                // we could cache the bitmap's alpha
                                // channel, instead of copying it
                                // again and again here.

                                // bitmap alpha channel is not fully
                                // opaque, have to restore it before
                                // applying our clip polygon.
                                Gdiplus::BitmapData aBitmapData;

                                aBitmapData.Width = maSize.getX();
                                aBitmapData.Height = maSize.getY();
                                aBitmapData.Stride = 4*aBitmapData.Width; // implicitely DWORD-aligned
                                aBitmapData.PixelFormat = PixelFormat32bppARGB;

                                try
                                {
                                    // TODO(P2): There exist a case where we can
                                    // avoid this temp mem copy here, but pass a
                                    // texture pointer directly as the Scan0 mem:
                                    // bitmap and texture layout are exactly the
                                    // same (which also means same size and
                                    // padding), and we don't need no clip alpha
                                    // buffer.
                                    aBitmapData.Scan0 = rtl_allocateMemory( aBitmapData.Stride*aBitmapData.Height );
                                    aBitmapData.Reserved = NULL;
                        
                                    // extract full bitmap content
                                    if( Gdiplus::Ok == mpBitmap->LockBits(
                                            NULL,
                                            Gdiplus::ImageLockModeRead|Gdiplus::ImageLockModeUserInputBuf,
                                            PixelFormat32bppARGB,
                                            &aBitmapData) )
                                    {
                                        mpBitmap->UnlockBits(&aBitmapData);
                                    }

                                    // since DirectX does not provide
                                    // appropriate clipping support,
                                    // we (mis)use the alpha channel
                                    // for that
                                    ::basegfx::B2DPolyPolygon aClipPath( tools::polyPolygonFromXPolyPolygon2D( mxClipPoly ) );

                                    // move clip by output offset
                                    ::basegfx::B2DHomMatrix aTranslate;
                                    aTranslate.translate( ANTIALIASING_BORDER,
                                                          ANTIALIASING_BORDER );
                                    aClipPath.transform( aTranslate );

                                    // render clip poly-polygon into
                                    // bitmap's alpha channel.  This is
                                    // performed by leaving the 'inside'
                                    // areas of the polygon intact, and
                                    // overwriting the 'outside' area of
                                    // the polygon with alpha values of 0.
                                    PolyPolygonRasterizer aRasterizer( aClipPath,
                                                                       PolyPolygonRasterizer::OFF_SPANS,
                                                                       aBitmapData.Width,
                                                                       aBitmapData.Height,
                                                                       aBitmapData.Stride,
                                                                       aBitmapData.Scan0 );

                                    aRasterizer.rasterConvert( ::basegfx::FillRule_EVEN_ODD );

                                    // TODO(F1): copyBits currently does not perform format conversion
                                    mpDXDevice->copyBits( mpDXTexture, maSize, (sal_uInt8*)aBitmapData.Scan0 );
                        
                                    rtl_freeMemory( aBitmapData.Scan0 );
                                }
                                catch(...)
                                {
                                    rtl_freeMemory( aBitmapData.Scan0 );
                                    throw;
                                }
                            }
                        }
                    }
                }

                mbClipDirty = false;
                mbSurfaceDirty = false;

                // push texture surface to primary surface
                mpDXDevice->render( mpDXTexture, 
                                    maPosition,
                                    mfAlpha );
            }
            else
            {
                VERBOSE_TRACE("CanvasCustomSprite::redraw(): emulating redraw via GdiPlus");

                // DirectX seems to be non-operational, push backbuffer
                // directly to primary surface

                // Setup an ImageAttributes with an alpha-modulating
                // color matrix.
                Gdiplus::ImageAttributes aImgAttr;
                tools::setModulateImageAttributes( aImgAttr,
                                                   1.0, 1.0, 1.0,
                                                   mfAlpha );

                // Assemble destination triangle (Gdiplus calcs the necessary 
                // affine transformation out of that)
                Gdiplus::Point destPoints[3] = 
                    {
                        Gdiplus::Point( ::basegfx::fround(maPosition.getX()),
                                        ::basegfx::fround(maPosition.getY()) ),
                        Gdiplus::Point( ::basegfx::fround(maPosition.getX() + maSize.getX()),
                                        ::basegfx::fround(maPosition.getY() ) ),
                        Gdiplus::Point( ::basegfx::fround(maPosition.getX() ),
                                        ::basegfx::fround(maPosition.getY() + maSize.getY()) )
                    };

                SurfaceGraphicsSharedPtr aGraphics( mpDXDevice->getPrimarySurface()->getGraphics() );

                if( Gdiplus::Ok != (*aGraphics)->ResetTransform() )
                    ENSURE_AND_THROW( false,
                                      "CanvasCustomSprite::redraw(): Cannot reset transformation" );

                // apply clip (if any)
                if( mxClipPoly.is() )
                {
                    GraphicsPathSharedPtr aClipPath( tools::graphicsPathFromXPolyPolygon2D( mxClipPoly ) );

                    if( aClipPath->GetPointCount() )
                    {
                        // TODO(P3): Cache clip. SetClip( GraphicsPath ) performs abyssmally on GDI+. 
                        // Try SetClip( Rect ) or similar for simple clip paths (need some support in 
                        // LinePolyPolygon, then)
                        if( Gdiplus::Ok != (*aGraphics)->SetClip( aClipPath.get(), 
                                                                  Gdiplus::CombineModeReplace ) )
                            ENSURE_AND_THROW( false,
                                              "CanvasCustomSprite::redraw(): Cannot set clip" );
                    }
                }

                (*aGraphics)->DrawImage( mpBitmap.get(), 
                                         destPoints,
                                         3,
                                         0,0,maSize.getX(),maSize.getY(),
                                         Gdiplus::UnitPixel,
                                         &aImgAttr,
                                         NULL,
                                         NULL );

                if( Gdiplus::Ok != (*aGraphics)->ResetClip() )
                    ENSURE_AND_THROW( false,
                                      "CanvasCustomSprite::redraw(): Cannot reset clip" );

                mbSurfaceDirty = false;
            }
        }
    }

    ::basegfx::B2DPoint CanvasCustomSprite::getPos() const
    {
        ::osl::MutexGuard aGuard( m_aMutex );

        return maPosition;
    }

    ::basegfx::B2DSize CanvasCustomSprite::getSize() const
    {
        ::osl::MutexGuard aGuard( m_aMutex );

        return ::basegfx::B2DSize( maOriginalSize.getX(), 
                                   maOriginalSize.getY() );
    }

    ::basegfx::B2DRectangle CanvasCustomSprite::getSpriteRect() const
    {
        // Internal! Only call with locked object mutex!
        return ::basegfx::B2DRectangle( maPosition.getX(),
                                        maPosition.getY(),
                                        maPosition.getX() + maSize.getX(),
                                        maPosition.getY() + maSize.getY() );
    }
}
