/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: dx_device.hxx,v $
 *
 *  $Revision: 1.4 $
 *
 *  last change: $Author: rt $ $Date: 2005/09/07 23:27:18 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/

#ifndef _DXCANVAS_DEVICE_HXX
#define _DXCANVAS_DEVICE_HXX

#include <dx_winstuff.hxx>
#include <dx_plainsurface.hxx>
#include <dx_texture.hxx>

#ifndef _BGFX_VECTOR_B2ISIZE_HXX
#include <basegfx/vector/b2isize.hxx>
#endif
#ifndef _BGFX_POINT_B2IPOINT_HXX
#include <basegfx/point/b2ipoint.hxx>
#endif
#ifndef _BGFX_POINT_B2DPOINT_HXX
#include <basegfx/point/b2dpoint.hxx>
#endif
#ifndef _BGFX_RANGE_B2IRECTANGLE_HXX
#include <basegfx/range/b2irectangle.hxx>
#endif
#ifndef _BGFX_RANGE_B2DRECTANGLE_HXX
#include <basegfx/range/b2drectangle.hxx>
#endif
#ifndef _BGFX_POLYGON_B2DPOLYPOLYGON_HXX
#include <basegfx/polygon/b2dpolypolygon.hxx>
#endif

#ifndef BOOST_SHARED_PTR_HPP_INCLUDED
#include <boost/shared_ptr.hpp>
#endif

#include <canvas/elapsedtime.hxx>

namespace dxcanvas
{
	/* Definition of Device class */

    /** Class encapsulating a DirectDraw device
     */
    class Device
    {
    public:
        /** Create a window device

        	@param renderHwnd
            The target window, all output is rendered into
         */
        Device( HWND renderHwnd );

        /** Create a fullscreen device

        	@param applHwnd
            A reference window associated with the fullscreen
            device. The fullscreen device is active when that window
            is.

            @param fFullscreenSize
            The requested dimension of the fullscreen device. The
            resulting device will have a size as close as possible to
            the one specified here.
         */
        Device( HWND applHwnd, const ::basegfx::B2ISize& rFullscreenSize );

        ~Device();

        /** Query actual size of the device

        	This is especially interesting for fullscreen devices
         */
        ::basegfx::B2ISize getSize() const;

        /** Query the amount of memory available for new surfaces

        	This might differ from getAvailableTextureMem()
            @see getAvailableTextureMem()

        	@return the amount of free surface mem
         */
        ::std::size_t	  getAvailableSurfaceMem() const;

        /** Query the amount of memory available for new textures

        	This might differ from getAvailableSurfaceMem()
            @see getAvailableSurfaceMem()

        	@return the amount of free texture mem
         */
        ::std::size_t     getAvailableTextureMem() const;

        /** Flip front- and backbuffer
         */
        bool flip();

        /** Flip front- and backbuffer, update only given area

			Note: Both update area and offset are ignored for
			fullscreen canvas, that uses page flipping (cannot, by
			definition, do anything else there except displaying the
			full backbuffer instead of the front buffer)

            @param rUpdateArea
            Area to copy from backbuffer to front

            @param rOffset
            Offset to apply to backbuffer content, when blitting it to front.
         */
        bool flip( const ::basegfx::B2IRectangle& rUpdateArea,
                   const ::basegfx::B2IPoint&	  rOffset );

        /** Get the primary drawing surface

			For double-buffered devices, this is always the
			backbuffer, thus, to make anything visible, you must call
			flip() after drawing. The primary surface always exists
			with a valid device.

            @see flip()
         */
        SurfaceSharedPtr getPrimarySurface() const;

        /** Render given surface to backbuffer

			@param rSurface
            Source surface

            @param rOutPos
            Output position of the surface in the backbuffer.
         */ 
        bool render( const PlainSurfaceSharedPtr& 	rSurface,
                     const ::basegfx::B2IPoint& 	rOutPos );

        /** Render given surface to backbuffer

			@param rSurface
            Source surface

            @param rOutPos
            Output position of the surface in the backbuffer.

            @param rSourceRect
            The part of the surface to blit into the backbuffer.
         */ 
        bool render( const PlainSurfaceSharedPtr& 	rSurface,
                     const ::basegfx::B2IPoint& 	rOutPos,
                     const ::basegfx::B2IRectangle& rSourceRect );

        /** Render given texture to backbuffer

			@param rTexture
            Source texture

            @param rOutPos
            Output position of the texture in the backbuffer. The texture content is
            always completely painted

            @param fAlpha
            Overall alpha blend of texture against background. Where
            0.0 means texture is completely transparent, and 1.0 means
            texture is drawn according to the unchanged texture alpha
            channel.
         */ 
        bool render( const TextureSharedPtr& 		rTexture,
                     const ::basegfx::B2DPoint& 	rOutPos,
                     double 						fAlpha );

        // TODO(F3): no format conversion for now
        // pData gets overwritten!
        bool copyBits( const TextureSharedPtr&		rTexture,
                       const ::basegfx::B2ISize&	rSize,
                       sal_uInt8*					pData );

        // TODO(F3): no format conversion for now
        bool applyClipMask( const TextureSharedPtr&				rTexture,
                            const ::basegfx::B2ISize&			rIntendedSize,
                            const ::basegfx::B2DPolyPolygon&	rClipPoly );

        bool clearSurface( const PlainSurfaceSharedPtr& rSurface ) const;
        bool clearSurface( const TextureSharedPtr& rSurface ) const;

        struct ModeSelectContext
        {
            DDSURFACEDESC		selectedDesc;

            ::basegfx::B2ISize	requestedSize;
        };

    private:
        friend class SurfaceManager; // needs access to the IDirectDraw ptr

        // default: disabled copy/assignment
        Device(const Device&);
        Device& operator=( const Device& );

        void setMode();
        bool setup3DDevice();

        void renderInfoText( const ::rtl::OUString& rStr,
                             const Gdiplus::PointF& rPos ) const;
        void renderFPSCounter() const;
        void renderMemAvailable() const;

        HWND									mhWnd;

        ModeSelectContext						maSelectedFullscreenMode;
        DDPIXELFORMAT							maTextureFormat;

        COMReference<IDirectDraw2> 				mpDirectDraw;
        COMReference<IDirectDrawSurface> 		mpPrimarySurface;
        COMReference<IDirectDrawSurface> 		mpBackBufferSurface;

        COMReference< IDirect3D2 >				mpDirect3D;
        COMReference< IDirect3DDevice2 >		mpDirect3DDevice;

        mutable ::canvas::tools::ElapsedTime	maLastUpdate;	// for the frame counter

        mutable PlainSurfaceSharedPtr			mpCachedPrimarySurface;
        const bool								mbPageFlipping;
    };

    typedef ::boost::shared_ptr< Device > 	DeviceSharedPtr;
    
}

#endif /* _DXCANVAS_DEVICE_HXX */
