/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: VectorBtree.java,v $
 *
 *  $Revision: 1.2 $
 *
 *  last change: $Author: rt $ $Date: 2005/09/09 16:26:03 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/
package com.sun.xmlsearch.db;

public class VectorBtree {
  protected VectorBlock _root;
  protected BlockManager _blockManager;
  protected VectorBtreeParameters _params;
  protected int _vecLen;		// should be final
  protected int _blockSize;
  protected int _maxEntries;
  protected int _leafDataLimit;
  protected int _vectorsOffset;

  // DEBUG
  protected EdgePrinter _printer = new EdgePrinter();

  protected class VectorBlock extends Block {
    public VectorBlock(int size) {
      super(size);
    }

    protected int findIndex(byte[] key) {
      int i = 0, j = _free - 1;
      while (i <= j) {
	int k = (i + j)/2;
	int test = memcmp(key, _data, vector(k), _vecLen);
	//	System.err.println("k = "+k+", test = "+test);
	if (test > 0)
	  i = k + 1;
	else if (test < 0)
	  j = k - 1;
	else
	  return -1 - k; // result always negative; "k" encoded
      }
      return i;
    }

    private int FindVectorsInLeaf(byte[] lo, byte[] hi,
				  int commLen, int prefLen,
				  byte[] buffer, int size)
    {
      int idx = 0, start;
    SEARCH: for (int nBytesEq = 0;;) {
      //      System.out.println("idx = " + idx);
      if (_data[idx] == nBytesEq) // at compression byte
	{
	  int i;
	  for (i = nBytesEq; i < _vecLen; i++)
	    if (lo[i] == _data[++idx])
	      ++nBytesEq;
	    else if ((lo[i]&0xFF) < (_data[idx]&0xFF))
	      if (nBytesEq >= commLen &&
		  (i >= prefLen || (hi[i]&0xFF) >= (_data[idx]&0xFF))) {
		start = nBytesEq;
		break SEARCH;
	      }
	      else
		return 0;
	    else {
	      idx += _vecLen - i; // skip
	      continue SEARCH;
	    }
	    
	  if (i == _vecLen)	// eq vec found
	    if ((_data[++idx]&0xFF) >= prefLen) {
	      start = _data[idx++]&0xFF;
	      break SEARCH;
	    }
	    else
	      return 0;
	}
      else if (_data[idx] < nBytesEq) // drop
	{
	  System.out.println(idx);
	  nBytesEq = (_data[idx++]);
	  System.out.println(nBytesEq);
	  if (nBytesEq < commLen)
	    return 0;
	  else if (lo[nBytesEq] < (_data[idx]&0xFF))
	    if (hi[nBytesEq] < (_data[idx]&0xFF))
	      return 0;
	    else {
	      start = nBytesEq;			// found
	      break SEARCH;
	    }
	  else
	    idx += _vecLen - nBytesEq;
	}
      else if ((_data[idx]&0xFF) == 0xFF)
	return 0;
      else			// compression is bigger
	idx += _vecLen + 1 - _data[idx];
    }

    int length = Math.min(size - start, _free - idx);
    buffer[0] = (byte)start;
    System.arraycopy(_data, idx, buffer, 1, length);
    buffer[length + 1] = 0;
    return length + 1;
    }
    
    protected boolean searchLeafBlock(byte[] key) {
      processLeafBlock(_printer);
      int nBytesEq = 0;
    SEARCH: for (int idx = 0;; idx += _vecLen + 1 - _data[idx]) {
      if (_data[idx] == nBytesEq) {
	int i, j;
	for (i = _data[idx], j = idx + 1; i < _vecLen; i++, j++)
	  if (key[i] == _data[j])
	    ++nBytesEq;
	  else if ((key[i]&0xFF) < (_data[j]&0xFF))
	    return false;
	  else			/* key[i] > _data[j] */
	    continue SEARCH;
	      
	if (i == _vecLen)		/* or nBytesEq == _vecLen */
	  return true;		/* equal vector found */
      }
      else if (_data[idx] < nBytesEq)
	return false;
    }
    }

    public boolean processLeafBlock(VectorProcessor processor) {
      byte[] buffer = processor.getVectorBuffer();
      for (int ix = 0; ix < _free; ix += _vecLen - _data[ix] + 1) {
	//	::memcpy(&buffer[_data[ix]], &_data[ix + 1], _vecLen - _data[ix]);
	System.arraycopy(_data, ix + 1, buffer, _data[ix], _vecLen - _data[ix]);
	if (processor.processVector())
	  return true;
      }
      return false;
    }
    
  } // VectorBlock

  protected VectorBtree() {/*empty*/}
  
  public VectorBtree(VectorBtreeParameters params) throws Exception {
    _params = params;
    _vecLen = params.getVectorLength();
    _blockSize = params.getBlockSize();
    _maxEntries=(_blockSize-Block.HEADERLEN-Block.IDLEN)/(_vecLen+Block.IDLEN);
    if ((_maxEntries & 1) == 0)	// needs to be odd
      _maxEntries--;
    
    _leafDataLimit = _blockSize - _vecLen -  Block.HEADERLEN - Block.IDLEN;
    
    _vectorsOffset = (_maxEntries + 1)*Block.IDLEN;
    _blockManager = new BlockManager(params, false, new BlockFactory() {
      public Block makeBlock() { return new VectorBlock(_blockSize); }
    });
    _root = accessBlock(params.getRootPosition());
  }
  
  public VectorBtreeIterator makeIterator() {
    return new VectorBtreeIterator(this, new byte[_vecLen]);
  }

  /*
  public static void main(String[] args) {
    try {
      Schema schema = new Schema("/files/resources/lexicon/LIF_ONT", false);
      VectorBtreeParameters params = new VectorBtreeParameters(schema, "EDGE");
      VectorBtree tree = new VectorBtree(params);
      VectorBtreeIterator iter = tree.makeIterator();
    }
    catch (Exception e) {
      e.printStackTrace();
    }
  }
  */

  public int getVectorLength() {
    return _vecLen;
  }

  protected void lock(Block bl) {
    _blockManager.lockBlock(bl._number);
  }
  
  protected void unlock(Block bl) {
    _blockManager.unlockBlock(bl._number);
  }
  
  public boolean find(byte[] vector) throws Exception {
    return treeSearch(_root, vector);
  }

  private boolean treeSearch(VectorBlock bl, byte[] vector) throws Exception {
    if (bl._isLeaf)
      return bl.searchLeafBlock(vector);
    else {
      int i = bl.findIndex(vector);
      return i < 0 || treeSearch(accessChild(bl, i), vector);
    }
  }

  protected int vector(int index) {
    return _vectorsOffset + _vecLen*index;
  }

  protected static int memcmp(byte[] v1, byte[] v2, int i2, int n) {
    for (int i = 0; i < n; i++, i2++)
      if (v1[i] != v2[i2])
	return (v1[i]&0xFF) - (v2[i2]&0xFF);
    return 0;
  }
  
  protected VectorBlock accessBlock(int index) throws Exception {
    return (VectorBlock)_blockManager.accessBlock(index);
  }

  protected VectorBlock accessChild(Block bl, int index) throws Exception {
    return accessBlock(bl.integerAt(index*4));
  }

  public static void assert(boolean assertion) {
    if (assertion == false)
      System.err.println("assertion failed");
  }

  public int getIteratorBufferLength() {
    return _blockSize + _vecLen*2;
  }

  public boolean mapProcessor(VectorBlock bl, VectorProcessor processor)
    throws Exception {
      System.out.println("block: " + bl._number + " " + bl._isLeaf + " " + bl._free);
    if (bl._isLeaf)
      return bl.processLeafBlock(processor);
    else {
      byte[] buffer = processor.getVectorBuffer();
    BLOCK: {
	lock(bl);
	for (int ix = 0; ix < bl._free; ix++) {
	  System.out.println("mapping " + ix);
	  if (mapProcessor(accessBlock(bl.integerAt(4*ix)), processor))
	    break BLOCK;
	  System.arraycopy(bl._data, vector(ix), buffer, 0, _vecLen);
	  System.out.println("internal ");
	  if (processor.processVector())
	    break BLOCK;
	}
      
	System.out.println("mapping " + bl._free);
	System.out.println("last internal ");
	System.out.println(bl._number + " at " + bl._free + " = " + bl.integerAt(4*bl._free));
	if (!mapProcessor(accessBlock(bl.integerAt(4*bl._free)), processor)) {
	    unlock(bl);
	    return false;
	  }
      }
      unlock(bl);
      return true;
    }
  }

  public boolean FindVectors1(byte[] lo, byte[] hi,
			      int commLen, int prefLen,
			      byte[] buffer, BlockStack stack)
    throws Exception
  {
    int index = 0;
    VectorBlock bl;
    stack.clear();
    // put on stack only those internal entries that would belong to
    // the output sequence
    for (bl = _root; !bl._isLeaf; bl = accessChild(bl, index)) {
	index = bl.findIndex(lo);
	if (index < 0)		// equal vector found in internal
	  {			// index encodes position
	    int idx = -index;	// idx is 1+ index of eq vector
	    // check for continuation at this level
	    if (idx<bl._free && memcmp(hi,bl._data,vector(idx),prefLen)>=0)
	      stack.push(bl._number, idx);
	    // descend on the leftmost path of the subtree
	    for (bl = accessChild(bl,idx); !bl._isLeaf; bl = accessChild(bl,0))
	      if (memcmp(hi, bl._data, vector(0), prefLen) >= 0)
		stack.push(bl._number, 0);
	    // bl points to a leaf where the seq should begin at the beginning
	    if (memcmp(lo, bl._data, 1, prefLen) <= 0 &&
		memcmp(hi, bl._data, 1, prefLen) >= 0)
	      {
		int delta = bl._free - 1 - commLen;
		buffer[0] = (byte)commLen;
		System.arraycopy(bl._data, commLen + 1, buffer, 1, delta);
		buffer[delta + 1] = 0;
		return true;
	      }
	    else
	      {
		assert(stack.size() == 0);
		return false;
	      }
	  }
	// indexed vector is bigger than input
	// check if it belongs to the seqence
	if (index<bl._free && memcmp(hi,bl._data,vector(index),prefLen)>=0)
	  stack.push(bl._number, index);
      }
  
    // at this point stack holds continuation points if any,
    // and bl points to a leaf block where the sequence should begin

    if (bl.FindVectorsInLeaf(lo, hi, commLen, prefLen,
			     buffer, getIteratorBufferLength()) > 0)
      return true;
    else if (stack.size() > 0) // the sequence will start from an internal vec
      {
	FindVectors2(lo, hi, commLen, prefLen, buffer, stack);
	return true;
      }
    else
      return false;
  }
  
  // continuation search for sequence of vectors bigger than input
  // using stack saved information

  public void FindVectors2(byte[] lo, byte[] hi, int commLen, int prefLen,
			   byte[] buffer, BlockStack stack)
    throws Exception {
    Block bl = accessBlock(stack.topBlockNumber());
    int index = stack.topIndex();

    // adding vector from internal block
    buffer[0] = (byte)commLen;
    System.arraycopy(bl._data,vector(index)+commLen,buffer,1,_vecLen-commLen);
    int length = _vecLen - commLen + 1;
    // advance index and check for continuation
    // note that continuation is guaranteed if not at the bottom of stack
    if (++index < bl._free
	&& (stack.size()>1 || memcmp(hi,bl._data,vector(index),prefLen)>=0))
      stack.increaseIndex();
    else
      stack.pop();
    // descend to the leaf
    for (bl = accessChild(bl, index); !bl._isLeaf; bl = accessChild(bl, 0))
      if (memcmp(hi, bl._data, vector(0), prefLen) >= 0)
	stack.push(bl._number, 0);
      
    // at leaf; if there is continuation, it has to be at the beginning
    int idx = 1; // past compr. byte
    if (memcmp(lo, bl._data, idx, prefLen) <= 0 &&
	memcmp(hi, bl._data, idx, prefLen) >= 0)
      {	 // matches
	idx += commLen;
	buffer[length++] = (byte)commLen; // default compression for cont
	int delta = bl._free - idx;
	System.arraycopy(bl._data, idx, buffer, length, delta);
	length += delta;
      }
    buffer[length] = 0;
  }
}
