/**
  @file log.c

  @author Johan Hedberg <johan.hedberg@nokia.com>

  Copyright (C) 2006 Nokia Corporation. All rights reserved.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License, version 2, as
  published by the Free Software Foundation.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA

*/
#include <stdio.h>
#include <stdlib.h>
#include <glib.h>

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#ifdef USE_OSSOLOG
# include <osso-log.h>
#else
# include <syslog.h>
# define DLOG_OPEN(name) openlog(name, LOG_PID, LOG_DAEMON)
# define DLOG_INFO(fmt, args...) syslog(LOG_INFO, fmt, ## args)
# define DLOG_ERR(fmt, args...) syslog(LOG_ERR, fmt, ## args)
# define DLOG_CRIT(fmt, args...) syslog(LOG_CRIT, fmt, ## args)
# define DLOG_WARN(fmt, args...) syslog(LOG_WARNING, fmt, ## args)
# define DLOG_DEBUG(fmt, args...) syslog(LOG_DEBUG, fmt, ## args)
#endif

#include "log.h"

static void loghandler_syslog(const gchar *log_domain,
                              GLogLevelFlags log_level,
                              const gchar *message,
                              gpointer unused_data)
{
    gboolean is_fatal = (log_level & G_LOG_FLAG_FATAL) != 0;

    switch (log_level & G_LOG_LEVEL_MASK) {
        case G_LOG_LEVEL_ERROR:
            DLOG_ERR("%s", message);
            break;
        case G_LOG_LEVEL_CRITICAL:
            DLOG_CRIT("%s", message);
            break;
        case G_LOG_LEVEL_WARNING:
            DLOG_WARN("%s", message);
            break;
        case G_LOG_LEVEL_MESSAGE:
            DLOG_INFO("%s", message);
            break;
        case G_LOG_LEVEL_INFO:
            DLOG_INFO("%s", message);
            break;
        case G_LOG_LEVEL_DEBUG:
            DLOG_DEBUG("%s", message);
            break;
        default:
            DLOG_DEBUG("%s", message);
            break;
    }

    if (is_fatal)
        DLOG_ERR("%s", "Aborting.");
}

static void loghandler_stdio(const gchar *log_domain,
                             GLogLevelFlags log_level,
                             const gchar *message,
                             gpointer unused_data)
{
    gboolean is_fatal = (log_level & G_LOG_FLAG_FATAL) != 0;
    const char *prefix;
    FILE *out;

    out = stdout;

    switch (log_level & G_LOG_LEVEL_MASK) {
        case G_LOG_LEVEL_ERROR:
            prefix = "ERROR";
            out = stderr;
            break;
        case G_LOG_LEVEL_CRITICAL:
            prefix = "CRITICAL";
            out = stderr;
            break;
        case G_LOG_LEVEL_WARNING:
            prefix = "WARNING";
            out = stderr;
            break;
        case G_LOG_LEVEL_MESSAGE:
            prefix = "MESSAGE";
            break;
        case G_LOG_LEVEL_INFO:
            prefix = "INFO";
            break;
        case G_LOG_LEVEL_DEBUG:
            prefix = "DEBUG";
            break;
        default:
            prefix = "LOG";
            break;
    }

    fprintf(out, "%-9s: %s\n", prefix, message);

    if (is_fatal)
        fprintf(stderr, "Aborting.\n");
}

static void print_info_log(const gchar *str)
{
    DLOG_INFO("%s", str);   
}

static void print_info_out(const gchar *str)
{
    printf("%s\n", str);
}

static void print_error_log(const gchar *str)
{
    DLOG_ERR("%s", str);
}

static void print_error_out(const gchar *str)
{
    fprintf(stderr, "%s\n", str);
}

void open_log(const char *name, gboolean use_syslog)
{
    if (use_syslog) {
        DLOG_OPEN(name);
        g_set_print_handler(print_info_log);
        g_set_printerr_handler(print_error_log);
        g_log_set_handler(NULL, G_LOG_LEVEL_MASK | G_LOG_FLAG_FATAL
                | G_LOG_FLAG_RECURSION, loghandler_syslog, NULL);
    }
    else {
        g_set_print_handler(print_info_out);
        g_set_printerr_handler(print_error_out);
        g_log_set_handler(NULL, G_LOG_LEVEL_MASK | G_LOG_FLAG_FATAL
                | G_LOG_FLAG_RECURSION, loghandler_stdio, NULL);
    }
}

