/* ----------------------------------------------------------------------------
 * init.c
 * funtions that prepare program workflow and initialization
 *
 * Copyright 2002 Matthias Grimm
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
 * ----------------------------------------------------------------------------*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif
#include "systems.h"

#include <stdio.h>
#include <fcntl.h>
#include <unistd.h>
#include <signal.h>
#include <string.h>
#include <syslog.h>
#include <getopt.h>
#include <errno.h>
#include <malloc.h>
#include <pwd.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include "pbbinput.h"

#include <glib.h>
#include <pbb.h>

#include "init.h"
#include "gettext_macros.h"

#include "input_manager.h"
#include "config_manager.h"

#ifdef WITH_MODULE_PMAC
#  include "module_pmac.h"
#endif

#include "module_system.h"
#include "module_powersave.h"
#include "module_display.h"
#include "module_cdrom.h"
#include "module_mixer.h"
#include "module_ossmixer.h"
#include "module_alsamixer.h"
#ifdef DEBUG
#  include "module_peep.h"
#endif

void
init_serverdata (struct serverdata *sd)
{
	int n = 0;

	sd->prgname	       = NULL;
	sd->quiet          = 0;
	sd->configreadfile = DEFAULT_CONFIG;
	sd->mainloop       = NULL;

	sd->module[n].initfunc = inputmanager_init;  sd->module[n].exitfunc    = inputmanager_exit;  /* pri 0 */
	sd->module[n].openfunc = NULL;               sd->module[n++].closefunc = NULL;
	sd->module[n].initfunc = configmanager_init; sd->module[n].exitfunc    = configmanager_exit; /* pri 0 */
	sd->module[n].openfunc = NULL;               sd->module[n++].closefunc = NULL;
	sd->module[n].initfunc = NULL;               sd->module[n].exitfunc    = NULL;               /* pri 0 */
	sd->module[n].openfunc = ipc_init_stub;      sd->module[n++].closefunc = ipc_exit;
	sd->module[n].initfunc = system_init;        sd->module[n].exitfunc    = system_exit;        /* pri 1 */
	sd->module[n].openfunc = system_open;        sd->module[n++].closefunc = system_close;
#ifdef WITH_MODULE_PMAC
	sd->module[n].initfunc = pmac_init;          sd->module[n].exitfunc    = pmac_exit;          /* pri 1 */
	sd->module[n].openfunc = pmac_open;          sd->module[n++].closefunc = pmac_close;
#endif
	sd->module[n].initfunc = power_init;         sd->module[n].exitfunc    = power_exit;         /* pri 2 */
	sd->module[n].openfunc = power_open;         sd->module[n++].closefunc = power_close;
	sd->module[n].initfunc = display_init;       sd->module[n].exitfunc    = display_exit;       /* pri 2 */
	sd->module[n].openfunc = display_open;       sd->module[n++].closefunc = display_close;
	sd->module[n].initfunc = cdrom_init;         sd->module[n].exitfunc    = cdrom_exit;         /* pri 2 */
	sd->module[n].openfunc = cdrom_open;         sd->module[n++].closefunc = cdrom_close;
	sd->module[n].initfunc = mixer_init;         sd->module[n].exitfunc    = mixer_exit;         /* pri 2 */
	sd->module[n].openfunc = mixer_open;         sd->module[n++].closefunc = mixer_close;
#ifdef WITH_OSS
	sd->module[n].initfunc = ossmixer_init;      sd->module[n].exitfunc    = ossmixer_exit;      /* pri 2 */
	sd->module[n].openfunc = NULL;               sd->module[n++].closefunc = NULL;
#endif
#ifdef WITH_ALSA
	sd->module[n].initfunc = alsamixer_init;     sd->module[n].exitfunc    = alsamixer_exit;     /* pri 2 */
	sd->module[n].openfunc = NULL;               sd->module[n++].closefunc = NULL;
#endif
#ifdef DEBUG
#  if 1
	sd->module[n].initfunc = peep_init;          sd->module[n].exitfunc    = peep_exit;          /* pri 3 */
	sd->module[n].openfunc = NULL;               sd->module[n++].closefunc = NULL;
#  else
	sd->module[n].initfunc = NULL;               sd->module[n].exitfunc    = peep_exit;
	sd->module[n].openfunc = peep_open;          sd->module[n++].closefunc = NULL;
#  endif
#endif

	for (; n < MODULECOUNT; n++) {
		sd->module[n].initfunc = NULL;
		sd->module[n].openfunc = NULL;
		sd->module[n].closefunc = NULL;
		sd->module[n].exitfunc = NULL;
	}
}

/* This function calls every known module so that they initialize themselves
 * and prepare for configuration. If the initialization of a module failed,
 * all already loaded modules would be cleaned up.
 */
int
init_modules(struct serverdata *sd)
{
	struct tagitem *taglist, *tl1, tl2[2] = {{TAG_END, 0}, {TAG_END, 0}};
	int n, rc = 0;

	for (n=0; n < MODULECOUNT; n++)
		if (sd->module[n].initfunc != NULL)
			if ((rc = sd->module[n].initfunc ())) {
				for (--n; n >= 0; n--)   /* cleanup already initialized modules */
					if (sd->module[n].exitfunc != NULL)
						sd->module[n].exitfunc ();
				break; /* critical error - program abortion */
			}

	if (rc == 0) {
		taglist = tl1 = readConfigFile(sd->configreadfile);	
		if (tl1 == NULL)
			taglist = tl2;
		taglist_add(taglist, TAG_CFGNAME, (long) sd->configreadfile);

		for (n=0; n < MODULECOUNT; n++)
			if (sd->module[n].openfunc != NULL)
				if ((rc = sd->module[n].openfunc (taglist))) {
					for (--n; n >= 0; n--)   /* cleanup already initialized modules */
						if (sd->module[n].closefunc != NULL)
							sd->module[n].closefunc ();

					for (n = MODULECOUNT-1; n >= 0; n--)
						if (sd->module[n].exitfunc != NULL)
							sd->module[n].exitfunc ();
					
					break; /* critical error - program abortion */
				}
		
		if (rc == 0) {
			process_queue (CONFIGQUEUE, taglist);

			/* check if someone has set a tagerror and abort if necessary */
			while (taglist->tag != TAG_END) {
				if (taglist->tag & FLG_ERROR) {
					if (taglist->data != E_NOSUPPORT) {
						exit_modules (sd);
						rc = taglist->data;   /* error code */
						break; /* critical error - program abortion */
					}
				}
				taglist++;
			}
		}
		
		if (tl1 != NULL)
			free (tl1);
	}
	return rc;
}

/* This funcion calls the exit funcion of every know modules. The sequence
 * is from last to first, that means the module that was initialized first
 * will be cleaned up last.
 */
void
exit_modules (struct serverdata *sd)
{
	int n;

	for (n = MODULECOUNT-1; n >= 0; n--)
		if (sd->module[n].closefunc != NULL)
			sd->module[n].closefunc ();

	for (n = MODULECOUNT-1; n >= 0; n--)
		if (sd->module[n].exitfunc != NULL)
			sd->module[n].exitfunc ();
}

/* This function rereads the configuration file and sends the resulting taglist to all
    modules. */

void
reconfig_modules(struct serverdata *sd)
{
	struct tagitem *taglist;
	struct passwd *pwddata;
	long username;

	taglist = readConfigFile(sd->configreadfile);	
	if (taglist != NULL) {
		ipc_filterclear();
		process_queue (CONFIGQUEUE, taglist);
		if ((username = tagfind (taglist, TAG_USERALLOWED, -1)) != -1) {
			if ((pwddata = getpwnam((char *) username)) != NULL)
				ipc_filteruser(pwddata->pw_uid);
			else
				ipc_filterall();      /* switch off IPC-receive */
		}
		free (taglist);
	}
}

int
evaluate_args(struct serverdata *sd, int argc, char *argv[])
{
	struct option const long_options[] = {
		  {"help", no_argument, 0, ARG_HELP},
		  {"version", no_argument, 0, ARG_VERSION},
		  {"detach", optional_argument, 0, ARG_DETACH},
		  {"quiet", no_argument, 0, ARG_QUIET},
		  {"configfile", required_argument, 0, ARG_CONFIG},
		  {NULL, 0, NULL, 0}
	};
	int c, err;
	char *optarg2;

	if((sd->prgname = strrchr(argv[0],'/')) == NULL)
		sd->prgname = argv[0];
	else sd->prgname++;		/* ignore first slash*/
	argv[0] = sd->prgname;

	while ((c = getopt_long (argc, argv, ARG_ALL, long_options, (int *) 0)) != EOF) {
		switch (c) {
			case ARG_VERSION:
				printf(_("%s, version %s"), PACKAGE, VERSION);
				printf(", (c) 2002-2005 Matthias Grimm\n");
				return E_INFO;
			case ARG_DETACH:
				optarg2 = (optarg) ? optarg : argv[optind];
				if (optarg2 == NULL || optarg2[0] == '-')
					optarg2 = DEFAULT_PIDFILE;
				prepare_daemon (sd->prgname, optarg2, PBBDF_FORCE);
				break;
			case ARG_QUIET:
				sd->quiet = 1;
				break;
			case ARG_CONFIG:
				if ((err = check_devorfile(sd->configreadfile = optarg, TYPE_FILE))) {
					printf (_("ERROR: Have problems reading configuration file [%s]: %s\n"), sd->configreadfile, strerror(errno));
					return err;
				}
				break;
			case ARG_HELP:
			default:
				printf(_("%s - daemon to support special features of laptops and notebooks.\n"), sd->prgname);
				printf(_("Usage: %s [OPTION]... \n"), sd->prgname);
				printf (_("Options:\n"
					"   -%c, --help               display this help text and exit\n"
					"   -%c, --version            display version information and exit\n"
					"   -%c, --quiet              suppress welcome message\n"
					"   -%c, --detach[=PIDFILE]   start %s as background process and\n"
					"                            optional use an alternative pid-file\n"
					"                            (default: %s)\n"
					"   -%c, --configfile=CONFIG  use alternative configuration file\n"
					"                            (default: %s)\n"
					"see configuration file for more options.\n"),
					ARG_HELP, ARG_VERSION, ARG_QUIET, ARG_DETACH, sd->prgname, DEFAULT_PIDFILE,
					ARG_CONFIG, sd->configreadfile);
				return E_INFO;
		}
	}
	return 0;
}

/* --- Signal Handler --- */

/* Pipe must be global because there is no other way for the
 * signal handler to access this data
 */
static int SigHandlerPipe[2];

void
cbSigHandlerTX (int signum)
{
	write (SigHandlerPipe[1], &signum, sizeof(int));
}

gboolean
cbSigHandlerRD (int fd, gpointer user_data)
{
	struct serverdata *base = user_data;
	int signum;
	gboolean rc = FALSE;

	if ((read (fd, &signum, sizeof(int))) == sizeof(int)) {
		switch (signum) {
			case SIGHUP:     /* reload config file and reconfigure the modules */
				scanEventDevices ();   /* look which HID are available */
				reconfig_modules (base); /* load config file again */
				rc = TRUE;
				break;
			case SIGUSR1:    /* reset hardware configuration */
				process_queue_single (CONFIGQUEUE, TAG_REINIT, 0);
				rc = TRUE;
				break;
			case SIGINT:
			case SIGTERM:    /* cleanup and exit the program */
				close (SigHandlerPipe[0]);
				close (SigHandlerPipe[1]);
				g_main_loop_quit (base->mainloop);
				break;
		}
	}
	return rc;
}

void
installSigHandler (struct serverdata *base)
{
	struct sigaction sa = { {cbSigHandlerTX}, {{0}}, SA_RESTART, 0 };
	InputSource *src;
	int n = 0;
	
	if ((pipe(SigHandlerPipe)) == 0) {
		src = addInputSource (SigHandlerPipe[0], cbSigHandlerRD, base, FALSE);

		if (!sigaction (SIGINT,  &sa, NULL)) n++;
		if (!sigaction (SIGTERM, &sa, NULL)) n++;
		if (!sigaction (SIGHUP,  &sa, NULL)) n++;
		if (!sigaction (SIGUSR1, &sa, NULL)) n++;
		
		if (n == 0) {
			g_source_remove (src->watch);
			close (SigHandlerPipe[0]);
			close (SigHandlerPipe[1]);
		} else if (n < 4)
			print_msg (PBB_WARN, _("Not all signal handlers could be installed.\n"));
	} else
		print_msg (PBB_WARN, _("Can't install any signal handler\n"));
	return;
}

int
ipc_init_stub (struct tagitem *taglist)
{
	int err;
	long username;
	struct passwd *pwddata;

	err = ipc_init (NULL, LIBMODE_SERVER, 0);
	switch (err) {
	   case E_MSGPORT:
		print_msg (PBB_ERR, _("Can't create message port for server: %s.\n"), strerror(errno));
		break;
	   case E_TWICE:
		print_msg (PBB_ERR, _("Server is already running. Sorry, only one instance allowed.\n"));
		break;
	   case E_OLDPORT:
		print_msg (PBB_WARN, _("Orphaned server port found and removed. All running clients have to be restarted.\n"));
		err = 0;
	   case 0:
		if ((username = tagfind (taglist, TAG_USERALLOWED, -1)) != -1) {
			if ((pwddata = getpwnam((char *) username)) != NULL)
				ipc_filteruser(pwddata->pw_uid);
			else
				ipc_filterall();      /* switch off IPC-receive */
		}
		break;
	}

	register_function (T10QUEUE, ipc_handler);  /* input via IPC messages */
	return err;
}
