# -*- coding: utf-8 -*-
# Copyright 2010-2011 Canonical Ltd.  This software is licensed under the
# GNU Lesser General Public License version 3 (see the file LICENSE).

from mock import patch
from unittest import TestCase
from wsgi_intercept import add_wsgi_intercept, remove_wsgi_intercept
from wsgi_intercept.httplib2_intercept import install, uninstall
import httplib2
import shutil
import tempfile

from piston_mini_client import (PistonAPI, returns_json, returns,
    returns_list_of, PistonResponseObject, PistonSerializable,
    OfflineModeException, safename)
from piston_mini_client.auth import BasicAuthorizer


class PistonAPITestCase(TestCase):
    class CoffeeAPI(PistonAPI):
        default_service_root = 'http://localhost:12345'
        def brew(self):
            self._get('/brew')

    def callback(self, environ, start_response):
        self.called = {
            'body': environ['wsgi.input'].read(),
            'path': environ['PATH_INFO'],
            'auth': environ.get('HTTP_AUTHORIZATION'),
            'content_type': environ.get('CONTENT_TYPE'),
            'query': environ.get('QUERY_STRING'),
            'scheme': environ.get('wsgi.url_scheme'),
            'method': environ.get('REQUEST_METHOD'),
        }
        return 'hello world'

    def setUp(self):
        install()
        add_wsgi_intercept('localhost', 12345, lambda: self.callback)

    def tearDown(self):
        remove_wsgi_intercept('localhost', 12345)
        uninstall()

    def test_request(self):
        api = self.CoffeeAPI()
        api._request('/foo', 'POST', body='foo=bar')
        self.assertEqual(self.called['body'], 'foo=bar')
        self.assertEqual(self.called['path'], '/foo')
        self.assertEqual(self.called['method'], 'POST')

    @patch('httplib2.Http.request')
    def test_request_cached(self, mock_request):
        path = "/foo"
        # setup mock cache
        tmpdir = tempfile.mkdtemp()
        http = httplib2.Http(cache=tmpdir)
        cachekey = self.CoffeeAPI.default_service_root + path
        http.cache.set(cachekey, "header\r\n\r\nmy_cached_body\n")
        # ensure that we trigger a error like when offline (no dns)
        mock_request.side_effect = httplib2.ServerNotFoundError("")
        api = self.CoffeeAPI(cachedir=tmpdir, offline_mode=True)
        res = api._request(path, 'GET')
        # check that we get the data we expect
        self.assertEqual(res, "my_cached_body\n")
        # check for nonexisting url
        res = api._request('/bar', 'GET')
        self.assertEqual(res, None)
        # ensure errors on POST, PUT
        self.assertRaises(OfflineModeException, api._request, path, 'POST')
        self.assertRaises(OfflineModeException, api._request, path, 'PUT')
        # cleanup
        shutil.rmtree(tmpdir)

    @patch('httplib2.Http.request')
    def test_request_cached_long_names(self, mock_request):
        # construct a really long path that triggers our safename code
        path = "/foo_with_a_" + 30 *"long_name"
        self.assertTrue(len(path) > 143)
        # setup mock cache
        tmpdir = tempfile.mkdtemp()
        cache = httplib2.FileCache(tmpdir, safe=safename)
        http = httplib2.Http(cache=cache)
        cachekey = self.CoffeeAPI.default_service_root + path
        http.cache.set(
            cachekey, "header\r\n\r\nmy_cached_body_from_long_path\n")
        # ensure that we trigger a error like when offline (no dns)
        mock_request.side_effect = httplib2.ServerNotFoundError("")
        api = self.CoffeeAPI(cachedir=tmpdir, offline_mode=True)
        res = api._request(path, 'GET')
        # check that we get the data we expect
        self.assertEqual(res, "my_cached_body_from_long_path\n")
        # cleanup
        shutil.rmtree(tmpdir)

    def test_auth_request(self):
        api = self.CoffeeAPI(auth=BasicAuthorizer(username='foo',
            password='bar'))
        api._request('/fee', 'GET')
        self.assertEqual(self.called['auth'], 'Basic Zm9vOmJhcg==')
        self.assertEqual(self.called['method'], 'GET')

    def test_post_no_content_type(self):
        api = self.CoffeeAPI()
        api._post('/serve', data={'foo': 'bar'})
        self.assertEqual(self.called['content_type'], 'application/json')
        self.assertEqual(self.called['method'], 'POST')

    def test_post_piston_serializable(self):
        class MyCoffeeRequest(PistonSerializable):
            _atts = ('strength',)
        api = self.CoffeeAPI()
        api._post('/serve', data=MyCoffeeRequest(strength='mild'))
        self.assertEqual(self.called['content_type'], 'application/json')
        self.assertEqual(self.called['method'], 'POST')

    def test_post_explicit_content_type(self):
        api = self.CoffeeAPI()
        api._post('/serve', data={'foo': 'bar'},
            content_type='application/x-www-form-urlencoded')
        self.assertEqual(self.called['content_type'],
            'application/x-www-form-urlencoded')
        self.assertEqual(self.called['method'], 'POST')

    def test_get_no_args(self):
        api = self.CoffeeAPI()
        api._get('/stew')
        self.assertEqual('/stew', self.called['path'])
        self.assertEqual('', self.called['query'])
        self.assertEqual(self.called['method'], 'GET')

    def test_get_with_args(self):
        api = self.CoffeeAPI()
        api._get('/stew', args={'foo': 'bar'})
        self.assertEqual('/stew', self.called['path'])
        self.assertEqual('foo=bar', self.called['query'])
        self.assertEqual(self.called['method'], 'GET')

    @patch('httplib2.Http.request')
    def test_valid_status_codes_dont_raise_exception(self, mock_request):
        for status in ['200', '201', '304']:
            response = {'status': status}
            expected_body = '"hello world!"'
            mock_request.return_value = (response, expected_body)
            api = self.CoffeeAPI()
            body = api._get('/simmer')
            self.assertEqual(expected_body, body)
            mock_request.assert_called_with('http://localhost:12345/simmer',
                body='', headers={}, method='GET')

    def test_get_with_extra_args(self):
        api = self.CoffeeAPI()
        api._get('/stew?zot=ping', args={'foo': 'bar'})
        self.assertEqual('/stew', self.called['path'])
        self.assertEqual('zot=ping&foo=bar', self.called['query'])
        self.assertEqual(self.called['method'], 'GET')

    def test_path2url_with_no_ending_slash(self):
        resource = PistonAPI('http://example.com/api')
        expected = 'http://example.com/api/frobble'
        self.assertEqual(expected, resource._path2url('frobble'))

    def test_path2url_with_ending_slash(self):
        resource = PistonAPI('http://example.com/api/')
        expected = 'http://example.com/api/frobble'
        self.assertEqual(expected, resource._path2url('frobble'))

    def test_instantiation_fails_with_no_service_root(self):
        try:
            self.CoffeeAPI.default_service_root = None
            self.assertRaises(ValueError, self.CoffeeAPI)
        finally:
            self.CoffeeAPI.default_service_root = 'http://localhost:12345'

    def test_instantiation_fails_with_invalid_scheme(self):
        self.assertRaises(ValueError, self.CoffeeAPI, 'ftp://foobar.baz')

    @patch('httplib2.Http.request')
    def test_request_scheme_switch_to_https(self, mock_request):
        mock_request.return_value = ({'status': '200'}, '""')
        api = self.CoffeeAPI()
        api._request('/foo', 'GET', scheme='https')
        mock_request.assert_called_with('https://localhost:12345/foo',
            body='', headers={}, method='GET')

    @patch('httplib2.Http.request')
    def test_get_scheme_switch_to_https(self, mock_request):
        mock_request.return_value = ({'status': '200'}, '""')
        api = self.CoffeeAPI()
        api._get('/foo', scheme='https')
        mock_request.assert_called_with('https://localhost:12345/foo',
            body='', headers={}, method='GET')

    @patch('httplib2.Http.request')
    def test_post_scheme_switch_to_https(self, mock_request):
        mock_request.return_value = ({'status': '200'}, '""')
        api = self.CoffeeAPI()
        api._post('/foo', scheme='https')
        mock_request.assert_called_with('https://localhost:12345/foo',
            body='null', headers={'Content-type': 'application/json',
            'X-Requested-With': 'XMLHttpRequest'},
            method='POST')

    def test_put_no_data(self):
        api = self.CoffeeAPI()
        api._put('/serve')
        self.assertEqual(self.called['body'], 'null')
        self.assertEqual(self.called['method'], 'PUT')

    def test_put_no_content_type(self):
        api = self.CoffeeAPI()
        api._put('/serve', data={'foo': 'bar'})
        self.assertEqual(self.called['content_type'], 'application/json')
        self.assertEqual(self.called['method'], 'PUT')

    def test_put_piston_serializable(self):
        class MyCoffeeRequest(PistonSerializable):
            _atts = ('strength',)
        api = self.CoffeeAPI()
        api._put('/serve', data=MyCoffeeRequest(strength='mild'))
        self.assertEqual(self.called['content_type'], 'application/json')
        self.assertEqual(self.called['method'], 'PUT')

    def test_put_no_scheme(self):
        api = self.CoffeeAPI()
        api._put('/serve')
        self.assertEqual(self.called['scheme'], 'http')
        self.assertEqual(self.called['method'], 'PUT')

    @patch('httplib2.Http.request')
    def test_put_scheme_switch_to_https(self, mock_request):
        mock_request.return_value = ({'status': '200'}, '""')
        api = self.CoffeeAPI()
        api._put('/foo', scheme='https')
        mock_request.assert_called_with('https://localhost:12345/foo',
            body='null', headers={'Content-type': 'application/json',
            'X-Requested-With': 'XMLHttpRequest'},
            method='PUT')

    def test_put(self):
        api = self.CoffeeAPI()
        api._put('/serve', data={'foo': 'bar'},
            content_type='application/x-www-form-urlencoded')
        self.assertEqual(self.called['body'], 'foo=bar')
        self.assertEqual(self.called['content_type'],
            'application/x-www-form-urlencoded')
        self.assertEqual(self.called['method'], 'PUT')


class PistonResponseObjectTestCase(TestCase):
    def test_from_response(self):
        obj = PistonResponseObject.from_response('{"foo": "bar"}')
        self.assertEqual('bar', obj.foo)

    def test_from_dict(self):
        obj = PistonResponseObject.from_dict({"foo": "bar"})
        self.assertEqual('bar', obj.foo)


class ReturnsJSONTestCase(TestCase):
    def test_returns_json(self):
        class MyAPI(PistonAPI):
            default_service_root = 'http://foo'
            @returns_json
            def func(self):
                return '{"foo": "bar", "baz": 42}'

        result = MyAPI().func()
        self.assertEqual({"foo": "bar", "baz": 42}, result)


class ReturnsTestCase(TestCase):
    def test_returns(self):
        class MyAPI(PistonAPI):
            default_service_root = 'http://foo'
            @returns(PistonResponseObject)
            def func(self):
                return '{"foo": "bar", "baz": 42}'

        result = MyAPI().func()
        self.assertTrue(isinstance(result, PistonResponseObject))

    def test_returns_none_allowed(self):
        class MyAPI(PistonAPI):
            default_service_root = 'http://foo'
            @returns(PistonResponseObject, none_allowed=True)
            def func(self):
                return 'null'

        result = MyAPI().func()
        self.assertEqual(result, None)

    def test_returns_none_allowed_normal_response(self):
        class MyAPI(PistonAPI):
            default_service_root = 'http://foo'
            @returns(PistonResponseObject, none_allowed=True)
            def func(self):
                return '{"foo": "bar", "baz": 42}'

        result = MyAPI().func()
        self.assertTrue(isinstance(result, PistonResponseObject))


class ReturnsListOfTestCase(TestCase):
    def test_returns(self):
        class MyAPI(PistonAPI):
            default_service_root = 'http://foo'
            @returns_list_of(PistonResponseObject)
            def func(self):
                return '[{"foo": "bar"}, {"baz": 42}]'

        result = MyAPI().func()
        self.assertEqual(2, len(result))
        self.assertEqual('bar', result[0].foo)
        self.assertEqual(42, result[1].baz)


class PistonSerializableTestCase(TestCase):
    class MySerializable(PistonSerializable):
        _atts = ('foo',)

    def test_init_with_extra_variables(self):
        obj = self.MySerializable(foo='bar', baz=42)
        self.assertEqual('bar', obj.foo)
        self.assertEqual(42, obj.baz)

    def test_init_with_missing_variables(self):
        obj = self.MySerializable()
        self.assertFalse(hasattr(obj, 'foo'))

    def test_missing_required_arguments(self):
        obj = self.MySerializable()
        self.assertRaises(ValueError, obj._as_serializable)

    def test_can_assign_required_arguments_after_init(self):
        obj = self.MySerializable()
        obj.foo = 'bar'
        self.assertEqual({'foo': 'bar'}, obj._as_serializable())

    def test_extra_args_arent_serialized(self):
        obj = self.MySerializable(foo='bar', baz=42)
        self.assertEqual({'foo': 'bar'}, obj._as_serializable())

