/**
 * pmount.c - policy wrapper around 'mount' to allow mounting removable devices
 *            for normal users
 *
 * Author: Martin Pitt <martin@piware.de>
 * (c) 2004 Canonical Ltd.
 * 
 * This software is distributed under the terms and licenses of the 
 * GNU General Public License. See file GPL for the full text of the license.
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <sys/wait.h>
#include <limits.h>
#include <unistd.h>
#include <fcntl.h>

#include "policy.h"

#define UIDMOUNTFLAGS "sync,atime,nodev,exec,noauto,nosuid,user,rw,uid=%i,gid=%i"
#define UIDFSLIST "vfat,udf,iso9660,hfsplus,hfs"

#define NOUIDMOUNTFLAGS "sync,atime,nodev,exec,noauto,nosuid,user,rw"
#define NOUIDFSLIST "ext3,ext2,reiserfs,xfs,jfs"

#define MOUNTPROG "/bin/mount"

/**
 * Replace each occurrence of char from by char to in string s.
 */
void
strreplace( char* s, char from, char to )
{
    char* i;
    for( i = s; *i; ++i )
	if( *i == from )
	    *i = to;
}

/**
 * Print some help.
 * @param exename Name of the executable (argv[0]).
 */
void
usage( const char* exename )
{
    printf( "Usage: %s <device> [<label>]\n\n\
Mount <device> to a directory below " MEDIADIR " if policy requirements\n\
are met. If <label> is given, the mount point will be /media/<label>,\n\
otherwise it will be /media/<device>. If the mount point does not exist,\n\
it will be created.\n\n\
Policy:\n\
- <device> is a block device in /dev/\n\
- <device> is not handled by /etc/fstab; if it is, '" MOUNTPROG " <device>'\n\
  is called to handle this case transparently\n\
- <device> is not already mounted according to /etc/mtab and /proc/mounts\n\
- if the mount point already exists, there is no device already mounted at it\n\
  and the directory is empty\n\
- <device> is removable (currently USB and FireWire)\n\n\
The device will be mounted with the following flags:\n\
sync,atime,nodev,exec,noauto,nosuid,user,rw\n", exename );
}

/**
 * Check whether the user is allowed to mount the given device to the given
 * mount point. Creates the mount point if it does not exist yet. 
 * @return 0 on failure, 1 on success
 */
int
check_policy( const char* device, const char* mntpt ) 
{
    return device_valid( device ) &&
	!device_mounted( device, 0, NULL ) &&
	device_removable( device ) &&
	mntpt_valid( mntpt ) &&
	!mntpt_mounted( mntpt, 0 );
}

/**
 * Entry point.
 */
int
main( int argc, char** argv )
{
    char mntpt[MEDIA_STRING_SIZE];
    char device[PATH_MAX];
    char options[255];
    int status;
    int devnull;

    /* are we root? */
    if( geteuid() ) {
	fputs( "Error: this program needs to be installed suid root to work\n", stderr );
	return -1;
    }

    /* drop root privileges until we really need them (still available as saved uid) */
    seteuid( getuid() );

    /* help wanted? */
    if( argc == 2 && ( !strcmp( argv[1], "-h" ) || !strcmp( argv[1], "--help") ) ) {
	usage( argv[0] );
	return 0;
    }

    /* invalid number of args? */
    if( argc < 2 || argc > 3 ) {
	usage( argv[0] );
	return -1;
    }

    if( !realpath( argv[1], device ) ) {
	perror( "Error: could not determine real path of the device" );
	return -1;
    }

    /* does the device start with DEVDIR? */
    if( strncmp( device, DEVDIR, sizeof( DEVDIR )-1 ) ) { 
	fprintf( stderr, "Error: invalid device %s (must be in /dev/)\n", device ); 
	return -2;
    }

    /* is the device already handled by fstab? */
    if( fstab_has_device( "/etc/fstab", device, NULL, NULL ) ) {
	if( argc == 3 ) {
	    fprintf( stderr, "Error: device %s is already handled by /etc/fstab, supplying a label is invalid\n", device );
	    return -5;
	}

	DEBUGLOG( printf( "device %s handled by fstab, calling mount\n", device ) );

	/* drop all privileges and transparently call mount */
	if( setreuid( -1, 0 ) || setuid( getuid() ) ) {
	    perror( "Internal error: could not change back to normal user (e)uid" );
	    return -99;
	}

	execl( MOUNTPROG, MOUNTPROG, device, NULL );
	perror( "Error: could not execute " MOUNTPROG );
	return -6;
    }

    /* determine mount point name */
    if( argc == 3 ) {
	if( strlen( argv[2] ) > MAX_LABEL_SIZE ) {
	    fprintf( stderr, "Error: label too long\n" );
	    return -2;
	}
	if( strchr( argv[2], '/' ) ) {
	    fprintf( stderr, "Error: '/' must not occur in label name\n" );
	    return -3;
	}
        snprintf( mntpt, sizeof( mntpt ), "%s%s", MEDIADIR, argv[2] );
    } else {
	if( strlen( argv[1] ) > MAX_LABEL_SIZE ) {
	    fprintf( stderr, "Error: device name too long\n" );
	    return -4;
	}

	/* chop the DEVDIR prefix */
	argv[1] += sizeof( DEVDIR )-1;

	/* get rid of slashes */
	strreplace( argv[1], '/', '_' );

        snprintf( mntpt, sizeof( mntpt ), "%s%s", MEDIADIR, argv[1] );
    }

    DEBUGLOG( printf( "mount point to be used: %s\n", mntpt ) );

    if( !check_policy( device, mntpt ) )
	return -5;

    DEBUGLOG( puts( "policy check passed" ) );

    DEBUGLOG( printf( "Executing command: %s %s %s %s %s %s %s\n", MOUNTPROG,
		"-t", UIDFSLIST, "-o", UIDMOUNTFLAGS, device, mntpt ) );

    /* execute mount, first try the file systems which allow uid= and gid= */
    snprintf( options, sizeof( options ), UIDMOUNTFLAGS, getuid(), getgid() );

    if( !fork() ) {
	if( setreuid( -1, 0 ) || setreuid( 0, 0 ) ) {
	    perror( "Internal error: could not change back to root (e)uid" );
	    return -99;
	}

	/* try to reroute stderr to /dev/null to suppress error messages */
	devnull = open( "/dev/null", O_WRONLY );
	if( devnull > 0 )
	    dup2( devnull, 2 );

	execl( MOUNTPROG, MOUNTPROG, "-t", UIDFSLIST, "-o", options, device, mntpt, NULL );
	perror( "Error: could not execute " MOUNTPROG );
	return -6;
    } else {
	if( wait( &status ) < 0 ) {
	    perror( "Error: could not wait for executed mount process" );
	    return -7;
	}
    }

    DEBUGLOG( printf( "mount attempt for " UIDFSLIST " terminated with status %i\n", status ) );

    DEBUGLOG( printf( "Executing command: %s %s %s %s %s %s %s\n", MOUNTPROG,
		"-t", NOUIDFSLIST, "-o", NOUIDMOUNTFLAGS, device, mntpt ) );

    if( !WIFEXITED( status ) || WEXITSTATUS( status ) != 0 ) {
	/* try the file systems which handle users on their own and don't allow
	   uid/gid specification */
	if( !fork() ) {
	    if( setreuid( -1, 0 ) || setreuid( 0, 0 ) ) {
		perror( "Internal error: could not change back to root (e)uid" );
		return -99;
	    }

	    execl( MOUNTPROG, MOUNTPROG, "-t", NOUIDFSLIST, "-o", NOUIDMOUNTFLAGS, device, mntpt, NULL );
	    perror( "Error: could not execute " MOUNTPROG );
	    return -6;
	} else {
	    if( wait( &status ) < 0 ) {
		perror( "Error: could not wait for executed mount process" );
		return -7;
	    }
	}
    }

    DEBUGLOG( printf( "mount attempt for " NOUIDFSLIST " terminated with status %i\n", status ) );

    if( !WIFEXITED( status ) || WEXITSTATUS( status ) != 0 ) {
	fprintf( stderr, "Error: mount failed\n" );
	setreuid( -1, 0 );
	if( rmdir( mntpt ) ) {
	    perror( "Error: could not delete mount point" );
	    return -9;
	}
	return -8;
    }

    return 0; 
}
