/**
 * pumount.c - policy wrapper around 'umount' to allow mounting removable
 *             devices for normal users
 *
 * Author: Martin Pitt <martin@piware.de>
 * (c) 2004 Canonical Ltd.
 *
 * This software is distributed under the terms and licenses of the 
 * GNU General Public License. See file GPL for the full text of the license.
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <sys/wait.h>
#include <limits.h>

#include "policy.h"

#define UMOUNTPROG "/bin/umount"

static char mntpt[MEDIA_STRING_SIZE];

/**
 * Print some help.
 * @param exename Name of the executable (argv[0]).
 */
void
usage( const char* exename )
{
    printf( "Usage: %s [-l] <device>\n\n\
Umount <device> from a directory below " MEDIADIR " if policy requirements\n\
are met. The mount point directory is removed afterwards. If -l is supplied,\n\
a lazy unmount is done, see umount(8).\n\n\
Policy:\n\
- <device> is a block device in /dev/ (only if -l is not supplied)\n\
- <device> is not handled by /etc/fstab; if it is, '" UMOUNTPROG " <device>'\n\
  is called to handle this case transparently\n\
- <device> is mounted according to /etc/mtab and /proc/mounts\n\
  with the calling user's uid\n\
- mount point is in /media\n", exename );
}

/**
 * Check whether the user is allowed to umount the given device.
 * @param do_lazy Flag if a lazy unmount is requested (in this case the device
 *        node does not need to exist)
 * @return 0 on failure, 1 on success
 */
int
check_policy( const char* device, int do_lazy ) 
{
    int devvalid;

    devvalid = ( do_lazy || device_valid( device ) ) &&
	device_mounted( device, 1, mntpt );

    if( !devvalid )
	return 0;

    /* paranoid check */
    if( !mntpt || !*mntpt ) {
	fputs( "Internal error: could not determine mount point\n", stderr );
	exit( -99 );
    }

    /* mount point must be below /media */
    if( strncmp( mntpt, MEDIADIR, sizeof( MEDIADIR )-1 ) ) {
	fprintf( stderr, "Error: mount point %s is not below " MEDIADIR "\n", mntpt );
	return 0;
    }

    return 1;
}

/**
 * Entry point.
 */
int
main( int argc, char** argv )
{
    char device[PATH_MAX];
    int dev_arg_idx = 1;
    int status;
    int do_lazy = 0;

    /* are we root? */
    if( geteuid() ) {
	fputs( "Error: this program needs to be installed suid root to work\n", stderr );
	return -1;
    }

    /* drop root privileges until we really need them (still available as saved uid) */
    seteuid( getuid() );

    /* no arguments? */
    if( argc <= 1 ) {
	usage( argv[0] );
	return 0;
    }

    /* help wanted? */
    if( !strcmp( argv[1], "-h" ) || !strcmp( argv[1], "--help") ) {
	usage( argv[0] );
	return 0;
    }

    /* lazy unmount? */
    if( !strcmp( argv[1], "-l" ) ) {
	do_lazy = 1;
	dev_arg_idx = 2;
    }

    /* invalid number of args? */
    if( argc != 2 + do_lazy ) {
	usage( argv[0] );
	return -1;
    }

    if( do_lazy ) {
	/* we cannot really check the real path here since the device node might not
	 * exist any more */
	if( !realpath( argv[dev_arg_idx], device ) )
	    snprintf( device, sizeof( device ), "%s", argv[dev_arg_idx] );
    } else {
	if( !realpath( argv[dev_arg_idx], device ) ) {
	    perror( "Error: could not determine real path of the device" );
	    return -1;
	}
    }

    /* does the device start with DEVDIR? */
    if( strncmp( device, DEVDIR, sizeof( DEVDIR )-1 ) ) {
	fprintf( stderr, "Error: invalid device %s (must be in /dev/)\n", device );
	return -2;
    }

    /* is the device already handled by fstab? */
    if( fstab_has_device( "/etc/fstab", device, NULL, NULL ) ) {
	/* drop all privileges and transparently call umount */
	if( setreuid( -1, 0 ) || setuid( getuid() ) ) {
	    perror( "Internal error: could not change back to normal user (e)uid" );
	    return -99;
	}

	DEBUGLOG( printf( "device %s handled by fstab, calling umount\n", device ) );

	if( do_lazy )
	    execl( UMOUNTPROG, UMOUNTPROG, "-l", device, NULL );
	else
	    execl( UMOUNTPROG, UMOUNTPROG, device, NULL );
	perror( "Error: could not execute " UMOUNTPROG );
	return -6;
    }

    if( !check_policy( device, do_lazy ) )
	return -5;

    DEBUGLOG( puts( "policy check passed" ) );

    DEBUGLOG( printf( "Executing command: %s %s\n", UMOUNTPROG, device ) );

    /* execute umount */
    if( !fork() ) {
	if( setreuid( -1, 0 ) || setreuid( 0, 0 ) ) {
	    perror( "Internal error: could not change back to root (e)uid" );
	    return -99;
	}

	if( do_lazy )
	    execl( UMOUNTPROG, UMOUNTPROG, "-l", device, NULL );
	else
	    execl( UMOUNTPROG, UMOUNTPROG, device, NULL );
	perror( "Error: could not execute " UMOUNTPROG );
	return -6;
    } else {
	if( wait( &status ) < 0 ) {
	    perror( "Error: could not wait for executed umount process" );
	    return -7;
	}
    }

    DEBUGLOG( printf( "umount program terminated with status %i\n", status ) );

    if( !WIFEXITED( status ) || WEXITSTATUS( status ) != 0 ) {
	fprintf( stderr, "Error: umount failed\n" );
	return -8;
    }

    setreuid( -1, 0 );
    rmdir( mntpt );

    return 0; 
}
